use std::net::IpAddr;
use tokio::sync::mpsc;
use {
    crate::{channels, decenc, error::Result, message},
    futures::future,
    log::info,
    tokio::{io, net, time},
};

pub struct Consumer {
    consumer_address: (IpAddr, u16),
}

impl Consumer {
    pub fn new(consumer_address: (IpAddr, u16)) -> Consumer {
        Consumer { consumer_address }
    }

    pub async fn run(
        &self,
        mut consumer_recv: channels::ChanPair<message::Message>,
        consumer_conn: mpsc::Sender<mpsc::Sender<message::Message>>,
    ) -> Result<()> {
        let (send, mut recv) = consumer_recv.split();

        loop {
            info!(
                "Connecting to {}:{}",
                self.consumer_address.0, self.consumer_address.1
            );
            let socket = net::TcpStream::connect(self.consumer_address).await;

            let (r, w) = io::split(match socket {
                Err(_) => {
                    time::sleep(time::Duration::from_secs(5)).await;
                    continue;
                }
                Ok(s) => {
                    info!("Connected!");
                    s
                }
            });

            let read_future = decenc::run_read(r, &send);
            let write_future = decenc::run_write(w, &mut recv);

            consumer_conn.send(send.clone()).await?;

            future::try_join(read_future, write_future).await?;
        }
    }
}
