//! Helper struct for managing the various communication channels between sockets and the
//! application

use tokio::sync::mpsc::{channel, Receiver, Sender};

/// A bi-directional channel pair allowing two async tasks to exchange data. The struct contains
/// one side of the bi-directional pair.
pub struct ChanPair<T> {
    send: Sender<T>,
    recv: Receiver<T>,
}

impl<T> ChanPair<T> {
    /// Constructs 2 new [ChanPair]s that are connected to each other. When both [ChanPair]s are
    /// given to different tasks, the tasks can communicate to each other.
    pub fn new(buffer: usize) -> (ChanPair<T>, ChanPair<T>) {
        let (send_a, recv_a) = channel(buffer);
        let (send_b, recv_b) = channel(buffer);

        (
            ChanPair::from(
                send_a,
                recv_b,
            ),
            ChanPair::from(
                send_b,
                recv_a,
            ),
        )
    }

    /// Construct a [ChanPair] from a [Sender] and [Receiver].
    pub fn from(send: Sender<T>, recv: Receiver<T>) -> ChanPair<T> {
        ChanPair { send, recv }
    }

    /// Split this [ChanPair] back into its [Sender] and [Receiver].
    pub fn split(&mut self) -> (&Sender<T>, &mut Receiver<T>) {
        (&self.send, &mut self.recv)
    }

    /// Retrieve the [Sender] part of the [ChanPair]
    pub fn send(&self) -> &Sender<T> {
        &self.send
    }

    /// Retrieve the [Receiver] part of the [ChanPair]
    pub fn recv(&mut self) -> &mut Receiver<T> {
        &mut self.recv
    }
}
