use {
    futures::future,
    log::info,
    std::{collections::VecDeque, iter::Iterator, ops::Index, sync::Arc},
    thfmr_protocol::message::{
        Message::{Next, Want},
        Song,
    },
    thfmr_protocol::{
        channels::ChanPair, consumer::Consumer, message::Message, producer::Producer,
    },
    thfmr_util::DropResult,
    tokio::{
        select,
        sync::{mpsc, RwLock},
    },
};

use std::net::IpAddr;

pub use thfmr_protocol::error::Result;

pub struct Duration(u64);

impl Duration {
    fn new(samples: u64) -> Duration {
        Duration(samples)
    }

    fn to_micros(&self) -> u64 {
        self.0 * 1000 / 48
    }
}

impl From<Duration> for std::time::Duration {
    fn from(duration: Duration) -> Self {
        Self::from_micros(duration.to_micros())
    }
}

pub struct Playlist {
    consumer: Consumer,
    producer: Producer,
}

pub struct PlaylistList(pub(crate) VecDeque<Song>);

impl PlaylistList {
    pub fn new() -> Arc<RwLock<Self>> {
        Arc::new(RwLock::new(Self(VecDeque::new())))
    }

    pub fn add_song(&mut self, song: Song) -> &mut Self {
        self.0.push_back(song);
        self
    }

    pub fn take_song(&mut self) -> Option<Song> {
        self.0.pop_front()
    }

    pub fn len(&self) -> usize {
        self.0.len()
    }

    pub fn duration(&self) -> Duration {
        let samples = self.0.iter().map(|song| song.stop - song.start).sum();
        Duration::new(samples)
    }
}

impl Index<usize> for PlaylistList {
    type Output = Song;

    fn index(&self, index: usize) -> &Self::Output {
        self.0.index(index)
    }
}

struct PlaylistProcess<'a> {
    consumer_send: &'a mpsc::Sender<Message>,
    consumer_recv: &'a mut mpsc::Receiver<Message>,
    consumer_conn: &'a mut mpsc::Receiver<mpsc::Sender<Message>>,
    producer_send: &'a mpsc::Sender<Message>,
    producer_recv: &'a mut mpsc::Receiver<Message>,
    producer_conn: &'a mut mpsc::Receiver<mpsc::Sender<Message>>,
}

impl Playlist {
    pub fn new(player_address: (IpAddr, u16), provider_address: (IpAddr, u16)) -> Playlist {
        Playlist {
            consumer: Consumer::new(player_address),
            producer: Producer::new(provider_address),
        }
    }

    pub async fn run(&self, playlist: Arc<RwLock<PlaylistList>>) -> Result<()> {
        let (consumer_recv, consumer_send) = ChanPair::new(1);
        let (consumer_conn_tx, consumer_conn_rx) = mpsc::channel::<mpsc::Sender<Message>>(1);
        let (producer_send, producer_recv) = ChanPair::new(1);
        let (producer_conn_tx, producer_conn_rx) = mpsc::channel::<mpsc::Sender<Message>>(1);

        let consumer_future = self.consumer.run(consumer_recv, consumer_conn_tx);
        let producer_future = self.producer.run(producer_send, producer_conn_tx);
        let playlist_future = Playlist::run_playlist(
            consumer_send,
            consumer_conn_rx,
            producer_recv,
            producer_conn_rx,
            playlist,
        );

        future::try_join3(consumer_future, producer_future, playlist_future)
            .await
            .drop_result()
    }

    pub async fn run_playlist(
        mut consumer_send: ChanPair<Message>,
        mut consumer_conn: mpsc::Receiver<mpsc::Sender<Message>>,
        mut producer_recv: ChanPair<Message>,
        mut producer_conn: mpsc::Receiver<mpsc::Sender<Message>>,
        playlist: Arc<RwLock<PlaylistList>>,
    ) -> Result<()> {
        let (consumer_send, consumer_recv) = consumer_send.split();
        let (producer_send, producer_recv) = producer_recv.split();

        let process = PlaylistProcess::new(
            consumer_send,
            consumer_recv,
            &mut consumer_conn,
            producer_send,
            producer_recv,
            &mut producer_conn,
        );

        process.run(playlist).await
    }
}

struct PlaylistState {
    playlist: Arc<RwLock<PlaylistList>>,
    starved: bool,
}

impl PlaylistState {
    fn new(playlist: Arc<RwLock<PlaylistList>>) -> PlaylistState {
        PlaylistState {
            playlist,
            starved: true,
        }
    }
}

impl<'a> PlaylistProcess<'a> {
    pub fn new(
        consumer_send: &'a mpsc::Sender<Message>,
        consumer_recv: &'a mut mpsc::Receiver<Message>,
        consumer_conn: &'a mut mpsc::Receiver<mpsc::Sender<Message>>,
        producer_send: &'a mpsc::Sender<Message>,
        producer_recv: &'a mut mpsc::Receiver<Message>,
        producer_conn: &'a mut mpsc::Receiver<mpsc::Sender<Message>>,
    ) -> PlaylistProcess<'a> {
        PlaylistProcess {
            consumer_send,
            consumer_recv,
            consumer_conn,
            producer_send,
            producer_recv,
            producer_conn,
        }
    }

    pub async fn run(self, playlist: Arc<RwLock<PlaylistList>>) -> Result<()> {
        let mut state = PlaylistState::new(playlist);

        loop {
            let res = select! {
                Some(sender) = self.consumer_conn.recv() => {
                    self.handle_new_consumer(sender, &mut state).await
                }
                Some(m) = self.consumer_recv.recv() => {
                    self.handle_consumer(m, &mut state).await
                },
                Some(sender) = self.producer_conn.recv() => {
                    self.handle_new_producer(sender, &mut state).await
                },
                Some(m) = self.producer_recv.recv() => {
                    self.handle_producer(m, &mut state).await
                },
            };
            res?;
        }
    }

    pub async fn handle_new_consumer(
        &self,
        _sender: mpsc::Sender<Message>,
        _state: &mut PlaylistState,
    ) -> Result<()> {
        info!("New consumer received");
        Ok(())
    }

    pub async fn handle_consumer(&self, message: Message, state: &mut PlaylistState) -> Result<()> {
        info!("Received consumer message: {:?}", message);
        let message = match message {
            Message::Want => {
                let mut guard = state.playlist.write().await;
                let song = guard.take_song();

                // Player requests a new song, lets find the next song in the playlist
                match song {
                    Some(song) => {
                        // Yay, give it to the player
                        self.consumer_send.send(Next(song)).await?;
                        let duration: std::time::Duration = guard.duration().into();

                        // And if the playlist is less than an hour long
                        if duration < std::time::Duration::from_secs(60 * 60) {
                            // Request more songs
                            Want
                        } else {
                            return Ok(());
                        }
                    }
                    None => {
                        // Whoops, no songs anymore, ask for a new one
                        state.starved = true;
                        Want
                    }
                }
            }

            Message::Loaded(info) => {
                info!("Song loaded: {:?}", info);
                Message::Loaded(info)
            }

            Message::Playing(info) => {
                info!("Song playing: {:?}", info);
                Message::Playing(info)
            }
            Message::Stopped(info) => {
                info!("Song stopped: {:?}", info);
                Message::Stopped(info)
            }

            message => message,
        };
        self.producer_send.send(message).await?;
        Ok(())
    }

    pub async fn handle_new_producer(
        &self,
        sender: mpsc::Sender<Message>,
        state: &mut PlaylistState,
    ) -> Result<()> {
        info!("New producer received");
        if state.starved {
            sender.send(Message::Want).await?;
        }
        Ok(())
    }

    pub async fn handle_producer(&self, message: Message, state: &mut PlaylistState) -> Result<()> {
        info!("Received producer message: {:?}", message);
        let message = match message {
            // Catch next messages
            Message::Next(song) => {
                // If the player was starved
                if state.starved {
                    // Mark as unstarved
                    state.starved = false;
                    // Send another want out to the provider so we have something to send later
                    self.producer_send.send(Message::Want).await?;
                    // And give the currently received song to the player
                    Message::Next(song)
                } else {
                    // Otherwise, start enqueueing the song
                    let mut guard = state.playlist.write().await;
                    guard.add_song(song);

                    let duration: std::time::Duration = guard.duration().into();

                    // And if the playlist is less than an hour long
                    if duration < std::time::Duration::from_secs(60 * 60) {
                        // Request more songs
                        self.producer_send.send(Message::Want).await?;
                    }

                    drop(guard);

                    // And we're done, don't send this to the player
                    return Ok(());
                }
            }

            // Pass through any unknown messages directly
            message => message,
        };
        self.consumer_send.send(message).await?;
        Ok(())
    }
}
