use {
    async_graphql::{
        http::{playground_source, GraphQLPlaygroundConfig},
        EmptyMutation, EmptySubscription, Schema,
    },
    async_graphql_warp::{BadRequest, Response},
    std::{convert::Infallible, net::IpAddr, sync::Arc},
    structopt::StructOpt,
    tokio::{
        sync::RwLock,
        task::{JoinError, JoinHandle},
    },
    warp::{
        http::{Response as HttpResponse, StatusCode},
        Filter, Rejection,
    },
};

use log::info;

use {
    crate::{graphql::QueryRoot, playlist::Playlist},
    futures::future,
};
// use thfmr-protocol::consumer::Consumer;

use crate::playlist::PlaylistList;

mod graphql;
mod playlist;

// use tokio::io::{AsyncReadExt, AsyncWriteExt};
// use tokio::net::TcpListener;

#[derive(Debug, StructOpt)]
#[structopt(
    name = "thfmr-playlist",
    about = "Playlist microservice of the TouHou.FM Radio software"
)]
struct Opt {
    #[structopt(long, default_value = "127.0.0.1")]
    pub graphql_address: IpAddr,

    #[structopt(long, default_value = "9000")]
    pub graphql_port: u16,

    #[structopt(long, default_value = "127.0.0.1")]
    pub player_control_address: IpAddr,

    #[structopt(long, default_value = "2000")]
    pub player_control_port: u16,

    #[structopt(long, default_value = "127.0.0.1")]
    pub playlist_control_address: IpAddr,

    #[structopt(long, default_value = "2001")]
    pub playlist_control_port: u16,
}

pub mod built_info {
    include!(concat!(env!("OUT_DIR"), "/built.rs"));
}

fn spawn_playlist(playlist: Playlist, playlist_list: Arc<RwLock<PlaylistList>>) -> JoinHandle<()> {
    tokio::spawn(async move {
        playlist
            .run(playlist_list.clone())
            .await
            .expect("thfmr-playlist encountered an error")
    })
}

fn spawn_graphql<Address: Into<(IpAddr, u16)>>(
    playlist: Arc<RwLock<PlaylistList>>,
    address: Address,
) -> JoinHandle<()> {
    let addr = address.into();

    tokio::spawn(async move {
        let log = warp::log("warp_server");

        let schema = Schema::build(QueryRoot, EmptyMutation, EmptySubscription)
            .data(playlist)
            .finish();

        let graphql_post = async_graphql_warp::graphql(schema).and_then(
            |(schema, request): (
                Schema<QueryRoot, EmptyMutation, EmptySubscription>,
                async_graphql::Request,
            )| async move {
                Ok::<_, Infallible>(Response::from(schema.execute(request).await))
            },
        );

        let graphql_playground = warp::path::end().and(warp::get()).map(|| {
            HttpResponse::builder()
                .header("content-type", "text/html")
                .body(playground_source(GraphQLPlaygroundConfig::new("/")))
        });

        let routes = graphql_playground
            .or(graphql_post)
            .recover(|err: Rejection| async move {
                if let Some(BadRequest(err)) = err.find() {
                    return Ok::<_, Infallible>(warp::reply::with_status(
                        err.to_string(),
                        StatusCode::BAD_REQUEST,
                    ));
                }

                Ok(warp::reply::with_status(
                    "INTERNAL_SERVER_ERROR".to_string(),
                    StatusCode::INTERNAL_SERVER_ERROR,
                ))
            })
            .with(log);

        warp::serve(routes).run(addr).await;
    })
}

#[tokio::main]
async fn main() -> Result<(), JoinError> {
    let opt = Opt::from_args();

    println!("{:?}", opt);

    env_logger::init();

    info!("Starting Playlist v{}", built_info::PKG_VERSION);

    let playlist = Playlist::new(
        (opt.player_control_address, opt.player_control_port),
        (opt.playlist_control_address, opt.playlist_control_port),
    );
    let playlist_list = PlaylistList::new();

    let playlist_handle = spawn_playlist(playlist, playlist_list.clone());
    let server_handle = spawn_graphql(playlist_list, (opt.graphql_address, opt.graphql_port));

    let res = future::join(playlist_handle, server_handle).await;
    if let Err(_) = res.0 {
        res.0
    } else if let Err(_) = res.1 {
        res.1
    } else {
        Ok(())
    }
}
