use {
    crate::playlist::PlaylistList,
    async_graphql::{
        connection::{query, Connection, Edge, EmptyFields},
        Object, Result, SimpleObject,
    },
    log::info,
    std::sync::Arc,
    thfmr_protocol::message,
    thfmr_util::graphql::Cursor,
    tokio::sync::RwLock,
};

pub struct QueryRoot;

#[derive(SimpleObject)]
struct Song {
    pub file: String,
    pub start: message::Timestamp,
    pub stop: message::Timestamp,
}

impl From<message::Song> for Song {
    fn from(song: message::Song) -> Self {
        let message::Song { file, start, stop } = song;
        Song { file, start, stop }
    }
}

#[Object]
impl QueryRoot {
    async fn playlist_version(&self) -> &str {
        crate::built_info::PKG_VERSION
    }

    async fn playlist(
        &self,
        ctx: &async_graphql::Context<'_>,
        after: Option<String>,
        before: Option<String>,
        first: Option<i32>,
        last: Option<i32>,
    ) -> Result<Connection<Cursor<usize>, Song, EmptyFields, EmptyFields>> {
        let playlist = ctx.data::<Arc<RwLock<PlaylistList>>>()?.clone();
        let playlist = playlist.read().await;
        let len = playlist.len();
        info!("Playlist length: {}", len);
        query(
            after,
            before,
            first,
            last,
            |after: Option<Cursor<usize>>, before: Option<Cursor<usize>>, first, last| async move {
                let mut start = after
                    .map(|c| c.into_prefix("PlaylistItem"))
                    .unwrap_or(Ok(0))?;
                let mut end = before
                    .map(|c| c.into_prefix("PlaylistItem"))
                    .unwrap_or(Ok(playlist.len()))?;
                if let Some(first) = first {
                    end = (start + first).min(end);
                }
                if let Some(last) = last {
                    start = if last > end - start { end } else { end - last };
                }
                let mut connection = Connection::new(start > 0, end < playlist.len());
                connection.append((start..end).into_iter().map(|n| {
                    Edge::with_additional_fields(
                        Cursor::new("PlaylistItem", n),
                        playlist[n].clone().into(),
                        EmptyFields,
                    )
                }));
                Ok(connection)
            },
        )
        .await
    }
}
