use embedded_hal::blocking::i2c;
use ndarray::ShapeError;
use std::error::Error as StdError;
use std::fmt;

#[derive(Clone, Copy, Debug, PartialEq)]
pub struct UndefinedInput(pub u8);

impl fmt::Display for UndefinedInput {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        write!(f, "Undefined input: {}", self.0)
    }
}

impl StdError for UndefinedInput {
    fn source(&self) -> Option<&(dyn StdError + 'static)> {
        None
    }
}

pub enum Error<I2C>
where
    I2C: i2c::Transactional,
    //<I2C as i2c::Transactional>::Error: StdError,
{
    I2c(I2C::Error),
    Shape(ndarray::ShapeError),
    UndefinedInput(UndefinedInput),
}

impl<I2C> fmt::Debug for Error<I2C>
where
    I2C: i2c::Transactional,
    //<I2C as i2c::Transactional>::Error: StdError,
    <I2C as i2c::Transactional>::Error: fmt::Debug,
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::I2c(e) => write!(f, "I2c({:?})", e),
            Self::Shape(e) => write!(f, "Shape({:?})", e),
            Self::UndefinedInput(e) => write!(f, "UndefinedInput({:?})", e),
        }
    }
}

impl<I2C> fmt::Display for Error<I2C>
where
    I2C: i2c::Transactional,
    //<I2C as i2c::Transactional>::Error: StdError,
    <I2C as i2c::Transactional>::Error: fmt::Display,
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match self {
            Self::I2c(e) => write!(f, "I2c error: {}", e),
            Self::Shape(e) => write!(f, "Shape error: {}", e),
            Self::UndefinedInput(e) => write!(f, "Undefined input: {}", e),
        }
    }
}

impl<I2C> StdError for Error<I2C>
where
    I2C: i2c::Transactional,
    <I2C as i2c::Transactional>::Error: StdError + 'static,
{
    fn source(&self) -> Option<&(dyn StdError + 'static)> {
        match self {
            Self::I2c(e) => Some(e),
            Self::Shape(e) => Some(e),
            Self::UndefinedInput(e) => Some(e),
        }
    }
}

impl<I2C> From<ShapeError> for Error<I2C>
where
    I2C: i2c::Transactional,
{
    fn from(e: ShapeError) -> Self {
        Self::Shape(e)
    }
}

impl<I2C> From<UndefinedInput> for Error<I2C>
where
    I2C: i2c::Transactional,
{
    fn from(e: UndefinedInput) -> Self {
        Self::UndefinedInput(e)
    }
}
