use ndarray::Array2;
use std::error::Error as StdError;

/// A trait for thermal cameras
pub trait ThermalCamera {
    type Error: 'static + StdError;

    /// Get the temperature of the camera.
    ///
    /// Many cameras have a built-in temperature sensor separate from the camera itself. The
    /// default implementation of this function returns the mean of the pixels in the image.
    fn temperature(&mut self) -> Option<f32> {
        let image = self.image().ok()?;
        let pixel_count = image.len();
        Some(image.into_iter().sum::<f32>() / pixel_count as f32)
    }

    /// Return a two-dimensional array representing an image from the thermal camera.
    /// Currently the order of the pixels in the image is camera-dependent, but that's probably
    /// going to change at some point.
    fn image(&mut self) -> Result<Array2<f32>, Self::Error>;
}
