use cursive::{
    Cursive,
    With,
    view::{Nameable, Scrollable},
    views::{SelectView, EditView, Dialog, LinearLayout, TextView, ResizedView, OnEventView, ViewRef},
    theme::{ColorStyle, Color, BaseColor, Style},
    event::{Event, Key},
    utils::markup::StyledString
};
use regex::RegexBuilder;
use rust_utils::utils;
use crate::obi_search;

pub fn show_search(view: &mut Cursive) {
    let input_style = ColorStyle::new(
        Color::Dark(BaseColor::Blue),
        Color::Light(BaseColor::White),
    );
    let mut term_input: EditView = EditView::new().style(input_style).filler(" ").on_submit(|v, _|{ search(v, false); });
    let last_term = utils::get_shared_val("last_term");
    let mut results_disp: SelectView<String> = SelectView::default()
    .on_submit(|view, name: &String| {
        let term: ViewRef<EditView> = view.find_name("term").expect("Houston, we have a problem!");
        let verse = super::VerseRef::from(name, super::Format::Paragraph);
        
        let regex = RegexBuilder::new(&term.get_content())
            .case_insensitive(true)
            .build()
            .expect("Invalid Regex!");
        let book_num = super::get_book_num(verse.book);
        let chapter_num = verse.chapter;
        let new_chapter = super::chapter::Chapter::get_with_results(book_num,chapter_num, regex);

        super::set_chapter(view, new_chapter);
        view.pop_layer();
    });

    if &last_term != "ERR" && &last_term != "" {
        term_input.set_content(&last_term);
        gen_results(&last_term, &mut results_disp);
    }

    let layout = ResizedView::with_max_size((70, 100),
        LinearLayout::vertical()
            .child(TextView::new("Search Term:\n"))
            .child(term_input.with_name("term"))
            .child(results_disp.with_name("results").scrollable())
    );
    
    view.add_layer(
        Dialog::around(layout)
            .button("Quit", |s| {
                s.pop_layer();
            })
            .button("Search", |v| { search(v, false); })
            .button("Reset", |v| { reset(v, false); })
            .title("Text Search")
            .wrap_with(OnEventView::new)
            .on_event(Event::Key(Key::Esc), |s| {
                s.pop_layer();
            })
    );
}

pub fn show_topic_search(view: &mut Cursive) {
    let input_style = ColorStyle::new(
        Color::Dark(BaseColor::Blue),
        Color::Light(BaseColor::White),
    );
    let mut term_input: EditView = EditView::new().style(input_style).filler(" ").on_submit(|v, _|{ search(v, true); });
    let last_term = utils::get_shared_val("last_topic");
    let mut results_disp: SelectView<String> = SelectView::default()
    .on_submit(|view, name: &String| {
        let term: ViewRef<EditView> = view.find_name("term").expect("Houston, we have a problem!");
        let verse = super::VerseRef::from(name, super::Format::Paragraph);
        let book_num = super::get_book_num(&verse.book);
        let chapter_num = verse.chapter;
        let new_chapter = super::chapter::Chapter::get_with_topic(book_num, chapter_num, &term.get_content(), verse);

        super::set_chapter(view, new_chapter);
        view.pop_layer();
    });

    if &last_term != "ERR" && &last_term != "" {
        term_input.set_content(&last_term);
        gen_topic_results(&last_term, &mut results_disp);
    }

    let layout = ResizedView::with_max_size((70, 100),
        LinearLayout::vertical()
            .child(TextView::new("What the Bible says about:\n"))
            .child(term_input.with_name("term"))
            .child(results_disp.with_name("results").scrollable())
            .child(TextView::new("\nPowered by openbible.info"))
    );
    
    view.add_layer(
        Dialog::around(layout)
            .button("Quit", |s| {
                s.pop_layer();
            })
            .button("Search", |v|{ search(v, true); })
            .button("Reset", |v| { reset(v, true); })
            .title("Topic Search")
            .wrap_with(OnEventView::new)
            .on_event(Event::Key(Key::Esc), |s| {
                s.pop_layer();
            })
    );
}

fn gen_results(term: &str, results_view: &mut SelectView<String>) {
    utils::set_shared_val("last_term", term);
    let regex = RegexBuilder::new(&term)
        .case_insensitive(true)
        .build()
        .expect("Invalid Regex!");
    let results = super::search_word(&regex);

    for verse in results {
        let mut disp = StyledString::new();
        let verse_data = super::VerseRef::from(&verse, super::Format::Paragraph);
        let book = super::books::BOOKS[super::get_book_num(&verse_data.book)];
        let verse_text = super::chapter::get_verse(&verse_data.book, verse_data.chapter, verse_data.verse);
        disp.append(StyledString::styled(format!("{} {}:{}: ", book, verse_data.chapter, verse_data.verse), Style::from(Color::Light(BaseColor::White))));
        for word in verse_text.split(" ") {
            if regex.is_match(&word) {
                disp.append(StyledString::styled(word, Style::from(Color::Light(BaseColor::Blue))));
            }
            else {
                disp.append(StyledString::styled(word, Style::from(Color::Dark(BaseColor::White))));
            }
            disp.append(" ");
        }
        results_view.add_item(disp, verse);
    }
}

fn gen_topic_results(term: &str, results_view: &mut SelectView<String>) {
    utils::set_shared_val("last_topic", term);
    let results = obi_search::get_topic(term);

    for verse in results {
        let mut disp = StyledString::new();
        let verse_data = super::VerseRef::from(&verse, super::Format::Paragraph);
        if verse_data.is_valid() {
            disp.append(StyledString::styled(verse_data.get_title(), Style::from(Color::Light(BaseColor::White))));
            disp.append(StyledString::styled(verse_data.get_text(), Style::from(Color::Dark(BaseColor::White))));
            results_view.add_item(disp, verse);
        }
    }
}

fn reset(view: &mut Cursive, is_topic_search: bool) {
    if is_topic_search{
        utils::set_shared_val("last_topic", "");    
    }
    else {
        utils::set_shared_val("last_term", "");
    }
    let mut results_view: ViewRef<SelectView<String>> = view.find_name("results").expect("Houston, we have a problem!");
    results_view.clear();
    let mut term: ViewRef<EditView> = view.find_name("term").expect("Houston, we have a problem!");
    term.set_content("");
}

fn search(view: &mut Cursive, istopic: bool) {
    let term: ViewRef<EditView> = view.find_name("term").expect("Houston, we have a problem!");
    let mut results_view: ViewRef<SelectView<String>> = view.find_name("results").expect("Houston, we have a problem!");
    results_view.clear();
    if istopic {
        gen_topic_results(&term.get_content(), &mut results_view);
    }
    else {
        gen_results(&term.get_content(), &mut results_view);
    }
}