use std::{fs, env};
use cursive::{
    Cursive,
    align::HAlign,
    event::{Event, Key},
    theme::{BaseColor, BorderStyle, Color, ColorStyle, PaletteColor, Theme},
    traits::*,
    view::{Nameable, Resizable, Scrollable},
    views::{Dialog, LinearLayout, OnEventView, ResizedView, SelectView, TextView, ViewRef}
};
use lazy_static::lazy_static;
use crate::{bible::*, config::*};

mod events;
mod search;

lazy_static! {
    static ref BOOK_CACHE: String = format!("{}/.cache/the_rock.book",env::var("HOME").unwrap());
    static ref CHAPTER_CACHE: String = format!("{}/.cache/the_rock.chapter",env::var("HOME").unwrap());
}

pub fn init(arg: String) {
    let status_style = ColorStyle::new(
        Color::Light(BaseColor::White),
        Color::TerminalDefault,
    );
    
    // go to Genesis 1 or previous open chapter (if the files exist in ~/.cache)
    let mut book_num: usize = fs::read_to_string(BOOK_CACHE.to_string()).unwrap_or("0".to_string()).parse().unwrap();
    let mut chapter_num: usize = fs::read_to_string(CHAPTER_CACHE.to_string()).unwrap_or("1".to_string()).parse().unwrap_or(1);

    // if there was a specific chapter at the command line, go to it
    if arg != "DEF"{
        let verse = VerseRef::from(arg, Format::Numbers);
        book_num = get_book_num(verse.book);
        chapter_num = verse.chapter;
        fs::write(BOOK_CACHE.to_string(), &book_num.to_string()).unwrap_or(());
        fs::write(CHAPTER_CACHE.to_string(), &chapter_num.to_string()).unwrap_or(());
    }

    let chapter = chapter::Chapter::get(book_num, chapter_num);
    let content = chapter.content.clone();
    let format = chapter.format;
    let mut root = cursive::default();
    let title = chapter.get_title();

    // the theme of the application
    let mut theme_def = Theme::default();
    theme_def.shadow = false;
    theme_def.palette[PaletteColor::Background] = Color::TerminalDefault;
    theme_def.palette[PaletteColor::Primary] = Color::Light(BaseColor::White);
    theme_def.palette[PaletteColor::View] = Color::TerminalDefault;
    theme_def.palette[PaletteColor::Highlight] = Color::Rgb(201,157,21);
    theme_def.palette[PaletteColor::HighlightText] = Color::Dark(BaseColor::Black);
    theme_def.palette[PaletteColor::Secondary] = Color::Dark(BaseColor::Blue);
    theme_def.palette[PaletteColor::TitlePrimary] = Color::Light(BaseColor::Blue);
    theme_def.borders = BorderStyle::Outset;

    root.set_theme(theme_def);

    // menu
    root.set_autohide_menu(false);
    root.add_global_callback(Key::Up, |s| s.select_menubar());
    root.menubar()
        .add_leaf("Books", goto_chapter)
        .add_leaf("Bookmarks", events::show_bookmarks)
        .add_leaf("Parables", events::show_parables)
        .add_leaf("Search", search::show_search)
        .add_leaf("Topics", search::show_topic_search)
        .add_leaf("Help", events::show_help);
    
    // bible text view
    root.screen_mut().add_fullscreen_layer(
        LinearLayout::vertical()
        .child(
            Dialog::around(ResizedView::with_full_screen(
                TextView::new(content)
                    .with_name("chapter")
                    .scrollable()
                    .wrap_with(OnEventView::new)
                    .on_pre_event_inner(Key::PageUp, events::scroll_up)
                    .on_pre_event_inner(Key::PageDown, events::scroll_down)
            ))
        
            .title(title)
            .title_position(HAlign::Center)
            .h_align(HAlign::Center)

            // go back to the previous chapter
            .button("Previous Chapter", prev_chapter)

            // advance to next chapter
            .button("Next Chapter", next_chapter)

            .with_name("dialog")
        )
        .child(ResizedView::with_full_width(
            TextView::new(format!("Verse view format: {}",format.desc()))
                .style(status_style)
                
                .with_name("status")
        ))
    );

    root.set_user_data(chapter);

    // exit the application
    root.add_global_callback('q', |s| s.quit());

    // toggle showing of parables
    root.add_global_callback('p', |s| {
        let mut settings = load_config();
        let show_parables = !settings.show_parables;
        settings.show_parables = show_parables;
        settings.update();
        reload(s);
    });

    // toggle showing of bookmarks
    root.add_global_callback('b', |s| {
        let mut settings = load_config();
        let show_bookmarks = !settings.show_bookmarks;
        settings.show_bookmarks = show_bookmarks;
        settings.update();
        reload(s);
    });

    // add a bookmark
    root.add_global_callback('a', events::add_bookmark);
    
    // toggle verse numbers
    root.add_global_callback('n', change_format);
    root.run();
}

pub fn prev_chapter(view: &mut Cursive) {
    let chapter = view.user_data::<chapter::Chapter>().unwrap().prev();
    set_chapter(view, chapter);
}

pub fn next_chapter(view: &mut Cursive) {
    let chapter = view.user_data::<chapter::Chapter>().unwrap().next(); 
    set_chapter(view, chapter);
}

// sets the text and title for the main chapter view
pub fn set_chapter(view: &mut Cursive, chapter: chapter::Chapter) {
    fs::write(BOOK_CACHE.to_string(), chapter.book_num.to_string()).unwrap_or(());
    fs::write(CHAPTER_CACHE.to_string(), chapter.chapter_num.to_string()).unwrap_or(());
    let content = chapter.content.clone();
   
    // set the content
    view.call_on_name("chapter", |view: &mut TextView|view.set_content(content));
    
    // set title
    view.call_on_name("dialog", |view: &mut Dialog|view.set_title(&chapter.get_title()));

    view.set_user_data(chapter);
}

fn goto_chapter(view: &mut Cursive){
    // book list
    let mut book_list: SelectView<String> = SelectView::new();

    for book in books::BOOKS.iter() {
        let book2 = book.to_string();
        book_list.add_item_str(book2);
    }
    view.add_layer(
        Dialog::around(
            book_list
            .on_submit(|s, name: &str| {
                let book_num = get_book_num(name);

                // chapter list
                let mut chapter_list: SelectView<String> = SelectView::new();
                for c in 1..books::CHAPTERS[book_num] + 1 {
                    chapter_list.add_item(c.to_string(), c.to_string());
                }
                s.add_layer(
                    Dialog::around(
                        chapter_list
                        .on_submit(move |s, chapter_string: &str| {
                            let chapter_num: usize = chapter_string.parse().expect("Invalid Chapter!");
                            let new_chapter = chapter::Chapter::get(book_num,chapter_num);

                            set_chapter(s, new_chapter);
                                            
                            s.pop_layer();
                            s.pop_layer();
                        })
                        .scrollable()
                        .fixed_size((20, 10))
                    )
                    .title("Select Chapter")
                    .dismiss_button("Back")
                    .wrap_with(OnEventView::new)
                    .on_event(Event::Key(Key::Esc), |s| {
                        s.pop_layer();
                    })
                );
            })
            .scrollable()
            .fixed_size((20, 10)),
        )
        .title("Select Book")
        .dismiss_button("Cancel")
        .with_name("dialog")
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })            
    );
}

fn change_format(view: &mut Cursive){
    // load the config
    let mut settings: Settings = load_config();
    let curr_chapter = view.user_data::<chapter::Chapter>().unwrap().clone();

    // set the new format
    let new_format = curr_chapter.format.next();
    settings.format = new_format;
    settings.update();

    let chapter = chapter::Chapter::get(curr_chapter.book_num, curr_chapter.chapter_num);
    let content = chapter.content.clone();

    // update the view with the new formatting
    let mut chapter_view: ViewRef<TextView> = view.find_name("chapter").expect("Houston, we have a problem!");
    let mut status: ViewRef<TextView> = view.find_name("status").expect("Houston, we have a problem!");

    status.set_content(format!("Verse view format: {}",settings.format.desc()));
    chapter_view.set_content(content);

    set_chapter(view, chapter);
}

fn reload(view: &mut Cursive) {
    let curr_chapter = view.user_data::<chapter::Chapter>().unwrap().clone();

    let book_num = curr_chapter.book_num;
    let chapter_num = curr_chapter.chapter_num;

    let chapter = chapter::Chapter::get(book_num, chapter_num);

    set_chapter(view, chapter)
}