use cursive::{
    Cursive,
    event::{Event, Key, EventResult},
    theme::{BaseColor, Color, ColorStyle, Style},
    traits::*,
    utils::markup::StyledString,
    view::{Nameable, Scrollable, scroll::Scroller},
    views::{Dialog, EditView, LinearLayout, OnEventView, SelectView, TextView, ScrollView, NamedView}
};
use lazy_static::lazy_static;
use crate::bible;
use rust_utils::utils;

lazy_static!{
    static ref HELP_TEXT: String = format!("The Rock: A King James Bible Viewer\nVersion {}\n\nKey Shortcuts:\nq: Quit\nn: set verse view format\na: Add verses in current chapter as named bookmark\nb: Show/Hide Bookmarks\np: Show/Hide Parables\n\n",env!("CARGO_PKG_VERSION"));
}

// scroll up
pub fn scroll_up(view: &mut ScrollView<NamedView<TextView>>, _event: &Event) -> Option<EventResult> {
    let scroller = view.get_scroller_mut();
    if scroller.can_scroll_up() {
        scroller.scroll_up(scroller.last_outer_size().y.saturating_sub(1));
    }
    Some(EventResult::Consumed(None))
}

// scroll down
pub fn scroll_down(view: &mut ScrollView<NamedView<TextView>>, _event: &Event) -> Option<EventResult> {
    let scroller = view.get_scroller_mut();
    if scroller.can_scroll_down() {
        scroller.scroll_down(scroller.last_outer_size().y.saturating_sub(1));
    }
    Some(EventResult::Consumed(None))
}

// show the help window
pub fn show_help(view: &mut Cursive){
    let parable_style = Style::from(Color::Light(BaseColor::Red));
    let bookmark_style = Style::from(Color::Light(BaseColor::Blue));
    let topic_style = Style::from(Color::from_256colors(165));

    let parable_title_style = ColorStyle::new(Color::Light(BaseColor::White),Color::Dark(BaseColor::Red));
    let bookmark_title_style = ColorStyle::new(Color::Light(BaseColor::White),Color::Dark(BaseColor::Blue));
    let topic_title_style = ColorStyle::new(Color::Light(BaseColor::White),Color::from_256colors(165));

    let mut annotation_legend = StyledString::plain("Annotation Legend:\n");
    annotation_legend.append(StyledString::styled("Parable Title\n",parable_title_style));
    annotation_legend.append(StyledString::styled("parable text\n\n",parable_style));
    annotation_legend.append(StyledString::styled("Bookmark Title\n",bookmark_title_style));
    annotation_legend.append(StyledString::styled("bookmark text\n\n",bookmark_style));
    annotation_legend.append(StyledString::styled("Topic Verse\n",topic_title_style));
    annotation_legend.append(StyledString::styled("topic verse\n",topic_style));

    let mut help_text = StyledString::plain(HELP_TEXT.to_string());
    help_text.append(annotation_legend);

    view.add_layer(
        Dialog::around(TextView::new(help_text))
        .dismiss_button("Back")
        .title("Help")
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })
    )
}

// show the parables window
pub fn show_parables(view: &mut Cursive) {
    let mut parables_list = SelectView::<&str>::new();
    for parable in bible::PARABLES.iter() {
        parables_list.add_item(parable.0, parable.1)
    }
    view.add_layer(
        Dialog::around(
            parables_list
            .on_submit(|view,name: &str|{
                let verse = bible::VerseRef::from(name, bible::Format::Paragraph);
                let book_num = bible::get_book_num(verse.book);
                let chapter_num = verse.chapter;
                let new_chapter = bible::chapter::Chapter::get(book_num,chapter_num);

                super::set_chapter(view, new_chapter);
                view.pop_layer();
            })
            .scrollable()
        )
        .title("Parables")
        .dismiss_button("Back")
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })
    )
}

// show the bookmarks window
pub fn show_bookmarks(view: &mut Cursive){
    let settings = super::load_config();
    let bookmarks = settings.bookmarks;
    let mut bookmark_list = SelectView::<String>::new();
    for bookmark in bookmarks{
        bookmark_list.add_item(bookmark.0,bookmark.1);
    }
    view.add_layer(
        Dialog::around(bookmark_list
            .on_select(|_,name|{
                utils::set_shared_val("bookmark_to_delete", name);
            })
            .on_submit(|view, name: &String |{
                let verse = bible::VerseRef::from(name, bible::Format::Paragraph);
                let book_num = bible::get_book_num(verse.book);
                let chapter_num = verse.chapter;
                let new_chapter = bible::chapter::Chapter::get(book_num,chapter_num);

                super::set_chapter(view, new_chapter);
                view.pop_layer();
            })
        )
        .dismiss_button("Back")
        .title("Bookmarks")
        .scrollable()
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })
        .on_event(Event::Key(Key::Del), |s| {
            s.add_layer(
                Dialog::around(TextView::new("Are you sure?")).dismiss_button("No")
                .button("Yes", |s|{
                    let mut settings = super::load_config();
                    let bookmarks = settings.bookmarks.clone();
                    let bookmark_to_delete = utils::get_shared_val("bookmark_to_delete");

                    for (i,bookmark) in bookmarks.iter().enumerate(){
                        if bookmark_to_delete == bookmark.1 {
                            settings.bookmarks.remove(i);
                        }
                    }
                    settings.update();
                    s.pop_layer();
                    s.pop_layer();
                    super::reload(s);
                })
                .wrap_with(OnEventView::new)
                .on_event(Event::Key(Key::Esc), |s| {
                    s.pop_layer();
                })
            )   
        })
        .with_name("bookmark_view")
    );
}

// add a new bookmark
pub fn add_bookmark(view: &mut Cursive){
    let input_style = ColorStyle::new(
        Color::Dark(BaseColor::Blue),
        Color::Light(BaseColor::White),
    );
    let curr_chapter = view.user_data::<bible::chapter::Chapter>().unwrap();
    let title_in = EditView::new().style(input_style).filler(" ");
    let verse_num_in = EditView::new().max_content_width(3).style(input_style).filler(" ");
    let book_name_in = EditView::new().max_content_width(20).content(bible::books::BOOKS[curr_chapter.book_num]).style(input_style).filler(" ");
    let chapter_num_in = EditView::new().max_content_width(2).content(curr_chapter.chapter_num.to_string()).style(input_style).filler(" ");
    let last_verse_num_in = EditView::new().max_content_width(3).content("0").style(input_style).filler(" ");

    view.add_layer(
        Dialog::around(
            LinearLayout::vertical()
            .child(TextView::new("Title:"))
            .child(title_in.with_name("title"))
            .child(TextView::new("Book:"))
            .child(book_name_in.with_name("book"))
            .child(TextView::new("Chapter:"))
            .child(chapter_num_in.with_name("chapter_num"))
            .child(TextView::new("Verse:"))
            .child(verse_num_in.with_name("verse_num"))
            .child(TextView::new("Ending verse:"))
            .child(last_verse_num_in.with_name("last_verse_num"))
            .child(TextView::new("^ Leave this at 0 for single verse"))
        )
        .button("Ok", move |view|{
            let mut settings = super::load_config();
            view.call_on_name("title",|view: &mut EditView| utils::set_shared_val("title", view.get_content().to_string()));
            view.call_on_name("book",|view: &mut EditView|{
                let book_num = bible::get_book_num(view.get_content().to_string());
                utils::set_shared_val("book", bible::books::BOOKS_SHORT2[book_num]);
            });
            view.call_on_name("chapter_num",|view: &mut EditView| utils::set_shared_val("chapter_num", view.get_content().to_string()));
            view.call_on_name("verse_num",|view: &mut EditView| utils::set_shared_val("verse_num", view.get_content().to_string()));
            view.call_on_name("last_verse_num",|view: &mut EditView| utils::set_shared_val("last_verse_num", view.get_content().to_string()));
            let title = utils::get_shared_val("title");
            let book = utils::get_shared_val("book");
            let chapter_num_str = utils::get_shared_val("chapter_num");
            let verse_num_str = utils::get_shared_val("verse_num");
            let last_verse_num_str = utils::get_shared_val("last_verse_num");

            let verse = if last_verse_num_str != "0" {
                format!("{}{}:{}-{}",book,chapter_num_str,verse_num_str,last_verse_num_str)
            }
            else {
                format!("{}{}:{}",book,chapter_num_str,verse_num_str)
            };
            settings.bookmarks.push((title, verse));
            settings.update();
            view.pop_layer();
            
            super::reload(view);
        })
        .title("Add Bookmark")
        .dismiss_button("Cancel")
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })
    )
}