use reqwest::blocking;
use regex::Regex;
use lazy_static::__Deref;
use std::{env, fs};

// openbible.info integration
// allows using the topical bible search from The Rock
pub fn get_topic(topic: &str) -> Vec<String> {
    // directory for cached topics
    let topic_cache = format!("{}/.cache/bible_topics", env::var("HOME").expect("Where is your home folder?"));

    let mut topic_f = String::new();
    for c in topic.chars() {
        if c == ' ' {
            topic_f.push('_');
        }
        else if c != '\"' {
            topic_f.push(c);
        }
    }

    // if there is a cached topic, return it instead
    match fs::read_to_string(format!("{}/{}", topic_cache, topic_f)) {
        Ok(cache) => {
            let verses: Vec<String> = cache.split("\n").map(|s| s.to_string()).collect();
            return verses;
        },
        Err(_) =>{}
    };

    // get the raw html from openbible.info
    let body = blocking::get(format!("https://www.openbible.info/topics/{}",topic_f)).expect("Unable to connect to openbible.info! Please check you internet connection!").text().unwrap();
    let lines: Vec<&str> = body.split("\n").collect();
    let mut verses: Vec<String> = Vec::new();
    let regex = Regex::new(r"<(\S*?)[^>]*>.*?|<.*? />").expect("Invalid regex!");
    let mut results = String::new();

    // parse the html to get the verse references
    for line in lines {
        if line.contains("bibleref") && !line.contains("style") & line.contains("a href") {
            let verse_t = regex.replace_all(line, "");
            let mut verse = verse_t.deref().to_string();
            verse.retain(|c| !c.is_whitespace());
            results.push_str(&verse);
            results.push('\n');
            verses.push(verse);
        }
    }
    results.truncate(results.len() - 1);
    fs::create_dir(&topic_cache).unwrap_or(());
    fs::write(format!("{}/{}", topic_cache, topic_f), results).expect("Cache error!");

    verses
}