use crate::bible::*;
use crate::obi_search;
use regex::RegexBuilder;
use colored::*;

pub fn parse_args(args: Vec<String>) {
    if args.len() == 0 {
        parse_refs("Ge1:1", false, false);
        return;
    }

    match args[0].as_str() {
        "-v" => println!("The Rock: A King James Bible Viewer\nVersion {}",env!("CARGO_PKG_VERSION")),
        "-h" => show_help(),
        "-c" => parse_refs(&args[1], false, true),
        "-p" => parse_refs(&args[1], true, false),
        "-pc" => parse_refs(&args[1], true, true),
        "-cp" => parse_refs(&args[1], true, true),
        "-s" => find_term(&args[1]),
        "-t" => topic_search(&args[1]),
        _=> parse_refs(&args[0], false, false)
    }
}

fn topic_search(topic: &str) {
    let results = obi_search::get_topic(topic);
    
    let title = format!("What the Bible says about {}:\n", topic).bright_white();
    println!("{}", title);

    for verse in results {
        let verse_data = VerseRef::from(verse, Format::Paragraph);
        if !verse_data.is_valid() {
            continue;
        }

        println!("{} {}", verse_data.get_title().bright_white(), verse_data.get_text().white())
    }

    println!("\nPowered by openbible.info");
}

fn find_term(term: &str) {
    let regex = RegexBuilder::new(term)
        .case_insensitive(true)
        .build()
        .expect("Invalid Regex!");

    let results = search_word(&regex);
    if results.len() == 0 {
        println!("The term \"{}\" did not return any results!", term);
        return;
    }

    for verse in results {
        let verse_data = VerseRef::from(verse, Format::Paragraph);
        let verse_text = verse_data.get_text();
        let mut formatted_verse = String::new();
        for word in verse_text.split(" ") {
            if regex.is_match(&word) {
                formatted_verse.push_str(&format!("{} ", word.blue().bold()));
            }
            else {
                formatted_verse.push_str(&format!("{} ", word.white()));
            }
        }

        println!("{} {}", verse_data.get_title().bright_white(), formatted_verse)
    }
}

// parse verse and passage references and output them
fn parse_refs(args: &str, isparagraph: bool, entire_chapter: bool) {
    let format = if isparagraph {
        Format::Paragraph
    }
    else {
        Format::Numbers
    };
    let verse_list = args.split(",");
    for verse_ref in verse_list {
        let verse = VerseRef::from(verse_ref, format);
        if entire_chapter {
            let (title, passage) = match format {
                Format::Paragraph => {
                    let title = format!("{} {}: ", verse.book, verse.chapter).bright_white();
                    let text = chapter::get_chapter(&verse.book, verse.chapter).join(" ");
                    (title, text)
                }

                Format::Numbers => {
                    let title = format!("{} {}\n\n", verse.book, verse.chapter).bright_white();
                    let mut text = String::new();
                    let chapter = chapter::get_chapter(&verse.book, verse.chapter);
                    for (i,verse_str) in chapter.iter().enumerate() {
                        text.push_str(&(i+1).to_string());
                        text.push(' ');
                        text.push_str(&verse_str);
                        text.push('\n');
                    }
                    (title, text)
                },
                _ => panic!("Invalid format!")
            };
            println!("{}{}", title, passage);
            continue;
        }
        print!("{}", verse.get_title().bright_white());
        println!("{}", verse.get_text());
    }   
}

// show the help for the program
fn show_help() {
    println!("{}\nbible <verse>\nbible <passage>\nbible <verse or passage>[,verse or passage,verse or passage...]\n","Usage:".bright_white());
    println!("{}\nbible Mark9:23\nbible matt24:4-14\nbible isa40:31,matt6:9-13,Genesis1\n","Examples:".bright_white());
    println!("Use -v to show current version.\nUse -h to show this help message again.\nUse -p <passage> to show the passage as a paragraph instead of verse list\nUse -s <word> to find all instances of a specific word.");
    println!("Use -t <topic> to search for a topic on openbible.info");
    println!("\n{} <book name><chapter>:<verse>[-<ending verse if passage>]","Passage and verse syntax:".bright_white());
}