use std::{fs, env};
use rust_utils::utils;
use cursive::{
    align::HAlign,
    Cursive,
    event::{Event, EventResult, Key},
    theme::{BaseColor, BorderStyle, Color, ColorStyle, PaletteColor, Style, Theme},
    traits::*,
    utils::markup::StyledString,
    view::{Nameable, Resizable, scroll::Scroller, Scrollable},
    views::{Dialog, EditView, LinearLayout, OnEventView, ResizedView, SelectView, TextView}
};
use lazy_static::lazy_static;
use crate::{bible::*,config::*};

pub fn show_ui(arg: String) {
    let status_style = ColorStyle::new(
        Color::Light(BaseColor::White),
        Color::TerminalDefault,
    );
    
    // go to Genesis 1 or previous open chapter (if the files exist in ~/.cache)
    let mut book_num: usize = fs::read_to_string(BOOK_CACHE.to_string()).unwrap_or("0".to_string()).parse().unwrap();
    let mut chapter_num: usize = fs::read_to_string(CHAPTER_CACHE.to_string()).unwrap_or("1".to_string()).parse().unwrap_or(1);

    // if there was a specific chapter at the command line, go to it
    if arg != "DEF"{
        let verse = parse_verse_ref(arg);
        book_num = get_book_num(verse.0);
        chapter_num = verse.1;
        fs::write(BOOK_CACHE.to_string(), &book_num.to_string()).unwrap_or(());
        fs::write(CHAPTER_CACHE.to_string(), &chapter_num.to_string()).unwrap_or(());
    }

    let chapter = chapter::Chapter::new(book_num, chapter_num);
    let content = chapter.content.clone();
    let format = chapter.format;
    let mut root = cursive::default();
    let title = chapter.get_title();

    // the theme of the application
    let mut theme_def = Theme::default();
    theme_def.shadow = false;
    theme_def.palette[PaletteColor::Background] = Color::TerminalDefault;
    theme_def.palette[PaletteColor::Primary] = Color::Light(BaseColor::White);
    theme_def.palette[PaletteColor::View] = Color::TerminalDefault;
    theme_def.palette[PaletteColor::Highlight] = Color::Rgb(201,157,21);
    theme_def.palette[PaletteColor::HighlightText] = Color::Dark(BaseColor::Black);
    theme_def.palette[PaletteColor::Secondary] = Color::Dark(BaseColor::Blue);
    theme_def.palette[PaletteColor::TitlePrimary] = Color::Light(BaseColor::Blue);
    theme_def.borders = BorderStyle::Outset;

    root.set_theme(theme_def);

    // bible text view
    root.screen_mut().add_fullscreen_layer(
        LinearLayout::vertical()
        .child(
            Dialog::around(ResizedView::with_full_screen(
                TextView::new(content)
                    .with_name("chapter")
                    .scrollable()
                    .wrap_with(OnEventView::new)
                    .on_pre_event_inner(Key::PageUp, |v, _| {
                        let scroller = v.get_scroller_mut();
                        if scroller.can_scroll_up() {
                            scroller.scroll_up(scroller.last_outer_size().y.saturating_sub(1));
                        }
                        Some(EventResult::Consumed(None))
                    })
                    .on_pre_event_inner(Key::PageDown, |v, _| {
                        let scroller = v.get_scroller_mut();
                        if scroller.can_scroll_down() {
                            scroller.scroll_down(scroller.last_outer_size().y.saturating_sub(1));
                        }
                        Some(EventResult::Consumed(None))
                    }
                )
            ))
        
            .title(title)
            .title_position(HAlign::Center)
            .h_align(HAlign::Center)

            // go back to the previous chapter
            .button("Previous Chapter", prev_chapter)

            // advance to next chapter
            .button("Next Chapter", next_chapter)

            // go to another chapter
            .button("Go to chapter", goto_chapter)

            // bookmarks
            .button("Bookmarks", bookmarks)

            // parables of Jesus Christ
            .button("Parables", parables)

            // help
            .button("Help", show_help)

            .with_name("dialog")
        )
        .child(ResizedView::with_full_width(
            TextView::new(format!("Verse view format: {}",format.desc()))
                .style(status_style)
                
                .with_name("status")
        ))
    );

    root.set_user_data(chapter);

    // exit the application
    root.add_global_callback('q', |s| s.quit());

    // toggle showing of parables
    root.add_global_callback('p', |s| {
        let mut settings = load_config();
        let show_parables = !settings.show_parables;
        settings.show_parables = show_parables;
        settings.update();
        reload(s);
    });

    // toggle showing of bookmarks
    root.add_global_callback('b', |s| {
        let mut settings = load_config();
        let show_bookmarks = !settings.show_bookmarks;
        settings.show_bookmarks = show_bookmarks;
        settings.update();
        reload(s);
    });

    // add a bookmark
    root.add_global_callback('a', add_bookmark);
    
    // toggle verse numbers
    root.add_global_callback('n', change_format);
    root.run();
}

fn prev_chapter(view: &mut Cursive) {
    let chapter = view.user_data::<chapter::Chapter>().unwrap().prev();
    set_chapter(view, chapter);
}

fn next_chapter(view: &mut Cursive) {
    let chapter = view.user_data::<chapter::Chapter>().unwrap().next();
    set_chapter(view, chapter);
}

// sets the text and title for the main chapter view
fn set_chapter(view: &mut Cursive, chapter: chapter::Chapter) {
    fs::write(BOOK_CACHE.to_string(), chapter.book_num.to_string()).unwrap_or(());
    fs::write(CHAPTER_CACHE.to_string(), chapter.chapter_num.to_string()).unwrap_or(());
    let content = chapter.content.clone();
   
    // set the content
    view.call_on_name("chapter", |view: &mut TextView|view.set_content(content));
    
    // set title
    view.call_on_name("dialog", |view: &mut Dialog|view.set_title(&chapter.get_title()));

    view.set_user_data(chapter);
}

fn goto_chapter(view: &mut Cursive){
    // book list
    let mut book_list = SelectView::<String>::new();

    for book in books::BOOKS.iter() {
        let book2 = book.to_string();
        book_list.add_item_str(book2);
    }
    view.add_layer(
        Dialog::around(
            book_list
            .on_submit(|s, name: &str| {
                let book_num = get_book_num(name);

                // chapter list
                let mut chapter_list = SelectView::<String>::new();
                for c in 1..books::CHAPTERS[book_num] + 1 {
                    chapter_list.add_item(c.to_string(), c.to_string());
                }
                s.add_layer(
                    Dialog::around(
                        chapter_list
                        .on_submit(move |s, chapter_string: &str| {
                            let chapter_num: usize = chapter_string.parse().expect("Invalid Chapter!");
                            let new_chapter = chapter::Chapter::new(book_num,chapter_num);

                            set_chapter(s, new_chapter);
                                            
                            s.pop_layer();
                            s.pop_layer();
                        })
                        .scrollable()
                        .fixed_size((20, 10))
                    )
                    .title("Select Chapter")
                    .dismiss_button("Back")
                    .wrap_with(OnEventView::new)
                    .on_event(Event::Key(Key::Esc), |s| {
                        s.pop_layer();
                    })
                );
            })
            .scrollable()
            .fixed_size((20, 10)),
        )
        .title("Select Book")
        .dismiss_button("Cancel")
        .with_name("dialog")
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })            
    );
}

fn change_format(view: &mut Cursive){
    // load the config
    let mut settings: Settings = load_config();
    let curr_chapter = view.user_data::<chapter::Chapter>().unwrap().clone();

    // set the new format
    let new_format = curr_chapter.format.next();
    settings.format = new_format;
    settings.update();

    let chapter = chapter::Chapter::new(curr_chapter.book_num, curr_chapter.chapter_num);
    let content = chapter.content.clone();

    // update the view with the new formatting
    view.call_on_name("chapter", |view: &mut TextView|view.set_content(content));

    view.call_on_name("status", |view: &mut TextView|{
        view.set_content(format!("Verse view format: {}",settings.format.desc()))
    });

    set_chapter(view, chapter);
}

fn add_bookmark(view: &mut Cursive){
    let input_style = ColorStyle::new(
        Color::Dark(BaseColor::Blue),
        Color::Light(BaseColor::White),
    );
    let curr_chapter = view.user_data::<chapter::Chapter>().unwrap();
    let title_in = EditView::new().style(input_style).filler(" ");
    let verse_num_in = EditView::new().max_content_width(3).style(input_style).filler(" ");
    let book_name_in = EditView::new().max_content_width(20).content(books::BOOKS[curr_chapter.book_num]).style(input_style).filler(" ");
    let chapter_num_in = EditView::new().max_content_width(2).content(curr_chapter.chapter_num.to_string()).style(input_style).filler(" ");
    let last_verse_num_in = EditView::new().max_content_width(3).content("0").style(input_style).filler(" ");

    view.add_layer(
        Dialog::around(
            LinearLayout::vertical()
            .child(TextView::new("Title:"))
            .child(title_in.with_name("title"))
            .child(TextView::new("Book:"))
            .child(book_name_in.with_name("book"))
            .child(TextView::new("Chapter:"))
            .child(chapter_num_in.with_name("chapter_num"))
            .child(TextView::new("Verse:"))
            .child(verse_num_in.with_name("verse_num"))
            .child(TextView::new("Ending verse:"))
            .child(last_verse_num_in.with_name("last_verse_num"))
            .child(TextView::new("^ Leave this at 0 for single verse"))
        )
        .button("Ok", move |view|{
            let mut settings = load_config();
            view.call_on_name("title",|view: &mut EditView|{
                utils::set_shared_val("title", view.get_content().to_string());
            });
            view.call_on_name("book",|view: &mut EditView|{
                let book_num = get_book_num(view.get_content().to_string());
                utils::set_shared_val("book", books::BOOKS_SHORT2[book_num]);
            });
            view.call_on_name("chapter_num",|view: &mut EditView|{
                utils::set_shared_val("chapter_num", view.get_content().to_string());
            });
            view.call_on_name("verse_num",|view: &mut EditView|{
                utils::set_shared_val("verse_num", view.get_content().to_string());
            });
            view.call_on_name("last_verse_num",|view: &mut EditView|{
                utils::set_shared_val("last_verse_num", view.get_content().to_string());
            });
            let title = utils::get_shared_val("title");
            let book = utils::get_shared_val("book");
            let chapter_num_str = utils::get_shared_val("chapter_num");
            let verse_num_str = utils::get_shared_val("verse_num");
            let last_verse_num_str = utils::get_shared_val("last_verse_num");

            let verse = if last_verse_num_str != "0" {
                format!("{}{}:{}-{}",book,chapter_num_str,verse_num_str,last_verse_num_str)
            }
            else {
                format!("{}{}:{}",book,chapter_num_str,verse_num_str)
            };
            settings.bookmarks.push((title, verse));
            settings.update();
            view.pop_layer();
            
            reload(view);
        })
        .title("Add Bookmark")
        .dismiss_button("Cancel")
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })
    )
}

fn bookmarks(view: &mut Cursive){
    let settings = load_config();
    let bookmarks = settings.bookmarks;
    let mut bookmark_list = SelectView::<String>::new();
    for bookmark in bookmarks{
        bookmark_list.add_item(bookmark.0,bookmark.1);
    }
    view.add_layer(
        Dialog::around(bookmark_list
            .on_select(|_,name|{
                utils::set_shared_val("bookmark_to_delete", name);
            })
            .on_submit(|view, name: &String |{
                let verse = parse_verse_ref(name);
                let book_num = get_book_num(verse.0);
                let chapter_num = verse.1;
                let new_chapter = chapter::Chapter::new(book_num,chapter_num);

                set_chapter(view, new_chapter);
                view.pop_layer();
            })
        )
        .dismiss_button("Back")
        .title("Bookmarks")
        .scrollable()
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })
        .on_event(Event::Key(Key::Del), |s| {
            s.add_layer(
                Dialog::around(TextView::new("Are you sure?")).dismiss_button("No")
                .button("Yes", |s|{
                    let mut settings = load_config();
                    let bookmarks = settings.bookmarks.clone();
                    let bookmark_to_delete = utils::get_shared_val("bookmark_to_delete");

                    for (i,bookmark) in bookmarks.iter().enumerate(){
                        if bookmark_to_delete == bookmark.1 {
                            settings.bookmarks.remove(i);
                        }
                    }
                    settings.update();
                    s.pop_layer();
                    s.pop_layer();
                    reload(s);
                })
                .wrap_with(OnEventView::new)
                .on_event(Event::Key(Key::Esc), |s| {
                    s.pop_layer();
                })
            )   
        })
        .with_name("bookmark_view")
    );
}

fn parables(view: &mut Cursive) {
    let mut parables_list = SelectView::<&str>::new();
    for parable in PARABLES.iter() {
        parables_list.add_item(parable.0, parable.1)
    }
    view.add_layer(
        Dialog::around(
            parables_list
            .on_submit(|view,name: &str|{
                let verse = parse_verse_ref(name);
                let book_num = get_book_num(verse.0);
                let chapter_num = verse.1;
                let new_chapter = chapter::Chapter::new(book_num,chapter_num);

                set_chapter(view, new_chapter);
                view.pop_layer();
            })
            .scrollable()
        )
        .title("Parables")
        .dismiss_button("Back")
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })
    )
}

fn show_help(view: &mut Cursive){
    let parable_style = Style::from(Color::Light(BaseColor::Red));
    let bookmark_style = Style::from(Color::Light(BaseColor::Blue));

    let parable_title_style = ColorStyle::new(Color::Light(BaseColor::White),Color::Dark(BaseColor::Red));
    let bookmark_title_style = ColorStyle::new(Color::Light(BaseColor::White),Color::Dark(BaseColor::Blue));

    let mut annotation_legend = StyledString::plain("Annotation Legend:\n");
    annotation_legend.append(StyledString::styled("Parable Title\n",parable_title_style));
    annotation_legend.append(StyledString::styled("parable text\n\n",parable_style));
    annotation_legend.append(StyledString::styled("Bookmark Title\n",bookmark_title_style));
    annotation_legend.append(StyledString::styled("bookmark text\n",bookmark_style));

    let mut help_text = StyledString::plain(HELP_TEXT.to_string());
    help_text.append(annotation_legend);

    view.add_layer(
        Dialog::around(TextView::new(help_text))
        .dismiss_button("Back")
        .title("Help")
        .wrap_with(OnEventView::new)
        .on_event(Event::Key(Key::Esc), |s| {
            s.pop_layer();
        })
    )
}

fn reload(view: &mut Cursive) {
    let curr_chapter = view.user_data::<chapter::Chapter>().unwrap().clone();

    let book_num = curr_chapter.book_num;
    let chapter_num = curr_chapter.chapter_num;

    let chapter = chapter::Chapter::new(book_num, chapter_num);

    set_chapter(view, chapter)
}

lazy_static! {
    static ref BOOK_CACHE: String = format!("{}/.cache/the_rock.book",env::var("HOME").unwrap());
    static ref CHAPTER_CACHE: String = format!("{}/.cache/the_rock.chapter",env::var("HOME").unwrap());
    static ref HELP_TEXT: String = format!("The Rock: A King James Bible Viewer\nVersion {}\n\nKey Shortcuts:\nq: Quit\nn: set verse view format\na: Add verses in current chapter as named bookmarkn\nb: Show/Hide Bookmarks\np: Show/Hide Parables\n\n",env!("CARGO_PKG_VERSION"));
}