use std::{env, fs};
use crate::bible::*;
use serde_derive::{Deserialize, Serialize};
use toml::Value;

#[derive(Deserialize,Serialize,Clone)]
pub struct Settings {
    // the format of the verses (numbered list, paragraph, or paragraph with numbers?)
    pub format: Format,

    // bookmarked passages/verses (by default, it is Matthew6:9-13, John3:16, Mark9:23, and Exodus 20)
    pub bookmarks: Vec<(String,String)>,

    // should parables be shown?
    pub show_parables: bool,

    // should bookmarks be shown?
    pub show_bookmarks: bool
}

impl Settings {
    // returns the default settings
    fn default() -> Settings {
        let passage_list = vec![
            (String::from("The Lord's Prayer"),String::from("Matt6:9-13")),
            (String::from("Eternal Promise"),String::from("John3:16")),
            (String::from("Everything is Possible"),String::from("Mark9:23")),
            (String::from("The 10 Commandments"),String::from("Exod20:1-26"))
        ];

        Settings {
            format: Format::Numbers,
            bookmarks: passage_list,
            show_parables: true,
            show_bookmarks: true,
        }
    }

    // updates the config file with new settings
    pub fn update(&self) {
        let config_dir = format!("{}/.config/the_rock/",env::var("HOME").unwrap());
        let config_file = format!("{}/.config/the_rock/config.toml",env::var("HOME").unwrap());
        fs::create_dir(config_dir).unwrap_or(());
        fs::write(config_file, toml::to_string(self).unwrap()).expect("unable to write to config file!");
    }
}

// gets the current settings or the defaults
pub fn load_config() -> Settings {
    let mut settings = Settings::default();
    let config_loc = format!("{}/.config/the_rock/config.toml",env::var("HOME").unwrap());
    let config_dir = format!("{}/.config/the_rock/",env::var("HOME").unwrap());
    let toml = match fs::read_to_string(&config_loc) {
        Ok(file) => {
            settings.bookmarks=Vec::new();
            file
        },

        // if the config file doesn't exist, create it
        Err(_) => {
            fs::create_dir(&config_dir).unwrap_or(());
            settings = Settings::default();
            fs::write(&config_loc, toml::to_string(&settings).unwrap()).expect("unable to create config file!");
            toml::to_string(&settings).unwrap()
        }
    };

    let settings_raw = toml.parse::<Value>().unwrap();

    if settings_raw.get("show_parables") != None {
        settings.show_parables = settings_raw["show_parables"].as_bool().unwrap_or(true);
    }

    if settings_raw.get("show_bookmarks") != None {
        settings.show_bookmarks = settings_raw["show_bookmarks"].as_bool().unwrap_or(true);
    }

    settings.format = Format::from_string(&mut settings_raw["format"].to_string());
    for bookmark in settings_raw["bookmarks"].as_array().unwrap() {
        settings.bookmarks.push((bookmark.as_array().unwrap()[0].as_str().unwrap().to_string(), bookmark.as_array().unwrap()[1].as_str().unwrap().to_string()))
    }

    settings
}