use std::{fs, process};
use serde_derive::{Deserialize, Serialize};
use rust_utils::utils;
use cursive::utils::markup::StyledString;
use cursive::theme::{Color, BaseColor, ColorStyle};
use cursive::theme::Style;

pub mod chapter;
pub mod books;

#[derive(Clone, Copy, Debug, PartialEq, Deserialize, Serialize)]
pub enum Format {
    // show passages with verse numbers
    Numbers,

    // show the passages as paragraphs
    Paragraph,

    // show as a paragraph, but with verse numbers before each verse
    NumParagraph
}

impl Format{
    pub fn from_string(text: &mut String) -> Self {
        text.remove(0);
        text.pop();
        match text.as_str() {
            "Numbers" => Format::Numbers,
            "Paragraph" => Format::Paragraph,
            "NumParagraph" => Format::NumParagraph,
            _ => {
                println!("Invalid config value for format: {}",text);
                println!("Using default value: Numbers");
                Format::Numbers
            }
        }
    }

    pub fn next(&self) -> Self {
        match *self {
            Format::Numbers => Format::Paragraph,
            Format::Paragraph => Format::NumParagraph,
            Format::NumParagraph => Format::Numbers
        }
    }

    pub fn desc(&self) -> &str {
        match *self {
            Format::Numbers => "Numbered List",
            Format::Paragraph => "Paragraph",
            Format::NumParagraph => "Paragraph with verse numbers"
        }
    }
}

// return every verse from a specific book
// panics if the book name is invalid
fn get_book(name: &str) -> Vec<String>{
    let bible_raw = open_bible();
    let book_num; // 0 is Genesis
    let mut book = Vec::new(); // book as vector of strings (the verses)
    let mut verse_num = 0; // line number from the bible.rawtext file
    let mut verse = &bible_raw[verse_num]; // the verse

    book_num = get_book_num(name);

    // error handler
    if book_num >= 66 {
        panic!("{} is not a valid book name!",name)
    }

    // book name as it is in bible.rawtext
    let book_name = books::BOOKS_SHORT1[book_num];

    // jump to first verse of the specified book
    while &verse[..book_name.len()]!= book_name {
        verse = &bible_raw[verse_num];
        verse_num += 1;
    }
    book.push(verse.to_string());

    // get the verses from the specified book
    while &verse[..book_name.len()]== book_name {
        // needed to return Revelation correctly
        if verse_num>=31102 {
            break;
        }

        verse = &bible_raw[verse_num];
        verse_num += 1;
        if &verse[..book_name.len()]== book_name {
            book.push(verse.to_string());
        }
    }
    
    // genesis 1:1 appears twice
    // this fixes that issue
    if book_num==0
    {
        book.remove(0);
    }

    book
}

// returns every verse from the bible.rawtext file as a vector of strings
fn open_bible() -> Vec<String>{
    let mut bible_raw = Vec::new();
    let file = match fs::read_to_string(BIBLE_FILE2) {
        Ok(file) => file,
        Err(_) => {
            fs::read_to_string(BIBLE_FILE).unwrap()
        }
    };

    for (count,line) in file.lines().enumerate() {
        // skip the first line (it is not part of the actual text)
        if count>0{
            bible_raw.push(String::from(line));
        }
    }

    bible_raw
}

// get the number of a specific book
// returns 66 if the name is invalid
pub fn get_book_num<N: Into<String>>(name: N) -> usize {
    let mut book_num = 0;

    // capitalize the first letter, if needed
    let namecap = utils::capitalize(name);

    // get the book number
    for book in &books::BOOKS {
        if namecap == book.to_string(){
            break;
        }
        book_num +=1;
    }

    if book_num >= 66{
        book_num = 0;
        for book in books::BOOKS_SHORT1.iter() {
            if namecap == book.to_string(){
                break;
            }
            book_num +=1;
        }
    }

    if book_num >= 66{
        book_num = 0;
        for book in books::BOOKS_SHORT2.iter() {
            if namecap == book.to_string(){
                break;
            }
            book_num +=1;
        }
    }
    
    book_num
}

// returns a bible verse reference as a 5 element tuple
// element 0: book name
// element 1: chapter
// element 2: first verse (if passage)
// element 3: last verse (if passage)
// element 4: is this a passage or verse?
pub fn parse_verse_ref<V: Into<String>>(verse_in: V) -> (String,usize,usize,usize,bool) {
    let verse = verse_in.into();
    let mut ispassage: bool = false;
    let mut book_t: Vec<char> = Vec::new();
    let mut book: String;
    let chapter_num: u8;
    let dashloc: usize;
    let mut colonloc: usize;
    let verse_num;
    let mut last_verse_num = 0;

    // parse the book name
    for (counter, c) in verse.chars().enumerate(){
        if counter == 0 ||!c.is_digit(10){
            book_t.push(c);
        }
    }
    book = book_t.iter().collect();

    // if there is a '-' char in the string, assume this is a passage
    dashloc = utils::find_char(&verse, '-');

    // parse the chapter number
    // if there isn't one, use 1 as the chapter number
    colonloc = utils::find_char(&verse,':');
    if colonloc == verse.len(){
        chapter_num = verse[book.len()..verse.len()].parse().unwrap_or(1);
    }
    else {
        if dashloc<verse.len(){
            chapter_num = verse[book.len()-2..colonloc].parse().unwrap_or(255);
        }
        else {
            chapter_num = verse[book.len()-1..colonloc].parse().unwrap_or(255);
        }
        if chapter_num == 255{
            println!("Invalid syntax!");
            process::exit(2);
        }
    }

    // parse the verse number
    // if there isn't one, use 1 as the verse number
    colonloc = utils::find_char(&verse,':');
    if colonloc == dashloc{
        verse_num = 1
    }
    else {
        verse_num = verse[colonloc+1..dashloc].parse().expect("Invalid syntax!");
    }

    if book.contains(':'){
        book.pop();
    }

    // if this is a passage, get the number of the last verse
    if dashloc<verse.len() {
        ispassage = true;
        last_verse_num = verse[dashloc+1..verse.len()].parse().expect("Invalid syntax!");
    }
    
    if get_book_num(&book) >=66 {
        book = book[..book.len()-1].to_string();
    }

    (book,chapter_num as usize,verse_num,last_verse_num,ispassage)
}

pub const BIBLE_FILE: &str = "bible.rawtext";
pub const BIBLE_FILE2: &str = "/usr/lib/the_rock/bible.rawtext";

// the parables of Jesus christ
// first tuple element is the name and the second is the verse reference
pub const PARABLES: [(&str, &str); 61]  = [
    ("New Cloth on an Old Coat 1", "Mat9:16"),
    ("New Cloth on an Old Coat 2", "Mark2:21"),
    ("New Cloth on an Old Coat 3", "Luke5:36"),

    ("New Wine in Old Wineskins 1", "Mark9:17"),
    ("New Wine in Old Wineskins 2", "Mark2:22"),
    ("New Wine in Old Wineskins 3", "Luke5:37-38"),

    ("The Lamp on a Stand 1", "Mat5:14-15"),
    ("The Lamp on a Stand 2", "Mark4:21-22"),
    ("The Lamp on a Stand 3", "Luke8:16"),
    ("The Lamp on a Stand 4", "Luke11:33"),
    
    ("The Wise and Foolish Builders 1", "Mat7:24-27"),
    ("The Wise and Foolish Builders 2", "Luke6:47-49"),
    
    ("The Moneylender forgiving unequal debts", "Luke7:41-43"),
    ("The Rich Fool", "Luke12:16-21"),
    
    ("The Servants Must Remain Watchful 1", "Mark13:35-37"),
    ("The Servants Must Remain Watchful 2", "Luke12:35-40"),
    
    ("The Wise and Foolish Servants 1", "Mat24:45-51"),
    ("The Wise and Foolish Servants 2", "Luke12:42-48"),
    
    ("The Unfruitful Fig Tree", "Luke13:6-9"),
    
    ("The Parable of the Soils 1", "Mat13:3-23"),
    ("The Parable of the Soils 2", "Mark4:1-20"),
    ("The Parable of the Soils 3", "Luke8:4-15"),
    
    ("The Weeds Among Good Plants", "Mat13:24-43"),
    ("The Growing Seed", "Mark4:26-29"),
    
    ("The Mustard Seed 1", "Mat13:31-32"),
    ("The Mustard Seed 2", "Mark4:30-32"),
    ("The Mustard Seed 3", "Luke13:18-19"),
    
    ("Yeast", "Mat13:31-32"),
    ("Hidden Treasure", "Mat13:44"),
    ("Valuable Pearl", "Mat13:45-46"),
    ("Fishing Net", "Mat13:47-50"),
    ("Owner of a House", "Mat13:52"),
    ("Lost Sheep", "Mat18:12-14"),
    ("The Master and His Servant", "Luke17:7-10"),
    ("The Unmerciful servant", "Mat18:23-34"),
    ("The Good Samaritan", "Luke10:30-37"),
    ("Friend in Need", "Luke11:5-8"),
    ("Lowest Seat at the Feast", "Luke14:7-14"),
    ("Invitation to a Great Banquet", "Luke14:16-24"),
    ("The Cost of Discipleship", "Luke14:28-33"),
    ("Lost Sheep", "Luke15:4-7"),
    ("Lost Coin", "Luke15:8-10"),
    ("The Prodigal Son", "Luke15:11-32"),
    ("The Shrewd Manager", "Luke16:1-8"),
    ("The Rich Man and Lazarus", "Luke16:19-31"),
    ("The Early and Late Workers in the Vineyard", "Mat20:1-16"),
    ("The Persistent Widow and Crooked Judge", "Mat18:1-8"),
    ("The Pharisee and Tax Collector", "Luke18:10-14"),
    ("The King’s Ten Servants Given Minas", "Luke19:12-27"),
    ("Two Sons (one obeys, one disobeys)", "Mat21:28-32"),
    
    ("Wicked Tenants 1", "Mat21:33-44"),
    ("Wicked Tenants 2", "Mark12:1-11"),
    ("Wicked Tenants 3", "Luke20:9-18"),
    
    ("Invitation to a Wedding Banquet", "Mat22:2-14"),
    
    ("The Fig Tree and Signs of the Future 1", "Mat24:32-35"),
    ("The Fig Tree and Signs of the Future 2", "Mark13:28-29"),
    ("The Fig Tree and Signs of the Future 3", "Luke21:29-31"),
    
    ("The Wise and Foolish Virgins", "Mat25:1-13"),
    ("The Talents", "Mat25:14-30"),
    ("The Sheep and the Goats", "Mat25:31-46"),
    ("The Sheep, Shepherd, and Gate", "John10:1-18")
];