use super::*;
use crate::config;

// object for chapter info
// used in the TUI
#[derive(Clone)]
pub struct Chapter {
    pub book_num: usize,
    pub chapter_num: usize,
    pub content: StyledString,
    pub format: Format,
}

impl Chapter {
    pub fn new(mut book_num: usize, mut chapter_num: usize) -> Chapter {
        if book_num > 65 {
            book_num = 0;
            chapter_num = 1;
        }

        let settings = config::load_config();
        let format = settings.format;
        let chapter = get_chapter(&books::BOOKS[book_num], chapter_num);
        let mut content = StyledString::from("");
        let parable_style = Style::from(Color::Light(BaseColor::Red));
        let bookmark_style = Style::from(Color::Light(BaseColor::Blue));

        let parable_title_style = ColorStyle::new(Color::Light(BaseColor::White),Color::Dark(BaseColor::Red));
        let bookmark_title_style = ColorStyle::new(Color::Light(BaseColor::White),Color::Dark(BaseColor::Blue));
        let mut prev_title = "N".to_string();
        let mut prev_title2 = "N".to_string();
        for (verse_num, verse) in chapter.iter().enumerate() {
            let line = match format {
                Format::Paragraph => format!(" {}",verse),
                Format::Numbers => format!("{} {}\n",verse_num + 1, verse),
                Format::NumParagraph => format!(" {} {}", verse_num + 1, verse)
            };

            let (title, is_parable) = is_parable(book_num, chapter_num, verse_num + 1);
            let (title2, is_bookmark) = is_bookmark(book_num, chapter_num, verse_num + 1);

            if is_parable && settings.show_parables {
                if title != prev_title && format ==  Format::Numbers {
                    content.append(StyledString::styled(
                        &title,
                        parable_title_style,
                    ));
                }

                content.append(StyledString::styled(
                    &line,
                    parable_style,
                ));
                prev_title = title;
            }

            else if is_bookmark && settings.show_bookmarks {
                if title2 != prev_title2 && format ==  Format::Numbers {
                    content.append(StyledString::styled(
                        &title2,
                        bookmark_title_style,
                    ));
                }
                content.append(StyledString::styled(
                    line,
                    bookmark_style,
                ));
            prev_title2 = title2;

            }
            else {
                content.append(StyledString::plain(line));
                prev_title = title;
            }
        }

        Chapter {
            book_num,
            chapter_num,
            content,
            format
        }
    }

    pub fn next(&mut self) -> Chapter {
        if self.chapter_num + 1 > books::CHAPTERS[self.book_num].into() {
            self.chapter_num = 0;
            self.book_num += 1;
        }
        Chapter::new(self.book_num, self.chapter_num+1)
    }

    pub fn prev(&mut self) -> Chapter {
        if self.chapter_num == 1 && self.book_num == 0 {
            self.chapter_num = 23;
            self.book_num = 65;
        }
        
        if self.chapter_num == 1 && self.book_num > 0 {
            self.chapter_num = books::CHAPTERS[self.book_num - 1] + 1;
            self.book_num -= 1;
        }

        Chapter::new(self.book_num, self.chapter_num - 1)
    }

    pub fn get_title(&self) -> String {
        format!("{} {}", books::BOOKS[self.book_num],self.chapter_num)
    }
}

// get a passage from a specific chapter
pub fn get_passage(book: &str, chapter: usize, begin: usize, end: usize) -> Vec<String>{
    let mut passage: Vec<String> = Vec::new();
    let chapter_cont = get_chapter(book, chapter);
    for i in begin - 1..end {
        passage.push(chapter_cont[i].clone())
    }
    
    passage
}

// get a specific verse
pub fn get_verse(book: &str, chapter: usize, verse_num: usize) -> String {
    let chapter_cont = get_chapter(book, chapter);

    chapter_cont[verse_num-1].clone()
}

// return verses from a specific chapter
pub fn get_chapter(name: &str, chapter: usize) -> Vec<String>{
    let book: Vec<String> = get_book(&name);
    let mut colonloc: usize;
    let mut spaceloc: usize;
    let mut chapter_num: usize;
    let book_num = get_book_num(name);
    let mut chapter_cont: Vec<String> = Vec::new();
    let book_short = books::BOOKS_SHORT1[book_num];

    for verse in book {
        // convert the chapter number in the verse header to integer
        colonloc = utils::find_char(&verse,':');
        chapter_num = verse[book_short.len()..colonloc].parse().unwrap();

        // is this the chapter number we want?
        if chapter_num == chapter {
            spaceloc = utils::find_char(&verse,' ');
            chapter_cont.push(verse[spaceloc+1..verse.len()].to_string());
        }
    }
    
    chapter_cont
}

// is the verse part of a bookmark?
fn is_bookmark(book_num: usize, chapter_num: usize, verse_num: usize) -> (String, bool) {
    let bookmarks = config::load_config().bookmarks;
    let verse = (books::BOOKS_SHORT2[book_num].to_string(),chapter_num,verse_num);

    for bookmark in bookmarks {
        let v = parse_verse_ref(utils::capitalize(bookmark.1));
        let title = format!("{}\n",bookmark.0);

        if v.4 {
            for cur_verse in v.2..=v.3 {
                let book = v.0.clone();
                if verse == (book,v.1,cur_verse) {
                    return (title,true);
                }
            }
        }
        else {
            if verse == (v.0,v.1,v.2) {
                return (title,true);
            }
        }
    }
    ("N".to_string(),false)
}

// is the verse part of a parable?
fn is_parable(book_num: usize, chapter_num: usize, verse_num: usize) -> (String, bool) {
    // the parables are only in the gospels
    // if this isn't a gospel book, don't bother looking for parables
    if book_num < 39 || book_num > 42 {
        return ("N".to_string(),false);
    }
    let verse = (books::BOOKS_SHORT1[book_num].to_string(),chapter_num,verse_num);
    for i in 0..61 {
        let parable = parse_verse_ref(PARABLES[i].1);
        let title = format!("{}\n",PARABLES[i].0);
        if parable.4 {
            for cur_verse in parable.2..=parable.3 {
                let book = parable.0.clone();
                if verse == (book,parable.1,cur_verse) {
                    return (title,true);
                }
            }
        }
        else {
            if verse == (parable.0,parable.1,parable.2) {
                return (title,true);
            }
        }
    }

    ("N".to_string(),false)
}