use crate::setup::Path;
use crate::structs;
use hyper_tls::HttpsConnector;
use hyper::{body, Request, Body};
use hyper::{Client};
use hyper::client::connect::HttpConnector;
use http::uri::Uri;
use serde_json;
use serde::{Deserialize, Serialize};

pub struct Api{
    keys_service_path: String,
    client: Client<HttpsConnector<HttpConnector>>,
}

impl Api{
    pub fn from_path(path: &Path) -> Option<Self> {
        let https = HttpsConnector::new();
        let client = Client::builder().build::<_, hyper::Body>(https);
        if let Some(keys_service_path) = path.get_keys_service(){
            return Some(Self{keys_service_path, client})
        }
        None
    }
    pub async fn get_key(&self, name: String) -> Result<structs::TinkoffApiKey, ()>{
        let uri = format!("{}/get?name={}", self.keys_service_path.clone(), name).parse();
        if let Ok(uri) = uri{
            if let Ok(resp) = self.client.get(uri).await{
                if resp.status() != 200{ return Err(())}
                if let Ok(b) = body::to_bytes(resp.into_body()).await{
                    if let Ok(s) = String::from_utf8(b.to_vec()){
                        if let Ok(key) = serde_json::from_str(&s){
                            return Ok(key)
                        }
                    }
                }
            }
        }
        Err(())
    }
    pub async fn get_keys(&self) -> Result<AllKeysResp, ()>{
        let uri = format!("{}/all", self.keys_service_path.clone()).parse();
        if let Ok(uri) = uri{
            if let Ok(resp) = self.client.get(uri).await{
                if resp.status() != 200{ return Err(())}
                if let Ok(b) = body::to_bytes(resp.into_body()).await{
                    if let Ok(s) = String::from_utf8(b.to_vec()){
                        if let Ok(key) = serde_json::from_str(&s){
                            return Ok(key)
                        }
                    }
                }
            }
        }
        Err(())
    }
    pub async fn set_key(&self, name: String, key: String) -> Result<i64, ()>{
        let uri = format!("{}/set", self.keys_service_path.clone()).parse();
        if let Ok(uri) = uri{
            let uri: Uri = uri;
            let body = serde_json::to_string(&KeySetReqBody{name, key});
            if let Ok(body) = body{
                let req = Request::post(uri).body(Body::from(body));
                if let Ok(req) = req{
                    if let Ok(resp) = self.client.request(req).await{
                        if resp.status() != 200{ return Err(())}
                        if let Ok(b) = body::to_bytes(resp.into_body()).await{
                            if let Ok(s) = String::from_utf8(b.to_vec()){
                                if let Ok(r) = serde_json::from_str(&s){
                                    let r: KeySetResp = r;
                                    return Ok(r.time)
                                }
                            }
                        }
                    }
                }
            }
        }
        Err(())
    }
}

#[derive(Deserialize, Serialize, Debug)]
pub struct KeySetReqBody{
    pub name: String,
    pub key: String,
}

#[derive(Deserialize, Serialize, Debug)]
pub struct KeySetResp{
    pub time: i64
}

#[derive(Deserialize, Serialize, Debug)]
pub struct AllKeysResp{
    pub keys: Vec<structs::TinkoffApiKey>,
}
