use crate::structs;
use rotorlib::protocol;
use std::convert::{TryFrom, Into, TryInto};
use url::{Url};
use rotorlib::protocol::Package;
use rotorlib::streaming_client::LightClient;
use rotorlib::rpc_client;
use crate::structs::TinkoffApiKey;


#[cfg(test)]
mod string_getter_tests{
    use crate::messages::get_next_string;

    #[test]
    fn test_normal_string(){
        let reference = "ABCD".to_string();
        let bytes = reference.clone().into_bytes();
        match get_next_string(&bytes, 0){
            Ok((string, end)) => {
                assert_eq!(string, reference);
                assert_eq!(end, 4);
            }
            Err(_) => {assert!(false, "Unexpected error")}
        }
    }

    #[test]
    fn test_void_string(){
        let reference = "".to_string();
        let bytes = reference.clone().into_bytes();
        match get_next_string(&bytes, 0){
            Ok((string, end)) => {
                assert_eq!(string, reference);
                assert_eq!(end, 0);
            }
            Err(_) => {assert!(false, "Unexpected error")}
        }
    }

    #[test]
    fn test_normal_string_with_prefix_data(){
        let reference = "ABCD".to_string();
        let mut bytes = vec![0,1,2,3,4,5];
        bytes.append(&mut reference.clone().into_bytes());
        match get_next_string(&bytes, 6){
            Ok((string, end)) => {
                assert_eq!(string, reference);
                assert_eq!(end, 10);
            }
            Err(_) => {assert!(false, "Unexpected error")}
        }
    }

    #[test]
    fn test_void_string_with_prefix_data(){
        let reference = "".to_string();
        let mut bytes = vec![0,1,2,3,4,5];
        bytes.append(&mut reference.clone().into_bytes());
        match get_next_string(&bytes, 6){
            Ok((string, end)) => {
                assert_eq!(string, reference);
                assert_eq!(end, 6);
            }
            Err(_) => {assert!(false, "Unexpected error")}
        }
    }

    #[test]
    fn test_normal_string_with_postfix_data(){
        let reference = "ABCD".to_string();
        let mut bytes = reference.clone().into_bytes();
        bytes.push(0); //separator
        bytes.append(&mut vec![0, 1, 2, 3, 4, 5]);
        match get_next_string(&bytes, 0){
            Ok((string, end)) => {
                assert_eq!(string, reference);
                assert_eq!(end, 5);
            }
            Err(_) => {assert!(false, "Unexpected error")}
        }
    }

    #[test]
    fn test_void_string_with_postfix_data(){
        let reference = "".to_string();
        let mut bytes = reference.clone().into_bytes();
        bytes.push(0); //separator
        bytes.append(&mut vec![0, 1, 2, 3, 4, 5]);
        match get_next_string(&bytes, 0){
            Ok((string, end)) => {
                assert_eq!(string, reference);
                assert_eq!(end, 1);
            }
            Err(_) => {assert!(false, "Unexpected error")}
        }
    }
}

#[cfg(test)]
mod tinkoff_api_ket_tests{
    use crate::structs::TinkoffApiKey;
    use std::convert::TryFrom;

    #[test]
    fn test_parsing(){
        let reference = TinkoffApiKey{
            name: "NAME".to_string(),
            key: "KEY".to_string(),
            time: 0
        };
        let bytes: Vec<u8> = reference.clone().into();
        match TinkoffApiKey::try_from(bytes){
            Ok(result) => {
                assert_eq!(result.name, reference.name);
                assert_eq!(result.key, reference.key);
                assert_eq!(result.time, reference.time);
            }
            Err(_) => {assert!(false, "Unexpected error")}
        }
    }

    #[test]
    fn test_parsing_with_void_fields(){
        let reference = TinkoffApiKey{
            name: "".to_string(),
            key: "".to_string(),
            time: 0
        };
        let bytes: Vec<u8> = reference.clone().into();
        match TinkoffApiKey::try_from(bytes){
            Ok(result) => {
                assert_eq!(result.name, reference.name);
                assert_eq!(result.key, reference.key);
                assert_eq!(result.time, reference.time);
            }
            Err(_) => {assert!(false, "Unexpected error")}
        }
    }
}

pub mod keys{
    pub const KEY_UPDATE: &str = "key:update";
}

fn get_next_string(bytes: &Vec<u8>, start: usize) -> Result<(String, usize), ()>{
    if start >= (bytes.len()){
        return Ok(("".to_string(), start))
    }
    if bytes[start] == 0{
        return Ok(("".to_string(), start+1))
    }
    let mut end: usize = 0;
    let mut next: usize = 0;
    for i in start..bytes.len() {
        if bytes[i] == 0{
            end = i;
            next = i+1;
            break
        }
        if i == bytes.len()-1{
            end = i+1;
            next = i+1;
            break
        }
    }
    match String::from_utf8(bytes[start..end].to_owned()){
        Ok(string) => {
            Ok((string, next))
        }
        Err(_) => { Err(()) }
    }
}


/*fn get_next_vec(bytes: &Vec<u8>, start: usize) -> Vec<u8>{
    if start >= (bytes.len()){
        return vec![];
    }
    bytes[start..].to_vec()
}*/

#[derive(Debug)]
pub enum ServiceMessage{
    Subscribe(protocol::SubscribePackage),
    Registration(protocol::RegistrationPackage)
}

#[derive(Debug)]
pub enum Payload{
    KeyUpdate(structs::TinkoffApiKey),
}

#[derive(Debug)]
pub enum Message{
    Payload(Payload),
    Service(ServiceMessage),
}

pub struct Sender{
    server_url: Url
}

impl Sender{
    pub fn new(server_url: Url) -> Self{ Self{server_url} }
    pub fn new_str(server_url: &str) -> Result<Self, ()>{
        let server_url: Url = match server_url.parse(){
            Ok(url) => {
                let url: Url = url;
                url
            }
            Err(_) => { return Err(()) }
        };
        Ok(Self{server_url})
    }
    pub async fn send(&self, payload: Payload) -> Result<(), ()>{
        rpc_client::send_msg(self.server_url.clone(),
                             payload.into()).await
    }
}

pub struct Api{
    client: LightClient
}

impl Api{
    pub fn new(client: LightClient) -> Self{ Self{client} }
    pub async fn open(url: Url) -> Result<Self, ()>{
        let client = LightClient::open(url).await;
        match client{
            Ok(client) => {
                Ok(Self::new(client))
            }
            Err(_) => {
                Err(())
            }
        }
    }
    pub async fn open_str(url: &str) -> Result<Self, ()>{
        let url: Url = match url.parse(){
            Ok(url) => {
                let url: Url = url;
                url
            }
            Err(_) => { return Err(()) }
        };
        Self::open(url).await
    }
    pub async fn close(&mut self) -> Result<(), ()>{
        match self.client.close().await{
            Ok(_) => { Ok(()) }
            Err(_) => { Err(()) }
        }
    }
    pub async fn next(&mut self) -> Result<Message, ()>{
        loop{
            match self.client.recv().await{
                Ok(package) => {
                    if let Ok(msg) = Message::try_from(package){
                        return Ok(msg)
                    }
                }
                Err(_) => {
                    return Err(())
                }
            }
        }
    }
    pub async fn next_payload(&mut self) -> Result<Payload, ()>{
        loop{
            match self.next().await{
                Ok(message) => {
                    if let Message::Payload(payload) = message{
                        return Ok(payload)
                    }
                }
                Err(_) => {
                    return Err(())
                }
            }
        }
    }
    pub async fn send(&mut self, msg: Message) -> Result<(), ()>{
        match self.client.send(msg.into()).await{
            Ok(_) => { Ok(()) }
            Err(_) => { Err(()) }
        }
    }
    pub async fn send_payload(&mut self, payload: Payload) -> Result<(), ()>{
        self.send(Message::Payload(payload)).await
    }
    pub async fn subscribe(&mut self, key: &str) -> Result<(), ()> {
        self.send(Message::Service(ServiceMessage::Subscribe(
            protocol::SubscribePackage{ is_sub: true, key: key.to_string() }
        ))).await
    }
    pub async fn unsubscribe(&mut self, key: &str) -> Result<(), ()> {
        self.send(Message::Service(ServiceMessage::Subscribe(
            protocol::SubscribePackage{ is_sub: false, key: key.to_string() }
        ))).await
    }
    pub async fn register(&mut self, is_router: bool) -> Result<(), ()>{
        self.send(Message::Service(ServiceMessage::Registration(
            protocol::RegistrationPackage{ is_router }
        ))).await
    }
}

impl Into<Package> for ServiceMessage{
    fn into(self) -> Package {
        match self{
            ServiceMessage::Subscribe(sub) => {
                Package::Sub(sub)
            }
            ServiceMessage::Registration(reg) => {
                Package::Reg(reg)
            }
        }
    }
}

impl TryFrom<protocol::Package> for ServiceMessage{
    type Error = ();

    fn try_from(value: Package) -> Result<Self, Self::Error> {
        match value{
            Package::Sub(sub) => {
                Ok(Self::Subscribe(sub))
            }
            Package::Reg(reg) => {
                Ok(Self::Registration(reg))
            }
            _ => { Err(()) }
        }
    }
}

impl Into<protocol::MessagePackage> for Payload{
    fn into(self) -> protocol::MessagePackage {
        match self{
            Payload::KeyUpdate(api_key) => {
                protocol::MessagePackage{
                    key: keys::KEY_UPDATE.to_string(),
                    payload: api_key.into(),
                }
            }
        }
    }
}

impl TryFrom<protocol::MessagePackage> for Payload{
    type Error = ();

    fn try_from(message: protocol::MessagePackage) -> Result<Self, Self::Error> {
        match message.key.as_str(){
            keys::KEY_UPDATE => {
                match structs::TinkoffApiKey::try_from(message.payload){
                    Ok(api_key) => {
                        Ok(Self::KeyUpdate(api_key))
                    }
                    Err(_) => { Err(()) }
                }
            }
            _ => { Err(()) }
        }
    }
}

impl Into<Package> for Message{
    fn into(self) -> Package {
        match self{
            Message::Payload(payload) => {
                Package::Msg(payload.into())
            }
            Message::Service(service) => {
                service.into()
            }
        }
    }
}

impl TryFrom<Package> for Message{
    type Error = ();

    fn try_from(package: Package) -> Result<Self, Self::Error> {
        match package{
            Package::Msg(msg) => {
                match Payload::try_from(msg){
                    Ok(payload) => {
                        Ok(Self::Payload(payload))
                    }
                    Err(_) => { Err(()) }
                }
            }
            _ => {
                match ServiceMessage::try_from(package){
                    Ok(service_msg) => {
                        Ok(Self::Service(service_msg))
                    }
                    Err(_) => { Err(()) }
                }
            }
        }
    }
}

impl Into<Vec<u8>> for structs::TinkoffApiKey{
    fn into(self) -> Vec<u8> {
        let mut ret = self.time.to_be_bytes().to_vec();
        ret.append(&mut self.name.into_bytes());
        ret.push(0); // separator
        ret.append(&mut self.key.into_bytes());
        ret
    }
}

impl TryFrom<Vec<u8>> for structs::TinkoffApiKey{
    type Error = ();

    fn try_from(bytes: Vec<u8>) -> Result<Self, Self::Error> {
        if bytes.len()<8{ return Err(()) }
        let time_arr = (&bytes[0..8]).try_into().unwrap();
        let time: i64 = i64::from_be_bytes(time_arr);
        let (name, end) = match get_next_string(&bytes, 8){
            Ok(res) => {res}
            Err(_) => { return Err(()) }
        };
        let (key, _) = match get_next_string(&bytes, end){
            Ok(res) => {res}
            Err(_) => { return Err(()) }
        };
        Ok(TinkoffApiKey{time, name, key})
    }
}
