use crate::setup::Path;
use crate::structs;
use rotorlib::protocol;
use std::convert::{TryFrom, Into};
use tokio::sync::mpsc::{Sender, Receiver, channel};
use rotorlib::protocol::{Package, Parsable};
use url::{Url};
use tokio_tungstenite::{MaybeTlsStream, WebSocketStream, connect_async};
use tokio_tungstenite::tungstenite::{Message};
use tokio::net::TcpStream;
use futures::{SinkExt, StreamExt};


pub mod keys{
    pub const KEY_UPDATE: &str = "key:update";
}

#[derive(Debug)]
pub enum Msg{
    KeyUpdate(structs::TinkoffApiKey),
    SUB(protocol::SubPackage),
    NODE(protocol::NodePackage),
}

pub struct Api{
    messages_bus: String,
    sender: Option<Sender<Vec<u8>>>,
    receiver: Option<Receiver<Vec<u8>>>,
}

impl Api{
    pub fn new(messages_bus: String) -> Self{
        Self{messages_bus, sender: None, receiver: None}
    }
    pub fn from_path(path: &Path) -> Option<Self> {
        match path.get_messages_bus(){
            None => { None }
            Some(messages_bus) => {
                Some(Self::new(messages_bus))
            }
        }
    }
    pub async fn open(&mut self, buffer: usize) -> Result<(), ()> {
        if let Some(_) = self.sender{ return Err(()) }
        if let Some(_) = self.receiver{ return Err(()) }
        match Url::parse(self.messages_bus.as_str()){
            Ok(url) => {
                let (up_sender, up_receiver) = channel(buffer);
                let (down_sender, down_receiver) = channel(buffer);
                match connect_async(url).await{
                    Ok(conn) => {
                        tokio::spawn(socket_listener(conn.0, up_sender, down_receiver));
                        self.sender = Some(down_sender);
                        self.receiver = Some(up_receiver);
                        Ok(())
                    }
                    Err(_) => { Err(()) }
                }
            }
            Err(_) => { Err(()) }
        }
    }
    pub async fn close(&mut self){
        if let Some(receiver) = self.receiver.as_mut(){
            receiver.close();
        }
        self.receiver = None;
        self.sender = None;
    }
    pub async fn next(&mut self) -> Result<Msg, ()>{
        match self.receiver.as_mut(){
            None => { Err(()) }
            Some(receiver) => {
                loop{
                    match receiver.recv().await{
                        None => { return Err(()) }
                        Some(bytes) => {
                            match rotorlib::protocol::Package::from_bytes(bytes){
                                None => { println!("rotor: strange msg"); }
                                Some(package) => {
                                    match Msg::try_from(package){
                                        Ok(msg) => {
                                            return Ok(msg)
                                        }
                                        Err(_) => {
                                            println!("bot: strange msg");
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    pub async fn send(&mut self, msg: Msg) -> Result<(), ()>{
        match self.sender.as_mut(){
            None => { Err(()) }
            Some(sender) => {
                let pack: protocol::Package = msg.into();
                let bytes = pack.to_bytes();
                match sender.send(bytes).await{
                    Ok(_) => { Ok(())}
                    Err(_) => { Err(()) }
                }
            }
        }
    }
    pub async fn subscribe(&mut self, key: &str) -> Result<(), ()> {
        let int_key = protocol::hash(&key.to_string());
        self.send(Msg::SUB(protocol::SubPackage{ is_sub: true, int_key })).await
    }
    pub async fn unsubscribe(&mut self, key: &str) -> Result<(), ()> {
        let int_key = protocol::hash(&key.to_string());
        self.send(Msg::SUB(protocol::SubPackage{ is_sub: false, int_key })).await
    }
    pub async fn set_node(&mut self, is_node: bool) -> Result<(), ()>{
        self.send(Msg::NODE(protocol::NodePackage{is_node})).await
    }
}

async fn socket_listener(mut socket: WebSocketStream<MaybeTlsStream<TcpStream>>, upstream_sender: Sender<Vec<u8>>, mut downstream_receiver: Receiver<Vec<u8>>){
    loop {
        tokio::select!{
            inc = socket.next() => {
                match inc{
                    None => { break }
                    Some(n) => {
                        match n{
                            Ok(msg) => {
                                match msg{
                                    Message::Binary(bin) => {
                                        if let Err(_) =  upstream_sender.send(bin).await{ break }
                                    }
                                    _ => {}
                                }
                            }
                            Err(_) => { break }
                        }
                    }
                }
            }
            outc = downstream_receiver.recv() => {
                match outc{
                    None => { break }
                    Some(bin) => {
                        if let Err(_) = socket.send(Message::Binary(bin)).await{ break }
                    }
                }
            }
        }
    }
    downstream_receiver.close();
    match socket.close(None).await{
        Ok(_) => {}
        Err(_) => {}
    };
}

impl Into<protocol::Package> for Msg{
    fn into(self) -> Package {
        match self{
            Msg::KeyUpdate(key_update) => {
                let str_key = keys::KEY_UPDATE.to_string();
                let int_key = protocol::hash(&str_key);
                protocol::Package::Msg(protocol::MsgPackage{
                    int_key,
                    str_key,
                    payload: key_update.into(),
                })
            }
            Msg::SUB(sub) => { protocol::Package::Sub(sub) }
            Msg::NODE(node) => { protocol::Package::Node(node) }
        }
    }
}

impl TryFrom<protocol::Package> for Msg{
    type Error = ();

    fn try_from(value: Package) -> Result<Self, Self::Error> {
        match value{
            Package::Sub(sub) => { Ok(Self::SUB(sub)) }
            Package::Node(node) => { Ok(Self::NODE(node)) }
            Package::Msg(msg) => {
                match msg.str_key.as_str(){
                    keys::KEY_UPDATE => {
                        match structs::TinkoffApiKey::try_from(msg.payload){
                            Ok(key_update) => {
                                Ok(Self::KeyUpdate(key_update))
                            }
                            Err(_) => { Err(()) }
                        }
                    }
                    _ => {
                        println!("unknown msg str key");
                        Err(())
                    }
                }
            }
        }
    }
}

impl Into<Vec<u8>> for structs::TinkoffApiKey{
    fn into(self) -> Vec<u8> {
        let mut ret: Vec<u8> = Vec::new();
        ret.append(&mut self.time.to_be_bytes().to_vec());
        ret.append(&mut self.name.into_bytes());
        ret.push(0);
        ret.append(&mut self.key.into_bytes());
        ret
    }
}

impl TryFrom<Vec<u8>> for structs::TinkoffApiKey{
    type Error = ();

    fn try_from(bytes: Vec<u8>) -> Result<Self, Self::Error> {
        let mut dst = [0u8; 8];
        dst.clone_from_slice(&bytes[0..8]);
        let time: f64 = f64::from_be_bytes(dst);
        let mut str_ends: usize = 8;
        for (nom, char) in bytes[8..].iter().enumerate(){
            str_ends = nom+8;
            if *char == 0{ break }
        }
        match String::from_utf8(bytes[5..str_ends].to_owned()){
            Ok(name) => {
                match String::from_utf8(bytes[str_ends+1..].to_vec()){
                    Ok(key) => {
                        Ok(Self{name, key, time})
                    }
                    Err(_) => {
                        println!("cant parse key");
                        Err(())
                    }
                }
            }
            Err(_) => {
                println!("cant parse name");
                Err(())
            }        }
    }
}
