/// Type representing a keypress.
#[derive(Eq, PartialEq, Debug, Clone)]
pub enum Key {
    String(String),
    Char(char),
    Bytes(Vec<u8>),
    Byte(u8),
    /// The associated value will be a byte corresponding to the lowercase
    /// letter for the control code. For instance, `^A` will be `Ctrl(b'a')`.
    Ctrl(u8),
    Meta(u8),
    Backspace,
    Escape,
    Up,
    Down,
    Right,
    Left,
    KeypadUp,
    KeypadDown,
    KeypadRight,
    KeypadLeft,
    Home,
    End,
    Insert,
    Delete,
    PageUp,
    PageDown,
    /// The associated value will be the number corresponding to the `F` key.
    /// For instance, `F1` will be `F(1)`.
    F(u8),
}

impl Key {
    /// Returns bytes generated by the given key press.
    #[must_use]
    pub fn into_bytes(self) -> Vec<u8> {
        match self {
            Key::String(s) => s.into_bytes(),
            Key::Char(c) => c.to_string().into_bytes(),
            Key::Bytes(s) => s,
            Key::Byte(c) => vec![c],
            Key::Ctrl(c) => vec![c - b'a' + 1],
            Key::Meta(c) => vec![b'\x1b', c],
            Key::Backspace => b"\x7f".to_vec(),
            Key::Escape => b"\x1b".to_vec(),
            Key::Up => b"\x1b[A".to_vec(),
            Key::Down => b"\x1b[B".to_vec(),
            Key::Right => b"\x1b[C".to_vec(),
            Key::Left => b"\x1b[D".to_vec(),
            Key::KeypadUp => b"\x1bOA".to_vec(),
            Key::KeypadDown => b"\x1bOB".to_vec(),
            Key::KeypadRight => b"\x1bOC".to_vec(),
            Key::KeypadLeft => b"\x1bOD".to_vec(),
            Key::Home => b"\x1b[H".to_vec(),
            Key::End => b"\x1b[F".to_vec(),
            Key::Insert => b"\x1b[2~".to_vec(),
            Key::Delete => b"\x1b[3~".to_vec(),
            Key::PageUp => b"\x1b[5~".to_vec(),
            Key::PageDown => b"\x1b[6~".to_vec(),
            Key::F(c) => match c {
                1 => b"\x1bOP".to_vec(),
                2 => b"\x1bOQ".to_vec(),
                3 => b"\x1bOR".to_vec(),
                4 => b"\x1bOS".to_vec(),
                5 => b"\x1b[15~".to_vec(),
                6 => b"\x1b[17~".to_vec(),
                7 => b"\x1b[18~".to_vec(),
                8 => b"\x1b[19~".to_vec(),
                9 => b"\x1b[20~".to_vec(),
                10 => b"\x1b[21~".to_vec(),
                11 => b"\x1b[23~".to_vec(),
                12 => b"\x1b[24~".to_vec(),
                13 => b"\x1b[25~".to_vec(),
                14 => b"\x1b[26~".to_vec(),
                15 => b"\x1b[28~".to_vec(),
                16 => b"\x1b[29~".to_vec(),
                17 => b"\x1b[31~".to_vec(),
                18 => b"\x1b[32~".to_vec(),
                19 => b"\x1b[33~".to_vec(),
                20 => b"\x1b[34~".to_vec(),
                _ => vec![],
            },
        }
    }
}
