use serde::{Serialize, Deserialize};

use crate::error::Error;

pub trait Response<T> {
    fn parse(self) -> Result<T, Error>;
}

#[derive(Serialize, Deserialize, Debug)]
pub struct BaseResponse {
    #[serde(rename = "ErrorCode")]
    pub error_code: u32,
    #[serde(rename = "ErrorInfo")]
    pub error_info: String,
    #[serde(rename = "ActionStatus")]
    pub action_status: String
}

impl Response<()> for BaseResponse {
    #[inline]
    fn parse(self) -> Result<(), Error> {
        if self.error_code == 0 {
            Ok(())
        } else {
            Err(Error::RemoteError{code: self.error_code, msg: self.error_info})
        }
    }
}

#[derive(Debug, Deserialize)]
pub struct RespHolder<T> {
    #[serde(flatten)]
    error: BaseResponse,
    #[serde(flatten)]
    data: Option<T>
}

impl<T> Response<T> for RespHolder<T> {
    fn parse(self) -> Result<T, Error> {
        if self.error.error_code != 0 {
            Err(Error::RemoteError{
                code: self.error.error_code,
                msg: self.error.error_info
            })
        } else {
            Ok(self.data.expect("tcloud return ok with empty body"))
        }
    }
}