use crate::client::Client;
use serde::{Serialize, Deserialize};


#[derive(Debug, Serialize, Deserialize)]
#[serde(untagged, rename_all = "PascalCase")]
pub enum MessageContent {
    Text {
        #[serde(rename = "Text")]
        text: String
    },
    Location {
        desc: String,
        latitude: f64,
        longitude: f64
    },
    Face {
        index: u32,
        data: String
    },
    Sound {
        // 语音地址
        url: String,
        // 语音标记
        #[serde(rename = "UUID")]
        uuid: String,
        // 语音大小： bytes
        size: u32,
        // 语音长度：秒
        second: u32,
        #[serde(rename = "Download_Flag")]
        // 语音下载方式标记。目前 Download_Flag 取值只能为2，表示可通过Url字段值的 URL 地址直接下载语音。
        download_flag: u32
    },
    Image {
        #[serde(rename = "UUID")]
        uuid: String,
        // 图片格式。JPG = 1，GIF = 2，PNG = 3，BMP = 4，其他 = 255。
        image_format: u32,
        image_info_array: Vec<ImageInfo>,
    },
    File {
        url: String,
        #[serde(rename = "UUID")]
        uuid: String,
        file_size: u32,
        file_name: String,
        #[serde(rename = "Download_Flag")]
        // 下载方式标记。目前 Download_Flag 取值只能为2，表示可通过Url字段值的 URL 地址直接下载语音。
        download_flag: u32
    },
    Video {
        video_url: String,
        #[serde(rename = "VideoUUID")]
        video_uuid: String,
        video_size: u32,
        video_second: u32,
        video_format: String,
        video_download_flag: u32,
        thumb_url: String,
        #[serde(rename = "thumbUUID")]
        thumb_uuid: String,
        thumb_size: u32,
        thumb_width: u32,
        thumb_height: u32,
        thumb_format: String,
        thumb_download_flag: u32
    },
    Custom {
        data: String,
        desc: String,
        ext: String,
        sound: String
    }
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(rename_all = "PascalCase")]
pub struct ImageInfo {
    // 图片类型： 1-原图，2-大图，3-缩略图。
    #[serde(rename = "type")]
    image_type: u32,
    //图片数据大小，单位：字节。
    size: u32,
    width: u32,
    height: u32,
    #[serde(rename = "URL")]
    url: String
}

#[derive(Debug, Serialize, Deserialize)]
pub enum MessageType {
    TIMTextElem,
    TIMLocationElem,
    TIMFaceElem,
    TIMSoundElem,
    TIMImageElem,
    TIMFileElem,
    TIMVideoFileElem,
    TIMCustomElem
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(rename_all = "PascalCase")]
pub struct MessageBody {
    pub msg_type: MessageType,
    pub msg_content: MessageContent
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(untagged)]
pub enum CallbackPolicy {
    /// 禁止发消息前回调
    ForbidBeforeSendMsgCallback,
    /// 禁止发消息后回调
    ForbidAfterSendMsgCallback 
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(untagged)]
pub enum SendPolicy {
    /// 该条消息不计入未读数
    NoUnread,
    /// 该条消息不更新会话列表
    NoLastMsg,
    /// 该条消息的接收方对发送方设置的免打扰选项生效（默认不生效）
    WithMuteNotifications
}


#[derive(Debug, Serialize, Deserialize)]
#[serde(rename_all = "PascalCase")]
pub struct Message {
    /// 1: sync 2:no_sync
    pub sync_other_machine: u32,
    #[serde(rename = "From_Account")]
    pub from_account: String,
    #[serde(rename = "To_Account")]
    pub to_account: String,
    ///0-604800 None: 7天，0：不发送离线
    #[serde(skip_serializing_if = "Option::is_none")]
    pub msg_life_time: Option<u32>,
    ///去重及排序
    #[serde(skip_serializing_if = "Option::is_none")]
    pub msg_seq: Option<u32>,
    /// 消息随机数，后台用于同一秒内的消息去重。请确保该字段填的是随机数
    pub msg_random: u32,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub msg_time_stamp: Option<u32>,
    #[serde(skip_serializing_if = "Vec::is_empty")]
    pub forbid_callback_control: Vec<CallbackPolicy>,
    #[serde(skip_serializing_if = "Vec::is_empty")]
    pub send_msg_control: Vec<SendPolicy>,
    pub msg_body: Vec<MessageBody>,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub cloud_custom_data: Option<String>,
    //暂不实现
    //pub offline_push_info: Option<OfflinePushInfo>,
}

#[derive(Debug, Deserialize)]
#[serde(rename_all = "PascalCase")]
pub struct SendResp {
    pub msg_time: u64,
    pub msg_key: String
}

#[derive(Debug, Serialize, Deserialize)]
#[serde(rename_all = "PascalCase")]
pub struct BatchMessage {
    /// 1: sync 2:no_sync
    pub sync_other_machine: bool,
    #[serde(rename = "From_Account")]
    pub from_account: String,
    #[serde(rename = "To_Account")]
    pub to_account: Vec<String>,
    ///0-604800 None: 7天，0：不发送离线
    #[serde(skip_serializing_if = "Option::is_none")]
    pub msg_life_time: Option<u32>,
    ///去重及排序
    #[serde(skip_serializing_if = "Option::is_none")]
    pub msg_seq: Option<u32>,
    /// 消息随机数，后台用于同一秒内的消息去重。请确保该字段填的是随机数
    pub msg_random: u32,
    #[serde(skip_serializing_if = "Vec::is_empty")]
    pub send_msg_control: Vec<SendPolicy>,
    pub msg_body: MessageBody,
    #[serde(skip_serializing_if = "Option::is_none")]
    pub cloud_custom_data: Option<String>
}

#[derive(Debug, Deserialize)]
#[serde(rename_all = "PascalCase")]
pub struct SendFailAccount {
    pub to_account: String,
    pub error_code: u32
}

#[derive(Debug, Deserialize)]
#[serde(rename_all = "PascalCase")]
pub struct BatchSendResp {
    pub error_list: Option<Vec<SendFailAccount>>,
    pub msg_key: String
}

#[derive(Debug, Deserialize, Serialize)]
#[serde(rename_all = "PascalCase")]
pub struct MessageDraw {
    #[serde(rename = "From_Account")]
    pub from_account: String,
    #[serde(rename = "To_Account")]
    pub to_account: String,
    pub msg_key: String
}

pub struct MessageApi<'a> {
    inner: &'a Client
}

impl<'a> MessageApi<'a> {
    define_api!(send, "/v4/openim/sendmsg", Message, SendResp);
    define_api!(batch_send, "/v4/openim/batchsendmsg", BatchMessage, BatchSendResp);
    // 撤回
    define_api!(draw, "/v4/openim/admin_msgwithdraw", MessageDraw, ());
    // define_api!(import, "v4/openim/importmsg", BatchMessage, BatchSendResp);
    // // 查询消息
    // define_api!(roamnsg, "v4/openim/admin_getroammsg", BatchMessage, BatchSendResp);
    // // 设置为已读
    // define_api!(mark_read, "v4/openim/admin_set_msg_read", BatchMessage, BatchSendResp);
    // // 未读计数
    // define_api!(unread_count, "v4/openim/get_c2c_unread_msg_num", BatchMessage, BatchSendResp);

    pub fn new(client: &'a Client) -> Self {
        Self {inner: client}
    }
}