use std::time::Duration;

use serde::{Serialize, Deserialize};

use crate::{client::Client, error::Error};

#[derive(Debug, Serialize, Deserialize)]
pub struct Account {
    #[serde(rename = "Identifier")]
    pub identifier: String,
    #[serde(rename = "Nick", skip_serializing_if = "Option::is_none")]
    pub nick: Option<String>,
    #[serde(rename = "FaceUrl", skip_serializing_if = "Option::is_none")]
    pub face_url: Option<String>
}


#[derive(Debug, Serialize, Deserialize)]
pub struct Accounts {
    #[serde(rename = "Accounts")]
    pub accounts: Vec<String>
}
impl From<Vec<String>> for Accounts {
    fn from(source: Vec<String>) -> Self {
        Self {accounts: source}
    }
}

#[derive(Debug, Deserialize)]
pub struct AccountsResp {
    #[serde(rename = "FailAccounts")]
    fail_accounts: Vec<String>
}

#[derive(Debug, Serialize, Deserialize)]
struct UserIdItem {
    #[serde(rename = "UserID")]
    user_id: String
}
#[derive(Debug, Serialize, Deserialize)]
pub struct DeleteAccounts {
    #[serde(rename = "DeleteItem")]
    delete_item: Vec<UserIdItem>
}
impl From<Vec<String>> for DeleteAccounts {
    fn from(source: Vec<String>) -> Self {
        let delete_item = source.into_iter()
            .map(|user_id| UserIdItem{user_id})
            .collect();
        Self {delete_item}
    }
}

#[derive(Debug, Deserialize)]
pub struct DeleteRespItem {
    #[serde(rename = "ResultCode")]
    result_code: u32,
    #[serde(rename = "ResultInfo")]
    result_info: String,
    #[serde(rename = "UserID")]
    user_id: String
}

#[derive(Debug, Deserialize)]
pub struct DeleteResp {
    #[serde(rename = "ResultItem")]
    items: Vec<DeleteRespItem>
}

#[derive(Debug, Serialize, Deserialize)]
pub struct CheckUsers {
    #[serde(rename = "CheckItem")]
    user_item: Vec<UserIdItem>
}
impl From<Vec<String>> for CheckUsers {
    fn from(source: Vec<String>) -> Self {
        let user_item = source.into_iter()
            .map(|user_id| UserIdItem{user_id})
            .collect();
        Self {user_item}
    }
}
#[derive(Debug, Deserialize)]
pub struct CheckRespItem {
    #[serde(rename = "ResultCode")]
    result_code: u32,
    #[serde(rename = "ResultInfo")]
    result_info: String,
    #[serde(rename = "UserID")]
    user_id: String,
    #[serde(rename = "AccountStatus")]
    account_status: String
}

#[derive(Debug, Deserialize)]
pub struct CheckResp {
    #[serde(rename = "ResultItem")]
    items: Vec<CheckRespItem>
}

#[derive(Debug, Serialize, Deserialize)]
pub struct Kick {
    #[serde(rename = "Identifier")]
    identifier: String  
}

impl From<String> for Kick {
    fn from(source: String) -> Self {
        Kick {identifier: source}
    }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct OnlineStatus {
    #[serde(rename = "IsNeedDetail")]
    need_detail: u32,
    #[serde(rename = "To_Account")]
    to_account: Vec<String>
}

impl OnlineStatus {
    pub fn new(need_detail: bool, accounts: Vec<String>) -> Self {
        Self {need_detail: if need_detail {1} else {0}, to_account: accounts}
    }
}

#[derive(Debug, Deserialize)]
pub struct OnlineDetail {
    #[serde(rename = "Platform")]
    platform: String,
    #[serde(rename = "Status")]
    status: String
}

#[derive(Debug, Deserialize)]
pub struct OnlineState {
    #[serde(rename = "To_Account")]
    account: String,
    #[serde(rename = "Status")]
    status: String,
    #[serde(rename = "Detail")]
    detail: Option<OnlineDetail>
}

#[derive(Debug, Deserialize)]
pub struct QueryError {
    #[serde(rename = "To_Account")]
    account: String,
    #[serde(rename = "ErrorCode")]
    code: u32
}

#[derive(Debug, Deserialize)]
pub struct QueryResp {
    #[serde(rename = "QueryResult")]
    result: Vec<OnlineState>,
    #[serde(rename = "ErrorList")]
    errors: Vec<QueryError>
}

pub struct AccountApi<'a> {
    inner: &'a Client
}

impl<'a> AccountApi<'a> {
    define_api!(import, "/v4/im_open_login_svc/account_import", Account, ());
    define_api!(multi_import, "/v4/im_open_login_svc/multiaccount_import",Accounts, AccountsResp);
    define_api!(delete, "v4/im_open_login_svc/account_delete", DeleteAccounts, DeleteResp);
    define_api!(check, "v4/im_open_login_svc/account_check", CheckUsers, CheckResp);
    define_api!(kick, "v4/im_open_login_svc/kick", Kick, ());
    define_api!(query_online_status, "v4/openim/query_online_status", OnlineStatus, QueryResp);

    pub fn gen_sign(&self, user_id: &str, expire: Duration) -> Result<String, Error> {
        super::usersig::gen_usersig(
            self.inner.sdkappid, 
            self.inner.key.as_str(),
            user_id, 
            expire
        )
    }

    pub fn new(client: &'a Client) -> Self {
        AccountApi{inner: client}
    }
}