use std::io::stdin;
use std::io::stdout;
use std::io::Stdout;
use std::io::Write;

use termion::input::TermRead;
use termion::raw::IntoRawMode;
use termion::{event::Key, raw::RawTerminal};

pub(crate) struct App<F>
where
    F: FnMut(String) -> Result<(), String>,
{
    history: Vec<String>,
    lookback_position: Option<usize>,
    command: String,
    cursor_pos: usize,
    raw_terminal: RawTerminal<Stdout>,
    handler: F,
}

pub(crate) enum AppAction {
    Continue,
    Error(String),
    Exit,
}

impl<F> App<F>
where
    F: FnMut(String) -> Result<(), String>,
{
    pub fn new(handler: F) -> Self {
        let raw_terminal = stdout().into_raw_mode().unwrap();
        App {
            history: Vec::new(),
            lookback_position: None,
            command: String::new(),
            cursor_pos: 0,
            raw_terminal,
            handler,
        }
    }

    fn command(&mut self, command: String) -> Result<(), String> {
        if command.is_empty() {
            return Ok(());
        }
        let command_to_send = format!("{}\r\n", command);
        (self.handler)(command_to_send)?;
        self.history.push(command);
        self.lookback_position = None;
        Ok(())
    }

    fn nth_last(&self, n: usize) -> Option<String> {
        return Some(self.history.iter().nth_back(n)?.clone());
    }

    fn up(&mut self) -> Option<String> {
        self.lookback_position = match self.lookback_position {
            Some(n) if n < self.history.len() => Some(n + 1),
            None => Some(0),
            _ => (self.lookback_position),
        };

        self.lookback_position?;

        self.nth_last(self.lookback_position.unwrap())
    }

    fn down(&mut self) -> Option<String> {
        self.lookback_position = match self.lookback_position {
            Some(n) if n > 0 => Some(n - 1),
            Some(n) if n == 0 => None,
            None => None,
            _ => unreachable!(),
        };

        self.lookback_position?;

        Some(self.nth_last(self.lookback_position.unwrap()).unwrap())
    }

    pub fn handle_key(&mut self, key: Key) -> AppAction {
        match key {
            Key::Ctrl('d') | Key::Ctrl('c') => return AppAction::Exit,
            Key::Char('\n') => {
                print!("\r\n");
                match self.command(self.command.clone()) {
                    Ok(()) => {}
                    Err(message) => return AppAction::Error(message),
                }
                self.command.clear();
                self.cursor_pos = 0;
            }
            Key::Char(c) => {
                self.command.insert(self.cursor_pos, c);
                self.cursor_pos += 1;
                self.refresh_command();
            }
            Key::Up => {
                self.command = self.up().unwrap_or_else(|| self.command.clone());
                self.swap_command();
                self.cursor_pos = self.command.len();
            }
            Key::Down => {
                self.command = self.down().unwrap_or_else(|| self.command.clone());
                self.swap_command();
                self.cursor_pos = self.command.len();
            }
            Key::Left if self.cursor_pos > 0 => {
                self.cursor_pos -= 1;
                print!("{}", termion::cursor::Left(1));
            }
            Key::Right if self.cursor_pos < self.command.len() => {
                self.cursor_pos += 1;
                print!("{}", termion::cursor::Right(1));
            }
            Key::End => {
                let right_shift = self.command.len() - self.cursor_pos;
                print!(
                    "{}",
                    termion::cursor::Right(right_shift.try_into().unwrap())
                );
                self.cursor_pos = self.command.len();
            }
            Key::Home => {
                print!(
                    "{}",
                    termion::cursor::Left(self.command.len().try_into().unwrap())
                );
                self.cursor_pos = 0;
            }
            Key::Delete if self.cursor_pos < self.command.len() => {
                self.command.remove(self.cursor_pos);
                self.refresh_command();
            }
            Key::Backspace if self.cursor_pos > 0 => {
                print!("{}", termion::cursor::Left(1));
                self.command.remove(self.cursor_pos - 1);
                self.cursor_pos -= 1;
                self.refresh_command();
            }
            _ => (),
        }

        self.raw_terminal.flush().unwrap();

        AppAction::Continue
    }

    fn refresh_command(&mut self) {
        print!("{}\r", termion::clear::CurrentLine);
        print!("{}\r", self.command);
        print!(
            "{}",
            termion::cursor::Right(self.cursor_pos.try_into().unwrap())
        );
    }

    fn swap_command(&mut self) {
        print!("\r{}", termion::clear::CurrentLine);
        print!("{}", self.command);
    }

    pub fn run(mut self) {
        let stdin = stdin();
        for key in stdin.keys() {
            let key = key.unwrap();
            let action = self.handle_key(key);
            match action {
                AppAction::Continue => {}
                AppAction::Exit => {
                    break;
                }
                AppAction::Error(message) => {
                    println!("\n\r{}\n\r", message);
                    break;
                }
            }
        }
    }
}
