use clap::arg;
use std::io::{stdin, stdout, Write};
use std::process::Stdio;
use termion::input::TermRead;
use termion::raw::IntoRawMode;

mod app;

use app::App;
use app::AppAction;

fn main() {
    let matches = clap::Command::new("tcli")
        .about("A CLI enhancement tool")
        .arg_required_else_help(true)
        .allow_external_subcommands(true)
        .allow_invalid_utf8_for_external_subcommands(true)
        .arg(arg!(<COMMAND> ... "CLI to run").allow_invalid_utf8(true))
        .get_matches();
    let command = matches
        .values_of_os("COMMAND")
        .unwrap_or_default()
        .collect::<Vec<_>>();

    let command_start = command.first().unwrap();

    let mut child_command = std::process::Command::new(command_start);

    if command.len() > 1 {
        child_command.args(&command[1..]);
    }

    let child = child_command
        .stdin(Stdio::piped())
        .stdout(Stdio::inherit())
        .spawn()
        .unwrap();

    let mut child_stdin = child.stdin.unwrap();

    let handler = |command: String| -> Result<(), String> {
        child_stdin
            .write_all(command.as_bytes())
            .map_err(|e| format!("Error writing to child process: {:?}", e.kind()))
    };

    let stdin = stdin();
    let stdout = stdout().into_raw_mode().unwrap();

    let mut app = App::new(stdout, handler);

    for key in stdin.keys() {
        let key = key.unwrap();
        let action = app.handle_key(key);
        match action {
            AppAction::Continue => {}
            AppAction::Exit => {
                break;
            }
            AppAction::Error(message) => {
                println!("\n\r{}\n\r", message);
                break;
            }
        }
    }
}
