use std::io::{Stdout, Write};

use termion::{event::Key, raw::RawTerminal};

pub(crate) struct App<F>
where
    F: FnMut(String) -> Result<(), String>,
{
    history: Vec<String>,
    lookback_position: Option<usize>,
    command: String,
    cursor_pos: usize,
    stdout: RawTerminal<Stdout>,
    handler: F,
}

pub(crate) enum AppAction {
    Continue,
    Error(String),
    Exit,
}

impl<F> App<F>
where
    F: FnMut(String) -> Result<(), String>,
{
    pub fn new(stdout: RawTerminal<Stdout>, handler: F) -> Self {
        App {
            history: Vec::new(),
            lookback_position: None,
            command: String::new(),
            cursor_pos: 0,
            stdout,
            handler,
        }
    }

    fn command(&mut self, command: String) -> Result<(), String> {
        if command.is_empty() {
            return Ok(());
        }
        let command_to_send = format!("{}\r\n", command);
        (self.handler)(command_to_send)?;
        self.history.push(command);
        self.lookback_position = None;
        Ok(())
    }

    fn nth_last(&self, n: usize) -> Option<String> {
        return Some(self.history.iter().nth_back(n)?.clone());
    }

    fn up(&mut self) -> Option<String> {
        self.lookback_position = match self.lookback_position {
            Some(n) if n < self.history.len() => Some(n + 1),
            None => Some(0),
            _ => (self.lookback_position),
        };

        if let None = self.lookback_position {
            return None;
        }

        self.nth_last(self.lookback_position.unwrap())
    }

    fn down(&mut self) -> Option<String> {
        self.lookback_position = match self.lookback_position {
            Some(n) if n > 0 => Some(n - 1),
            Some(n) if n == 0 => None,
            None => None,
            _ => unreachable!(),
        };

        if let None = self.lookback_position {
            return None;
        }

        Some(self.nth_last(self.lookback_position.unwrap()).unwrap())
    }

    pub fn handle_key(&mut self, key: Key) -> AppAction {
        match key {
            Key::Ctrl('d') | Key::Ctrl('c') => return AppAction::Exit,
            Key::Char('\n') => {
                match self.command(self.command.clone()) {
                    Ok(()) => {}
                    Err(message) => return AppAction::Error(message),
                }
                self.command.clear();
                self.cursor_pos = 0;
                print!("\r\n")
            }
            Key::Char(c) => {
                self.command.insert(self.cursor_pos, c);
                self.cursor_pos += 1;
                self.refresh_command(true);
            }
            Key::Up => {
                self.command = self.up().unwrap_or(self.command.clone());
                self.swap_command();
                self.cursor_pos = self.command.len();
            }
            Key::Down => {
                self.command = self.down().unwrap_or(self.command.clone());
                self.swap_command();
                self.cursor_pos = self.command.len();
            }
            Key::Left if self.cursor_pos > 0 => {
                self.cursor_pos -= 1;
                write!(self.stdout, "{}", termion::cursor::Left(1)).unwrap();
            }
            Key::Right if self.cursor_pos < self.command.len() => {
                self.cursor_pos += 1;
                write!(self.stdout, "{}", termion::cursor::Right(1)).unwrap();
            }
            Key::End => {
                let right_shift = self.command.len() - self.cursor_pos;
                write!(
                    self.stdout,
                    "{}",
                    termion::cursor::Right(right_shift.try_into().unwrap())
                )
                .unwrap();
                self.cursor_pos = self.command.len();
            }
            Key::Home => {
                write!(
                    self.stdout,
                    "{}",
                    termion::cursor::Left(self.command.len().try_into().unwrap())
                )
                .unwrap();
                self.cursor_pos = 0;
            }
            Key::Delete if self.cursor_pos < self.command.len() => {
                self.command.remove(self.cursor_pos);
                self.refresh_command(false);
            }
            Key::Backspace if self.cursor_pos > 0 => {
                write!(self.stdout, "{}", termion::cursor::Left(1)).unwrap();
                self.command.remove(self.cursor_pos - 1);
                self.cursor_pos -= 1;
                self.refresh_command(false);
            }
            _ => (),
        }

        self.stdout.flush().unwrap();

        AppAction::Continue
    }
    fn refresh_command(&mut self, move_right: bool) {
        write!(
            self.stdout,
            "{}{}{}",
            termion::cursor::Save,
            '\r',
            termion::clear::CurrentLine
        )
        .unwrap();
        print!("{}", self.command);
        write!(self.stdout, "{}", termion::cursor::Restore).unwrap();
        if move_right {
            write!(self.stdout, "{}", termion::cursor::Right(1)).unwrap();
        }
    }

    fn swap_command(&mut self) {
        write!(self.stdout, "{}{}", '\r', termion::clear::CurrentLine).unwrap();
        print!("{}", self.command);
    }
}
