pub extern crate lmdb_zero as lmdb;
pub use std::collections::HashMap;
use std::time::Duration;

///
/// Keeps the hash of all transaction data
/// as well as other things for finding
/// specific transactions
///
#[repr(C)]
#[derive(Debug, Hash, Default, PartialEq, Eq, PartialOrd, Copy, Ord, Clone)]
pub struct TransactionHeader {
    tx_hash: &'static str,
}

///
/// The vehicle for transacting value on
/// the blockchain
///
#[repr(C)]
#[derive(Debug, Hash, Default, PartialEq, Eq, PartialOrd, Copy, Ord, Clone)]
pub struct Transaction {
    pub header: TransactionHeader,
    timestamp: Duration,
    sender: &'static str,
    receiver: &'static str,
    data: &'static str,
}

///
/// Holds info related to the block
/// such as index and hashed block data
///
#[repr(C)]
#[derive(Debug, Default, Hash, PartialEq, Ord, Eq, PartialOrd, Clone, Copy)]
pub struct BlockHeader {
    pub index: u64,
    proof: &'static str,
    block_hash: &'static str,
    time_mined: Duration,
}

///
/// Holds a list of transactions
/// and are added to the blockchain
/// by the miners (Coming soon)
///
#[repr(C)]
#[derive(Debug, Default, Hash, Ord, PartialEq, Eq, Copy, PartialOrd, Clone)]
pub struct Block<'a> {
    pub header: BlockHeader,
    tx_list: &'a [Transaction],
}

///
/// Uses a HashMap of blocks
/// along with the hashed data
/// to ensure immutability and
/// availability once distributed
///
#[derive(Debug, Clone, Default)]
pub struct Blockchain<'a> {
    pub tip: Block<'a>,
    pub map: HashMap<BlockHeader, Block<'a>>,
}
