pub use super::block::*;

unsafe impl lmdb::traits::LmdbRaw for Block<'static> {}
unsafe impl lmdb::traits::LmdbOrdKey for Block<'static> {}

impl Blockchain<'_> {
    ///
    /// Gets the last block mined from the network
    /// and checks to see if ut is already saved.
    /// If not saved, will grab from peer in
    /// peer list
    ///
    /// #Examples
    /// ```
    /// use super::chain::get_tip;
    ///
    /// fn main() {
    ///     let bc = Blockchain::default();
    ///     bc.get_tip();
    /// }
    /// ```
    ///
    pub fn get_tip(&self) {
        // Check network for current block
    }
}

///
/// Read blockchain from file and load
/// into hashmap
///
/// #Examples
/// ```
/// use super::chain::init_blockchain;
///
/// fn main() {
///     let blockchain = init_blockchain("test_blockchaindb").unwrap();
/// }
/// ```
///
pub fn init_blockchain(db_name: &'static str) -> Result<Blockchain, lmdb::Error> {
    // Get handle to blockchain db
    let db: lmdb::Database = get_lmdb_connection(db_name)?;

    // Create read transaction for getting blocks
    let tx = lmdb::ReadTransaction::new(db.env())?;

    // Initialize blockchain structure
    let mut blockchain = Blockchain::default();

    // Create cursor and put on last database entry
    let mut curs = tx.cursor(&db).unwrap();

    // Assign accessor to database
    let access = tx.access();

    // Get last block saved in database
    let last_block: &Block = curs.last::<str, Block>(&access)?.1;

    {
        blockchain.tip = *last_block;
    }

    Ok(blockchain)
}

///
/// Creates a file for storing the blockchain
/// database
///
/// # Examples:
/// ```
/// use super::chain::get_lmdb_connection;
///
/// fn main() {
///     let test_db = get_lmdb_connection("test_blockchaindb");
/// }
/// ```
pub fn get_lmdb_connection(env: &'static str) -> Result<lmdb::Database, lmdb::Error> {
    // Create directory for lmdb
    std::fs::create_dir_all(env).unwrap();

    // Setup lmdb enviornment
    let db_env: lmdb::Environment =
        unsafe { lmdb::EnvBuilder::new()?.open(&env, lmdb::open::Flags::empty(), 0o600)? };

    // Set Database options
    let ops = lmdb::DatabaseOptions::create_map::<Block>();

    // Open database
    let db: lmdb::Database = lmdb::Database::open(db_env, None, &ops)?;

    // Return database handle
    Ok(db)
}
