use std::time::{SystemTime, UNIX_EPOCH};

use chrono::prelude::Utc;
use crypto::mac::MacResult;
use crypto::sha2::Sha256;
use crypto::{digest::Digest, mac::Mac};

use clap::{command, Arg, Command};
use rustc_hash::FxHashMap as HashMap;
use serde_json::Value;

const API_HOST: &str = "dnspod.tencentcloudapi.com";
const SERVICE: &str = "dnspod";

fn main() {
    let matches = command!()
        .propagate_version(true)
        .subcommand_required(true)
        .arg_required_else_help(true)
        .subcommand(
            Command::new("listRecord")
                .about("Get DNS records list under a given domain")
                .arg(
                    Arg::new("domain")
                        .short('d')
                        .long("domain")
                        .takes_value(true)
                        .default_value("bluewater.vip")
                        .help("The target Domain"),
                ),
        )
        .get_matches();

    match matches.subcommand() {
        Some(("listRecord", sub_matches)) => {
            let domain = sub_matches.value_of("domain").unwrap();
            get_record_list(domain);
        }
        _ => unreachable!("Exhausted list of subcommands and subcommand_required prevents `None`"),
    }
}

fn get_record_list(domain: &str) {
    let mut map = HashMap::default();
    map.insert("Domain", domain.to_owned());
    call_tencent_api("DescribeRecordList", map)
}

fn call_tencent_api(action: &str, params: HashMap<&str, String>) {
    let app_secret_path = dirs::home_dir()
        .unwrap()
        .join(".config")
        .join("tcloud")
        .join("app_secret.toml");
    let toml_str = std::fs::read_to_string(app_secret_path).unwrap();
    let v: toml::Value = toml::from_str(toml_str.as_str()).unwrap();
    let api_secret_key = v
        .as_table()
        .unwrap()
        .get("api_secret_key")
        .unwrap()
        .as_str()
        .unwrap();
    let api_secret_id = v
        .as_table()
        .unwrap()
        .get("api_secret_id")
        .unwrap()
        .as_str()
        .unwrap();

    let timestamp = SystemTime::now()
        .duration_since(UNIX_EPOCH)
        .unwrap()
        .as_secs()
        .to_string();

    let utc_date = Utc::now()
        .to_rfc3339()
        .split('T')
        .into_iter()
        .next()
        .unwrap()
        .to_owned();

    let body = map_to_json_str(&params);

    let hashed_body = gen_sha256_hex_lower_case(&body);

    let canonical_request = format!(
        "POST\n/\n\ncontent-type:application/json\nhost:dnspod.tencentcloudapi.com\n\ncontent-type;host\n{hashed_body}");

    let hashed_canonical_request = gen_sha256_hex_lower_case(&canonical_request);
    let credential_scope = format!("{utc_date}/{SERVICE}/tc3_request");
    let str_to_sign =
        format!("TC3-HMAC-SHA256\n{timestamp}\n{credential_scope}\n{hashed_canonical_request}");

    // println!("{str_to_sign}");

    let secret_date = gen_hmac_sha256(format!("TC3{api_secret_key}").as_bytes(), &utc_date);
    let secret_service = gen_hmac_sha256(secret_date.code(), &"dnspod".to_owned());
    let secret_signing = gen_hmac_sha256(secret_service.code(), &"tc3_request".to_owned());
    let signature = hex::encode(gen_hmac_sha256(secret_signing.code(), &str_to_sign).code());

    // println!("{signature}");

    let authz =
            format!("TC3-HMAC-SHA256 Credential={api_secret_id}/{credential_scope},SignedHeaders=content-type;host,Signature={signature}");
    // println!("{authz}");

    let url = format!("https://{API_HOST}");
    let client = reqwest::blocking::Client::new();

    let resp: Value = client
        .post(url)
        .header("Content-Type", "application/json")
        .header("Host", API_HOST)
        .header("X-TC-Action", action)
        .header("X-TC-Timestamp", timestamp)
        .header("X-TC-Version", "2021-03-23")
        .header("Authorization", authz)
        .body(body)
        .send()
        .unwrap()
        .json()
        .unwrap();
    println!("{}", serde_json::to_string_pretty(&resp).unwrap())
}

fn gen_sha256_hex_lower_case(str: &String) -> String {
    let mut hasher = Sha256::new();
    // write input message
    hasher.input_str(&str.as_str());

    // read hash digest
    let hex = hasher.result_str();
    return hex;
}

fn gen_hmac_sha256(key: &[u8], data: &String) -> MacResult {
    let mut hasher = crypto::hmac::Hmac::new(Sha256::new(), key);
    hasher.input(data.as_bytes());
    hasher.result()
}

fn map_to_json_str(map: &HashMap<&str, String>) -> String {
    format!(
        "{{{}}}",
        map.into_iter()
            .map(|(key, value)| format!("\"{key}\":\"{value}\""))
            .collect::<Vec<String>>()
            .join(",")
    )
}

#[cfg(test)]
mod test {
    use std::time::{SystemTime, UNIX_EPOCH};

    use chrono::Utc;
    use rustc_hash::FxHashMap;
    use serde_json::Value;

    use crate::{call_tencent_api, gen_hmac_sha256, gen_sha256_hex_lower_case, map_to_json_str};

    #[test]
    fn time_test() {
        let x = SystemTime::now()
            .duration_since(UNIX_EPOCH)
            .unwrap()
            .as_secs()
            .to_string();
        println!("{}", x);
        println!(
            "{}",
            Utc::now()
                .to_rfc3339()
                .split('T')
                .into_iter()
                .next()
                .unwrap()
        )
    }

    #[test]
    fn sha256_test() {
        let json = gen_sha256_hex_lower_case(&"{\"Domain\":\"bluewater.vip\"}".to_owned());
        println!("{json}")
    }

    #[test]
    fn test_api() {
        let mut m = FxHashMap::default();
        m.insert("Domain", "bluewater.vip".to_owned());
        call_tencent_api("DescribeRecordList", m);
    }

    #[test]
    fn test_all_string() {
        let mut params = FxHashMap::default();
        params.insert("Domain", "bluewater.vip".to_owned());

        let timestamp = SystemTime::now()
            .duration_since(UNIX_EPOCH)
            .unwrap()
            .as_secs()
            .to_string();

        let utc_date = Utc::now()
            .to_rfc3339()
            .split('T')
            .into_iter()
            .next()
            .unwrap()
            .to_owned();

        let body = map_to_json_str(&params);
        assert_eq!(r#"{"Domain":"bluewater.vip"}"#, body);

        let hashed_body = gen_sha256_hex_lower_case(&body);

        let canonical_request = format!(
        "POST\n/\n\ncontent-type:application/json\nhost:dnspod.tencentcloudapi.com\n\ncontent-type;host\n{hashed_body}");

        assert_eq!(
            canonical_request,
            "POST
/

content-type:application/json
host:dnspod.tencentcloudapi.com

content-type;host
cf13291310143d0b2c5c9e434239085b3142d1777fba9b569d2c9dc1ac8136c3"
        );

        let credential_scope = format!("{utc_date}/dnspod/tc3_request");
        let hashed_canonical_request = gen_sha256_hex_lower_case(&canonical_request);
        let str_to_sign =
            format!("TC3-HMAC-SHA256\n{timestamp}\n{credential_scope}\n{hashed_canonical_request}");
        println!("{str_to_sign}");

        let api_secret_key: &str = "av878tPq79xWeXAetgDUP6WfgS6VG5i5";
        let api_secret_id: &str = "AKIDMUMxPbnnnWyrW1fHsDH6eDl381nJnWE1";
        const API_HOST: &str = "dnspod.tencentcloudapi.com";

        let secret_date = gen_hmac_sha256(format!("TC3{api_secret_key}").as_bytes(), &utc_date);
        let secret_service = gen_hmac_sha256(secret_date.code(), &"dnspod".to_owned());
        let secret_signing = gen_hmac_sha256(secret_service.code(), &"tc3_request".to_owned());
        let signature = hex::encode(gen_hmac_sha256(secret_signing.code(), &str_to_sign).code());

        println!("{signature}");

        let authz =
            format!("TC3-HMAC-SHA256 Credential={api_secret_id}/{credential_scope},SignedHeaders=content-type;host,Signature={signature}");
        println!("{authz}");

        let url = format!("https://{API_HOST}");
        let client = reqwest::blocking::Client::new();

        let resp: Value = client
            .post(url)
            .header("Content-Type", "application/json")
            .header("Host", API_HOST)
            .header("X-TC-Action", "DescribeRecordList")
            .header("X-TC-Timestamp", timestamp)
            .header("X-TC-Version", "2021-03-23")
            .header("Authorization", authz)
            .body(body)
            .send()
            .unwrap()
            .json()
            .unwrap();
        println!("{}", serde_json::to_string_pretty(&resp).unwrap())
    }
}
