use crate::attribute::Attribute;
use crate::misc::{Source, Zone};
use crate::usize_wrapper::{CardID, PlayerID};
use std::collections::{BTreeMap, BTreeSet};

#[derive(Serialize, Deserialize, Clone, Debug, PartialEq, Eq)]
pub struct Card<CardType, AttName>
where
    AttName: Ord,
{
    id: CardID,
    owner: Option<PlayerID>,
    pub src: String,
    pub zone: Zone,
    pub card_type: CardType,
    pub created_by: Source,
    attributes: BTreeMap<AttName, Attribute>,
}

impl<CardType, AttName> Card<CardType, AttName>
where
    AttName: Default + Ord,
{
    pub fn new(
        id: CardID,
        card_type: CardType,
        owner: Option<PlayerID>,
    ) -> Card<CardType, AttName> {
        Card {
            id,
            owner: owner,
            src: "Default".to_string(),
            zone: Zone::Nowhere,
            card_type,
            created_by: Source::Game,
            attributes: Default::default(),
        }
    }

    pub fn id(&self) -> CardID {
        self.id
    }

    pub fn owner(&self) -> Option<PlayerID> {
        self.owner
    }

    pub fn int(&self, att: AttName) -> Option<isize> {
        self.attributes.get(&att)?.int()
    }

    pub fn text(&self, att: AttName) -> Option<&str> {
        self.attributes.get(&att)?.text()
    }

    pub fn flag(&self, att: AttName) -> Option<bool> {
        self.attributes.get(&att)?.flag()
    }

    pub fn set(&self, att: AttName) -> Option<&BTreeSet<Attribute>> {
        self.attributes.get(&att)?.set()
    }

    pub fn att(&mut self, name: AttName, att: Attribute) {
        self.attributes.insert(name, att);
    }
}
