use crate::usize_wrapper::{CardID, DeckID, GridID, PlayerID};
use crate::cardinal::Point;
#[derive(Serialize, Deserialize, Copy, Clone, PartialEq, Eq, Hash, Debug)]
pub enum Zone {
    Grid(GridID, Point, Point),
    Deck(DeckID),
    Nowhere,
}

impl Zone {
    pub fn is_board(&self) -> bool {
        match self {
            Zone::Grid(_, _, _) => true,
            _ => false,
        }
    }

    pub fn is_deck(&self) -> bool {
        match self {
            Zone::Deck(_) => true,
            _ => false,
        }
    }

    pub fn is_nowhere(&self) -> bool {
        match self {
            Zone::Nowhere => true,
            _ => false,
        }
    }
}

#[derive(Serialize, Deserialize, Copy, Clone, PartialEq, Eq, Debug)]
pub enum Source {
    Game,
    Player(PlayerID),
    Card(CardID),
    Deck(DeckID),
    Grid(GridID),
    GridPoint(GridID, Point),
}

#[derive(Serialize, Deserialize, Clone, PartialEq, Eq, Default)]
pub struct Phase<PhaseType>
where
    PhaseType: PartialEq + Eq,
{
    player: PlayerID,
    phase_type: PhaseType,
    time_limit: usize,
    time_current: usize,
}

impl<PhaseType> Phase<PhaseType>
where
    PhaseType: PartialEq + Eq,
{
    pub fn new(phase_type: PhaseType, time_limit: usize, player: PlayerID) -> Phase<PhaseType> {
        Phase {
            player,
            phase_type,
            time_limit,
            time_current: 0,
        }
    }

    pub fn player(&self) -> PlayerID {
        self.player
    }

    pub fn phase_type(&self) -> &PhaseType {
        &self.phase_type
    }

    pub fn is(&self, pt: PhaseType) -> bool {
        self.phase_type == pt
    }
}
#[derive(Serialize, Deserialize, Clone, PartialEq, Eq, Default)]
pub struct LayeredEffect<Effect, EffectLayer> {
    pub layer: EffectLayer,
    pub effect: Effect,
    pub player: Option<PlayerID>,
    pub card: Option<CardID>,
}
