#![allow(unused_imports)]
#![cfg_attr(rustfmt, rustfmt_skip)]
/* THIS FILE IS AUTOMATICALLY GENERATED. DO NOT EDIT */
use crate::{Client, ClientBuilder, Credentials, Retry};
use anyhow::Error;
use serde_json::Value;
use std::time::Duration;
use crate::util::urlencode;

/// Queue Service
///
/// The queue service is responsible for accepting tasks and tracking their state
/// as they are executed by workers, in order to ensure they are eventually
/// resolved.
///
/// ## Artifact Storage Types
///
/// * **Object artifacts** contain arbitrary data, stored via the object service.
/// * **Redirect artifacts**, will redirect the caller to URL when fetched
/// with a a 303 (See Other) response.  Clients will not apply any kind of
/// authentication to that URL.
/// * **Link artifacts**, will be treated as if the caller requested the linked
/// artifact on the same task.  Links may be chained, but cycles are forbidden.
/// The caller must have scopes for the linked artifact, or a 403 response will
/// be returned.
/// * **Error artifacts**, only consists of meta-data which the queue will
/// store for you. These artifacts are only meant to indicate that you the
/// worker or the task failed to generate a specific artifact, that you
/// would otherwise have uploaded. For example docker-worker will upload an
/// error artifact, if the file it was supposed to upload doesn't exists or
/// turns out to be a directory. Clients requesting an error artifact will
/// get a `424` (Failed Dependency) response. This is mainly designed to
/// ensure that dependent tasks can distinguish between artifacts that were
/// suppose to be generated and artifacts for which the name is misspelled.
/// * **S3 artifacts** are used for static files which will be
/// stored on S3. When creating an S3 artifact the queue will return a
/// pre-signed URL to which you can do a `PUT` request to upload your
/// artifact. Note that `PUT` request **must** specify the `content-length`
/// header and **must** give the `content-type` header the same value as in
/// the request to `createArtifact`. S3 artifacts will be deprecated soon,
/// and users should prefer object artifacts instead.
///
/// ## Artifact immutability
///
/// Generally speaking you cannot overwrite an artifact when created.
/// But if you repeat the request with the same properties the request will
/// succeed as the operation is idempotent.
/// This is useful if you need to refresh a signed URL while uploading.
/// Do not abuse this to overwrite artifacts created by another entity!
/// Such as worker-host overwriting artifact created by worker-code.
///
/// The queue defines the following *immutability special cases*:
///
/// * A `reference` artifact can replace an existing `reference` artifact.
/// * A `link` artifact can replace an existing `reference` artifact.
/// * Any artifact's `expires` can be extended (made later, but not earlier).
pub struct Queue (Client);

#[allow(non_snake_case)]
impl Queue {
    /// Create a new undefined instance, based on the given client.
    pub fn new<CB: Into<ClientBuilder>>(client_builder: CB) -> Result<Self, Error> {
        Ok(Self(client_builder
            .into()
            .path_prefix("api/queue/v1/")
            .build()?))
    }

    /// Ping Server
    /// 
    /// Respond without doing anything.
    /// This endpoint is used to check that the service is up.
    pub async fn ping(&self) -> Result<(), Error> {
        let method = "GET";
        let (path, query) = Self::ping_details();
        let body = None;
        let resp = self.0.request(method, path, query, body).await?;
        resp.bytes().await?;
        Ok(())
    }

    /// Generate an unsigned URL for the ping endpoint
    pub fn ping_url(&self) -> Result<String, Error> {
        let (path, query) = Self::ping_details();
        self.0.make_url(path, query)
    }

    /// Generate a signed URL for the ping endpoint
    pub fn ping_signed_url(&self, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::ping_details();
        self.0.make_signed_url(path, query, ttl)
    }

    /// Determine the HTTP request details for ping
    fn ping_details<'a>() -> (&'static str, Option<Vec<(&'static str, &'a str)>>) {
        let path = "ping";
        let query = None;

        (path, query)
    }

    /// Get Task Definition
    /// 
    /// This end-point will return the task-definition. Notice that the task
    /// definition may have been modified by queue, if an optional property is
    /// not specified the queue may provide a default value.
    pub async fn task(&self, taskId: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::task_details(taskId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the task endpoint
    pub fn task_url(&self, taskId: &str) -> Result<String, Error> {
        let (path, query) = Self::task_details(taskId);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the task endpoint
    pub fn task_signed_url(&self, taskId: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::task_details(taskId);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for task
    fn task_details<'a>(taskId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}", urlencode(taskId));
        let query = None;

        (path, query)
    }

    /// Get task status
    /// 
    /// Get task status structure from `taskId`
    pub async fn status(&self, taskId: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::status_details(taskId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the status endpoint
    pub fn status_url(&self, taskId: &str) -> Result<String, Error> {
        let (path, query) = Self::status_details(taskId);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the status endpoint
    pub fn status_signed_url(&self, taskId: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::status_details(taskId);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for status
    fn status_details<'a>(taskId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/status", urlencode(taskId));
        let query = None;

        (path, query)
    }

    /// List Task Group
    /// 
    /// List tasks sharing the same `taskGroupId`.
    /// 
    /// As a task-group may contain an unbounded number of tasks, this end-point
    /// may return a `continuationToken`. To continue listing tasks you must call
    /// the `listTaskGroup` again with the `continuationToken` as the
    /// query-string option `continuationToken`.
    /// 
    /// By default this end-point will try to return up to 1000 members in one
    /// request. But it **may return less**, even if more tasks are available.
    /// It may also return a `continuationToken` even though there are no more
    /// results. However, you can only be sure to have seen all results if you
    /// keep calling `listTaskGroup` with the last `continuationToken` until you
    /// get a result without a `continuationToken`.
    /// 
    /// If you are not interested in listing all the members at once, you may
    /// use the query-string option `limit` to return fewer.
    pub async fn listTaskGroup(&self, taskGroupId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::listTaskGroup_details(taskGroupId, continuationToken, limit);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the listTaskGroup endpoint
    pub fn listTaskGroup_url(&self, taskGroupId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<String, Error> {
        let (path, query) = Self::listTaskGroup_details(taskGroupId, continuationToken, limit);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the listTaskGroup endpoint
    pub fn listTaskGroup_signed_url(&self, taskGroupId: &str, continuationToken: Option<&str>, limit: Option<&str>, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::listTaskGroup_details(taskGroupId, continuationToken, limit);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for listTaskGroup
    fn listTaskGroup_details<'a>(taskGroupId: &'a str, continuationToken: Option<&'a str>, limit: Option<&'a str>) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task-group/{}/list", urlencode(taskGroupId));
        let mut query = None;
        if let Some(q) = continuationToken {
            query.get_or_insert_with(Vec::new).push(("continuationToken", q));
        }
        if let Some(q) = limit {
            query.get_or_insert_with(Vec::new).push(("limit", q));
        }

        (path, query)
    }

    /// List Dependent Tasks
    /// 
    /// List tasks that depend on the given `taskId`.
    /// 
    /// As many tasks from different task-groups may dependent on a single tasks,
    /// this end-point may return a `continuationToken`. To continue listing
    /// tasks you must call `listDependentTasks` again with the
    /// `continuationToken` as the query-string option `continuationToken`.
    /// 
    /// By default this end-point will try to return up to 1000 tasks in one
    /// request. But it **may return less**, even if more tasks are available.
    /// It may also return a `continuationToken` even though there are no more
    /// results. However, you can only be sure to have seen all results if you
    /// keep calling `listDependentTasks` with the last `continuationToken` until
    /// you get a result without a `continuationToken`.
    /// 
    /// If you are not interested in listing all the tasks at once, you may
    /// use the query-string option `limit` to return fewer.
    pub async fn listDependentTasks(&self, taskId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::listDependentTasks_details(taskId, continuationToken, limit);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the listDependentTasks endpoint
    pub fn listDependentTasks_url(&self, taskId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<String, Error> {
        let (path, query) = Self::listDependentTasks_details(taskId, continuationToken, limit);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the listDependentTasks endpoint
    pub fn listDependentTasks_signed_url(&self, taskId: &str, continuationToken: Option<&str>, limit: Option<&str>, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::listDependentTasks_details(taskId, continuationToken, limit);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for listDependentTasks
    fn listDependentTasks_details<'a>(taskId: &'a str, continuationToken: Option<&'a str>, limit: Option<&'a str>) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/dependents", urlencode(taskId));
        let mut query = None;
        if let Some(q) = continuationToken {
            query.get_or_insert_with(Vec::new).push(("continuationToken", q));
        }
        if let Some(q) = limit {
            query.get_or_insert_with(Vec::new).push(("limit", q));
        }

        (path, query)
    }

    /// Create New Task
    /// 
    /// Create a new task, this is an **idempotent** operation, so repeat it if
    /// you get an internal server error or network connection is dropped.
    /// 
    /// **Task `deadline`**: the deadline property can be no more than 5 days
    /// into the future. This is to limit the amount of pending tasks not being
    /// taken care of. Ideally, you should use a much shorter deadline.
    /// 
    /// **Task expiration**: the `expires` property must be greater than the
    /// task `deadline`. If not provided it will default to `deadline` + one
    /// year. Notice that artifacts created by a task must expire before the
    /// task's expiration.
    /// 
    /// **Task specific routing-keys**: using the `task.routes` property you may
    /// define task specific routing-keys. If a task has a task specific 
    /// routing-key: `<route>`, then when the AMQP message about the task is
    /// published, the message will be CC'ed with the routing-key: 
    /// `route.<route>`. This is useful if you want another component to listen
    /// for completed tasks you have posted.  The caller must have scope
    /// `queue:route:<route>` for each route.
    /// 
    /// **Dependencies**: any tasks referenced in `task.dependencies` must have
    /// already been created at the time of this call.
    /// 
    /// **Scopes**: Note that the scopes required to complete this API call depend
    /// on the content of the `scopes`, `routes`, `schedulerId`, `priority`,
    /// `provisionerId`, and `workerType` properties of the task definition.
    pub async fn createTask(&self, taskId: &str, payload: &Value) -> Result<Value, Error> {
        let method = "PUT";
        let (path, query) = Self::createTask_details(taskId);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for createTask
    fn createTask_details<'a>(taskId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}", urlencode(taskId));
        let query = None;

        (path, query)
    }

    /// Schedule Defined Task
    /// 
    /// scheduleTask will schedule a task to be executed, even if it has
    /// unresolved dependencies. A task would otherwise only be scheduled if
    /// its dependencies were resolved.
    /// 
    /// This is useful if you have defined a task that depends on itself or on
    /// some other task that has not been resolved, but you wish the task to be
    /// scheduled immediately.
    /// 
    /// This will announce the task as pending and workers will be allowed to
    /// claim it and resolve the task.
    /// 
    /// **Note** this operation is **idempotent** and will not fail or complain
    /// if called with a `taskId` that is already scheduled, or even resolved.
    /// To reschedule a task previously resolved, use `rerunTask`.
    pub async fn scheduleTask(&self, taskId: &str) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::scheduleTask_details(taskId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for scheduleTask
    fn scheduleTask_details<'a>(taskId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/schedule", urlencode(taskId));
        let query = None;

        (path, query)
    }

    /// Rerun a Resolved Task
    /// 
    /// This method _reruns_ a previously resolved task, even if it was
    /// _completed_. This is useful if your task completes unsuccessfully, and
    /// you just want to run it from scratch again. This will also reset the
    /// number of `retries` allowed. It will schedule a task that is _unscheduled_
    /// regardless of the state of its dependencies.
    /// 
    /// This method is deprecated in favour of creating a new task with the same
    /// task definition (but with a new taskId).
    /// 
    /// Remember that `retries` in the task status counts the number of runs that
    /// the queue have started because the worker stopped responding, for example
    /// because a spot node died.
    /// 
    /// **Remark** this operation is idempotent: if it is invoked for a task that
    /// is `pending` or `running`, it will just return the current task status.
    pub async fn rerunTask(&self, taskId: &str) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::rerunTask_details(taskId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for rerunTask
    fn rerunTask_details<'a>(taskId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/rerun", urlencode(taskId));
        let query = None;

        (path, query)
    }

    /// Cancel Task
    /// 
    /// This method will cancel a task that is either `unscheduled`, `pending` or
    /// `running`. It will resolve the current run as `exception` with
    /// `reasonResolved` set to `canceled`. If the task isn't scheduled yet, ie.
    /// it doesn't have any runs, an initial run will be added and resolved as
    /// described above. Hence, after canceling a task, it cannot be scheduled
    /// with `queue.scheduleTask`, but a new run can be created with
    /// `queue.rerun`. These semantics is equivalent to calling
    /// `queue.scheduleTask` immediately followed by `queue.cancelTask`.
    /// 
    /// **Remark** this operation is idempotent, if you try to cancel a task that
    /// isn't `unscheduled`, `pending` or `running`, this operation will just
    /// return the current task status.
    pub async fn cancelTask(&self, taskId: &str) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::cancelTask_details(taskId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for cancelTask
    fn cancelTask_details<'a>(taskId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/cancel", urlencode(taskId));
        let query = None;

        (path, query)
    }

    /// Claim Work
    /// 
    /// Claim pending task(s) for the given task queue.
    /// 
    /// If any work is available (even if fewer than the requested number of
    /// tasks, this will return immediately. Otherwise, it will block for tens of
    /// seconds waiting for work.  If no work appears, it will return an emtpy
    /// list of tasks.  Callers should sleep a short while (to avoid denial of
    /// service in an error condition) and call the endpoint again.  This is a
    /// simple implementation of "long polling".
    pub async fn claimWork(&self, taskQueueId: &str, payload: &Value) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::claimWork_details(taskQueueId);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for claimWork
    fn claimWork_details<'a>(taskQueueId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("claim-work/{}", urlencode(taskQueueId));
        let query = None;

        (path, query)
    }

    /// Claim Task
    /// 
    /// claim a task - never documented
    pub async fn claimTask(&self, taskId: &str, runId: &str, payload: &Value) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::claimTask_details(taskId, runId);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for claimTask
    fn claimTask_details<'a>(taskId: &'a str, runId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/claim", urlencode(taskId), urlencode(runId));
        let query = None;

        (path, query)
    }

    /// Reclaim task
    /// 
    /// Refresh the claim for a specific `runId` for given `taskId`. This updates
    /// the `takenUntil` property and returns a new set of temporary credentials
    /// for performing requests on behalf of the task. These credentials should
    /// be used in-place of the credentials returned by `claimWork`.
    /// 
    /// The `reclaimTask` requests serves to:
    ///  * Postpone `takenUntil` preventing the queue from resolving
    ///    `claim-expired`,
    ///  * Refresh temporary credentials used for processing the task, and
    ///  * Abort execution if the task/run have been resolved.
    /// 
    /// If the `takenUntil` timestamp is exceeded the queue will resolve the run
    /// as _exception_ with reason `claim-expired`, and proceeded to retry to the
    /// task. This ensures that tasks are retried, even if workers disappear
    /// without warning.
    /// 
    /// If the task is resolved, this end-point will return `409` reporting
    /// `RequestConflict`. This typically happens if the task have been canceled
    /// or the `task.deadline` have been exceeded. If reclaiming fails, workers
    /// should abort the task and forget about the given `runId`. There is no
    /// need to resolve the run or upload artifacts.
    pub async fn reclaimTask(&self, taskId: &str, runId: &str) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::reclaimTask_details(taskId, runId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for reclaimTask
    fn reclaimTask_details<'a>(taskId: &'a str, runId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/reclaim", urlencode(taskId), urlencode(runId));
        let query = None;

        (path, query)
    }

    /// Report Run Completed
    /// 
    /// Report a task completed, resolving the run as `completed`.
    pub async fn reportCompleted(&self, taskId: &str, runId: &str) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::reportCompleted_details(taskId, runId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for reportCompleted
    fn reportCompleted_details<'a>(taskId: &'a str, runId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/completed", urlencode(taskId), urlencode(runId));
        let query = None;

        (path, query)
    }

    /// Report Run Failed
    /// 
    /// Report a run failed, resolving the run as `failed`. Use this to resolve
    /// a run that failed because the task specific code behaved unexpectedly.
    /// For example the task exited non-zero, or didn't produce expected output.
    /// 
    /// Do not use this if the task couldn't be run because if malformed
    /// payload, or other unexpected condition. In these cases we have a task
    /// exception, which should be reported with `reportException`.
    pub async fn reportFailed(&self, taskId: &str, runId: &str) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::reportFailed_details(taskId, runId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for reportFailed
    fn reportFailed_details<'a>(taskId: &'a str, runId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/failed", urlencode(taskId), urlencode(runId));
        let query = None;

        (path, query)
    }

    /// Report Task Exception
    /// 
    /// Resolve a run as _exception_. Generally, you will want to report tasks as
    /// failed instead of exception. You should `reportException` if,
    /// 
    ///   * The `task.payload` is invalid,
    ///   * Non-existent resources are referenced,
    ///   * Declared actions cannot be executed due to unavailable resources,
    ///   * The worker had to shutdown prematurely,
    ///   * The worker experienced an unknown error, or,
    ///   * The task explicitly requested a retry.
    /// 
    /// Do not use this to signal that some user-specified code crashed for any
    /// reason specific to this code. If user-specific code hits a resource that
    /// is temporarily unavailable worker should report task _failed_.
    pub async fn reportException(&self, taskId: &str, runId: &str, payload: &Value) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::reportException_details(taskId, runId);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for reportException
    fn reportException_details<'a>(taskId: &'a str, runId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/exception", urlencode(taskId), urlencode(runId));
        let query = None;

        (path, query)
    }

    /// Create Artifact
    /// 
    /// This API end-point creates an artifact for a specific run of a task. This
    /// should **only** be used by a worker currently operating on this task, or
    /// from a process running within the task (ie. on the worker).
    /// 
    /// All artifacts must specify when they expire. The queue will
    /// automatically take care of deleting artifacts past their
    /// expiration point. This feature makes it feasible to upload large
    /// intermediate artifacts from data processing applications, as the
    /// artifacts can be set to expire a few days later.
    pub async fn createArtifact(&self, taskId: &str, runId: &str, name: &str, payload: &Value) -> Result<Value, Error> {
        let method = "POST";
        let (path, query) = Self::createArtifact_details(taskId, runId, name);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for createArtifact
    fn createArtifact_details<'a>(taskId: &'a str, runId: &'a str, name: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/artifacts/{}", urlencode(taskId), urlencode(runId), urlencode(name));
        let query = None;

        (path, query)
    }

    /// Finish Artifact
    /// 
    /// This endpoint marks an artifact as present for the given task, and
    /// should be called when the artifact data is fully uploaded.
    /// 
    /// The storage types `reference`, `link`, and `error` do not need to
    /// be finished, as they are finished immediately by `createArtifact`.
    /// The storage type `s3` does not support this functionality and cannot
    /// be finished.  In all such cases, calling this method is an input error
    /// (400).
    pub async fn finishArtifact(&self, taskId: &str, runId: &str, name: &str, payload: &Value) -> Result<(), Error> {
        let method = "PUT";
        let (path, query) = Self::finishArtifact_details(taskId, runId, name);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        resp.bytes().await?;
        Ok(())
    }

    /// Determine the HTTP request details for finishArtifact
    fn finishArtifact_details<'a>(taskId: &'a str, runId: &'a str, name: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/artifacts/{}", urlencode(taskId), urlencode(runId), urlencode(name));
        let query = None;

        (path, query)
    }

    /// Get Artifact Data from Run
    /// 
    /// Get artifact by `<name>` from a specific run.
    /// 
    /// **Artifact Access**, in order to get an artifact you need the scope
    /// `queue:get-artifact:<name>`, where `<name>` is the name of the artifact.
    /// To allow access to fetch artifacts with a client like `curl` or a web
    /// browser, without using Taskcluster credentials, include a scope in the
    /// `anonymous` role.  The convention is to include
    /// `queue:get-artifact:public/*`.
    /// 
    /// **Response**: the HTTP response to this method is a 303 redirect to the
    /// URL from which the artifact can be downloaded.  The body of that response
    /// contains the data described in the output schema, contianing the same URL.
    /// Callers are encouraged to use whichever method of gathering the URL is
    /// most convenient.  Standard HTTP clients will follow the redirect, while
    /// API client libraries will return the JSON body.
    /// 
    /// In order to download an artifact the following must be done:
    /// 
    /// 1. Obtain queue url.  Building a signed url with a taskcluster client is
    /// recommended
    /// 1. Make a GET request which does not follow redirects
    /// 1. In all cases, if specified, the
    /// x-taskcluster-location-{content,transfer}-{sha256,length} values must be
    /// validated to be equal to the Content-Length and Sha256 checksum of the
    /// final artifact downloaded. as well as any intermediate redirects
    /// 1. If this response is a 500-series error, retry using an exponential
    /// backoff.  No more than 5 retries should be attempted
    /// 1. If this response is a 400-series error, treat it appropriately for
    /// your context.  This might be an error in responding to this request or
    /// an Error storage type body.  This request should not be retried.
    /// 1. If this response is a 200-series response, the response body is the artifact.
    /// If the x-taskcluster-location-{content,transfer}-{sha256,length} and
    /// x-taskcluster-location-content-encoding are specified, they should match
    /// this response body
    /// 1. If the response type is a 300-series redirect, the artifact will be at the
    /// location specified by the `Location` header.  There are multiple artifact storage
    /// types which use a 300-series redirect.
    /// 1. For all redirects followed, the user must verify that the content-sha256, content-length,
    /// transfer-sha256, transfer-length and content-encoding match every further request.  The final
    /// artifact must also be validated against the values specified in the original queue response
    /// 1. Caching of requests with an x-taskcluster-artifact-storage-type value of `reference`
    /// must not occur
    /// 
    /// **Headers**
    /// The following important headers are set on the response to this method:
    /// 
    /// * location: the url of the artifact if a redirect is to be performed
    /// * x-taskcluster-artifact-storage-type: the storage type.  Example: s3
    pub async fn getArtifact(&self, taskId: &str, runId: &str, name: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::getArtifact_details(taskId, runId, name);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the getArtifact endpoint
    pub fn getArtifact_url(&self, taskId: &str, runId: &str, name: &str) -> Result<String, Error> {
        let (path, query) = Self::getArtifact_details(taskId, runId, name);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the getArtifact endpoint
    pub fn getArtifact_signed_url(&self, taskId: &str, runId: &str, name: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::getArtifact_details(taskId, runId, name);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for getArtifact
    fn getArtifact_details<'a>(taskId: &'a str, runId: &'a str, name: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/artifacts/{}", urlencode(taskId), urlencode(runId), urlencode(name));
        let query = None;

        (path, query)
    }

    /// Get Artifact Data from Latest Run
    /// 
    /// Get artifact by `<name>` from the last run of a task.
    /// 
    /// **Artifact Access**, in order to get an artifact you need the scope
    /// `queue:get-artifact:<name>`, where `<name>` is the name of the artifact.
    /// To allow access to fetch artifacts with a client like `curl` or a web
    /// browser, without using Taskcluster credentials, include a scope in the
    /// `anonymous` role.  The convention is to include
    /// `queue:get-artifact:public/*`.
    /// 
    /// **API Clients**, this method will redirect you to the artifact, if it is
    /// stored externally. Either way, the response may not be JSON. So API
    /// client users might want to generate a signed URL for this end-point and
    /// use that URL with a normal HTTP client.
    /// 
    /// **Remark**, this end-point is slightly slower than
    /// `queue.getArtifact`, so consider that if you already know the `runId` of
    /// the latest run. Otherwise, just us the most convenient API end-point.
    pub async fn getLatestArtifact(&self, taskId: &str, name: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::getLatestArtifact_details(taskId, name);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the getLatestArtifact endpoint
    pub fn getLatestArtifact_url(&self, taskId: &str, name: &str) -> Result<String, Error> {
        let (path, query) = Self::getLatestArtifact_details(taskId, name);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the getLatestArtifact endpoint
    pub fn getLatestArtifact_signed_url(&self, taskId: &str, name: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::getLatestArtifact_details(taskId, name);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for getLatestArtifact
    fn getLatestArtifact_details<'a>(taskId: &'a str, name: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/artifacts/{}", urlencode(taskId), urlencode(name));
        let query = None;

        (path, query)
    }

    /// Get Artifacts from Run
    /// 
    /// Returns a list of artifacts and associated meta-data for a given run.
    /// 
    /// As a task may have many artifacts paging may be necessary. If this
    /// end-point returns a `continuationToken`, you should call the end-point
    /// again with the `continuationToken` as the query-string option:
    /// `continuationToken`.
    /// 
    /// By default this end-point will list up-to 1000 artifacts in a single page
    /// you may limit this with the query-string parameter `limit`.
    pub async fn listArtifacts(&self, taskId: &str, runId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::listArtifacts_details(taskId, runId, continuationToken, limit);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the listArtifacts endpoint
    pub fn listArtifacts_url(&self, taskId: &str, runId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<String, Error> {
        let (path, query) = Self::listArtifacts_details(taskId, runId, continuationToken, limit);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the listArtifacts endpoint
    pub fn listArtifacts_signed_url(&self, taskId: &str, runId: &str, continuationToken: Option<&str>, limit: Option<&str>, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::listArtifacts_details(taskId, runId, continuationToken, limit);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for listArtifacts
    fn listArtifacts_details<'a>(taskId: &'a str, runId: &'a str, continuationToken: Option<&'a str>, limit: Option<&'a str>) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/artifacts", urlencode(taskId), urlencode(runId));
        let mut query = None;
        if let Some(q) = continuationToken {
            query.get_or_insert_with(Vec::new).push(("continuationToken", q));
        }
        if let Some(q) = limit {
            query.get_or_insert_with(Vec::new).push(("limit", q));
        }

        (path, query)
    }

    /// Get Artifacts from Latest Run
    /// 
    /// Returns a list of artifacts and associated meta-data for the latest run
    /// from the given task.
    /// 
    /// As a task may have many artifacts paging may be necessary. If this
    /// end-point returns a `continuationToken`, you should call the end-point
    /// again with the `continuationToken` as the query-string option:
    /// `continuationToken`.
    /// 
    /// By default this end-point will list up-to 1000 artifacts in a single page
    /// you may limit this with the query-string parameter `limit`.
    pub async fn listLatestArtifacts(&self, taskId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::listLatestArtifacts_details(taskId, continuationToken, limit);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the listLatestArtifacts endpoint
    pub fn listLatestArtifacts_url(&self, taskId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<String, Error> {
        let (path, query) = Self::listLatestArtifacts_details(taskId, continuationToken, limit);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the listLatestArtifacts endpoint
    pub fn listLatestArtifacts_signed_url(&self, taskId: &str, continuationToken: Option<&str>, limit: Option<&str>, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::listLatestArtifacts_details(taskId, continuationToken, limit);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for listLatestArtifacts
    fn listLatestArtifacts_details<'a>(taskId: &'a str, continuationToken: Option<&'a str>, limit: Option<&'a str>) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/artifacts", urlencode(taskId));
        let mut query = None;
        if let Some(q) = continuationToken {
            query.get_or_insert_with(Vec::new).push(("continuationToken", q));
        }
        if let Some(q) = limit {
            query.get_or_insert_with(Vec::new).push(("limit", q));
        }

        (path, query)
    }

    /// Get Artifact Information From Run
    /// 
    /// Returns associated metadata for a given artifact, in the given task run.
    /// The metadata is the same as that returned from `listArtifacts`, and does
    /// not grant access to the artifact data.
    /// 
    /// Note that this method does *not* automatically follow link artifacts.
    pub async fn artifactInfo(&self, taskId: &str, runId: &str, name: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::artifactInfo_details(taskId, runId, name);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the artifactInfo endpoint
    pub fn artifactInfo_url(&self, taskId: &str, runId: &str, name: &str) -> Result<String, Error> {
        let (path, query) = Self::artifactInfo_details(taskId, runId, name);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the artifactInfo endpoint
    pub fn artifactInfo_signed_url(&self, taskId: &str, runId: &str, name: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::artifactInfo_details(taskId, runId, name);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for artifactInfo
    fn artifactInfo_details<'a>(taskId: &'a str, runId: &'a str, name: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/artifact-info/{}", urlencode(taskId), urlencode(runId), urlencode(name));
        let query = None;

        (path, query)
    }

    /// Get Artifact Information From Latest Run
    /// 
    /// Returns associated metadata for a given artifact, in the latest run of the
    /// task.  The metadata is the same as that returned from `listArtifacts`,
    /// and does not grant access to the artifact data.
    /// 
    /// Note that this method does *not* automatically follow link artifacts.
    pub async fn latestArtifactInfo(&self, taskId: &str, name: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::latestArtifactInfo_details(taskId, name);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the latestArtifactInfo endpoint
    pub fn latestArtifactInfo_url(&self, taskId: &str, name: &str) -> Result<String, Error> {
        let (path, query) = Self::latestArtifactInfo_details(taskId, name);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the latestArtifactInfo endpoint
    pub fn latestArtifactInfo_signed_url(&self, taskId: &str, name: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::latestArtifactInfo_details(taskId, name);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for latestArtifactInfo
    fn latestArtifactInfo_details<'a>(taskId: &'a str, name: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/artifact-info/{}", urlencode(taskId), urlencode(name));
        let query = None;

        (path, query)
    }

    /// Get Artifact Content From Run
    /// 
    /// Returns information about the content of the artifact, in the given task run.
    /// 
    /// Depending on the storage type, the endpoint returns the content of the artifact
    /// or enough information to access that content.
    /// 
    /// This method follows link artifacts, so it will not return content
    /// for a link artifact.
    pub async fn artifact(&self, taskId: &str, runId: &str, name: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::artifact_details(taskId, runId, name);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the artifact endpoint
    pub fn artifact_url(&self, taskId: &str, runId: &str, name: &str) -> Result<String, Error> {
        let (path, query) = Self::artifact_details(taskId, runId, name);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the artifact endpoint
    pub fn artifact_signed_url(&self, taskId: &str, runId: &str, name: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::artifact_details(taskId, runId, name);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for artifact
    fn artifact_details<'a>(taskId: &'a str, runId: &'a str, name: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/runs/{}/artifact-content/{}", urlencode(taskId), urlencode(runId), urlencode(name));
        let query = None;

        (path, query)
    }

    /// Get Artifact Content From Latest Run
    /// 
    /// Returns information about the content of the artifact, in the latest task run.
    /// 
    /// Depending on the storage type, the endpoint returns the content of the artifact
    /// or enough information to access that content.
    /// 
    /// This method follows link artifacts, so it will not return content
    /// for a link artifact.
    pub async fn latestArtifact(&self, taskId: &str, name: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::latestArtifact_details(taskId, name);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the latestArtifact endpoint
    pub fn latestArtifact_url(&self, taskId: &str, name: &str) -> Result<String, Error> {
        let (path, query) = Self::latestArtifact_details(taskId, name);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the latestArtifact endpoint
    pub fn latestArtifact_signed_url(&self, taskId: &str, name: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::latestArtifact_details(taskId, name);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for latestArtifact
    fn latestArtifact_details<'a>(taskId: &'a str, name: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task/{}/artifact-content/{}", urlencode(taskId), urlencode(name));
        let query = None;

        (path, query)
    }

    /// Get a list of all active provisioners
    /// 
    /// Get all active provisioners.
    /// 
    /// The term "provisioner" is taken broadly to mean anything with a provisionerId.
    /// This does not necessarily mean there is an associated service performing any
    /// provisioning activity.
    /// 
    /// The response is paged. If this end-point returns a `continuationToken`, you
    /// should call the end-point again with the `continuationToken` as a query-string
    /// option. By default this end-point will list up to 1000 provisioners in a single
    /// page. You may limit this with the query-string parameter `limit`.
    pub async fn listProvisioners(&self, continuationToken: Option<&str>, limit: Option<&str>) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::listProvisioners_details(continuationToken, limit);
        let body = None;
        let resp = self.0.request(method, path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the listProvisioners endpoint
    pub fn listProvisioners_url(&self, continuationToken: Option<&str>, limit: Option<&str>) -> Result<String, Error> {
        let (path, query) = Self::listProvisioners_details(continuationToken, limit);
        self.0.make_url(path, query)
    }

    /// Generate a signed URL for the listProvisioners endpoint
    pub fn listProvisioners_signed_url(&self, continuationToken: Option<&str>, limit: Option<&str>, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::listProvisioners_details(continuationToken, limit);
        self.0.make_signed_url(path, query, ttl)
    }

    /// Determine the HTTP request details for listProvisioners
    fn listProvisioners_details<'a>(continuationToken: Option<&'a str>, limit: Option<&'a str>) -> (&'static str, Option<Vec<(&'static str, &'a str)>>) {
        let path = "provisioners";
        let mut query = None;
        if let Some(q) = continuationToken {
            query.get_or_insert_with(Vec::new).push(("continuationToken", q));
        }
        if let Some(q) = limit {
            query.get_or_insert_with(Vec::new).push(("limit", q));
        }

        (path, query)
    }

    /// Get an active provisioner
    /// 
    /// Get an active provisioner.
    /// 
    /// The term "provisioner" is taken broadly to mean anything with a provisionerId.
    /// This does not necessarily mean there is an associated service performing any
    /// provisioning activity.
    pub async fn getProvisioner(&self, provisionerId: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::getProvisioner_details(provisionerId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the getProvisioner endpoint
    pub fn getProvisioner_url(&self, provisionerId: &str) -> Result<String, Error> {
        let (path, query) = Self::getProvisioner_details(provisionerId);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the getProvisioner endpoint
    pub fn getProvisioner_signed_url(&self, provisionerId: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::getProvisioner_details(provisionerId);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for getProvisioner
    fn getProvisioner_details<'a>(provisionerId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}", urlencode(provisionerId));
        let query = None;

        (path, query)
    }

    /// Update a provisioner
    /// 
    /// Declare a provisioner, supplying some details about it.
    /// 
    /// `declareProvisioner` allows updating one or more properties of a provisioner as long as the required scopes are
    /// possessed. For example, a request to update the `my-provisioner`
    /// provisioner with a body `{description: 'This provisioner is great'}` would require you to have the scope
    /// `queue:declare-provisioner:my-provisioner#description`.
    /// 
    /// The term "provisioner" is taken broadly to mean anything with a provisionerId.
    /// This does not necessarily mean there is an associated service performing any
    /// provisioning activity.
    pub async fn declareProvisioner(&self, provisionerId: &str, payload: &Value) -> Result<Value, Error> {
        let method = "PUT";
        let (path, query) = Self::declareProvisioner_details(provisionerId);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for declareProvisioner
    fn declareProvisioner_details<'a>(provisionerId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}", urlencode(provisionerId));
        let query = None;

        (path, query)
    }

    /// Get Number of Pending Tasks
    /// 
    /// Get an approximate number of pending tasks for the given `taskQueueId`.
    /// 
    /// The underlying Azure Storage Queues only promises to give us an estimate.
    /// Furthermore, we cache the result in memory for 20 seconds. So consumers
    /// should be no means expect this to be an accurate number.
    /// It is, however, a solid estimate of the number of pending tasks.
    pub async fn pendingTasks(&self, taskQueueId: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::pendingTasks_details(taskQueueId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the pendingTasks endpoint
    pub fn pendingTasks_url(&self, taskQueueId: &str) -> Result<String, Error> {
        let (path, query) = Self::pendingTasks_details(taskQueueId);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the pendingTasks endpoint
    pub fn pendingTasks_signed_url(&self, taskQueueId: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::pendingTasks_details(taskQueueId);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for pendingTasks
    fn pendingTasks_details<'a>(taskQueueId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("pending/{}", urlencode(taskQueueId));
        let query = None;

        (path, query)
    }

    /// Get a list of all active worker-types
    /// 
    /// Get all active worker-types for the given provisioner.
    /// 
    /// The response is paged. If this end-point returns a `continuationToken`, you
    /// should call the end-point again with the `continuationToken` as a query-string
    /// option. By default this end-point will list up to 1000 worker-types in a single
    /// page. You may limit this with the query-string parameter `limit`.
    pub async fn listWorkerTypes(&self, provisionerId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::listWorkerTypes_details(provisionerId, continuationToken, limit);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the listWorkerTypes endpoint
    pub fn listWorkerTypes_url(&self, provisionerId: &str, continuationToken: Option<&str>, limit: Option<&str>) -> Result<String, Error> {
        let (path, query) = Self::listWorkerTypes_details(provisionerId, continuationToken, limit);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the listWorkerTypes endpoint
    pub fn listWorkerTypes_signed_url(&self, provisionerId: &str, continuationToken: Option<&str>, limit: Option<&str>, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::listWorkerTypes_details(provisionerId, continuationToken, limit);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for listWorkerTypes
    fn listWorkerTypes_details<'a>(provisionerId: &'a str, continuationToken: Option<&'a str>, limit: Option<&'a str>) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}/worker-types", urlencode(provisionerId));
        let mut query = None;
        if let Some(q) = continuationToken {
            query.get_or_insert_with(Vec::new).push(("continuationToken", q));
        }
        if let Some(q) = limit {
            query.get_or_insert_with(Vec::new).push(("limit", q));
        }

        (path, query)
    }

    /// Get a worker-type
    /// 
    /// Get a worker-type from a provisioner.
    pub async fn getWorkerType(&self, provisionerId: &str, workerType: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::getWorkerType_details(provisionerId, workerType);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the getWorkerType endpoint
    pub fn getWorkerType_url(&self, provisionerId: &str, workerType: &str) -> Result<String, Error> {
        let (path, query) = Self::getWorkerType_details(provisionerId, workerType);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the getWorkerType endpoint
    pub fn getWorkerType_signed_url(&self, provisionerId: &str, workerType: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::getWorkerType_details(provisionerId, workerType);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for getWorkerType
    fn getWorkerType_details<'a>(provisionerId: &'a str, workerType: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}/worker-types/{}", urlencode(provisionerId), urlencode(workerType));
        let query = None;

        (path, query)
    }

    /// Update a worker-type
    /// 
    /// Declare a workerType, supplying some details about it.
    /// 
    /// `declareWorkerType` allows updating one or more properties of a worker-type as long as the required scopes are
    /// possessed. For example, a request to update the `highmem` worker-type within the `my-provisioner`
    /// provisioner with a body `{description: 'This worker type is great'}` would require you to have the scope
    /// `queue:declare-worker-type:my-provisioner/highmem#description`.
    pub async fn declareWorkerType(&self, provisionerId: &str, workerType: &str, payload: &Value) -> Result<Value, Error> {
        let method = "PUT";
        let (path, query) = Self::declareWorkerType_details(provisionerId, workerType);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for declareWorkerType
    fn declareWorkerType_details<'a>(provisionerId: &'a str, workerType: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}/worker-types/{}", urlencode(provisionerId), urlencode(workerType));
        let query = None;

        (path, query)
    }

    /// Get a list of all active task queues
    /// 
    /// Get all active task queues.
    /// 
    /// The response is paged. If this end-point returns a `continuationToken`, you
    /// should call the end-point again with the `continuationToken` as a query-string
    /// option. By default this end-point will list up to 1000 task queues in a single
    /// page. You may limit this with the query-string parameter `limit`.
    pub async fn listTaskQueues(&self, continuationToken: Option<&str>, limit: Option<&str>) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::listTaskQueues_details(continuationToken, limit);
        let body = None;
        let resp = self.0.request(method, path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the listTaskQueues endpoint
    pub fn listTaskQueues_url(&self, continuationToken: Option<&str>, limit: Option<&str>) -> Result<String, Error> {
        let (path, query) = Self::listTaskQueues_details(continuationToken, limit);
        self.0.make_url(path, query)
    }

    /// Generate a signed URL for the listTaskQueues endpoint
    pub fn listTaskQueues_signed_url(&self, continuationToken: Option<&str>, limit: Option<&str>, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::listTaskQueues_details(continuationToken, limit);
        self.0.make_signed_url(path, query, ttl)
    }

    /// Determine the HTTP request details for listTaskQueues
    fn listTaskQueues_details<'a>(continuationToken: Option<&'a str>, limit: Option<&'a str>) -> (&'static str, Option<Vec<(&'static str, &'a str)>>) {
        let path = "task-queues";
        let mut query = None;
        if let Some(q) = continuationToken {
            query.get_or_insert_with(Vec::new).push(("continuationToken", q));
        }
        if let Some(q) = limit {
            query.get_or_insert_with(Vec::new).push(("limit", q));
        }

        (path, query)
    }

    /// Get a task queue
    /// 
    /// Get a task queue.
    pub async fn getTaskQueue(&self, taskQueueId: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::getTaskQueue_details(taskQueueId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the getTaskQueue endpoint
    pub fn getTaskQueue_url(&self, taskQueueId: &str) -> Result<String, Error> {
        let (path, query) = Self::getTaskQueue_details(taskQueueId);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the getTaskQueue endpoint
    pub fn getTaskQueue_signed_url(&self, taskQueueId: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::getTaskQueue_details(taskQueueId);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for getTaskQueue
    fn getTaskQueue_details<'a>(taskQueueId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("task-queues/{}", urlencode(taskQueueId));
        let query = None;

        (path, query)
    }

    /// Get a list of all active workers of a workerType
    /// 
    /// Get a list of all active workers of a workerType.
    /// 
    /// `listWorkers` allows a response to be filtered by quarantined and non quarantined workers.
    /// To filter the query, you should call the end-point with `quarantined` as a query-string option with a
    /// true or false value.
    /// 
    /// The response is paged. If this end-point returns a `continuationToken`, you
    /// should call the end-point again with the `continuationToken` as a query-string
    /// option. By default this end-point will list up to 1000 workers in a single
    /// page. You may limit this with the query-string parameter `limit`.
    pub async fn listWorkers(&self, provisionerId: &str, workerType: &str, continuationToken: Option<&str>, limit: Option<&str>, quarantined: Option<&str>) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::listWorkers_details(provisionerId, workerType, continuationToken, limit, quarantined);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the listWorkers endpoint
    pub fn listWorkers_url(&self, provisionerId: &str, workerType: &str, continuationToken: Option<&str>, limit: Option<&str>, quarantined: Option<&str>) -> Result<String, Error> {
        let (path, query) = Self::listWorkers_details(provisionerId, workerType, continuationToken, limit, quarantined);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the listWorkers endpoint
    pub fn listWorkers_signed_url(&self, provisionerId: &str, workerType: &str, continuationToken: Option<&str>, limit: Option<&str>, quarantined: Option<&str>, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::listWorkers_details(provisionerId, workerType, continuationToken, limit, quarantined);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for listWorkers
    fn listWorkers_details<'a>(provisionerId: &'a str, workerType: &'a str, continuationToken: Option<&'a str>, limit: Option<&'a str>, quarantined: Option<&'a str>) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}/worker-types/{}/workers", urlencode(provisionerId), urlencode(workerType));
        let mut query = None;
        if let Some(q) = continuationToken {
            query.get_or_insert_with(Vec::new).push(("continuationToken", q));
        }
        if let Some(q) = limit {
            query.get_or_insert_with(Vec::new).push(("limit", q));
        }
        if let Some(q) = quarantined {
            query.get_or_insert_with(Vec::new).push(("quarantined", q));
        }

        (path, query)
    }

    /// Get a worker-type
    /// 
    /// Get a worker from a worker-type.
    pub async fn getWorker(&self, provisionerId: &str, workerType: &str, workerGroup: &str, workerId: &str) -> Result<Value, Error> {
        let method = "GET";
        let (path, query) = Self::getWorker_details(provisionerId, workerType, workerGroup, workerId);
        let body = None;
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Generate an unsigned URL for the getWorker endpoint
    pub fn getWorker_url(&self, provisionerId: &str, workerType: &str, workerGroup: &str, workerId: &str) -> Result<String, Error> {
        let (path, query) = Self::getWorker_details(provisionerId, workerType, workerGroup, workerId);
        self.0.make_url(&path, query)
    }

    /// Generate a signed URL for the getWorker endpoint
    pub fn getWorker_signed_url(&self, provisionerId: &str, workerType: &str, workerGroup: &str, workerId: &str, ttl: Duration) -> Result<String, Error> {
        let (path, query) = Self::getWorker_details(provisionerId, workerType, workerGroup, workerId);
        self.0.make_signed_url(&path, query, ttl)
    }

    /// Determine the HTTP request details for getWorker
    fn getWorker_details<'a>(provisionerId: &'a str, workerType: &'a str, workerGroup: &'a str, workerId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}/worker-types/{}/workers/{}/{}", urlencode(provisionerId), urlencode(workerType), urlencode(workerGroup), urlencode(workerId));
        let query = None;

        (path, query)
    }

    /// Quarantine a worker
    /// 
    /// Quarantine a worker
    pub async fn quarantineWorker(&self, provisionerId: &str, workerType: &str, workerGroup: &str, workerId: &str, payload: &Value) -> Result<Value, Error> {
        let method = "PUT";
        let (path, query) = Self::quarantineWorker_details(provisionerId, workerType, workerGroup, workerId);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for quarantineWorker
    fn quarantineWorker_details<'a>(provisionerId: &'a str, workerType: &'a str, workerGroup: &'a str, workerId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}/worker-types/{}/workers/{}/{}", urlencode(provisionerId), urlencode(workerType), urlencode(workerGroup), urlencode(workerId));
        let query = None;

        (path, query)
    }

    /// Declare a worker
    /// 
    /// Declare a worker, supplying some details about it.
    /// 
    /// `declareWorker` allows updating one or more properties of a worker as long as the required scopes are
    /// possessed.
    pub async fn declareWorker(&self, provisionerId: &str, workerType: &str, workerGroup: &str, workerId: &str, payload: &Value) -> Result<Value, Error> {
        let method = "PUT";
        let (path, query) = Self::declareWorker_details(provisionerId, workerType, workerGroup, workerId);
        let body = Some(payload);
        let resp = self.0.request(method, &path, query, body).await?;
        Ok(resp.json().await?)
    }

    /// Determine the HTTP request details for declareWorker
    fn declareWorker_details<'a>(provisionerId: &'a str, workerType: &'a str, workerGroup: &'a str, workerId: &'a str) -> (String, Option<Vec<(&'static str, &'a str)>>) {
        let path = format!("provisioners/{}/worker-types/{}/{}/{}", urlencode(provisionerId), urlencode(workerType), urlencode(workerGroup), urlencode(workerId));
        let query = None;

        (path, query)
    }
}
