#![doc = include_str!("../README.md")]
use clap::{ArgEnum, Parser};
use heck::ToUpperCamelCase;
use itertools::Itertools;
use regex::Regex;
use serde::Serialize;
#[cfg(not(target_os = "windows"))]
use std::os::unix::fs::symlink;
#[cfg(target_os = "windows")]
use std::os::windows::fs::symlink_dir;
use std::{
    collections::HashMap,
    fs::{copy, read_to_string, write},
    path::{Path, PathBuf},
    process::Command,
};
use tempfile::{tempdir, TempDir};
use tinytemplate::TinyTemplate;

#[derive(ArgEnum, Clone, Debug)]
enum Visibility {
    Public,
    Crate,
}

#[derive(Debug, Parser)]
#[clap(author, version, about)]
struct Args {
    /// The `tailwind.config.json` file for your project.
    #[clap(short, long)]
    tailwind_config: PathBuf,
    /// The `css` file used as input for tailwindcss.
    #[clap(short, long)]
    input: PathBuf,
    /// The path to the file which should be generated. The directory that
    /// contains the file must already exist.
    #[clap(short, long)]
    output: PathBuf,
    /// The visibility of the generated structs and consts. This defaults to
    /// "crate", which marks them as `pub(crate)`.
    #[clap(short, long, arg_enum, default_value_t = Visibility::Crate)]
    visibility: Visibility,
    /// If this is true, then rustfmt will be run on the generated
    /// code. Defaults to false.
    #[clap(short, long)]
    rustfmt: bool,
}

fn main() {
    let args = Args::parse();

    let structs = structs_from_css_file(&args.tailwind_config, &args.input);

    let visibility = match args.visibility {
        Visibility::Public => "pub",
        Visibility::Crate => "pub(crate)",
    };
    let mut code = format!(
        "#![allow(dead_code)]\n\n/* generated by tailwindcss-to-rust {} */\n\n",
        env!("CARGO_PKG_VERSION"),
    );
    code.push_str(&modifiers_code(visibility));
    code.push_str(&classes_code(&structs, visibility));

    write(&args.output, &code).unwrap_or_else(|e| {
        panic!(
            "Could not write generated code to {}: {}",
            args.output.to_string_lossy(),
            e
        )
    });

    if args.rustfmt {
        Command::new("rustfmt")
            .arg(&args.output)
            .output()
            .unwrap_or_else(|e| panic!("Could not run rustfmt: {}", e));
    }
}

fn structs_from_css_file(
    tw: &Path,
    input: &Path,
) -> HashMap<&'static str, HashMap<String, String>> {
    let tempdir = tempdir().unwrap_or_else(|e| panic!("Could not create temp dir: {}", e));
    let css = write_css_file(tw, input, &tempdir);
    let content = read_to_string(&css).unwrap_or_else(|e| {
        panic!(
            "Could not read css file at {}: {}",
            css.to_string_lossy(),
            e
        )
    });

    // Matches any class like ".foo" but not classes with modifiers like
    // ".foo::-moz-foo".
    let re = Regex::new(r"(?ms)^\.([^ :]+) \{").unwrap();
    let classes = re
        .captures_iter(&content)
        .map(|c| c.get(1).unwrap().as_str())
        .filter(|c| !c.starts_with('-'))
        .collect::<Vec<_>>();

    let groups = make_groups();

    // { group => { struct_field => css_name } }
    let mut structs: HashMap<&'static str, HashMap<String, String>> = HashMap::new();
    for class in classes {
        let clean = class.replace('\\', "");
        let group = group_for(&groups, &clean);

        let members = if let Some(s) = structs.get_mut(group) {
            s
        } else {
            let members = HashMap::new();
            structs.insert(group, members);
            structs.get_mut(group).unwrap()
        };
        members.insert(class_to_rust_var(class), clean);
    }

    structs
}

fn write_css_file(tw: &Path, input: &Path, tempdir: &TempDir) -> PathBuf {
    let mut config = read_to_string(tw).unwrap_or_else(|e| {
        panic!(
            "Could not read tailwind config at {}: {}",
            tw.to_string_lossy(),
            e
        )
    });
    // Putting our new config items at the very end ensures they'll override
    // any existing keys with the same names.
    config = Regex::new(r"};?\s*\z")
        .unwrap()
        .replace(&config, "\nsafelist: [ { pattern: /.*/ } ],\n}\n")
        .to_string();

    let mut node_modules = tw.parent().unwrap().to_path_buf();
    node_modules.push("node_modules");
    if node_modules.exists() {
        let mut new_node_modules = tempdir.path().to_path_buf();
        new_node_modules.push("node_modules");
        #[cfg(target_os = "windows")]
        symlink_dir(&node_modules, &new_node_modules).unwrap_or_else(|e| {
            panic!(
                "Could not make a symlink from {} to {} directory: {}",
                node_modules.to_string_lossy(),
                new_node_modules.to_string_lossy(),
                e
            )
        });
        #[cfg(not(target_os = "windows"))]
        symlink(&node_modules, &new_node_modules).unwrap_or_else(|e| {
            panic!(
                "Could not make a symlink from {} to {} directory: {}",
                node_modules.to_string_lossy(),
                new_node_modules.to_string_lossy(),
                e
            )
        });
    }

    let mut new_tw = tempdir.path().to_path_buf();
    new_tw.push("tailwind.config.js");
    write(&new_tw, config).unwrap_or_else(|e| {
        panic!(
            "Could not write new tailwind config to {}: {}",
            new_tw.to_string_lossy(),
            e
        )
    });

    let mut new_css = tempdir.path().to_path_buf();
    new_css.push("tailwind.css");
    copy(&input, &new_css).unwrap_or_else(|e| {
        panic!(
            "Could not copy {} to {}: {}",
            input.to_string_lossy(),
            new_css.to_string_lossy(),
            e,
        )
    });

    let mut css = tempdir.path().to_path_buf();
    css.push("tailwind-output.css");
    Command::new("tailwindcss")
        .arg("--config")
        .arg(&new_tw)
        .arg("--input")
        .arg(&new_css)
        .arg("--output")
        .arg(&css)
        .output()
        .unwrap_or_else(|e| panic!("Could not run tailwindcss: {}", e));

    css
}

fn group_for(groups: &HashMap<&'static str, &'static str>, class: &str) -> &'static str {
    if let Some(g) = groups.get(class) {
        return *g;
    }
    // There are a bunch of classes that Tailwind generates that aren't
    // explicitly mentioned in the docs (maybe for back compat?).
    if class.starts_with("aspect-") {
        "aspect"
    } else if class.starts_with("backdrop-") {
        "filters"
    } else if class.starts_with("bg-opacity-") {
        "backgrounds"
    } else if class.starts_with("blur-") {
        "filters"
    } else if class.starts_with("border-opacity-") {
        "borders"
    } else if class.starts_with("declaration-") {
        "typography"
    } else if class.starts_with("decoration-") {
        "layout"
    } else if class.starts_with("filter") {
        "filters"
    } else if class.starts_with("flex-") {
        "flex_and_grid"
    } else if class.starts_with("line-clamp-") {
        "line_clamp"
    } else if class.starts_with("overflow-") {
        "layout"
    } else if class.starts_with("placeholder-") {
        "placeholders"
    } else if class.starts_with("prose") {
        "prose"
    } else if class.starts_with("ring-opacity-") {
        "borders"
    } else if class.starts_with("text-opacity-") {
        "typography"
    } else if class.starts_with("transform") {
        "transforms"
    } else {
        "unknown"
    }
}

// Taken from
// https://tailwindcss.com/docs/hover-focus-and-other-states#quick-reference
static MODIFIERS: &[&str] = &[
    "hover",
    "focus",
    "focus-within",
    "focus-visible",
    "active",
    "visited",
    "target",
    "first",
    "last",
    "only",
    "odd",
    "even",
    "first-of-type",
    "last-of-type",
    "only-of-type",
    "empty",
    "disabled",
    "checked",
    "indeterminate",
    "default",
    "required",
    "valid",
    "invalid",
    "in-range",
    "out-of-range",
    "placeholder-shown",
    "autofill",
    "read-only",
    "open",
    "before",
    "after",
    "first-letter",
    "first-line",
    "marker",
    "selection",
    "file",
    "placeholder",
    "sm",
    "md",
    "lg",
    "xl",
    "2xl",
    "dark",
    "portrait",
    "landscape",
    "motion-safe",
    "motion-reduce",
    "print",
    "rtl",
    "ltr",
];

#[derive(Serialize)]
struct ModifiersContext {
    visibility: &'static str,
    fields: Vec<(String, String)>,
}

const MODIFIERS_TEMPLATE: &str = r#"
#[derive(Clone, Copy)]
{visibility} struct Modifiers \{
{{ for field in fields }}    {visibility} {field.0}: &'static str,
{{ endfor }}
}

{visibility} const M: Modifiers = Modifiers \{
{{ for field in fields }}    {field.0}: "{field.1}",
{{ endfor }}
};

"#;

fn modifiers_code(visibility: &'static str) -> String {
    let mut tt = TinyTemplate::new();
    tt.add_template("modifiers", MODIFIERS_TEMPLATE)
        .expect("Could not parse modifiers template");

    let context = ModifiersContext {
        visibility,
        fields: MODIFIERS
            .iter()
            .map(|m| (class_to_rust_var(m), m.to_string()))
            .sorted()
            .collect(),
    };
    tt.render("modifiers", &context)
        .expect("Could not render modifiers template")
}

#[derive(Serialize)]
struct StructsContext<'a> {
    visibility: &'static str,
    name: String,
    const_name: String,
    fields: Vec<(&'a str, &'a str)>,
}

const ONE_STRUCT_TEMPLATE: &str = r#"
#[derive(Clone, Copy)]
{visibility} struct {name} \{
{{ for field in fields }}    {visibility} {field.0}: &'static str,
{{ endfor }}
}

{visibility} const {const_name}: {name} = {name} \{
{{ for field in fields }}    {field.0}: "{field.1}",
{{ endfor }}
};

"#;

#[derive(Serialize)]
struct GroupsContext<'a> {
    visibility: &'static str,
    groups: Vec<(&'a str, String, String)>,
}

const GROUPS_TEMPLATE: &str = r#"
#[derive(Clone, Copy)]
{visibility} struct C \{
{{ for group in groups }}    {visibility} {group.0}: {group.1},
{{ endfor }}
}

{visibility} const C: C = C \{
{{ for group in groups }}    {group.0}: {group.2},
{{ endfor }}
};

"#;

fn classes_code(
    structs: &HashMap<&str, HashMap<String, String>>,
    visibility: &'static str,
) -> String {
    let mut tt = TinyTemplate::new();
    tt.add_template("structs", ONE_STRUCT_TEMPLATE)
        .expect("Could not parse one struct template");

    let mut groups: Vec<(&str, String, String)> = vec![];
    let mut code = String::new();
    for group in structs.keys().sorted() {
        let members = structs.get(group).unwrap();
        let context = StructsContext {
            visibility,
            name: group.to_upper_camel_case(),
            const_name: group.to_uppercase(),
            fields: structs
                .get(group)
                .unwrap()
                .keys()
                .sorted()
                .map(|f| (maybe_escape_field(f), members.get(f).unwrap().as_str()))
                .collect(),
        };
        code.push_str(
            &tt.render("structs", &context)
                .expect("Could not render structs template"),
        );
        groups.push((
            group_abbreviation(group),
            group.to_upper_camel_case(),
            group.to_uppercase(),
        ));
    }

    tt.add_template("groups", GROUPS_TEMPLATE)
        .expect("Could not parse groups template");

    let context = GroupsContext { visibility, groups };
    code.push_str(
        &tt.render("groups", &context)
            .expect("Could not render groups template"),
    );

    code
}

fn maybe_escape_field(field: &str) -> &str {
    if field == "static" {
        r##"r#static"##
    } else {
        field
    }
}

fn class_to_rust_var(class: &str) -> String {
    class
        .starts_with('2')
        .then(|| class.replacen('2', "two_", 1))
        .as_deref()
        .unwrap_or(class)
        .replace('\\', "")
        .replace('-', "_")
        .replace('.', "_p_")
        .replace("/", "_of_")
}

fn group_abbreviation(group: &str) -> &str {
    match group {
        "animation" => "anim",
        "backgrounds" => "bg",
        "flex_and_grid" => "fg",
        "interactivity" => "intr",
        "line_clamp" => "lc",
        "spacing" => "spc",
        "tables" => "tbl",
        "transforms" => "trn",
        _ => &group[0..3],
    }
}

fn make_groups() -> HashMap<&'static str, &'static str> {
    HashMap::from([
        ("not-sr-only", "accessibility"),
        ("sr-only", "accessibility"),
        ("animate-bounce", "animation"),
        ("animate-none", "animation"),
        ("animate-ping", "animation"),
        ("animate-pulse", "animation"),
        ("animate-spin", "animation"),
        ("delay-100", "animation"),
        ("delay-1000", "animation"),
        ("delay-150", "animation"),
        ("delay-200", "animation"),
        ("delay-300", "animation"),
        ("delay-500", "animation"),
        ("delay-700", "animation"),
        ("delay-75", "animation"),
        ("duration-100", "animation"),
        ("duration-1000", "animation"),
        ("duration-150", "animation"),
        ("duration-200", "animation"),
        ("duration-300", "animation"),
        ("duration-500", "animation"),
        ("duration-700", "animation"),
        ("duration-75", "animation"),
        ("ease-in", "animation"),
        ("ease-in-out", "animation"),
        ("ease-linear", "animation"),
        ("ease-out", "animation"),
        ("transition", "animation"),
        ("transition-all", "animation"),
        ("transition-colors", "animation"),
        ("transition-none", "animation"),
        ("transition-opacity", "animation"),
        ("transition-shadow", "animation"),
        ("transition-transform", "animation"),
        ("bg-amber-100", "backgrounds"),
        ("bg-amber-200", "backgrounds"),
        ("bg-amber-300", "backgrounds"),
        ("bg-amber-400", "backgrounds"),
        ("bg-amber-50", "backgrounds"),
        ("bg-amber-500", "backgrounds"),
        ("bg-amber-600", "backgrounds"),
        ("bg-amber-700", "backgrounds"),
        ("bg-amber-800", "backgrounds"),
        ("bg-amber-900", "backgrounds"),
        ("bg-auto", "backgrounds"),
        ("bg-black", "backgrounds"),
        ("bg-blue-100", "backgrounds"),
        ("bg-blue-200", "backgrounds"),
        ("bg-blue-300", "backgrounds"),
        ("bg-blue-400", "backgrounds"),
        ("bg-blue-50", "backgrounds"),
        ("bg-blue-500", "backgrounds"),
        ("bg-blue-600", "backgrounds"),
        ("bg-blue-700", "backgrounds"),
        ("bg-blue-800", "backgrounds"),
        ("bg-blue-900", "backgrounds"),
        ("bg-bottom", "backgrounds"),
        ("bg-center", "backgrounds"),
        ("bg-clip-border", "backgrounds"),
        ("bg-clip-content", "backgrounds"),
        ("bg-clip-padding", "backgrounds"),
        ("bg-clip-text", "backgrounds"),
        ("bg-contain", "backgrounds"),
        ("bg-cover", "backgrounds"),
        ("bg-current", "backgrounds"),
        ("bg-cyan-100", "backgrounds"),
        ("bg-cyan-200", "backgrounds"),
        ("bg-cyan-300", "backgrounds"),
        ("bg-cyan-400", "backgrounds"),
        ("bg-cyan-50", "backgrounds"),
        ("bg-cyan-500", "backgrounds"),
        ("bg-cyan-600", "backgrounds"),
        ("bg-cyan-700", "backgrounds"),
        ("bg-cyan-800", "backgrounds"),
        ("bg-cyan-900", "backgrounds"),
        ("bg-emerald-100", "backgrounds"),
        ("bg-emerald-200", "backgrounds"),
        ("bg-emerald-300", "backgrounds"),
        ("bg-emerald-400", "backgrounds"),
        ("bg-emerald-50", "backgrounds"),
        ("bg-emerald-500", "backgrounds"),
        ("bg-emerald-600", "backgrounds"),
        ("bg-emerald-700", "backgrounds"),
        ("bg-emerald-800", "backgrounds"),
        ("bg-emerald-900", "backgrounds"),
        ("bg-fixed", "backgrounds"),
        ("bg-fuchsia-100", "backgrounds"),
        ("bg-fuchsia-200", "backgrounds"),
        ("bg-fuchsia-300", "backgrounds"),
        ("bg-fuchsia-400", "backgrounds"),
        ("bg-fuchsia-50", "backgrounds"),
        ("bg-fuchsia-500", "backgrounds"),
        ("bg-fuchsia-600", "backgrounds"),
        ("bg-fuchsia-700", "backgrounds"),
        ("bg-fuchsia-800", "backgrounds"),
        ("bg-fuchsia-900", "backgrounds"),
        ("bg-gradient-to-b", "backgrounds"),
        ("bg-gradient-to-bl", "backgrounds"),
        ("bg-gradient-to-br", "backgrounds"),
        ("bg-gradient-to-l", "backgrounds"),
        ("bg-gradient-to-r", "backgrounds"),
        ("bg-gradient-to-t", "backgrounds"),
        ("bg-gradient-to-tl", "backgrounds"),
        ("bg-gradient-to-tr", "backgrounds"),
        ("bg-gray-100", "backgrounds"),
        ("bg-gray-200", "backgrounds"),
        ("bg-gray-300", "backgrounds"),
        ("bg-gray-400", "backgrounds"),
        ("bg-gray-50", "backgrounds"),
        ("bg-gray-500", "backgrounds"),
        ("bg-gray-600", "backgrounds"),
        ("bg-gray-700", "backgrounds"),
        ("bg-gray-800", "backgrounds"),
        ("bg-gray-900", "backgrounds"),
        ("bg-green-100", "backgrounds"),
        ("bg-green-200", "backgrounds"),
        ("bg-green-300", "backgrounds"),
        ("bg-green-400", "backgrounds"),
        ("bg-green-50", "backgrounds"),
        ("bg-green-500", "backgrounds"),
        ("bg-green-600", "backgrounds"),
        ("bg-green-700", "backgrounds"),
        ("bg-green-800", "backgrounds"),
        ("bg-green-900", "backgrounds"),
        ("bg-indigo-100", "backgrounds"),
        ("bg-indigo-200", "backgrounds"),
        ("bg-indigo-300", "backgrounds"),
        ("bg-indigo-400", "backgrounds"),
        ("bg-indigo-50", "backgrounds"),
        ("bg-indigo-500", "backgrounds"),
        ("bg-indigo-600", "backgrounds"),
        ("bg-indigo-700", "backgrounds"),
        ("bg-indigo-800", "backgrounds"),
        ("bg-indigo-900", "backgrounds"),
        ("bg-inherit", "backgrounds"),
        ("bg-left", "backgrounds"),
        ("bg-left-bottom", "backgrounds"),
        ("bg-left-top", "backgrounds"),
        ("bg-lime-100", "backgrounds"),
        ("bg-lime-200", "backgrounds"),
        ("bg-lime-300", "backgrounds"),
        ("bg-lime-400", "backgrounds"),
        ("bg-lime-50", "backgrounds"),
        ("bg-lime-500", "backgrounds"),
        ("bg-lime-600", "backgrounds"),
        ("bg-lime-700", "backgrounds"),
        ("bg-lime-800", "backgrounds"),
        ("bg-lime-900", "backgrounds"),
        ("bg-local", "backgrounds"),
        ("bg-neutral-100", "backgrounds"),
        ("bg-neutral-200", "backgrounds"),
        ("bg-neutral-300", "backgrounds"),
        ("bg-neutral-400", "backgrounds"),
        ("bg-neutral-50", "backgrounds"),
        ("bg-neutral-500", "backgrounds"),
        ("bg-neutral-600", "backgrounds"),
        ("bg-neutral-700", "backgrounds"),
        ("bg-neutral-800", "backgrounds"),
        ("bg-neutral-900", "backgrounds"),
        ("bg-no-repeat", "backgrounds"),
        ("bg-none", "backgrounds"),
        ("bg-orange-100", "backgrounds"),
        ("bg-orange-200", "backgrounds"),
        ("bg-orange-300", "backgrounds"),
        ("bg-orange-400", "backgrounds"),
        ("bg-orange-50", "backgrounds"),
        ("bg-orange-500", "backgrounds"),
        ("bg-orange-600", "backgrounds"),
        ("bg-orange-700", "backgrounds"),
        ("bg-orange-800", "backgrounds"),
        ("bg-orange-900", "backgrounds"),
        ("bg-origin-border", "backgrounds"),
        ("bg-origin-content", "backgrounds"),
        ("bg-origin-padding", "backgrounds"),
        ("bg-pink-100", "backgrounds"),
        ("bg-pink-200", "backgrounds"),
        ("bg-pink-300", "backgrounds"),
        ("bg-pink-400", "backgrounds"),
        ("bg-pink-50", "backgrounds"),
        ("bg-pink-500", "backgrounds"),
        ("bg-pink-600", "backgrounds"),
        ("bg-pink-700", "backgrounds"),
        ("bg-pink-800", "backgrounds"),
        ("bg-pink-900", "backgrounds"),
        ("bg-purple-100", "backgrounds"),
        ("bg-purple-200", "backgrounds"),
        ("bg-purple-300", "backgrounds"),
        ("bg-purple-400", "backgrounds"),
        ("bg-purple-50", "backgrounds"),
        ("bg-purple-500", "backgrounds"),
        ("bg-purple-600", "backgrounds"),
        ("bg-purple-700", "backgrounds"),
        ("bg-purple-800", "backgrounds"),
        ("bg-purple-900", "backgrounds"),
        ("bg-red-100", "backgrounds"),
        ("bg-red-200", "backgrounds"),
        ("bg-red-300", "backgrounds"),
        ("bg-red-400", "backgrounds"),
        ("bg-red-50", "backgrounds"),
        ("bg-red-500", "backgrounds"),
        ("bg-red-600", "backgrounds"),
        ("bg-red-700", "backgrounds"),
        ("bg-red-800", "backgrounds"),
        ("bg-red-900", "backgrounds"),
        ("bg-repeat", "backgrounds"),
        ("bg-repeat-round", "backgrounds"),
        ("bg-repeat-space", "backgrounds"),
        ("bg-repeat-x", "backgrounds"),
        ("bg-repeat-y", "backgrounds"),
        ("bg-right", "backgrounds"),
        ("bg-right-bottom", "backgrounds"),
        ("bg-right-top", "backgrounds"),
        ("bg-rose-100", "backgrounds"),
        ("bg-rose-200", "backgrounds"),
        ("bg-rose-300", "backgrounds"),
        ("bg-rose-400", "backgrounds"),
        ("bg-rose-50", "backgrounds"),
        ("bg-rose-500", "backgrounds"),
        ("bg-rose-600", "backgrounds"),
        ("bg-rose-700", "backgrounds"),
        ("bg-rose-800", "backgrounds"),
        ("bg-rose-900", "backgrounds"),
        ("bg-scroll", "backgrounds"),
        ("bg-sky-100", "backgrounds"),
        ("bg-sky-200", "backgrounds"),
        ("bg-sky-300", "backgrounds"),
        ("bg-sky-400", "backgrounds"),
        ("bg-sky-50", "backgrounds"),
        ("bg-sky-500", "backgrounds"),
        ("bg-sky-600", "backgrounds"),
        ("bg-sky-700", "backgrounds"),
        ("bg-sky-800", "backgrounds"),
        ("bg-sky-900", "backgrounds"),
        ("bg-slate-100", "backgrounds"),
        ("bg-slate-200", "backgrounds"),
        ("bg-slate-300", "backgrounds"),
        ("bg-slate-400", "backgrounds"),
        ("bg-slate-50", "backgrounds"),
        ("bg-slate-500", "backgrounds"),
        ("bg-slate-600", "backgrounds"),
        ("bg-slate-700", "backgrounds"),
        ("bg-slate-800", "backgrounds"),
        ("bg-slate-900", "backgrounds"),
        ("bg-stone-100", "backgrounds"),
        ("bg-stone-200", "backgrounds"),
        ("bg-stone-300", "backgrounds"),
        ("bg-stone-400", "backgrounds"),
        ("bg-stone-50", "backgrounds"),
        ("bg-stone-500", "backgrounds"),
        ("bg-stone-600", "backgrounds"),
        ("bg-stone-700", "backgrounds"),
        ("bg-stone-800", "backgrounds"),
        ("bg-stone-900", "backgrounds"),
        ("bg-teal-100", "backgrounds"),
        ("bg-teal-200", "backgrounds"),
        ("bg-teal-300", "backgrounds"),
        ("bg-teal-400", "backgrounds"),
        ("bg-teal-50", "backgrounds"),
        ("bg-teal-500", "backgrounds"),
        ("bg-teal-600", "backgrounds"),
        ("bg-teal-700", "backgrounds"),
        ("bg-teal-800", "backgrounds"),
        ("bg-teal-900", "backgrounds"),
        ("bg-top", "backgrounds"),
        ("bg-transparent", "backgrounds"),
        ("bg-violet-100", "backgrounds"),
        ("bg-violet-200", "backgrounds"),
        ("bg-violet-300", "backgrounds"),
        ("bg-violet-400", "backgrounds"),
        ("bg-violet-50", "backgrounds"),
        ("bg-violet-500", "backgrounds"),
        ("bg-violet-600", "backgrounds"),
        ("bg-violet-700", "backgrounds"),
        ("bg-violet-800", "backgrounds"),
        ("bg-violet-900", "backgrounds"),
        ("bg-white", "backgrounds"),
        ("bg-yellow-100", "backgrounds"),
        ("bg-yellow-200", "backgrounds"),
        ("bg-yellow-300", "backgrounds"),
        ("bg-yellow-400", "backgrounds"),
        ("bg-yellow-50", "backgrounds"),
        ("bg-yellow-500", "backgrounds"),
        ("bg-yellow-600", "backgrounds"),
        ("bg-yellow-700", "backgrounds"),
        ("bg-yellow-800", "backgrounds"),
        ("bg-yellow-900", "backgrounds"),
        ("bg-zinc-100", "backgrounds"),
        ("bg-zinc-200", "backgrounds"),
        ("bg-zinc-300", "backgrounds"),
        ("bg-zinc-400", "backgrounds"),
        ("bg-zinc-50", "backgrounds"),
        ("bg-zinc-500", "backgrounds"),
        ("bg-zinc-600", "backgrounds"),
        ("bg-zinc-700", "backgrounds"),
        ("bg-zinc-800", "backgrounds"),
        ("bg-zinc-900", "backgrounds"),
        ("from-amber-100", "backgrounds"),
        ("from-amber-200", "backgrounds"),
        ("from-amber-300", "backgrounds"),
        ("from-amber-400", "backgrounds"),
        ("from-amber-50", "backgrounds"),
        ("from-amber-500", "backgrounds"),
        ("from-amber-600", "backgrounds"),
        ("from-amber-700", "backgrounds"),
        ("from-amber-800", "backgrounds"),
        ("from-amber-900", "backgrounds"),
        ("from-black", "backgrounds"),
        ("from-blue-100", "backgrounds"),
        ("from-blue-200", "backgrounds"),
        ("from-blue-300", "backgrounds"),
        ("from-blue-400", "backgrounds"),
        ("from-blue-50", "backgrounds"),
        ("from-blue-500", "backgrounds"),
        ("from-blue-600", "backgrounds"),
        ("from-blue-700", "backgrounds"),
        ("from-blue-800", "backgrounds"),
        ("from-blue-900", "backgrounds"),
        ("from-current", "backgrounds"),
        ("from-cyan-100", "backgrounds"),
        ("from-cyan-200", "backgrounds"),
        ("from-cyan-300", "backgrounds"),
        ("from-cyan-400", "backgrounds"),
        ("from-cyan-50", "backgrounds"),
        ("from-cyan-500", "backgrounds"),
        ("from-cyan-600", "backgrounds"),
        ("from-cyan-700", "backgrounds"),
        ("from-cyan-800", "backgrounds"),
        ("from-cyan-900", "backgrounds"),
        ("from-emerald-100", "backgrounds"),
        ("from-emerald-200", "backgrounds"),
        ("from-emerald-300", "backgrounds"),
        ("from-emerald-400", "backgrounds"),
        ("from-emerald-50", "backgrounds"),
        ("from-emerald-500", "backgrounds"),
        ("from-emerald-600", "backgrounds"),
        ("from-emerald-700", "backgrounds"),
        ("from-emerald-800", "backgrounds"),
        ("from-emerald-900", "backgrounds"),
        ("from-fuchsia-100", "backgrounds"),
        ("from-fuchsia-200", "backgrounds"),
        ("from-fuchsia-300", "backgrounds"),
        ("from-fuchsia-400", "backgrounds"),
        ("from-fuchsia-50", "backgrounds"),
        ("from-fuchsia-500", "backgrounds"),
        ("from-fuchsia-600", "backgrounds"),
        ("from-fuchsia-700", "backgrounds"),
        ("from-fuchsia-800", "backgrounds"),
        ("from-fuchsia-900", "backgrounds"),
        ("from-gray-100", "backgrounds"),
        ("from-gray-200", "backgrounds"),
        ("from-gray-300", "backgrounds"),
        ("from-gray-400", "backgrounds"),
        ("from-gray-50", "backgrounds"),
        ("from-gray-500", "backgrounds"),
        ("from-gray-600", "backgrounds"),
        ("from-gray-700", "backgrounds"),
        ("from-gray-800", "backgrounds"),
        ("from-gray-900", "backgrounds"),
        ("from-green-100", "backgrounds"),
        ("from-green-200", "backgrounds"),
        ("from-green-300", "backgrounds"),
        ("from-green-400", "backgrounds"),
        ("from-green-50", "backgrounds"),
        ("from-green-500", "backgrounds"),
        ("from-green-600", "backgrounds"),
        ("from-green-700", "backgrounds"),
        ("from-green-800", "backgrounds"),
        ("from-green-900", "backgrounds"),
        ("from-indigo-100", "backgrounds"),
        ("from-indigo-200", "backgrounds"),
        ("from-indigo-300", "backgrounds"),
        ("from-indigo-400", "backgrounds"),
        ("from-indigo-50", "backgrounds"),
        ("from-indigo-500", "backgrounds"),
        ("from-indigo-600", "backgrounds"),
        ("from-indigo-700", "backgrounds"),
        ("from-indigo-800", "backgrounds"),
        ("from-indigo-900", "backgrounds"),
        ("from-inherit", "backgrounds"),
        ("from-lime-100", "backgrounds"),
        ("from-lime-200", "backgrounds"),
        ("from-lime-300", "backgrounds"),
        ("from-lime-400", "backgrounds"),
        ("from-lime-50", "backgrounds"),
        ("from-lime-500", "backgrounds"),
        ("from-lime-600", "backgrounds"),
        ("from-lime-700", "backgrounds"),
        ("from-lime-800", "backgrounds"),
        ("from-lime-900", "backgrounds"),
        ("from-neutral-100", "backgrounds"),
        ("from-neutral-200", "backgrounds"),
        ("from-neutral-300", "backgrounds"),
        ("from-neutral-400", "backgrounds"),
        ("from-neutral-50", "backgrounds"),
        ("from-neutral-500", "backgrounds"),
        ("from-neutral-600", "backgrounds"),
        ("from-neutral-700", "backgrounds"),
        ("from-neutral-800", "backgrounds"),
        ("from-neutral-900", "backgrounds"),
        ("from-orange-100", "backgrounds"),
        ("from-orange-200", "backgrounds"),
        ("from-orange-300", "backgrounds"),
        ("from-orange-400", "backgrounds"),
        ("from-orange-50", "backgrounds"),
        ("from-orange-500", "backgrounds"),
        ("from-orange-600", "backgrounds"),
        ("from-orange-700", "backgrounds"),
        ("from-orange-800", "backgrounds"),
        ("from-orange-900", "backgrounds"),
        ("from-pink-100", "backgrounds"),
        ("from-pink-200", "backgrounds"),
        ("from-pink-300", "backgrounds"),
        ("from-pink-400", "backgrounds"),
        ("from-pink-50", "backgrounds"),
        ("from-pink-500", "backgrounds"),
        ("from-pink-600", "backgrounds"),
        ("from-pink-700", "backgrounds"),
        ("from-pink-800", "backgrounds"),
        ("from-pink-900", "backgrounds"),
        ("from-purple-100", "backgrounds"),
        ("from-purple-200", "backgrounds"),
        ("from-purple-300", "backgrounds"),
        ("from-purple-400", "backgrounds"),
        ("from-purple-50", "backgrounds"),
        ("from-purple-500", "backgrounds"),
        ("from-purple-600", "backgrounds"),
        ("from-purple-700", "backgrounds"),
        ("from-purple-800", "backgrounds"),
        ("from-purple-900", "backgrounds"),
        ("from-red-100", "backgrounds"),
        ("from-red-200", "backgrounds"),
        ("from-red-300", "backgrounds"),
        ("from-red-400", "backgrounds"),
        ("from-red-50", "backgrounds"),
        ("from-red-500", "backgrounds"),
        ("from-red-600", "backgrounds"),
        ("from-red-700", "backgrounds"),
        ("from-red-800", "backgrounds"),
        ("from-red-900", "backgrounds"),
        ("from-rose-100", "backgrounds"),
        ("from-rose-200", "backgrounds"),
        ("from-rose-300", "backgrounds"),
        ("from-rose-400", "backgrounds"),
        ("from-rose-50", "backgrounds"),
        ("from-rose-500", "backgrounds"),
        ("from-rose-600", "backgrounds"),
        ("from-rose-700", "backgrounds"),
        ("from-rose-800", "backgrounds"),
        ("from-rose-900", "backgrounds"),
        ("from-sky-100", "backgrounds"),
        ("from-sky-200", "backgrounds"),
        ("from-sky-300", "backgrounds"),
        ("from-sky-400", "backgrounds"),
        ("from-sky-50", "backgrounds"),
        ("from-sky-500", "backgrounds"),
        ("from-sky-600", "backgrounds"),
        ("from-sky-700", "backgrounds"),
        ("from-sky-800", "backgrounds"),
        ("from-sky-900", "backgrounds"),
        ("from-slate-100", "backgrounds"),
        ("from-slate-200", "backgrounds"),
        ("from-slate-300", "backgrounds"),
        ("from-slate-400", "backgrounds"),
        ("from-slate-50", "backgrounds"),
        ("from-slate-500", "backgrounds"),
        ("from-slate-600", "backgrounds"),
        ("from-slate-700", "backgrounds"),
        ("from-slate-800", "backgrounds"),
        ("from-slate-900", "backgrounds"),
        ("from-stone-100", "backgrounds"),
        ("from-stone-200", "backgrounds"),
        ("from-stone-300", "backgrounds"),
        ("from-stone-400", "backgrounds"),
        ("from-stone-50", "backgrounds"),
        ("from-stone-500", "backgrounds"),
        ("from-stone-600", "backgrounds"),
        ("from-stone-700", "backgrounds"),
        ("from-stone-800", "backgrounds"),
        ("from-stone-900", "backgrounds"),
        ("from-teal-100", "backgrounds"),
        ("from-teal-200", "backgrounds"),
        ("from-teal-300", "backgrounds"),
        ("from-teal-400", "backgrounds"),
        ("from-teal-50", "backgrounds"),
        ("from-teal-500", "backgrounds"),
        ("from-teal-600", "backgrounds"),
        ("from-teal-700", "backgrounds"),
        ("from-teal-800", "backgrounds"),
        ("from-teal-900", "backgrounds"),
        ("from-transparent", "backgrounds"),
        ("from-violet-100", "backgrounds"),
        ("from-violet-200", "backgrounds"),
        ("from-violet-300", "backgrounds"),
        ("from-violet-400", "backgrounds"),
        ("from-violet-50", "backgrounds"),
        ("from-violet-500", "backgrounds"),
        ("from-violet-600", "backgrounds"),
        ("from-violet-700", "backgrounds"),
        ("from-violet-800", "backgrounds"),
        ("from-violet-900", "backgrounds"),
        ("from-white", "backgrounds"),
        ("from-yellow-100", "backgrounds"),
        ("from-yellow-200", "backgrounds"),
        ("from-yellow-300", "backgrounds"),
        ("from-yellow-400", "backgrounds"),
        ("from-yellow-50", "backgrounds"),
        ("from-yellow-500", "backgrounds"),
        ("from-yellow-600", "backgrounds"),
        ("from-yellow-700", "backgrounds"),
        ("from-yellow-800", "backgrounds"),
        ("from-yellow-900", "backgrounds"),
        ("from-zinc-100", "backgrounds"),
        ("from-zinc-200", "backgrounds"),
        ("from-zinc-300", "backgrounds"),
        ("from-zinc-400", "backgrounds"),
        ("from-zinc-50", "backgrounds"),
        ("from-zinc-500", "backgrounds"),
        ("from-zinc-600", "backgrounds"),
        ("from-zinc-700", "backgrounds"),
        ("from-zinc-800", "backgrounds"),
        ("from-zinc-900", "backgrounds"),
        ("to-amber-100", "backgrounds"),
        ("to-amber-200", "backgrounds"),
        ("to-amber-300", "backgrounds"),
        ("to-amber-400", "backgrounds"),
        ("to-amber-50", "backgrounds"),
        ("to-amber-500", "backgrounds"),
        ("to-amber-600", "backgrounds"),
        ("to-amber-700", "backgrounds"),
        ("to-amber-800", "backgrounds"),
        ("to-amber-900", "backgrounds"),
        ("to-black", "backgrounds"),
        ("to-blue-100", "backgrounds"),
        ("to-blue-200", "backgrounds"),
        ("to-blue-300", "backgrounds"),
        ("to-blue-400", "backgrounds"),
        ("to-blue-50", "backgrounds"),
        ("to-blue-500", "backgrounds"),
        ("to-blue-600", "backgrounds"),
        ("to-blue-700", "backgrounds"),
        ("to-blue-800", "backgrounds"),
        ("to-blue-900", "backgrounds"),
        ("to-current", "backgrounds"),
        ("to-cyan-100", "backgrounds"),
        ("to-cyan-200", "backgrounds"),
        ("to-cyan-300", "backgrounds"),
        ("to-cyan-400", "backgrounds"),
        ("to-cyan-50", "backgrounds"),
        ("to-cyan-500", "backgrounds"),
        ("to-cyan-600", "backgrounds"),
        ("to-cyan-700", "backgrounds"),
        ("to-cyan-800", "backgrounds"),
        ("to-cyan-900", "backgrounds"),
        ("to-emerald-100", "backgrounds"),
        ("to-emerald-200", "backgrounds"),
        ("to-emerald-300", "backgrounds"),
        ("to-emerald-400", "backgrounds"),
        ("to-emerald-50", "backgrounds"),
        ("to-emerald-500", "backgrounds"),
        ("to-emerald-600", "backgrounds"),
        ("to-emerald-700", "backgrounds"),
        ("to-emerald-800", "backgrounds"),
        ("to-emerald-900", "backgrounds"),
        ("to-fuchsia-100", "backgrounds"),
        ("to-fuchsia-200", "backgrounds"),
        ("to-fuchsia-300", "backgrounds"),
        ("to-fuchsia-400", "backgrounds"),
        ("to-fuchsia-50", "backgrounds"),
        ("to-fuchsia-500", "backgrounds"),
        ("to-fuchsia-600", "backgrounds"),
        ("to-fuchsia-700", "backgrounds"),
        ("to-fuchsia-800", "backgrounds"),
        ("to-fuchsia-900", "backgrounds"),
        ("to-gray-100", "backgrounds"),
        ("to-gray-200", "backgrounds"),
        ("to-gray-300", "backgrounds"),
        ("to-gray-400", "backgrounds"),
        ("to-gray-50", "backgrounds"),
        ("to-gray-500", "backgrounds"),
        ("to-gray-600", "backgrounds"),
        ("to-gray-700", "backgrounds"),
        ("to-gray-800", "backgrounds"),
        ("to-gray-900", "backgrounds"),
        ("to-green-100", "backgrounds"),
        ("to-green-200", "backgrounds"),
        ("to-green-300", "backgrounds"),
        ("to-green-400", "backgrounds"),
        ("to-green-50", "backgrounds"),
        ("to-green-500", "backgrounds"),
        ("to-green-600", "backgrounds"),
        ("to-green-700", "backgrounds"),
        ("to-green-800", "backgrounds"),
        ("to-green-900", "backgrounds"),
        ("to-indigo-100", "backgrounds"),
        ("to-indigo-200", "backgrounds"),
        ("to-indigo-300", "backgrounds"),
        ("to-indigo-400", "backgrounds"),
        ("to-indigo-50", "backgrounds"),
        ("to-indigo-500", "backgrounds"),
        ("to-indigo-600", "backgrounds"),
        ("to-indigo-700", "backgrounds"),
        ("to-indigo-800", "backgrounds"),
        ("to-indigo-900", "backgrounds"),
        ("to-inherit", "backgrounds"),
        ("to-lime-100", "backgrounds"),
        ("to-lime-200", "backgrounds"),
        ("to-lime-300", "backgrounds"),
        ("to-lime-400", "backgrounds"),
        ("to-lime-50", "backgrounds"),
        ("to-lime-500", "backgrounds"),
        ("to-lime-600", "backgrounds"),
        ("to-lime-700", "backgrounds"),
        ("to-lime-800", "backgrounds"),
        ("to-lime-900", "backgrounds"),
        ("to-neutral-100", "backgrounds"),
        ("to-neutral-200", "backgrounds"),
        ("to-neutral-300", "backgrounds"),
        ("to-neutral-400", "backgrounds"),
        ("to-neutral-50", "backgrounds"),
        ("to-neutral-500", "backgrounds"),
        ("to-neutral-600", "backgrounds"),
        ("to-neutral-700", "backgrounds"),
        ("to-neutral-800", "backgrounds"),
        ("to-neutral-900", "backgrounds"),
        ("to-orange-100", "backgrounds"),
        ("to-orange-200", "backgrounds"),
        ("to-orange-300", "backgrounds"),
        ("to-orange-400", "backgrounds"),
        ("to-orange-50", "backgrounds"),
        ("to-orange-500", "backgrounds"),
        ("to-orange-600", "backgrounds"),
        ("to-orange-700", "backgrounds"),
        ("to-orange-800", "backgrounds"),
        ("to-orange-900", "backgrounds"),
        ("to-pink-100", "backgrounds"),
        ("to-pink-200", "backgrounds"),
        ("to-pink-300", "backgrounds"),
        ("to-pink-400", "backgrounds"),
        ("to-pink-50", "backgrounds"),
        ("to-pink-500", "backgrounds"),
        ("to-pink-600", "backgrounds"),
        ("to-pink-700", "backgrounds"),
        ("to-pink-800", "backgrounds"),
        ("to-pink-900", "backgrounds"),
        ("to-purple-100", "backgrounds"),
        ("to-purple-200", "backgrounds"),
        ("to-purple-300", "backgrounds"),
        ("to-purple-400", "backgrounds"),
        ("to-purple-50", "backgrounds"),
        ("to-purple-500", "backgrounds"),
        ("to-purple-600", "backgrounds"),
        ("to-purple-700", "backgrounds"),
        ("to-purple-800", "backgrounds"),
        ("to-purple-900", "backgrounds"),
        ("to-red-100", "backgrounds"),
        ("to-red-200", "backgrounds"),
        ("to-red-300", "backgrounds"),
        ("to-red-400", "backgrounds"),
        ("to-red-50", "backgrounds"),
        ("to-red-500", "backgrounds"),
        ("to-red-600", "backgrounds"),
        ("to-red-700", "backgrounds"),
        ("to-red-800", "backgrounds"),
        ("to-red-900", "backgrounds"),
        ("to-rose-100", "backgrounds"),
        ("to-rose-200", "backgrounds"),
        ("to-rose-300", "backgrounds"),
        ("to-rose-400", "backgrounds"),
        ("to-rose-50", "backgrounds"),
        ("to-rose-500", "backgrounds"),
        ("to-rose-600", "backgrounds"),
        ("to-rose-700", "backgrounds"),
        ("to-rose-800", "backgrounds"),
        ("to-rose-900", "backgrounds"),
        ("to-sky-100", "backgrounds"),
        ("to-sky-200", "backgrounds"),
        ("to-sky-300", "backgrounds"),
        ("to-sky-400", "backgrounds"),
        ("to-sky-50", "backgrounds"),
        ("to-sky-500", "backgrounds"),
        ("to-sky-600", "backgrounds"),
        ("to-sky-700", "backgrounds"),
        ("to-sky-800", "backgrounds"),
        ("to-sky-900", "backgrounds"),
        ("to-slate-100", "backgrounds"),
        ("to-slate-200", "backgrounds"),
        ("to-slate-300", "backgrounds"),
        ("to-slate-400", "backgrounds"),
        ("to-slate-50", "backgrounds"),
        ("to-slate-500", "backgrounds"),
        ("to-slate-600", "backgrounds"),
        ("to-slate-700", "backgrounds"),
        ("to-slate-800", "backgrounds"),
        ("to-slate-900", "backgrounds"),
        ("to-stone-100", "backgrounds"),
        ("to-stone-200", "backgrounds"),
        ("to-stone-300", "backgrounds"),
        ("to-stone-400", "backgrounds"),
        ("to-stone-50", "backgrounds"),
        ("to-stone-500", "backgrounds"),
        ("to-stone-600", "backgrounds"),
        ("to-stone-700", "backgrounds"),
        ("to-stone-800", "backgrounds"),
        ("to-stone-900", "backgrounds"),
        ("to-teal-100", "backgrounds"),
        ("to-teal-200", "backgrounds"),
        ("to-teal-300", "backgrounds"),
        ("to-teal-400", "backgrounds"),
        ("to-teal-50", "backgrounds"),
        ("to-teal-500", "backgrounds"),
        ("to-teal-600", "backgrounds"),
        ("to-teal-700", "backgrounds"),
        ("to-teal-800", "backgrounds"),
        ("to-teal-900", "backgrounds"),
        ("to-transparent", "backgrounds"),
        ("to-violet-100", "backgrounds"),
        ("to-violet-200", "backgrounds"),
        ("to-violet-300", "backgrounds"),
        ("to-violet-400", "backgrounds"),
        ("to-violet-50", "backgrounds"),
        ("to-violet-500", "backgrounds"),
        ("to-violet-600", "backgrounds"),
        ("to-violet-700", "backgrounds"),
        ("to-violet-800", "backgrounds"),
        ("to-violet-900", "backgrounds"),
        ("to-white", "backgrounds"),
        ("to-yellow-100", "backgrounds"),
        ("to-yellow-200", "backgrounds"),
        ("to-yellow-300", "backgrounds"),
        ("to-yellow-400", "backgrounds"),
        ("to-yellow-50", "backgrounds"),
        ("to-yellow-500", "backgrounds"),
        ("to-yellow-600", "backgrounds"),
        ("to-yellow-700", "backgrounds"),
        ("to-yellow-800", "backgrounds"),
        ("to-yellow-900", "backgrounds"),
        ("to-zinc-100", "backgrounds"),
        ("to-zinc-200", "backgrounds"),
        ("to-zinc-300", "backgrounds"),
        ("to-zinc-400", "backgrounds"),
        ("to-zinc-50", "backgrounds"),
        ("to-zinc-500", "backgrounds"),
        ("to-zinc-600", "backgrounds"),
        ("to-zinc-700", "backgrounds"),
        ("to-zinc-800", "backgrounds"),
        ("to-zinc-900", "backgrounds"),
        ("via-amber-100", "backgrounds"),
        ("via-amber-200", "backgrounds"),
        ("via-amber-300", "backgrounds"),
        ("via-amber-400", "backgrounds"),
        ("via-amber-50", "backgrounds"),
        ("via-amber-500", "backgrounds"),
        ("via-amber-600", "backgrounds"),
        ("via-amber-700", "backgrounds"),
        ("via-amber-800", "backgrounds"),
        ("via-amber-900", "backgrounds"),
        ("via-black", "backgrounds"),
        ("via-blue-100", "backgrounds"),
        ("via-blue-200", "backgrounds"),
        ("via-blue-300", "backgrounds"),
        ("via-blue-400", "backgrounds"),
        ("via-blue-50", "backgrounds"),
        ("via-blue-500", "backgrounds"),
        ("via-blue-600", "backgrounds"),
        ("via-blue-700", "backgrounds"),
        ("via-blue-800", "backgrounds"),
        ("via-blue-900", "backgrounds"),
        ("via-current", "backgrounds"),
        ("via-cyan-100", "backgrounds"),
        ("via-cyan-200", "backgrounds"),
        ("via-cyan-300", "backgrounds"),
        ("via-cyan-400", "backgrounds"),
        ("via-cyan-50", "backgrounds"),
        ("via-cyan-500", "backgrounds"),
        ("via-cyan-600", "backgrounds"),
        ("via-cyan-700", "backgrounds"),
        ("via-cyan-800", "backgrounds"),
        ("via-cyan-900", "backgrounds"),
        ("via-emerald-100", "backgrounds"),
        ("via-emerald-200", "backgrounds"),
        ("via-emerald-300", "backgrounds"),
        ("via-emerald-400", "backgrounds"),
        ("via-emerald-50", "backgrounds"),
        ("via-emerald-500", "backgrounds"),
        ("via-emerald-600", "backgrounds"),
        ("via-emerald-700", "backgrounds"),
        ("via-emerald-800", "backgrounds"),
        ("via-emerald-900", "backgrounds"),
        ("via-fuchsia-100", "backgrounds"),
        ("via-fuchsia-200", "backgrounds"),
        ("via-fuchsia-300", "backgrounds"),
        ("via-fuchsia-400", "backgrounds"),
        ("via-fuchsia-50", "backgrounds"),
        ("via-fuchsia-500", "backgrounds"),
        ("via-fuchsia-600", "backgrounds"),
        ("via-fuchsia-700", "backgrounds"),
        ("via-fuchsia-800", "backgrounds"),
        ("via-fuchsia-900", "backgrounds"),
        ("via-gray-100", "backgrounds"),
        ("via-gray-200", "backgrounds"),
        ("via-gray-300", "backgrounds"),
        ("via-gray-400", "backgrounds"),
        ("via-gray-50", "backgrounds"),
        ("via-gray-500", "backgrounds"),
        ("via-gray-600", "backgrounds"),
        ("via-gray-700", "backgrounds"),
        ("via-gray-800", "backgrounds"),
        ("via-gray-900", "backgrounds"),
        ("via-green-100", "backgrounds"),
        ("via-green-200", "backgrounds"),
        ("via-green-300", "backgrounds"),
        ("via-green-400", "backgrounds"),
        ("via-green-50", "backgrounds"),
        ("via-green-500", "backgrounds"),
        ("via-green-600", "backgrounds"),
        ("via-green-700", "backgrounds"),
        ("via-green-800", "backgrounds"),
        ("via-green-900", "backgrounds"),
        ("via-indigo-100", "backgrounds"),
        ("via-indigo-200", "backgrounds"),
        ("via-indigo-300", "backgrounds"),
        ("via-indigo-400", "backgrounds"),
        ("via-indigo-50", "backgrounds"),
        ("via-indigo-500", "backgrounds"),
        ("via-indigo-600", "backgrounds"),
        ("via-indigo-700", "backgrounds"),
        ("via-indigo-800", "backgrounds"),
        ("via-indigo-900", "backgrounds"),
        ("via-inherit", "backgrounds"),
        ("via-lime-100", "backgrounds"),
        ("via-lime-200", "backgrounds"),
        ("via-lime-300", "backgrounds"),
        ("via-lime-400", "backgrounds"),
        ("via-lime-50", "backgrounds"),
        ("via-lime-500", "backgrounds"),
        ("via-lime-600", "backgrounds"),
        ("via-lime-700", "backgrounds"),
        ("via-lime-800", "backgrounds"),
        ("via-lime-900", "backgrounds"),
        ("via-neutral-100", "backgrounds"),
        ("via-neutral-200", "backgrounds"),
        ("via-neutral-300", "backgrounds"),
        ("via-neutral-400", "backgrounds"),
        ("via-neutral-50", "backgrounds"),
        ("via-neutral-500", "backgrounds"),
        ("via-neutral-600", "backgrounds"),
        ("via-neutral-700", "backgrounds"),
        ("via-neutral-800", "backgrounds"),
        ("via-neutral-900", "backgrounds"),
        ("via-orange-100", "backgrounds"),
        ("via-orange-200", "backgrounds"),
        ("via-orange-300", "backgrounds"),
        ("via-orange-400", "backgrounds"),
        ("via-orange-50", "backgrounds"),
        ("via-orange-500", "backgrounds"),
        ("via-orange-600", "backgrounds"),
        ("via-orange-700", "backgrounds"),
        ("via-orange-800", "backgrounds"),
        ("via-orange-900", "backgrounds"),
        ("via-pink-100", "backgrounds"),
        ("via-pink-200", "backgrounds"),
        ("via-pink-300", "backgrounds"),
        ("via-pink-400", "backgrounds"),
        ("via-pink-50", "backgrounds"),
        ("via-pink-500", "backgrounds"),
        ("via-pink-600", "backgrounds"),
        ("via-pink-700", "backgrounds"),
        ("via-pink-800", "backgrounds"),
        ("via-pink-900", "backgrounds"),
        ("via-purple-100", "backgrounds"),
        ("via-purple-200", "backgrounds"),
        ("via-purple-300", "backgrounds"),
        ("via-purple-400", "backgrounds"),
        ("via-purple-50", "backgrounds"),
        ("via-purple-500", "backgrounds"),
        ("via-purple-600", "backgrounds"),
        ("via-purple-700", "backgrounds"),
        ("via-purple-800", "backgrounds"),
        ("via-purple-900", "backgrounds"),
        ("via-red-100", "backgrounds"),
        ("via-red-200", "backgrounds"),
        ("via-red-300", "backgrounds"),
        ("via-red-400", "backgrounds"),
        ("via-red-50", "backgrounds"),
        ("via-red-500", "backgrounds"),
        ("via-red-600", "backgrounds"),
        ("via-red-700", "backgrounds"),
        ("via-red-800", "backgrounds"),
        ("via-red-900", "backgrounds"),
        ("via-rose-100", "backgrounds"),
        ("via-rose-200", "backgrounds"),
        ("via-rose-300", "backgrounds"),
        ("via-rose-400", "backgrounds"),
        ("via-rose-50", "backgrounds"),
        ("via-rose-500", "backgrounds"),
        ("via-rose-600", "backgrounds"),
        ("via-rose-700", "backgrounds"),
        ("via-rose-800", "backgrounds"),
        ("via-rose-900", "backgrounds"),
        ("via-sky-100", "backgrounds"),
        ("via-sky-200", "backgrounds"),
        ("via-sky-300", "backgrounds"),
        ("via-sky-400", "backgrounds"),
        ("via-sky-50", "backgrounds"),
        ("via-sky-500", "backgrounds"),
        ("via-sky-600", "backgrounds"),
        ("via-sky-700", "backgrounds"),
        ("via-sky-800", "backgrounds"),
        ("via-sky-900", "backgrounds"),
        ("via-slate-100", "backgrounds"),
        ("via-slate-200", "backgrounds"),
        ("via-slate-300", "backgrounds"),
        ("via-slate-400", "backgrounds"),
        ("via-slate-50", "backgrounds"),
        ("via-slate-500", "backgrounds"),
        ("via-slate-600", "backgrounds"),
        ("via-slate-700", "backgrounds"),
        ("via-slate-800", "backgrounds"),
        ("via-slate-900", "backgrounds"),
        ("via-stone-100", "backgrounds"),
        ("via-stone-200", "backgrounds"),
        ("via-stone-300", "backgrounds"),
        ("via-stone-400", "backgrounds"),
        ("via-stone-50", "backgrounds"),
        ("via-stone-500", "backgrounds"),
        ("via-stone-600", "backgrounds"),
        ("via-stone-700", "backgrounds"),
        ("via-stone-800", "backgrounds"),
        ("via-stone-900", "backgrounds"),
        ("via-teal-100", "backgrounds"),
        ("via-teal-200", "backgrounds"),
        ("via-teal-300", "backgrounds"),
        ("via-teal-400", "backgrounds"),
        ("via-teal-50", "backgrounds"),
        ("via-teal-500", "backgrounds"),
        ("via-teal-600", "backgrounds"),
        ("via-teal-700", "backgrounds"),
        ("via-teal-800", "backgrounds"),
        ("via-teal-900", "backgrounds"),
        ("via-transparent", "backgrounds"),
        ("via-violet-100", "backgrounds"),
        ("via-violet-200", "backgrounds"),
        ("via-violet-300", "backgrounds"),
        ("via-violet-400", "backgrounds"),
        ("via-violet-50", "backgrounds"),
        ("via-violet-500", "backgrounds"),
        ("via-violet-600", "backgrounds"),
        ("via-violet-700", "backgrounds"),
        ("via-violet-800", "backgrounds"),
        ("via-violet-900", "backgrounds"),
        ("via-white", "backgrounds"),
        ("via-yellow-100", "backgrounds"),
        ("via-yellow-200", "backgrounds"),
        ("via-yellow-300", "backgrounds"),
        ("via-yellow-400", "backgrounds"),
        ("via-yellow-50", "backgrounds"),
        ("via-yellow-500", "backgrounds"),
        ("via-yellow-600", "backgrounds"),
        ("via-yellow-700", "backgrounds"),
        ("via-yellow-800", "backgrounds"),
        ("via-yellow-900", "backgrounds"),
        ("via-zinc-100", "backgrounds"),
        ("via-zinc-200", "backgrounds"),
        ("via-zinc-300", "backgrounds"),
        ("via-zinc-400", "backgrounds"),
        ("via-zinc-50", "backgrounds"),
        ("via-zinc-500", "backgrounds"),
        ("via-zinc-600", "backgrounds"),
        ("via-zinc-700", "backgrounds"),
        ("via-zinc-800", "backgrounds"),
        ("via-zinc-900", "backgrounds"),
        ("border", "borders"),
        ("border-0", "borders"),
        ("border-2", "borders"),
        ("border-4", "borders"),
        ("border-8", "borders"),
        ("border-amber-100", "borders"),
        ("border-amber-200", "borders"),
        ("border-amber-300", "borders"),
        ("border-amber-400", "borders"),
        ("border-amber-50", "borders"),
        ("border-amber-500", "borders"),
        ("border-amber-600", "borders"),
        ("border-amber-700", "borders"),
        ("border-amber-800", "borders"),
        ("border-amber-900", "borders"),
        ("border-b", "borders"),
        ("border-b-0", "borders"),
        ("border-b-2", "borders"),
        ("border-b-4", "borders"),
        ("border-b-8", "borders"),
        ("border-b-amber-100", "borders"),
        ("border-b-amber-200", "borders"),
        ("border-b-amber-300", "borders"),
        ("border-b-amber-400", "borders"),
        ("border-b-amber-50", "borders"),
        ("border-b-amber-500", "borders"),
        ("border-b-amber-600", "borders"),
        ("border-b-amber-700", "borders"),
        ("border-b-amber-800", "borders"),
        ("border-b-amber-900", "borders"),
        ("border-b-black", "borders"),
        ("border-b-blue-100", "borders"),
        ("border-b-blue-200", "borders"),
        ("border-b-blue-300", "borders"),
        ("border-b-blue-400", "borders"),
        ("border-b-blue-50", "borders"),
        ("border-b-blue-500", "borders"),
        ("border-b-blue-600", "borders"),
        ("border-b-blue-700", "borders"),
        ("border-b-blue-800", "borders"),
        ("border-b-blue-900", "borders"),
        ("border-b-current", "borders"),
        ("border-b-cyan-100", "borders"),
        ("border-b-cyan-200", "borders"),
        ("border-b-cyan-300", "borders"),
        ("border-b-cyan-400", "borders"),
        ("border-b-cyan-50", "borders"),
        ("border-b-cyan-500", "borders"),
        ("border-b-cyan-600", "borders"),
        ("border-b-cyan-700", "borders"),
        ("border-b-cyan-800", "borders"),
        ("border-b-cyan-900", "borders"),
        ("border-b-emerald-100", "borders"),
        ("border-b-emerald-200", "borders"),
        ("border-b-emerald-300", "borders"),
        ("border-b-emerald-400", "borders"),
        ("border-b-emerald-50", "borders"),
        ("border-b-emerald-500", "borders"),
        ("border-b-emerald-600", "borders"),
        ("border-b-emerald-700", "borders"),
        ("border-b-emerald-800", "borders"),
        ("border-b-emerald-900", "borders"),
        ("border-b-fuchsia-100", "borders"),
        ("border-b-fuchsia-200", "borders"),
        ("border-b-fuchsia-300", "borders"),
        ("border-b-fuchsia-400", "borders"),
        ("border-b-fuchsia-50", "borders"),
        ("border-b-fuchsia-500", "borders"),
        ("border-b-fuchsia-600", "borders"),
        ("border-b-fuchsia-700", "borders"),
        ("border-b-fuchsia-800", "borders"),
        ("border-b-fuchsia-900", "borders"),
        ("border-b-gray-100", "borders"),
        ("border-b-gray-200", "borders"),
        ("border-b-gray-300", "borders"),
        ("border-b-gray-400", "borders"),
        ("border-b-gray-50", "borders"),
        ("border-b-gray-500", "borders"),
        ("border-b-gray-600", "borders"),
        ("border-b-gray-700", "borders"),
        ("border-b-gray-800", "borders"),
        ("border-b-gray-900", "borders"),
        ("border-b-green-100", "borders"),
        ("border-b-green-200", "borders"),
        ("border-b-green-300", "borders"),
        ("border-b-green-400", "borders"),
        ("border-b-green-50", "borders"),
        ("border-b-green-500", "borders"),
        ("border-b-green-600", "borders"),
        ("border-b-green-700", "borders"),
        ("border-b-green-800", "borders"),
        ("border-b-green-900", "borders"),
        ("border-b-indigo-100", "borders"),
        ("border-b-indigo-200", "borders"),
        ("border-b-indigo-300", "borders"),
        ("border-b-indigo-400", "borders"),
        ("border-b-indigo-50", "borders"),
        ("border-b-indigo-500", "borders"),
        ("border-b-indigo-600", "borders"),
        ("border-b-indigo-700", "borders"),
        ("border-b-indigo-800", "borders"),
        ("border-b-indigo-900", "borders"),
        ("border-b-inherit", "borders"),
        ("border-b-lime-100", "borders"),
        ("border-b-lime-200", "borders"),
        ("border-b-lime-300", "borders"),
        ("border-b-lime-400", "borders"),
        ("border-b-lime-50", "borders"),
        ("border-b-lime-500", "borders"),
        ("border-b-lime-600", "borders"),
        ("border-b-lime-700", "borders"),
        ("border-b-lime-800", "borders"),
        ("border-b-lime-900", "borders"),
        ("border-b-neutral-100", "borders"),
        ("border-b-neutral-200", "borders"),
        ("border-b-neutral-300", "borders"),
        ("border-b-neutral-400", "borders"),
        ("border-b-neutral-50", "borders"),
        ("border-b-neutral-500", "borders"),
        ("border-b-neutral-600", "borders"),
        ("border-b-neutral-700", "borders"),
        ("border-b-neutral-800", "borders"),
        ("border-b-neutral-900", "borders"),
        ("border-b-orange-100", "borders"),
        ("border-b-orange-200", "borders"),
        ("border-b-orange-300", "borders"),
        ("border-b-orange-400", "borders"),
        ("border-b-orange-50", "borders"),
        ("border-b-orange-500", "borders"),
        ("border-b-orange-600", "borders"),
        ("border-b-orange-700", "borders"),
        ("border-b-orange-800", "borders"),
        ("border-b-orange-900", "borders"),
        ("border-b-pink-100", "borders"),
        ("border-b-pink-200", "borders"),
        ("border-b-pink-300", "borders"),
        ("border-b-pink-400", "borders"),
        ("border-b-pink-50", "borders"),
        ("border-b-pink-500", "borders"),
        ("border-b-pink-600", "borders"),
        ("border-b-pink-700", "borders"),
        ("border-b-pink-800", "borders"),
        ("border-b-pink-900", "borders"),
        ("border-b-purple-100", "borders"),
        ("border-b-purple-200", "borders"),
        ("border-b-purple-300", "borders"),
        ("border-b-purple-400", "borders"),
        ("border-b-purple-50", "borders"),
        ("border-b-purple-500", "borders"),
        ("border-b-purple-600", "borders"),
        ("border-b-purple-700", "borders"),
        ("border-b-purple-800", "borders"),
        ("border-b-purple-900", "borders"),
        ("border-b-red-100", "borders"),
        ("border-b-red-200", "borders"),
        ("border-b-red-300", "borders"),
        ("border-b-red-400", "borders"),
        ("border-b-red-50", "borders"),
        ("border-b-red-500", "borders"),
        ("border-b-red-600", "borders"),
        ("border-b-red-700", "borders"),
        ("border-b-red-800", "borders"),
        ("border-b-red-900", "borders"),
        ("border-b-rose-100", "borders"),
        ("border-b-rose-200", "borders"),
        ("border-b-rose-300", "borders"),
        ("border-b-rose-400", "borders"),
        ("border-b-rose-50", "borders"),
        ("border-b-rose-500", "borders"),
        ("border-b-rose-600", "borders"),
        ("border-b-rose-700", "borders"),
        ("border-b-rose-800", "borders"),
        ("border-b-rose-900", "borders"),
        ("border-b-sky-100", "borders"),
        ("border-b-sky-200", "borders"),
        ("border-b-sky-300", "borders"),
        ("border-b-sky-400", "borders"),
        ("border-b-sky-50", "borders"),
        ("border-b-sky-500", "borders"),
        ("border-b-sky-600", "borders"),
        ("border-b-sky-700", "borders"),
        ("border-b-sky-800", "borders"),
        ("border-b-sky-900", "borders"),
        ("border-b-slate-100", "borders"),
        ("border-b-slate-200", "borders"),
        ("border-b-slate-300", "borders"),
        ("border-b-slate-400", "borders"),
        ("border-b-slate-50", "borders"),
        ("border-b-slate-500", "borders"),
        ("border-b-slate-600", "borders"),
        ("border-b-slate-700", "borders"),
        ("border-b-slate-800", "borders"),
        ("border-b-slate-900", "borders"),
        ("border-b-stone-100", "borders"),
        ("border-b-stone-200", "borders"),
        ("border-b-stone-300", "borders"),
        ("border-b-stone-400", "borders"),
        ("border-b-stone-50", "borders"),
        ("border-b-stone-500", "borders"),
        ("border-b-stone-600", "borders"),
        ("border-b-stone-700", "borders"),
        ("border-b-stone-800", "borders"),
        ("border-b-stone-900", "borders"),
        ("border-b-teal-100", "borders"),
        ("border-b-teal-200", "borders"),
        ("border-b-teal-300", "borders"),
        ("border-b-teal-400", "borders"),
        ("border-b-teal-50", "borders"),
        ("border-b-teal-500", "borders"),
        ("border-b-teal-600", "borders"),
        ("border-b-teal-700", "borders"),
        ("border-b-teal-800", "borders"),
        ("border-b-teal-900", "borders"),
        ("border-b-transparent", "borders"),
        ("border-b-violet-100", "borders"),
        ("border-b-violet-200", "borders"),
        ("border-b-violet-300", "borders"),
        ("border-b-violet-400", "borders"),
        ("border-b-violet-50", "borders"),
        ("border-b-violet-500", "borders"),
        ("border-b-violet-600", "borders"),
        ("border-b-violet-700", "borders"),
        ("border-b-violet-800", "borders"),
        ("border-b-violet-900", "borders"),
        ("border-b-white", "borders"),
        ("border-b-yellow-100", "borders"),
        ("border-b-yellow-200", "borders"),
        ("border-b-yellow-300", "borders"),
        ("border-b-yellow-400", "borders"),
        ("border-b-yellow-50", "borders"),
        ("border-b-yellow-500", "borders"),
        ("border-b-yellow-600", "borders"),
        ("border-b-yellow-700", "borders"),
        ("border-b-yellow-800", "borders"),
        ("border-b-yellow-900", "borders"),
        ("border-b-zinc-100", "borders"),
        ("border-b-zinc-200", "borders"),
        ("border-b-zinc-300", "borders"),
        ("border-b-zinc-400", "borders"),
        ("border-b-zinc-50", "borders"),
        ("border-b-zinc-500", "borders"),
        ("border-b-zinc-600", "borders"),
        ("border-b-zinc-700", "borders"),
        ("border-b-zinc-800", "borders"),
        ("border-b-zinc-900", "borders"),
        ("border-black", "borders"),
        ("border-blue-100", "borders"),
        ("border-blue-200", "borders"),
        ("border-blue-300", "borders"),
        ("border-blue-400", "borders"),
        ("border-blue-50", "borders"),
        ("border-blue-500", "borders"),
        ("border-blue-600", "borders"),
        ("border-blue-700", "borders"),
        ("border-blue-800", "borders"),
        ("border-blue-900", "borders"),
        ("border-current", "borders"),
        ("border-cyan-100", "borders"),
        ("border-cyan-200", "borders"),
        ("border-cyan-300", "borders"),
        ("border-cyan-400", "borders"),
        ("border-cyan-50", "borders"),
        ("border-cyan-500", "borders"),
        ("border-cyan-600", "borders"),
        ("border-cyan-700", "borders"),
        ("border-cyan-800", "borders"),
        ("border-cyan-900", "borders"),
        ("border-dashed", "borders"),
        ("border-dotted", "borders"),
        ("border-double", "borders"),
        ("border-emerald-100", "borders"),
        ("border-emerald-200", "borders"),
        ("border-emerald-300", "borders"),
        ("border-emerald-400", "borders"),
        ("border-emerald-50", "borders"),
        ("border-emerald-500", "borders"),
        ("border-emerald-600", "borders"),
        ("border-emerald-700", "borders"),
        ("border-emerald-800", "borders"),
        ("border-emerald-900", "borders"),
        ("border-fuchsia-100", "borders"),
        ("border-fuchsia-200", "borders"),
        ("border-fuchsia-300", "borders"),
        ("border-fuchsia-400", "borders"),
        ("border-fuchsia-50", "borders"),
        ("border-fuchsia-500", "borders"),
        ("border-fuchsia-600", "borders"),
        ("border-fuchsia-700", "borders"),
        ("border-fuchsia-800", "borders"),
        ("border-fuchsia-900", "borders"),
        ("border-gray-100", "borders"),
        ("border-gray-200", "borders"),
        ("border-gray-300", "borders"),
        ("border-gray-400", "borders"),
        ("border-gray-50", "borders"),
        ("border-gray-500", "borders"),
        ("border-gray-600", "borders"),
        ("border-gray-700", "borders"),
        ("border-gray-800", "borders"),
        ("border-gray-900", "borders"),
        ("border-green-100", "borders"),
        ("border-green-200", "borders"),
        ("border-green-300", "borders"),
        ("border-green-400", "borders"),
        ("border-green-50", "borders"),
        ("border-green-500", "borders"),
        ("border-green-600", "borders"),
        ("border-green-700", "borders"),
        ("border-green-800", "borders"),
        ("border-green-900", "borders"),
        ("border-hidden", "borders"),
        ("border-indigo-100", "borders"),
        ("border-indigo-200", "borders"),
        ("border-indigo-300", "borders"),
        ("border-indigo-400", "borders"),
        ("border-indigo-50", "borders"),
        ("border-indigo-500", "borders"),
        ("border-indigo-600", "borders"),
        ("border-indigo-700", "borders"),
        ("border-indigo-800", "borders"),
        ("border-indigo-900", "borders"),
        ("border-inherit", "borders"),
        ("border-l", "borders"),
        ("border-l-0", "borders"),
        ("border-l-2", "borders"),
        ("border-l-4", "borders"),
        ("border-l-8", "borders"),
        ("border-l-amber-100", "borders"),
        ("border-l-amber-200", "borders"),
        ("border-l-amber-300", "borders"),
        ("border-l-amber-400", "borders"),
        ("border-l-amber-50", "borders"),
        ("border-l-amber-500", "borders"),
        ("border-l-amber-600", "borders"),
        ("border-l-amber-700", "borders"),
        ("border-l-amber-800", "borders"),
        ("border-l-amber-900", "borders"),
        ("border-l-black", "borders"),
        ("border-l-blue-100", "borders"),
        ("border-l-blue-200", "borders"),
        ("border-l-blue-300", "borders"),
        ("border-l-blue-400", "borders"),
        ("border-l-blue-50", "borders"),
        ("border-l-blue-500", "borders"),
        ("border-l-blue-600", "borders"),
        ("border-l-blue-700", "borders"),
        ("border-l-blue-800", "borders"),
        ("border-l-blue-900", "borders"),
        ("border-l-current", "borders"),
        ("border-l-cyan-100", "borders"),
        ("border-l-cyan-200", "borders"),
        ("border-l-cyan-300", "borders"),
        ("border-l-cyan-400", "borders"),
        ("border-l-cyan-50", "borders"),
        ("border-l-cyan-500", "borders"),
        ("border-l-cyan-600", "borders"),
        ("border-l-cyan-700", "borders"),
        ("border-l-cyan-800", "borders"),
        ("border-l-cyan-900", "borders"),
        ("border-l-emerald-100", "borders"),
        ("border-l-emerald-200", "borders"),
        ("border-l-emerald-300", "borders"),
        ("border-l-emerald-400", "borders"),
        ("border-l-emerald-50", "borders"),
        ("border-l-emerald-500", "borders"),
        ("border-l-emerald-600", "borders"),
        ("border-l-emerald-700", "borders"),
        ("border-l-emerald-800", "borders"),
        ("border-l-emerald-900", "borders"),
        ("border-l-fuchsia-100", "borders"),
        ("border-l-fuchsia-200", "borders"),
        ("border-l-fuchsia-300", "borders"),
        ("border-l-fuchsia-400", "borders"),
        ("border-l-fuchsia-50", "borders"),
        ("border-l-fuchsia-500", "borders"),
        ("border-l-fuchsia-600", "borders"),
        ("border-l-fuchsia-700", "borders"),
        ("border-l-fuchsia-800", "borders"),
        ("border-l-fuchsia-900", "borders"),
        ("border-l-gray-100", "borders"),
        ("border-l-gray-200", "borders"),
        ("border-l-gray-300", "borders"),
        ("border-l-gray-400", "borders"),
        ("border-l-gray-50", "borders"),
        ("border-l-gray-500", "borders"),
        ("border-l-gray-600", "borders"),
        ("border-l-gray-700", "borders"),
        ("border-l-gray-800", "borders"),
        ("border-l-gray-900", "borders"),
        ("border-l-green-100", "borders"),
        ("border-l-green-200", "borders"),
        ("border-l-green-300", "borders"),
        ("border-l-green-400", "borders"),
        ("border-l-green-50", "borders"),
        ("border-l-green-500", "borders"),
        ("border-l-green-600", "borders"),
        ("border-l-green-700", "borders"),
        ("border-l-green-800", "borders"),
        ("border-l-green-900", "borders"),
        ("border-l-indigo-100", "borders"),
        ("border-l-indigo-200", "borders"),
        ("border-l-indigo-300", "borders"),
        ("border-l-indigo-400", "borders"),
        ("border-l-indigo-50", "borders"),
        ("border-l-indigo-500", "borders"),
        ("border-l-indigo-600", "borders"),
        ("border-l-indigo-700", "borders"),
        ("border-l-indigo-800", "borders"),
        ("border-l-indigo-900", "borders"),
        ("border-l-inherit", "borders"),
        ("border-l-lime-100", "borders"),
        ("border-l-lime-200", "borders"),
        ("border-l-lime-300", "borders"),
        ("border-l-lime-400", "borders"),
        ("border-l-lime-50", "borders"),
        ("border-l-lime-500", "borders"),
        ("border-l-lime-600", "borders"),
        ("border-l-lime-700", "borders"),
        ("border-l-lime-800", "borders"),
        ("border-l-lime-900", "borders"),
        ("border-l-neutral-100", "borders"),
        ("border-l-neutral-200", "borders"),
        ("border-l-neutral-300", "borders"),
        ("border-l-neutral-400", "borders"),
        ("border-l-neutral-50", "borders"),
        ("border-l-neutral-500", "borders"),
        ("border-l-neutral-600", "borders"),
        ("border-l-neutral-700", "borders"),
        ("border-l-neutral-800", "borders"),
        ("border-l-neutral-900", "borders"),
        ("border-l-orange-100", "borders"),
        ("border-l-orange-200", "borders"),
        ("border-l-orange-300", "borders"),
        ("border-l-orange-400", "borders"),
        ("border-l-orange-50", "borders"),
        ("border-l-orange-500", "borders"),
        ("border-l-orange-600", "borders"),
        ("border-l-orange-700", "borders"),
        ("border-l-orange-800", "borders"),
        ("border-l-orange-900", "borders"),
        ("border-l-pink-100", "borders"),
        ("border-l-pink-200", "borders"),
        ("border-l-pink-300", "borders"),
        ("border-l-pink-400", "borders"),
        ("border-l-pink-50", "borders"),
        ("border-l-pink-500", "borders"),
        ("border-l-pink-600", "borders"),
        ("border-l-pink-700", "borders"),
        ("border-l-pink-800", "borders"),
        ("border-l-pink-900", "borders"),
        ("border-l-purple-100", "borders"),
        ("border-l-purple-200", "borders"),
        ("border-l-purple-300", "borders"),
        ("border-l-purple-400", "borders"),
        ("border-l-purple-50", "borders"),
        ("border-l-purple-500", "borders"),
        ("border-l-purple-600", "borders"),
        ("border-l-purple-700", "borders"),
        ("border-l-purple-800", "borders"),
        ("border-l-purple-900", "borders"),
        ("border-l-red-100", "borders"),
        ("border-l-red-200", "borders"),
        ("border-l-red-300", "borders"),
        ("border-l-red-400", "borders"),
        ("border-l-red-50", "borders"),
        ("border-l-red-500", "borders"),
        ("border-l-red-600", "borders"),
        ("border-l-red-700", "borders"),
        ("border-l-red-800", "borders"),
        ("border-l-red-900", "borders"),
        ("border-l-rose-100", "borders"),
        ("border-l-rose-200", "borders"),
        ("border-l-rose-300", "borders"),
        ("border-l-rose-400", "borders"),
        ("border-l-rose-50", "borders"),
        ("border-l-rose-500", "borders"),
        ("border-l-rose-600", "borders"),
        ("border-l-rose-700", "borders"),
        ("border-l-rose-800", "borders"),
        ("border-l-rose-900", "borders"),
        ("border-l-sky-100", "borders"),
        ("border-l-sky-200", "borders"),
        ("border-l-sky-300", "borders"),
        ("border-l-sky-400", "borders"),
        ("border-l-sky-50", "borders"),
        ("border-l-sky-500", "borders"),
        ("border-l-sky-600", "borders"),
        ("border-l-sky-700", "borders"),
        ("border-l-sky-800", "borders"),
        ("border-l-sky-900", "borders"),
        ("border-l-slate-100", "borders"),
        ("border-l-slate-200", "borders"),
        ("border-l-slate-300", "borders"),
        ("border-l-slate-400", "borders"),
        ("border-l-slate-50", "borders"),
        ("border-l-slate-500", "borders"),
        ("border-l-slate-600", "borders"),
        ("border-l-slate-700", "borders"),
        ("border-l-slate-800", "borders"),
        ("border-l-slate-900", "borders"),
        ("border-l-stone-100", "borders"),
        ("border-l-stone-200", "borders"),
        ("border-l-stone-300", "borders"),
        ("border-l-stone-400", "borders"),
        ("border-l-stone-50", "borders"),
        ("border-l-stone-500", "borders"),
        ("border-l-stone-600", "borders"),
        ("border-l-stone-700", "borders"),
        ("border-l-stone-800", "borders"),
        ("border-l-stone-900", "borders"),
        ("border-l-teal-100", "borders"),
        ("border-l-teal-200", "borders"),
        ("border-l-teal-300", "borders"),
        ("border-l-teal-400", "borders"),
        ("border-l-teal-50", "borders"),
        ("border-l-teal-500", "borders"),
        ("border-l-teal-600", "borders"),
        ("border-l-teal-700", "borders"),
        ("border-l-teal-800", "borders"),
        ("border-l-teal-900", "borders"),
        ("border-l-transparent", "borders"),
        ("border-l-violet-100", "borders"),
        ("border-l-violet-200", "borders"),
        ("border-l-violet-300", "borders"),
        ("border-l-violet-400", "borders"),
        ("border-l-violet-50", "borders"),
        ("border-l-violet-500", "borders"),
        ("border-l-violet-600", "borders"),
        ("border-l-violet-700", "borders"),
        ("border-l-violet-800", "borders"),
        ("border-l-violet-900", "borders"),
        ("border-l-white", "borders"),
        ("border-l-yellow-100", "borders"),
        ("border-l-yellow-200", "borders"),
        ("border-l-yellow-300", "borders"),
        ("border-l-yellow-400", "borders"),
        ("border-l-yellow-50", "borders"),
        ("border-l-yellow-500", "borders"),
        ("border-l-yellow-600", "borders"),
        ("border-l-yellow-700", "borders"),
        ("border-l-yellow-800", "borders"),
        ("border-l-yellow-900", "borders"),
        ("border-l-zinc-100", "borders"),
        ("border-l-zinc-200", "borders"),
        ("border-l-zinc-300", "borders"),
        ("border-l-zinc-400", "borders"),
        ("border-l-zinc-50", "borders"),
        ("border-l-zinc-500", "borders"),
        ("border-l-zinc-600", "borders"),
        ("border-l-zinc-700", "borders"),
        ("border-l-zinc-800", "borders"),
        ("border-l-zinc-900", "borders"),
        ("border-lime-100", "borders"),
        ("border-lime-200", "borders"),
        ("border-lime-300", "borders"),
        ("border-lime-400", "borders"),
        ("border-lime-50", "borders"),
        ("border-lime-500", "borders"),
        ("border-lime-600", "borders"),
        ("border-lime-700", "borders"),
        ("border-lime-800", "borders"),
        ("border-lime-900", "borders"),
        ("border-neutral-100", "borders"),
        ("border-neutral-200", "borders"),
        ("border-neutral-300", "borders"),
        ("border-neutral-400", "borders"),
        ("border-neutral-50", "borders"),
        ("border-neutral-500", "borders"),
        ("border-neutral-600", "borders"),
        ("border-neutral-700", "borders"),
        ("border-neutral-800", "borders"),
        ("border-neutral-900", "borders"),
        ("border-none", "borders"),
        ("border-orange-100", "borders"),
        ("border-orange-200", "borders"),
        ("border-orange-300", "borders"),
        ("border-orange-400", "borders"),
        ("border-orange-50", "borders"),
        ("border-orange-500", "borders"),
        ("border-orange-600", "borders"),
        ("border-orange-700", "borders"),
        ("border-orange-800", "borders"),
        ("border-orange-900", "borders"),
        ("border-pink-100", "borders"),
        ("border-pink-200", "borders"),
        ("border-pink-300", "borders"),
        ("border-pink-400", "borders"),
        ("border-pink-50", "borders"),
        ("border-pink-500", "borders"),
        ("border-pink-600", "borders"),
        ("border-pink-700", "borders"),
        ("border-pink-800", "borders"),
        ("border-pink-900", "borders"),
        ("border-purple-100", "borders"),
        ("border-purple-200", "borders"),
        ("border-purple-300", "borders"),
        ("border-purple-400", "borders"),
        ("border-purple-50", "borders"),
        ("border-purple-500", "borders"),
        ("border-purple-600", "borders"),
        ("border-purple-700", "borders"),
        ("border-purple-800", "borders"),
        ("border-purple-900", "borders"),
        ("border-r", "borders"),
        ("border-r-0", "borders"),
        ("border-r-2", "borders"),
        ("border-r-4", "borders"),
        ("border-r-8", "borders"),
        ("border-r-amber-100", "borders"),
        ("border-r-amber-200", "borders"),
        ("border-r-amber-300", "borders"),
        ("border-r-amber-400", "borders"),
        ("border-r-amber-50", "borders"),
        ("border-r-amber-500", "borders"),
        ("border-r-amber-600", "borders"),
        ("border-r-amber-700", "borders"),
        ("border-r-amber-800", "borders"),
        ("border-r-amber-900", "borders"),
        ("border-r-black", "borders"),
        ("border-r-blue-100", "borders"),
        ("border-r-blue-200", "borders"),
        ("border-r-blue-300", "borders"),
        ("border-r-blue-400", "borders"),
        ("border-r-blue-50", "borders"),
        ("border-r-blue-500", "borders"),
        ("border-r-blue-600", "borders"),
        ("border-r-blue-700", "borders"),
        ("border-r-blue-800", "borders"),
        ("border-r-blue-900", "borders"),
        ("border-r-current", "borders"),
        ("border-r-cyan-100", "borders"),
        ("border-r-cyan-200", "borders"),
        ("border-r-cyan-300", "borders"),
        ("border-r-cyan-400", "borders"),
        ("border-r-cyan-50", "borders"),
        ("border-r-cyan-500", "borders"),
        ("border-r-cyan-600", "borders"),
        ("border-r-cyan-700", "borders"),
        ("border-r-cyan-800", "borders"),
        ("border-r-cyan-900", "borders"),
        ("border-r-emerald-100", "borders"),
        ("border-r-emerald-200", "borders"),
        ("border-r-emerald-300", "borders"),
        ("border-r-emerald-400", "borders"),
        ("border-r-emerald-50", "borders"),
        ("border-r-emerald-500", "borders"),
        ("border-r-emerald-600", "borders"),
        ("border-r-emerald-700", "borders"),
        ("border-r-emerald-800", "borders"),
        ("border-r-emerald-900", "borders"),
        ("border-r-fuchsia-100", "borders"),
        ("border-r-fuchsia-200", "borders"),
        ("border-r-fuchsia-300", "borders"),
        ("border-r-fuchsia-400", "borders"),
        ("border-r-fuchsia-50", "borders"),
        ("border-r-fuchsia-500", "borders"),
        ("border-r-fuchsia-600", "borders"),
        ("border-r-fuchsia-700", "borders"),
        ("border-r-fuchsia-800", "borders"),
        ("border-r-fuchsia-900", "borders"),
        ("border-r-gray-100", "borders"),
        ("border-r-gray-200", "borders"),
        ("border-r-gray-300", "borders"),
        ("border-r-gray-400", "borders"),
        ("border-r-gray-50", "borders"),
        ("border-r-gray-500", "borders"),
        ("border-r-gray-600", "borders"),
        ("border-r-gray-700", "borders"),
        ("border-r-gray-800", "borders"),
        ("border-r-gray-900", "borders"),
        ("border-r-green-100", "borders"),
        ("border-r-green-200", "borders"),
        ("border-r-green-300", "borders"),
        ("border-r-green-400", "borders"),
        ("border-r-green-50", "borders"),
        ("border-r-green-500", "borders"),
        ("border-r-green-600", "borders"),
        ("border-r-green-700", "borders"),
        ("border-r-green-800", "borders"),
        ("border-r-green-900", "borders"),
        ("border-r-indigo-100", "borders"),
        ("border-r-indigo-200", "borders"),
        ("border-r-indigo-300", "borders"),
        ("border-r-indigo-400", "borders"),
        ("border-r-indigo-50", "borders"),
        ("border-r-indigo-500", "borders"),
        ("border-r-indigo-600", "borders"),
        ("border-r-indigo-700", "borders"),
        ("border-r-indigo-800", "borders"),
        ("border-r-indigo-900", "borders"),
        ("border-r-inherit", "borders"),
        ("border-r-lime-100", "borders"),
        ("border-r-lime-200", "borders"),
        ("border-r-lime-300", "borders"),
        ("border-r-lime-400", "borders"),
        ("border-r-lime-50", "borders"),
        ("border-r-lime-500", "borders"),
        ("border-r-lime-600", "borders"),
        ("border-r-lime-700", "borders"),
        ("border-r-lime-800", "borders"),
        ("border-r-lime-900", "borders"),
        ("border-r-neutral-100", "borders"),
        ("border-r-neutral-200", "borders"),
        ("border-r-neutral-300", "borders"),
        ("border-r-neutral-400", "borders"),
        ("border-r-neutral-50", "borders"),
        ("border-r-neutral-500", "borders"),
        ("border-r-neutral-600", "borders"),
        ("border-r-neutral-700", "borders"),
        ("border-r-neutral-800", "borders"),
        ("border-r-neutral-900", "borders"),
        ("border-r-orange-100", "borders"),
        ("border-r-orange-200", "borders"),
        ("border-r-orange-300", "borders"),
        ("border-r-orange-400", "borders"),
        ("border-r-orange-50", "borders"),
        ("border-r-orange-500", "borders"),
        ("border-r-orange-600", "borders"),
        ("border-r-orange-700", "borders"),
        ("border-r-orange-800", "borders"),
        ("border-r-orange-900", "borders"),
        ("border-r-pink-100", "borders"),
        ("border-r-pink-200", "borders"),
        ("border-r-pink-300", "borders"),
        ("border-r-pink-400", "borders"),
        ("border-r-pink-50", "borders"),
        ("border-r-pink-500", "borders"),
        ("border-r-pink-600", "borders"),
        ("border-r-pink-700", "borders"),
        ("border-r-pink-800", "borders"),
        ("border-r-pink-900", "borders"),
        ("border-r-purple-100", "borders"),
        ("border-r-purple-200", "borders"),
        ("border-r-purple-300", "borders"),
        ("border-r-purple-400", "borders"),
        ("border-r-purple-50", "borders"),
        ("border-r-purple-500", "borders"),
        ("border-r-purple-600", "borders"),
        ("border-r-purple-700", "borders"),
        ("border-r-purple-800", "borders"),
        ("border-r-purple-900", "borders"),
        ("border-r-red-100", "borders"),
        ("border-r-red-200", "borders"),
        ("border-r-red-300", "borders"),
        ("border-r-red-400", "borders"),
        ("border-r-red-50", "borders"),
        ("border-r-red-500", "borders"),
        ("border-r-red-600", "borders"),
        ("border-r-red-700", "borders"),
        ("border-r-red-800", "borders"),
        ("border-r-red-900", "borders"),
        ("border-r-rose-100", "borders"),
        ("border-r-rose-200", "borders"),
        ("border-r-rose-300", "borders"),
        ("border-r-rose-400", "borders"),
        ("border-r-rose-50", "borders"),
        ("border-r-rose-500", "borders"),
        ("border-r-rose-600", "borders"),
        ("border-r-rose-700", "borders"),
        ("border-r-rose-800", "borders"),
        ("border-r-rose-900", "borders"),
        ("border-r-sky-100", "borders"),
        ("border-r-sky-200", "borders"),
        ("border-r-sky-300", "borders"),
        ("border-r-sky-400", "borders"),
        ("border-r-sky-50", "borders"),
        ("border-r-sky-500", "borders"),
        ("border-r-sky-600", "borders"),
        ("border-r-sky-700", "borders"),
        ("border-r-sky-800", "borders"),
        ("border-r-sky-900", "borders"),
        ("border-r-slate-100", "borders"),
        ("border-r-slate-200", "borders"),
        ("border-r-slate-300", "borders"),
        ("border-r-slate-400", "borders"),
        ("border-r-slate-50", "borders"),
        ("border-r-slate-500", "borders"),
        ("border-r-slate-600", "borders"),
        ("border-r-slate-700", "borders"),
        ("border-r-slate-800", "borders"),
        ("border-r-slate-900", "borders"),
        ("border-r-stone-100", "borders"),
        ("border-r-stone-200", "borders"),
        ("border-r-stone-300", "borders"),
        ("border-r-stone-400", "borders"),
        ("border-r-stone-50", "borders"),
        ("border-r-stone-500", "borders"),
        ("border-r-stone-600", "borders"),
        ("border-r-stone-700", "borders"),
        ("border-r-stone-800", "borders"),
        ("border-r-stone-900", "borders"),
        ("border-r-teal-100", "borders"),
        ("border-r-teal-200", "borders"),
        ("border-r-teal-300", "borders"),
        ("border-r-teal-400", "borders"),
        ("border-r-teal-50", "borders"),
        ("border-r-teal-500", "borders"),
        ("border-r-teal-600", "borders"),
        ("border-r-teal-700", "borders"),
        ("border-r-teal-800", "borders"),
        ("border-r-teal-900", "borders"),
        ("border-r-transparent", "borders"),
        ("border-r-violet-100", "borders"),
        ("border-r-violet-200", "borders"),
        ("border-r-violet-300", "borders"),
        ("border-r-violet-400", "borders"),
        ("border-r-violet-50", "borders"),
        ("border-r-violet-500", "borders"),
        ("border-r-violet-600", "borders"),
        ("border-r-violet-700", "borders"),
        ("border-r-violet-800", "borders"),
        ("border-r-violet-900", "borders"),
        ("border-r-white", "borders"),
        ("border-r-yellow-100", "borders"),
        ("border-r-yellow-200", "borders"),
        ("border-r-yellow-300", "borders"),
        ("border-r-yellow-400", "borders"),
        ("border-r-yellow-50", "borders"),
        ("border-r-yellow-500", "borders"),
        ("border-r-yellow-600", "borders"),
        ("border-r-yellow-700", "borders"),
        ("border-r-yellow-800", "borders"),
        ("border-r-yellow-900", "borders"),
        ("border-r-zinc-100", "borders"),
        ("border-r-zinc-200", "borders"),
        ("border-r-zinc-300", "borders"),
        ("border-r-zinc-400", "borders"),
        ("border-r-zinc-50", "borders"),
        ("border-r-zinc-500", "borders"),
        ("border-r-zinc-600", "borders"),
        ("border-r-zinc-700", "borders"),
        ("border-r-zinc-800", "borders"),
        ("border-r-zinc-900", "borders"),
        ("border-red-100", "borders"),
        ("border-red-200", "borders"),
        ("border-red-300", "borders"),
        ("border-red-400", "borders"),
        ("border-red-50", "borders"),
        ("border-red-500", "borders"),
        ("border-red-600", "borders"),
        ("border-red-700", "borders"),
        ("border-red-800", "borders"),
        ("border-red-900", "borders"),
        ("border-rose-100", "borders"),
        ("border-rose-200", "borders"),
        ("border-rose-300", "borders"),
        ("border-rose-400", "borders"),
        ("border-rose-50", "borders"),
        ("border-rose-500", "borders"),
        ("border-rose-600", "borders"),
        ("border-rose-700", "borders"),
        ("border-rose-800", "borders"),
        ("border-rose-900", "borders"),
        ("border-sky-100", "borders"),
        ("border-sky-200", "borders"),
        ("border-sky-300", "borders"),
        ("border-sky-400", "borders"),
        ("border-sky-50", "borders"),
        ("border-sky-500", "borders"),
        ("border-sky-600", "borders"),
        ("border-sky-700", "borders"),
        ("border-sky-800", "borders"),
        ("border-sky-900", "borders"),
        ("border-slate-100", "borders"),
        ("border-slate-200", "borders"),
        ("border-slate-300", "borders"),
        ("border-slate-400", "borders"),
        ("border-slate-50", "borders"),
        ("border-slate-500", "borders"),
        ("border-slate-600", "borders"),
        ("border-slate-700", "borders"),
        ("border-slate-800", "borders"),
        ("border-slate-900", "borders"),
        ("border-solid", "borders"),
        ("border-stone-100", "borders"),
        ("border-stone-200", "borders"),
        ("border-stone-300", "borders"),
        ("border-stone-400", "borders"),
        ("border-stone-50", "borders"),
        ("border-stone-500", "borders"),
        ("border-stone-600", "borders"),
        ("border-stone-700", "borders"),
        ("border-stone-800", "borders"),
        ("border-stone-900", "borders"),
        ("border-t", "borders"),
        ("border-t-0", "borders"),
        ("border-t-2", "borders"),
        ("border-t-4", "borders"),
        ("border-t-8", "borders"),
        ("border-t-amber-100", "borders"),
        ("border-t-amber-200", "borders"),
        ("border-t-amber-300", "borders"),
        ("border-t-amber-400", "borders"),
        ("border-t-amber-50", "borders"),
        ("border-t-amber-500", "borders"),
        ("border-t-amber-600", "borders"),
        ("border-t-amber-700", "borders"),
        ("border-t-amber-800", "borders"),
        ("border-t-amber-900", "borders"),
        ("border-t-black", "borders"),
        ("border-t-blue-100", "borders"),
        ("border-t-blue-200", "borders"),
        ("border-t-blue-300", "borders"),
        ("border-t-blue-400", "borders"),
        ("border-t-blue-50", "borders"),
        ("border-t-blue-500", "borders"),
        ("border-t-blue-600", "borders"),
        ("border-t-blue-700", "borders"),
        ("border-t-blue-800", "borders"),
        ("border-t-blue-900", "borders"),
        ("border-t-current", "borders"),
        ("border-t-cyan-100", "borders"),
        ("border-t-cyan-200", "borders"),
        ("border-t-cyan-300", "borders"),
        ("border-t-cyan-400", "borders"),
        ("border-t-cyan-50", "borders"),
        ("border-t-cyan-500", "borders"),
        ("border-t-cyan-600", "borders"),
        ("border-t-cyan-700", "borders"),
        ("border-t-cyan-800", "borders"),
        ("border-t-cyan-900", "borders"),
        ("border-t-emerald-100", "borders"),
        ("border-t-emerald-200", "borders"),
        ("border-t-emerald-300", "borders"),
        ("border-t-emerald-400", "borders"),
        ("border-t-emerald-50", "borders"),
        ("border-t-emerald-500", "borders"),
        ("border-t-emerald-600", "borders"),
        ("border-t-emerald-700", "borders"),
        ("border-t-emerald-800", "borders"),
        ("border-t-emerald-900", "borders"),
        ("border-t-fuchsia-100", "borders"),
        ("border-t-fuchsia-200", "borders"),
        ("border-t-fuchsia-300", "borders"),
        ("border-t-fuchsia-400", "borders"),
        ("border-t-fuchsia-50", "borders"),
        ("border-t-fuchsia-500", "borders"),
        ("border-t-fuchsia-600", "borders"),
        ("border-t-fuchsia-700", "borders"),
        ("border-t-fuchsia-800", "borders"),
        ("border-t-fuchsia-900", "borders"),
        ("border-t-gray-100", "borders"),
        ("border-t-gray-200", "borders"),
        ("border-t-gray-300", "borders"),
        ("border-t-gray-400", "borders"),
        ("border-t-gray-50", "borders"),
        ("border-t-gray-500", "borders"),
        ("border-t-gray-600", "borders"),
        ("border-t-gray-700", "borders"),
        ("border-t-gray-800", "borders"),
        ("border-t-gray-900", "borders"),
        ("border-t-green-100", "borders"),
        ("border-t-green-200", "borders"),
        ("border-t-green-300", "borders"),
        ("border-t-green-400", "borders"),
        ("border-t-green-50", "borders"),
        ("border-t-green-500", "borders"),
        ("border-t-green-600", "borders"),
        ("border-t-green-700", "borders"),
        ("border-t-green-800", "borders"),
        ("border-t-green-900", "borders"),
        ("border-t-indigo-100", "borders"),
        ("border-t-indigo-200", "borders"),
        ("border-t-indigo-300", "borders"),
        ("border-t-indigo-400", "borders"),
        ("border-t-indigo-50", "borders"),
        ("border-t-indigo-500", "borders"),
        ("border-t-indigo-600", "borders"),
        ("border-t-indigo-700", "borders"),
        ("border-t-indigo-800", "borders"),
        ("border-t-indigo-900", "borders"),
        ("border-t-inherit", "borders"),
        ("border-t-lime-100", "borders"),
        ("border-t-lime-200", "borders"),
        ("border-t-lime-300", "borders"),
        ("border-t-lime-400", "borders"),
        ("border-t-lime-50", "borders"),
        ("border-t-lime-500", "borders"),
        ("border-t-lime-600", "borders"),
        ("border-t-lime-700", "borders"),
        ("border-t-lime-800", "borders"),
        ("border-t-lime-900", "borders"),
        ("border-t-neutral-100", "borders"),
        ("border-t-neutral-200", "borders"),
        ("border-t-neutral-300", "borders"),
        ("border-t-neutral-400", "borders"),
        ("border-t-neutral-50", "borders"),
        ("border-t-neutral-500", "borders"),
        ("border-t-neutral-600", "borders"),
        ("border-t-neutral-700", "borders"),
        ("border-t-neutral-800", "borders"),
        ("border-t-neutral-900", "borders"),
        ("border-t-orange-100", "borders"),
        ("border-t-orange-200", "borders"),
        ("border-t-orange-300", "borders"),
        ("border-t-orange-400", "borders"),
        ("border-t-orange-50", "borders"),
        ("border-t-orange-500", "borders"),
        ("border-t-orange-600", "borders"),
        ("border-t-orange-700", "borders"),
        ("border-t-orange-800", "borders"),
        ("border-t-orange-900", "borders"),
        ("border-t-pink-100", "borders"),
        ("border-t-pink-200", "borders"),
        ("border-t-pink-300", "borders"),
        ("border-t-pink-400", "borders"),
        ("border-t-pink-50", "borders"),
        ("border-t-pink-500", "borders"),
        ("border-t-pink-600", "borders"),
        ("border-t-pink-700", "borders"),
        ("border-t-pink-800", "borders"),
        ("border-t-pink-900", "borders"),
        ("border-t-purple-100", "borders"),
        ("border-t-purple-200", "borders"),
        ("border-t-purple-300", "borders"),
        ("border-t-purple-400", "borders"),
        ("border-t-purple-50", "borders"),
        ("border-t-purple-500", "borders"),
        ("border-t-purple-600", "borders"),
        ("border-t-purple-700", "borders"),
        ("border-t-purple-800", "borders"),
        ("border-t-purple-900", "borders"),
        ("border-t-red-100", "borders"),
        ("border-t-red-200", "borders"),
        ("border-t-red-300", "borders"),
        ("border-t-red-400", "borders"),
        ("border-t-red-50", "borders"),
        ("border-t-red-500", "borders"),
        ("border-t-red-600", "borders"),
        ("border-t-red-700", "borders"),
        ("border-t-red-800", "borders"),
        ("border-t-red-900", "borders"),
        ("border-t-rose-100", "borders"),
        ("border-t-rose-200", "borders"),
        ("border-t-rose-300", "borders"),
        ("border-t-rose-400", "borders"),
        ("border-t-rose-50", "borders"),
        ("border-t-rose-500", "borders"),
        ("border-t-rose-600", "borders"),
        ("border-t-rose-700", "borders"),
        ("border-t-rose-800", "borders"),
        ("border-t-rose-900", "borders"),
        ("border-t-sky-100", "borders"),
        ("border-t-sky-200", "borders"),
        ("border-t-sky-300", "borders"),
        ("border-t-sky-400", "borders"),
        ("border-t-sky-50", "borders"),
        ("border-t-sky-500", "borders"),
        ("border-t-sky-600", "borders"),
        ("border-t-sky-700", "borders"),
        ("border-t-sky-800", "borders"),
        ("border-t-sky-900", "borders"),
        ("border-t-slate-100", "borders"),
        ("border-t-slate-200", "borders"),
        ("border-t-slate-300", "borders"),
        ("border-t-slate-400", "borders"),
        ("border-t-slate-50", "borders"),
        ("border-t-slate-500", "borders"),
        ("border-t-slate-600", "borders"),
        ("border-t-slate-700", "borders"),
        ("border-t-slate-800", "borders"),
        ("border-t-slate-900", "borders"),
        ("border-t-stone-100", "borders"),
        ("border-t-stone-200", "borders"),
        ("border-t-stone-300", "borders"),
        ("border-t-stone-400", "borders"),
        ("border-t-stone-50", "borders"),
        ("border-t-stone-500", "borders"),
        ("border-t-stone-600", "borders"),
        ("border-t-stone-700", "borders"),
        ("border-t-stone-800", "borders"),
        ("border-t-stone-900", "borders"),
        ("border-t-teal-100", "borders"),
        ("border-t-teal-200", "borders"),
        ("border-t-teal-300", "borders"),
        ("border-t-teal-400", "borders"),
        ("border-t-teal-50", "borders"),
        ("border-t-teal-500", "borders"),
        ("border-t-teal-600", "borders"),
        ("border-t-teal-700", "borders"),
        ("border-t-teal-800", "borders"),
        ("border-t-teal-900", "borders"),
        ("border-t-transparent", "borders"),
        ("border-t-violet-100", "borders"),
        ("border-t-violet-200", "borders"),
        ("border-t-violet-300", "borders"),
        ("border-t-violet-400", "borders"),
        ("border-t-violet-50", "borders"),
        ("border-t-violet-500", "borders"),
        ("border-t-violet-600", "borders"),
        ("border-t-violet-700", "borders"),
        ("border-t-violet-800", "borders"),
        ("border-t-violet-900", "borders"),
        ("border-t-white", "borders"),
        ("border-t-yellow-100", "borders"),
        ("border-t-yellow-200", "borders"),
        ("border-t-yellow-300", "borders"),
        ("border-t-yellow-400", "borders"),
        ("border-t-yellow-50", "borders"),
        ("border-t-yellow-500", "borders"),
        ("border-t-yellow-600", "borders"),
        ("border-t-yellow-700", "borders"),
        ("border-t-yellow-800", "borders"),
        ("border-t-yellow-900", "borders"),
        ("border-t-zinc-100", "borders"),
        ("border-t-zinc-200", "borders"),
        ("border-t-zinc-300", "borders"),
        ("border-t-zinc-400", "borders"),
        ("border-t-zinc-50", "borders"),
        ("border-t-zinc-500", "borders"),
        ("border-t-zinc-600", "borders"),
        ("border-t-zinc-700", "borders"),
        ("border-t-zinc-800", "borders"),
        ("border-t-zinc-900", "borders"),
        ("border-teal-100", "borders"),
        ("border-teal-200", "borders"),
        ("border-teal-300", "borders"),
        ("border-teal-400", "borders"),
        ("border-teal-50", "borders"),
        ("border-teal-500", "borders"),
        ("border-teal-600", "borders"),
        ("border-teal-700", "borders"),
        ("border-teal-800", "borders"),
        ("border-teal-900", "borders"),
        ("border-transparent", "borders"),
        ("border-violet-100", "borders"),
        ("border-violet-200", "borders"),
        ("border-violet-300", "borders"),
        ("border-violet-400", "borders"),
        ("border-violet-50", "borders"),
        ("border-violet-500", "borders"),
        ("border-violet-600", "borders"),
        ("border-violet-700", "borders"),
        ("border-violet-800", "borders"),
        ("border-violet-900", "borders"),
        ("border-white", "borders"),
        ("border-x", "borders"),
        ("border-x-0", "borders"),
        ("border-x-2", "borders"),
        ("border-x-4", "borders"),
        ("border-x-8", "borders"),
        ("border-x-amber-100", "borders"),
        ("border-x-amber-200", "borders"),
        ("border-x-amber-300", "borders"),
        ("border-x-amber-400", "borders"),
        ("border-x-amber-50", "borders"),
        ("border-x-amber-500", "borders"),
        ("border-x-amber-600", "borders"),
        ("border-x-amber-700", "borders"),
        ("border-x-amber-800", "borders"),
        ("border-x-amber-900", "borders"),
        ("border-x-black", "borders"),
        ("border-x-blue-100", "borders"),
        ("border-x-blue-200", "borders"),
        ("border-x-blue-300", "borders"),
        ("border-x-blue-400", "borders"),
        ("border-x-blue-50", "borders"),
        ("border-x-blue-500", "borders"),
        ("border-x-blue-600", "borders"),
        ("border-x-blue-700", "borders"),
        ("border-x-blue-800", "borders"),
        ("border-x-blue-900", "borders"),
        ("border-x-current", "borders"),
        ("border-x-cyan-100", "borders"),
        ("border-x-cyan-200", "borders"),
        ("border-x-cyan-300", "borders"),
        ("border-x-cyan-400", "borders"),
        ("border-x-cyan-50", "borders"),
        ("border-x-cyan-500", "borders"),
        ("border-x-cyan-600", "borders"),
        ("border-x-cyan-700", "borders"),
        ("border-x-cyan-800", "borders"),
        ("border-x-cyan-900", "borders"),
        ("border-x-emerald-100", "borders"),
        ("border-x-emerald-200", "borders"),
        ("border-x-emerald-300", "borders"),
        ("border-x-emerald-400", "borders"),
        ("border-x-emerald-50", "borders"),
        ("border-x-emerald-500", "borders"),
        ("border-x-emerald-600", "borders"),
        ("border-x-emerald-700", "borders"),
        ("border-x-emerald-800", "borders"),
        ("border-x-emerald-900", "borders"),
        ("border-x-fuchsia-100", "borders"),
        ("border-x-fuchsia-200", "borders"),
        ("border-x-fuchsia-300", "borders"),
        ("border-x-fuchsia-400", "borders"),
        ("border-x-fuchsia-50", "borders"),
        ("border-x-fuchsia-500", "borders"),
        ("border-x-fuchsia-600", "borders"),
        ("border-x-fuchsia-700", "borders"),
        ("border-x-fuchsia-800", "borders"),
        ("border-x-fuchsia-900", "borders"),
        ("border-x-gray-100", "borders"),
        ("border-x-gray-200", "borders"),
        ("border-x-gray-300", "borders"),
        ("border-x-gray-400", "borders"),
        ("border-x-gray-50", "borders"),
        ("border-x-gray-500", "borders"),
        ("border-x-gray-600", "borders"),
        ("border-x-gray-700", "borders"),
        ("border-x-gray-800", "borders"),
        ("border-x-gray-900", "borders"),
        ("border-x-green-100", "borders"),
        ("border-x-green-200", "borders"),
        ("border-x-green-300", "borders"),
        ("border-x-green-400", "borders"),
        ("border-x-green-50", "borders"),
        ("border-x-green-500", "borders"),
        ("border-x-green-600", "borders"),
        ("border-x-green-700", "borders"),
        ("border-x-green-800", "borders"),
        ("border-x-green-900", "borders"),
        ("border-x-indigo-100", "borders"),
        ("border-x-indigo-200", "borders"),
        ("border-x-indigo-300", "borders"),
        ("border-x-indigo-400", "borders"),
        ("border-x-indigo-50", "borders"),
        ("border-x-indigo-500", "borders"),
        ("border-x-indigo-600", "borders"),
        ("border-x-indigo-700", "borders"),
        ("border-x-indigo-800", "borders"),
        ("border-x-indigo-900", "borders"),
        ("border-x-inherit", "borders"),
        ("border-x-lime-100", "borders"),
        ("border-x-lime-200", "borders"),
        ("border-x-lime-300", "borders"),
        ("border-x-lime-400", "borders"),
        ("border-x-lime-50", "borders"),
        ("border-x-lime-500", "borders"),
        ("border-x-lime-600", "borders"),
        ("border-x-lime-700", "borders"),
        ("border-x-lime-800", "borders"),
        ("border-x-lime-900", "borders"),
        ("border-x-neutral-100", "borders"),
        ("border-x-neutral-200", "borders"),
        ("border-x-neutral-300", "borders"),
        ("border-x-neutral-400", "borders"),
        ("border-x-neutral-50", "borders"),
        ("border-x-neutral-500", "borders"),
        ("border-x-neutral-600", "borders"),
        ("border-x-neutral-700", "borders"),
        ("border-x-neutral-800", "borders"),
        ("border-x-neutral-900", "borders"),
        ("border-x-orange-100", "borders"),
        ("border-x-orange-200", "borders"),
        ("border-x-orange-300", "borders"),
        ("border-x-orange-400", "borders"),
        ("border-x-orange-50", "borders"),
        ("border-x-orange-500", "borders"),
        ("border-x-orange-600", "borders"),
        ("border-x-orange-700", "borders"),
        ("border-x-orange-800", "borders"),
        ("border-x-orange-900", "borders"),
        ("border-x-pink-100", "borders"),
        ("border-x-pink-200", "borders"),
        ("border-x-pink-300", "borders"),
        ("border-x-pink-400", "borders"),
        ("border-x-pink-50", "borders"),
        ("border-x-pink-500", "borders"),
        ("border-x-pink-600", "borders"),
        ("border-x-pink-700", "borders"),
        ("border-x-pink-800", "borders"),
        ("border-x-pink-900", "borders"),
        ("border-x-purple-100", "borders"),
        ("border-x-purple-200", "borders"),
        ("border-x-purple-300", "borders"),
        ("border-x-purple-400", "borders"),
        ("border-x-purple-50", "borders"),
        ("border-x-purple-500", "borders"),
        ("border-x-purple-600", "borders"),
        ("border-x-purple-700", "borders"),
        ("border-x-purple-800", "borders"),
        ("border-x-purple-900", "borders"),
        ("border-x-red-100", "borders"),
        ("border-x-red-200", "borders"),
        ("border-x-red-300", "borders"),
        ("border-x-red-400", "borders"),
        ("border-x-red-50", "borders"),
        ("border-x-red-500", "borders"),
        ("border-x-red-600", "borders"),
        ("border-x-red-700", "borders"),
        ("border-x-red-800", "borders"),
        ("border-x-red-900", "borders"),
        ("border-x-rose-100", "borders"),
        ("border-x-rose-200", "borders"),
        ("border-x-rose-300", "borders"),
        ("border-x-rose-400", "borders"),
        ("border-x-rose-50", "borders"),
        ("border-x-rose-500", "borders"),
        ("border-x-rose-600", "borders"),
        ("border-x-rose-700", "borders"),
        ("border-x-rose-800", "borders"),
        ("border-x-rose-900", "borders"),
        ("border-x-sky-100", "borders"),
        ("border-x-sky-200", "borders"),
        ("border-x-sky-300", "borders"),
        ("border-x-sky-400", "borders"),
        ("border-x-sky-50", "borders"),
        ("border-x-sky-500", "borders"),
        ("border-x-sky-600", "borders"),
        ("border-x-sky-700", "borders"),
        ("border-x-sky-800", "borders"),
        ("border-x-sky-900", "borders"),
        ("border-x-slate-100", "borders"),
        ("border-x-slate-200", "borders"),
        ("border-x-slate-300", "borders"),
        ("border-x-slate-400", "borders"),
        ("border-x-slate-50", "borders"),
        ("border-x-slate-500", "borders"),
        ("border-x-slate-600", "borders"),
        ("border-x-slate-700", "borders"),
        ("border-x-slate-800", "borders"),
        ("border-x-slate-900", "borders"),
        ("border-x-stone-100", "borders"),
        ("border-x-stone-200", "borders"),
        ("border-x-stone-300", "borders"),
        ("border-x-stone-400", "borders"),
        ("border-x-stone-50", "borders"),
        ("border-x-stone-500", "borders"),
        ("border-x-stone-600", "borders"),
        ("border-x-stone-700", "borders"),
        ("border-x-stone-800", "borders"),
        ("border-x-stone-900", "borders"),
        ("border-x-teal-100", "borders"),
        ("border-x-teal-200", "borders"),
        ("border-x-teal-300", "borders"),
        ("border-x-teal-400", "borders"),
        ("border-x-teal-50", "borders"),
        ("border-x-teal-500", "borders"),
        ("border-x-teal-600", "borders"),
        ("border-x-teal-700", "borders"),
        ("border-x-teal-800", "borders"),
        ("border-x-teal-900", "borders"),
        ("border-x-transparent", "borders"),
        ("border-x-violet-100", "borders"),
        ("border-x-violet-200", "borders"),
        ("border-x-violet-300", "borders"),
        ("border-x-violet-400", "borders"),
        ("border-x-violet-50", "borders"),
        ("border-x-violet-500", "borders"),
        ("border-x-violet-600", "borders"),
        ("border-x-violet-700", "borders"),
        ("border-x-violet-800", "borders"),
        ("border-x-violet-900", "borders"),
        ("border-x-white", "borders"),
        ("border-x-yellow-100", "borders"),
        ("border-x-yellow-200", "borders"),
        ("border-x-yellow-300", "borders"),
        ("border-x-yellow-400", "borders"),
        ("border-x-yellow-50", "borders"),
        ("border-x-yellow-500", "borders"),
        ("border-x-yellow-600", "borders"),
        ("border-x-yellow-700", "borders"),
        ("border-x-yellow-800", "borders"),
        ("border-x-yellow-900", "borders"),
        ("border-x-zinc-100", "borders"),
        ("border-x-zinc-200", "borders"),
        ("border-x-zinc-300", "borders"),
        ("border-x-zinc-400", "borders"),
        ("border-x-zinc-50", "borders"),
        ("border-x-zinc-500", "borders"),
        ("border-x-zinc-600", "borders"),
        ("border-x-zinc-700", "borders"),
        ("border-x-zinc-800", "borders"),
        ("border-x-zinc-900", "borders"),
        ("border-y", "borders"),
        ("border-y-0", "borders"),
        ("border-y-2", "borders"),
        ("border-y-4", "borders"),
        ("border-y-8", "borders"),
        ("border-y-amber-100", "borders"),
        ("border-y-amber-200", "borders"),
        ("border-y-amber-300", "borders"),
        ("border-y-amber-400", "borders"),
        ("border-y-amber-50", "borders"),
        ("border-y-amber-500", "borders"),
        ("border-y-amber-600", "borders"),
        ("border-y-amber-700", "borders"),
        ("border-y-amber-800", "borders"),
        ("border-y-amber-900", "borders"),
        ("border-y-black", "borders"),
        ("border-y-blue-100", "borders"),
        ("border-y-blue-200", "borders"),
        ("border-y-blue-300", "borders"),
        ("border-y-blue-400", "borders"),
        ("border-y-blue-50", "borders"),
        ("border-y-blue-500", "borders"),
        ("border-y-blue-600", "borders"),
        ("border-y-blue-700", "borders"),
        ("border-y-blue-800", "borders"),
        ("border-y-blue-900", "borders"),
        ("border-y-current", "borders"),
        ("border-y-cyan-100", "borders"),
        ("border-y-cyan-200", "borders"),
        ("border-y-cyan-300", "borders"),
        ("border-y-cyan-400", "borders"),
        ("border-y-cyan-50", "borders"),
        ("border-y-cyan-500", "borders"),
        ("border-y-cyan-600", "borders"),
        ("border-y-cyan-700", "borders"),
        ("border-y-cyan-800", "borders"),
        ("border-y-cyan-900", "borders"),
        ("border-y-emerald-100", "borders"),
        ("border-y-emerald-200", "borders"),
        ("border-y-emerald-300", "borders"),
        ("border-y-emerald-400", "borders"),
        ("border-y-emerald-50", "borders"),
        ("border-y-emerald-500", "borders"),
        ("border-y-emerald-600", "borders"),
        ("border-y-emerald-700", "borders"),
        ("border-y-emerald-800", "borders"),
        ("border-y-emerald-900", "borders"),
        ("border-y-fuchsia-100", "borders"),
        ("border-y-fuchsia-200", "borders"),
        ("border-y-fuchsia-300", "borders"),
        ("border-y-fuchsia-400", "borders"),
        ("border-y-fuchsia-50", "borders"),
        ("border-y-fuchsia-500", "borders"),
        ("border-y-fuchsia-600", "borders"),
        ("border-y-fuchsia-700", "borders"),
        ("border-y-fuchsia-800", "borders"),
        ("border-y-fuchsia-900", "borders"),
        ("border-y-gray-100", "borders"),
        ("border-y-gray-200", "borders"),
        ("border-y-gray-300", "borders"),
        ("border-y-gray-400", "borders"),
        ("border-y-gray-50", "borders"),
        ("border-y-gray-500", "borders"),
        ("border-y-gray-600", "borders"),
        ("border-y-gray-700", "borders"),
        ("border-y-gray-800", "borders"),
        ("border-y-gray-900", "borders"),
        ("border-y-green-100", "borders"),
        ("border-y-green-200", "borders"),
        ("border-y-green-300", "borders"),
        ("border-y-green-400", "borders"),
        ("border-y-green-50", "borders"),
        ("border-y-green-500", "borders"),
        ("border-y-green-600", "borders"),
        ("border-y-green-700", "borders"),
        ("border-y-green-800", "borders"),
        ("border-y-green-900", "borders"),
        ("border-y-indigo-100", "borders"),
        ("border-y-indigo-200", "borders"),
        ("border-y-indigo-300", "borders"),
        ("border-y-indigo-400", "borders"),
        ("border-y-indigo-50", "borders"),
        ("border-y-indigo-500", "borders"),
        ("border-y-indigo-600", "borders"),
        ("border-y-indigo-700", "borders"),
        ("border-y-indigo-800", "borders"),
        ("border-y-indigo-900", "borders"),
        ("border-y-inherit", "borders"),
        ("border-y-lime-100", "borders"),
        ("border-y-lime-200", "borders"),
        ("border-y-lime-300", "borders"),
        ("border-y-lime-400", "borders"),
        ("border-y-lime-50", "borders"),
        ("border-y-lime-500", "borders"),
        ("border-y-lime-600", "borders"),
        ("border-y-lime-700", "borders"),
        ("border-y-lime-800", "borders"),
        ("border-y-lime-900", "borders"),
        ("border-y-neutral-100", "borders"),
        ("border-y-neutral-200", "borders"),
        ("border-y-neutral-300", "borders"),
        ("border-y-neutral-400", "borders"),
        ("border-y-neutral-50", "borders"),
        ("border-y-neutral-500", "borders"),
        ("border-y-neutral-600", "borders"),
        ("border-y-neutral-700", "borders"),
        ("border-y-neutral-800", "borders"),
        ("border-y-neutral-900", "borders"),
        ("border-y-orange-100", "borders"),
        ("border-y-orange-200", "borders"),
        ("border-y-orange-300", "borders"),
        ("border-y-orange-400", "borders"),
        ("border-y-orange-50", "borders"),
        ("border-y-orange-500", "borders"),
        ("border-y-orange-600", "borders"),
        ("border-y-orange-700", "borders"),
        ("border-y-orange-800", "borders"),
        ("border-y-orange-900", "borders"),
        ("border-y-pink-100", "borders"),
        ("border-y-pink-200", "borders"),
        ("border-y-pink-300", "borders"),
        ("border-y-pink-400", "borders"),
        ("border-y-pink-50", "borders"),
        ("border-y-pink-500", "borders"),
        ("border-y-pink-600", "borders"),
        ("border-y-pink-700", "borders"),
        ("border-y-pink-800", "borders"),
        ("border-y-pink-900", "borders"),
        ("border-y-purple-100", "borders"),
        ("border-y-purple-200", "borders"),
        ("border-y-purple-300", "borders"),
        ("border-y-purple-400", "borders"),
        ("border-y-purple-50", "borders"),
        ("border-y-purple-500", "borders"),
        ("border-y-purple-600", "borders"),
        ("border-y-purple-700", "borders"),
        ("border-y-purple-800", "borders"),
        ("border-y-purple-900", "borders"),
        ("border-y-red-100", "borders"),
        ("border-y-red-200", "borders"),
        ("border-y-red-300", "borders"),
        ("border-y-red-400", "borders"),
        ("border-y-red-50", "borders"),
        ("border-y-red-500", "borders"),
        ("border-y-red-600", "borders"),
        ("border-y-red-700", "borders"),
        ("border-y-red-800", "borders"),
        ("border-y-red-900", "borders"),
        ("border-y-rose-100", "borders"),
        ("border-y-rose-200", "borders"),
        ("border-y-rose-300", "borders"),
        ("border-y-rose-400", "borders"),
        ("border-y-rose-50", "borders"),
        ("border-y-rose-500", "borders"),
        ("border-y-rose-600", "borders"),
        ("border-y-rose-700", "borders"),
        ("border-y-rose-800", "borders"),
        ("border-y-rose-900", "borders"),
        ("border-y-sky-100", "borders"),
        ("border-y-sky-200", "borders"),
        ("border-y-sky-300", "borders"),
        ("border-y-sky-400", "borders"),
        ("border-y-sky-50", "borders"),
        ("border-y-sky-500", "borders"),
        ("border-y-sky-600", "borders"),
        ("border-y-sky-700", "borders"),
        ("border-y-sky-800", "borders"),
        ("border-y-sky-900", "borders"),
        ("border-y-slate-100", "borders"),
        ("border-y-slate-200", "borders"),
        ("border-y-slate-300", "borders"),
        ("border-y-slate-400", "borders"),
        ("border-y-slate-50", "borders"),
        ("border-y-slate-500", "borders"),
        ("border-y-slate-600", "borders"),
        ("border-y-slate-700", "borders"),
        ("border-y-slate-800", "borders"),
        ("border-y-slate-900", "borders"),
        ("border-y-stone-100", "borders"),
        ("border-y-stone-200", "borders"),
        ("border-y-stone-300", "borders"),
        ("border-y-stone-400", "borders"),
        ("border-y-stone-50", "borders"),
        ("border-y-stone-500", "borders"),
        ("border-y-stone-600", "borders"),
        ("border-y-stone-700", "borders"),
        ("border-y-stone-800", "borders"),
        ("border-y-stone-900", "borders"),
        ("border-y-teal-100", "borders"),
        ("border-y-teal-200", "borders"),
        ("border-y-teal-300", "borders"),
        ("border-y-teal-400", "borders"),
        ("border-y-teal-50", "borders"),
        ("border-y-teal-500", "borders"),
        ("border-y-teal-600", "borders"),
        ("border-y-teal-700", "borders"),
        ("border-y-teal-800", "borders"),
        ("border-y-teal-900", "borders"),
        ("border-y-transparent", "borders"),
        ("border-y-violet-100", "borders"),
        ("border-y-violet-200", "borders"),
        ("border-y-violet-300", "borders"),
        ("border-y-violet-400", "borders"),
        ("border-y-violet-50", "borders"),
        ("border-y-violet-500", "borders"),
        ("border-y-violet-600", "borders"),
        ("border-y-violet-700", "borders"),
        ("border-y-violet-800", "borders"),
        ("border-y-violet-900", "borders"),
        ("border-y-white", "borders"),
        ("border-y-yellow-100", "borders"),
        ("border-y-yellow-200", "borders"),
        ("border-y-yellow-300", "borders"),
        ("border-y-yellow-400", "borders"),
        ("border-y-yellow-50", "borders"),
        ("border-y-yellow-500", "borders"),
        ("border-y-yellow-600", "borders"),
        ("border-y-yellow-700", "borders"),
        ("border-y-yellow-800", "borders"),
        ("border-y-yellow-900", "borders"),
        ("border-y-zinc-100", "borders"),
        ("border-y-zinc-200", "borders"),
        ("border-y-zinc-300", "borders"),
        ("border-y-zinc-400", "borders"),
        ("border-y-zinc-50", "borders"),
        ("border-y-zinc-500", "borders"),
        ("border-y-zinc-600", "borders"),
        ("border-y-zinc-700", "borders"),
        ("border-y-zinc-800", "borders"),
        ("border-y-zinc-900", "borders"),
        ("border-yellow-100", "borders"),
        ("border-yellow-200", "borders"),
        ("border-yellow-300", "borders"),
        ("border-yellow-400", "borders"),
        ("border-yellow-50", "borders"),
        ("border-yellow-500", "borders"),
        ("border-yellow-600", "borders"),
        ("border-yellow-700", "borders"),
        ("border-yellow-800", "borders"),
        ("border-yellow-900", "borders"),
        ("border-zinc-100", "borders"),
        ("border-zinc-200", "borders"),
        ("border-zinc-300", "borders"),
        ("border-zinc-400", "borders"),
        ("border-zinc-50", "borders"),
        ("border-zinc-500", "borders"),
        ("border-zinc-600", "borders"),
        ("border-zinc-700", "borders"),
        ("border-zinc-800", "borders"),
        ("border-zinc-900", "borders"),
        ("divide-amber-100 ", "borders"),
        ("divide-amber-200 ", "borders"),
        ("divide-amber-300 ", "borders"),
        ("divide-amber-400 ", "borders"),
        ("divide-amber-50 ", "borders"),
        ("divide-amber-500 ", "borders"),
        ("divide-amber-600 ", "borders"),
        ("divide-amber-700 ", "borders"),
        ("divide-amber-800 ", "borders"),
        ("divide-amber-900 ", "borders"),
        ("divide-black ", "borders"),
        ("divide-blue-100 ", "borders"),
        ("divide-blue-200 ", "borders"),
        ("divide-blue-300 ", "borders"),
        ("divide-blue-400 ", "borders"),
        ("divide-blue-50 ", "borders"),
        ("divide-blue-500 ", "borders"),
        ("divide-blue-600 ", "borders"),
        ("divide-blue-700 ", "borders"),
        ("divide-blue-800 ", "borders"),
        ("divide-blue-900 ", "borders"),
        ("divide-current ", "borders"),
        ("divide-cyan-100 ", "borders"),
        ("divide-cyan-200 ", "borders"),
        ("divide-cyan-300 ", "borders"),
        ("divide-cyan-400 ", "borders"),
        ("divide-cyan-50 ", "borders"),
        ("divide-cyan-500 ", "borders"),
        ("divide-cyan-600 ", "borders"),
        ("divide-cyan-700 ", "borders"),
        ("divide-cyan-800 ", "borders"),
        ("divide-cyan-900 ", "borders"),
        ("divide-dashed ", "borders"),
        ("divide-dotted ", "borders"),
        ("divide-double ", "borders"),
        ("divide-emerald-100 ", "borders"),
        ("divide-emerald-200 ", "borders"),
        ("divide-emerald-300 ", "borders"),
        ("divide-emerald-400 ", "borders"),
        ("divide-emerald-50 ", "borders"),
        ("divide-emerald-500 ", "borders"),
        ("divide-emerald-600 ", "borders"),
        ("divide-emerald-700 ", "borders"),
        ("divide-emerald-800 ", "borders"),
        ("divide-emerald-900 ", "borders"),
        ("divide-fuchsia-100 ", "borders"),
        ("divide-fuchsia-200 ", "borders"),
        ("divide-fuchsia-300 ", "borders"),
        ("divide-fuchsia-400 ", "borders"),
        ("divide-fuchsia-50 ", "borders"),
        ("divide-fuchsia-500 ", "borders"),
        ("divide-fuchsia-600 ", "borders"),
        ("divide-fuchsia-700 ", "borders"),
        ("divide-fuchsia-800 ", "borders"),
        ("divide-fuchsia-900 ", "borders"),
        ("divide-gray-100 ", "borders"),
        ("divide-gray-200 ", "borders"),
        ("divide-gray-300 ", "borders"),
        ("divide-gray-400 ", "borders"),
        ("divide-gray-50 ", "borders"),
        ("divide-gray-500 ", "borders"),
        ("divide-gray-600 ", "borders"),
        ("divide-gray-700 ", "borders"),
        ("divide-gray-800 ", "borders"),
        ("divide-gray-900 ", "borders"),
        ("divide-green-100 ", "borders"),
        ("divide-green-200 ", "borders"),
        ("divide-green-300 ", "borders"),
        ("divide-green-400 ", "borders"),
        ("divide-green-50 ", "borders"),
        ("divide-green-500 ", "borders"),
        ("divide-green-600 ", "borders"),
        ("divide-green-700 ", "borders"),
        ("divide-green-800 ", "borders"),
        ("divide-green-900 ", "borders"),
        ("divide-indigo-100 ", "borders"),
        ("divide-indigo-200 ", "borders"),
        ("divide-indigo-300 ", "borders"),
        ("divide-indigo-400 ", "borders"),
        ("divide-indigo-50 ", "borders"),
        ("divide-indigo-500 ", "borders"),
        ("divide-indigo-600 ", "borders"),
        ("divide-indigo-700 ", "borders"),
        ("divide-indigo-800 ", "borders"),
        ("divide-indigo-900 ", "borders"),
        ("divide-inherit ", "borders"),
        ("divide-lime-100 ", "borders"),
        ("divide-lime-200 ", "borders"),
        ("divide-lime-300 ", "borders"),
        ("divide-lime-400 ", "borders"),
        ("divide-lime-50 ", "borders"),
        ("divide-lime-500 ", "borders"),
        ("divide-lime-600 ", "borders"),
        ("divide-lime-700 ", "borders"),
        ("divide-lime-800 ", "borders"),
        ("divide-lime-900 ", "borders"),
        ("divide-neutral-100 ", "borders"),
        ("divide-neutral-200 ", "borders"),
        ("divide-neutral-300 ", "borders"),
        ("divide-neutral-400 ", "borders"),
        ("divide-neutral-50 ", "borders"),
        ("divide-neutral-500 ", "borders"),
        ("divide-neutral-600 ", "borders"),
        ("divide-neutral-700 ", "borders"),
        ("divide-neutral-800 ", "borders"),
        ("divide-neutral-900 ", "borders"),
        ("divide-none ", "borders"),
        ("divide-orange-100 ", "borders"),
        ("divide-orange-200 ", "borders"),
        ("divide-orange-300 ", "borders"),
        ("divide-orange-400 ", "borders"),
        ("divide-orange-50 ", "borders"),
        ("divide-orange-500 ", "borders"),
        ("divide-orange-600 ", "borders"),
        ("divide-orange-700 ", "borders"),
        ("divide-orange-800 ", "borders"),
        ("divide-orange-900 ", "borders"),
        ("divide-pink-100 ", "borders"),
        ("divide-pink-200 ", "borders"),
        ("divide-pink-300 ", "borders"),
        ("divide-pink-400 ", "borders"),
        ("divide-pink-50 ", "borders"),
        ("divide-pink-500 ", "borders"),
        ("divide-pink-600 ", "borders"),
        ("divide-pink-700 ", "borders"),
        ("divide-pink-800 ", "borders"),
        ("divide-pink-900 ", "borders"),
        ("divide-purple-100 ", "borders"),
        ("divide-purple-200 ", "borders"),
        ("divide-purple-300 ", "borders"),
        ("divide-purple-400 ", "borders"),
        ("divide-purple-50 ", "borders"),
        ("divide-purple-500 ", "borders"),
        ("divide-purple-600 ", "borders"),
        ("divide-purple-700 ", "borders"),
        ("divide-purple-800 ", "borders"),
        ("divide-purple-900 ", "borders"),
        ("divide-red-100 ", "borders"),
        ("divide-red-200 ", "borders"),
        ("divide-red-300 ", "borders"),
        ("divide-red-400 ", "borders"),
        ("divide-red-50 ", "borders"),
        ("divide-red-500 ", "borders"),
        ("divide-red-600 ", "borders"),
        ("divide-red-700 ", "borders"),
        ("divide-red-800 ", "borders"),
        ("divide-red-900 ", "borders"),
        ("divide-rose-100 ", "borders"),
        ("divide-rose-200 ", "borders"),
        ("divide-rose-300 ", "borders"),
        ("divide-rose-400 ", "borders"),
        ("divide-rose-50 ", "borders"),
        ("divide-rose-500 ", "borders"),
        ("divide-rose-600 ", "borders"),
        ("divide-rose-700 ", "borders"),
        ("divide-rose-800 ", "borders"),
        ("divide-rose-900 ", "borders"),
        ("divide-sky-100 ", "borders"),
        ("divide-sky-200 ", "borders"),
        ("divide-sky-300 ", "borders"),
        ("divide-sky-400 ", "borders"),
        ("divide-sky-50 ", "borders"),
        ("divide-sky-500 ", "borders"),
        ("divide-sky-600 ", "borders"),
        ("divide-sky-700 ", "borders"),
        ("divide-sky-800 ", "borders"),
        ("divide-sky-900 ", "borders"),
        ("divide-slate-100 ", "borders"),
        ("divide-slate-200 ", "borders"),
        ("divide-slate-300 ", "borders"),
        ("divide-slate-400 ", "borders"),
        ("divide-slate-50 ", "borders"),
        ("divide-slate-500 ", "borders"),
        ("divide-slate-600 ", "borders"),
        ("divide-slate-700 ", "borders"),
        ("divide-slate-800 ", "borders"),
        ("divide-slate-900 ", "borders"),
        ("divide-solid ", "borders"),
        ("divide-stone-100 ", "borders"),
        ("divide-stone-200 ", "borders"),
        ("divide-stone-300 ", "borders"),
        ("divide-stone-400 ", "borders"),
        ("divide-stone-50 ", "borders"),
        ("divide-stone-500 ", "borders"),
        ("divide-stone-600 ", "borders"),
        ("divide-stone-700 ", "borders"),
        ("divide-stone-800 ", "borders"),
        ("divide-stone-900 ", "borders"),
        ("divide-teal-100 ", "borders"),
        ("divide-teal-200 ", "borders"),
        ("divide-teal-300 ", "borders"),
        ("divide-teal-400 ", "borders"),
        ("divide-teal-50 ", "borders"),
        ("divide-teal-500 ", "borders"),
        ("divide-teal-600 ", "borders"),
        ("divide-teal-700 ", "borders"),
        ("divide-teal-800 ", "borders"),
        ("divide-teal-900 ", "borders"),
        ("divide-transparent ", "borders"),
        ("divide-violet-100 ", "borders"),
        ("divide-violet-200 ", "borders"),
        ("divide-violet-300 ", "borders"),
        ("divide-violet-400 ", "borders"),
        ("divide-violet-50 ", "borders"),
        ("divide-violet-500 ", "borders"),
        ("divide-violet-600 ", "borders"),
        ("divide-violet-700 ", "borders"),
        ("divide-violet-800 ", "borders"),
        ("divide-violet-900 ", "borders"),
        ("divide-white ", "borders"),
        ("divide-x ", "borders"),
        ("divide-x-0 ", "borders"),
        ("divide-x-2 ", "borders"),
        ("divide-x-4 ", "borders"),
        ("divide-x-8 ", "borders"),
        ("divide-x-reverse ", "borders"),
        ("divide-y ", "borders"),
        ("divide-y-0 ", "borders"),
        ("divide-y-2 ", "borders"),
        ("divide-y-4 ", "borders"),
        ("divide-y-8 ", "borders"),
        ("divide-y-reverse ", "borders"),
        ("divide-yellow-100 ", "borders"),
        ("divide-yellow-200 ", "borders"),
        ("divide-yellow-300 ", "borders"),
        ("divide-yellow-400 ", "borders"),
        ("divide-yellow-50 ", "borders"),
        ("divide-yellow-500 ", "borders"),
        ("divide-yellow-600 ", "borders"),
        ("divide-yellow-700 ", "borders"),
        ("divide-yellow-800 ", "borders"),
        ("divide-yellow-900 ", "borders"),
        ("divide-zinc-100 ", "borders"),
        ("divide-zinc-200 ", "borders"),
        ("divide-zinc-300 ", "borders"),
        ("divide-zinc-400 ", "borders"),
        ("divide-zinc-50 ", "borders"),
        ("divide-zinc-500 ", "borders"),
        ("divide-zinc-600 ", "borders"),
        ("divide-zinc-700 ", "borders"),
        ("divide-zinc-800 ", "borders"),
        ("divide-zinc-900 ", "borders"),
        ("outline", "borders"),
        ("outline-0", "borders"),
        ("outline-1", "borders"),
        ("outline-2", "borders"),
        ("outline-4", "borders"),
        ("outline-8", "borders"),
        ("outline-amber-100", "borders"),
        ("outline-amber-200", "borders"),
        ("outline-amber-300", "borders"),
        ("outline-amber-400", "borders"),
        ("outline-amber-50", "borders"),
        ("outline-amber-500", "borders"),
        ("outline-amber-600", "borders"),
        ("outline-amber-700", "borders"),
        ("outline-amber-800", "borders"),
        ("outline-amber-900", "borders"),
        ("outline-black", "borders"),
        ("outline-blue-100", "borders"),
        ("outline-blue-200", "borders"),
        ("outline-blue-300", "borders"),
        ("outline-blue-400", "borders"),
        ("outline-blue-50", "borders"),
        ("outline-blue-500", "borders"),
        ("outline-blue-600", "borders"),
        ("outline-blue-700", "borders"),
        ("outline-blue-800", "borders"),
        ("outline-blue-900", "borders"),
        ("outline-current", "borders"),
        ("outline-cyan-100", "borders"),
        ("outline-cyan-200", "borders"),
        ("outline-cyan-300", "borders"),
        ("outline-cyan-400", "borders"),
        ("outline-cyan-50", "borders"),
        ("outline-cyan-500", "borders"),
        ("outline-cyan-600", "borders"),
        ("outline-cyan-700", "borders"),
        ("outline-cyan-800", "borders"),
        ("outline-cyan-900", "borders"),
        ("outline-dashed", "borders"),
        ("outline-dotted", "borders"),
        ("outline-double", "borders"),
        ("outline-emerald-100", "borders"),
        ("outline-emerald-200", "borders"),
        ("outline-emerald-300", "borders"),
        ("outline-emerald-400", "borders"),
        ("outline-emerald-50", "borders"),
        ("outline-emerald-500", "borders"),
        ("outline-emerald-600", "borders"),
        ("outline-emerald-700", "borders"),
        ("outline-emerald-800", "borders"),
        ("outline-emerald-900", "borders"),
        ("outline-fuchsia-100", "borders"),
        ("outline-fuchsia-200", "borders"),
        ("outline-fuchsia-300", "borders"),
        ("outline-fuchsia-400", "borders"),
        ("outline-fuchsia-50", "borders"),
        ("outline-fuchsia-500", "borders"),
        ("outline-fuchsia-600", "borders"),
        ("outline-fuchsia-700", "borders"),
        ("outline-fuchsia-800", "borders"),
        ("outline-fuchsia-900", "borders"),
        ("outline-gray-100", "borders"),
        ("outline-gray-200", "borders"),
        ("outline-gray-300", "borders"),
        ("outline-gray-400", "borders"),
        ("outline-gray-50", "borders"),
        ("outline-gray-500", "borders"),
        ("outline-gray-600", "borders"),
        ("outline-gray-700", "borders"),
        ("outline-gray-800", "borders"),
        ("outline-gray-900", "borders"),
        ("outline-green-100", "borders"),
        ("outline-green-200", "borders"),
        ("outline-green-300", "borders"),
        ("outline-green-400", "borders"),
        ("outline-green-50", "borders"),
        ("outline-green-500", "borders"),
        ("outline-green-600", "borders"),
        ("outline-green-700", "borders"),
        ("outline-green-800", "borders"),
        ("outline-green-900", "borders"),
        ("outline-hidden", "borders"),
        ("outline-indigo-100", "borders"),
        ("outline-indigo-200", "borders"),
        ("outline-indigo-300", "borders"),
        ("outline-indigo-400", "borders"),
        ("outline-indigo-50", "borders"),
        ("outline-indigo-500", "borders"),
        ("outline-indigo-600", "borders"),
        ("outline-indigo-700", "borders"),
        ("outline-indigo-800", "borders"),
        ("outline-indigo-900", "borders"),
        ("outline-inherit", "borders"),
        ("outline-lime-100", "borders"),
        ("outline-lime-200", "borders"),
        ("outline-lime-300", "borders"),
        ("outline-lime-400", "borders"),
        ("outline-lime-50", "borders"),
        ("outline-lime-500", "borders"),
        ("outline-lime-600", "borders"),
        ("outline-lime-700", "borders"),
        ("outline-lime-800", "borders"),
        ("outline-lime-900", "borders"),
        ("outline-neutral-100", "borders"),
        ("outline-neutral-200", "borders"),
        ("outline-neutral-300", "borders"),
        ("outline-neutral-400", "borders"),
        ("outline-neutral-50", "borders"),
        ("outline-neutral-500", "borders"),
        ("outline-neutral-600", "borders"),
        ("outline-neutral-700", "borders"),
        ("outline-neutral-800", "borders"),
        ("outline-neutral-900", "borders"),
        ("outline-none", "borders"),
        ("outline-offset-0", "borders"),
        ("outline-offset-1", "borders"),
        ("outline-offset-2", "borders"),
        ("outline-offset-4", "borders"),
        ("outline-offset-8", "borders"),
        ("outline-orange-100", "borders"),
        ("outline-orange-200", "borders"),
        ("outline-orange-300", "borders"),
        ("outline-orange-400", "borders"),
        ("outline-orange-50", "borders"),
        ("outline-orange-500", "borders"),
        ("outline-orange-600", "borders"),
        ("outline-orange-700", "borders"),
        ("outline-orange-800", "borders"),
        ("outline-orange-900", "borders"),
        ("outline-pink-100", "borders"),
        ("outline-pink-200", "borders"),
        ("outline-pink-300", "borders"),
        ("outline-pink-400", "borders"),
        ("outline-pink-50", "borders"),
        ("outline-pink-500", "borders"),
        ("outline-pink-600", "borders"),
        ("outline-pink-700", "borders"),
        ("outline-pink-800", "borders"),
        ("outline-pink-900", "borders"),
        ("outline-purple-100", "borders"),
        ("outline-purple-200", "borders"),
        ("outline-purple-300", "borders"),
        ("outline-purple-400", "borders"),
        ("outline-purple-50", "borders"),
        ("outline-purple-500", "borders"),
        ("outline-purple-600", "borders"),
        ("outline-purple-700", "borders"),
        ("outline-purple-800", "borders"),
        ("outline-purple-900", "borders"),
        ("outline-red-100", "borders"),
        ("outline-red-200", "borders"),
        ("outline-red-300", "borders"),
        ("outline-red-400", "borders"),
        ("outline-red-50", "borders"),
        ("outline-red-500", "borders"),
        ("outline-red-600", "borders"),
        ("outline-red-700", "borders"),
        ("outline-red-800", "borders"),
        ("outline-red-900", "borders"),
        ("outline-rose-100", "borders"),
        ("outline-rose-200", "borders"),
        ("outline-rose-300", "borders"),
        ("outline-rose-400", "borders"),
        ("outline-rose-50", "borders"),
        ("outline-rose-500", "borders"),
        ("outline-rose-600", "borders"),
        ("outline-rose-700", "borders"),
        ("outline-rose-800", "borders"),
        ("outline-rose-900", "borders"),
        ("outline-sky-100", "borders"),
        ("outline-sky-200", "borders"),
        ("outline-sky-300", "borders"),
        ("outline-sky-400", "borders"),
        ("outline-sky-50", "borders"),
        ("outline-sky-500", "borders"),
        ("outline-sky-600", "borders"),
        ("outline-sky-700", "borders"),
        ("outline-sky-800", "borders"),
        ("outline-sky-900", "borders"),
        ("outline-slate-100", "borders"),
        ("outline-slate-200", "borders"),
        ("outline-slate-300", "borders"),
        ("outline-slate-400", "borders"),
        ("outline-slate-50", "borders"),
        ("outline-slate-500", "borders"),
        ("outline-slate-600", "borders"),
        ("outline-slate-700", "borders"),
        ("outline-slate-800", "borders"),
        ("outline-slate-900", "borders"),
        ("outline-stone-100", "borders"),
        ("outline-stone-200", "borders"),
        ("outline-stone-300", "borders"),
        ("outline-stone-400", "borders"),
        ("outline-stone-50", "borders"),
        ("outline-stone-500", "borders"),
        ("outline-stone-600", "borders"),
        ("outline-stone-700", "borders"),
        ("outline-stone-800", "borders"),
        ("outline-stone-900", "borders"),
        ("outline-teal-100", "borders"),
        ("outline-teal-200", "borders"),
        ("outline-teal-300", "borders"),
        ("outline-teal-400", "borders"),
        ("outline-teal-50", "borders"),
        ("outline-teal-500", "borders"),
        ("outline-teal-600", "borders"),
        ("outline-teal-700", "borders"),
        ("outline-teal-800", "borders"),
        ("outline-teal-900", "borders"),
        ("outline-transparent", "borders"),
        ("outline-violet-100", "borders"),
        ("outline-violet-200", "borders"),
        ("outline-violet-300", "borders"),
        ("outline-violet-400", "borders"),
        ("outline-violet-50", "borders"),
        ("outline-violet-500", "borders"),
        ("outline-violet-600", "borders"),
        ("outline-violet-700", "borders"),
        ("outline-violet-800", "borders"),
        ("outline-violet-900", "borders"),
        ("outline-white", "borders"),
        ("outline-yellow-100", "borders"),
        ("outline-yellow-200", "borders"),
        ("outline-yellow-300", "borders"),
        ("outline-yellow-400", "borders"),
        ("outline-yellow-50", "borders"),
        ("outline-yellow-500", "borders"),
        ("outline-yellow-600", "borders"),
        ("outline-yellow-700", "borders"),
        ("outline-yellow-800", "borders"),
        ("outline-yellow-900", "borders"),
        ("outline-zinc-100", "borders"),
        ("outline-zinc-200", "borders"),
        ("outline-zinc-300", "borders"),
        ("outline-zinc-400", "borders"),
        ("outline-zinc-50", "borders"),
        ("outline-zinc-500", "borders"),
        ("outline-zinc-600", "borders"),
        ("outline-zinc-700", "borders"),
        ("outline-zinc-800", "borders"),
        ("outline-zinc-900", "borders"),
        ("ring", "borders"),
        ("ring-0", "borders"),
        ("ring-1", "borders"),
        ("ring-2", "borders"),
        ("ring-4", "borders"),
        ("ring-8", "borders"),
        ("ring-amber-100", "borders"),
        ("ring-amber-200", "borders"),
        ("ring-amber-300", "borders"),
        ("ring-amber-400", "borders"),
        ("ring-amber-50", "borders"),
        ("ring-amber-500", "borders"),
        ("ring-amber-600", "borders"),
        ("ring-amber-700", "borders"),
        ("ring-amber-800", "borders"),
        ("ring-amber-900", "borders"),
        ("ring-black", "borders"),
        ("ring-blue-100", "borders"),
        ("ring-blue-200", "borders"),
        ("ring-blue-300", "borders"),
        ("ring-blue-400", "borders"),
        ("ring-blue-50", "borders"),
        ("ring-blue-500", "borders"),
        ("ring-blue-600", "borders"),
        ("ring-blue-700", "borders"),
        ("ring-blue-800", "borders"),
        ("ring-blue-900", "borders"),
        ("ring-current", "borders"),
        ("ring-cyan-100", "borders"),
        ("ring-cyan-200", "borders"),
        ("ring-cyan-300", "borders"),
        ("ring-cyan-400", "borders"),
        ("ring-cyan-50", "borders"),
        ("ring-cyan-500", "borders"),
        ("ring-cyan-600", "borders"),
        ("ring-cyan-700", "borders"),
        ("ring-cyan-800", "borders"),
        ("ring-cyan-900", "borders"),
        ("ring-emerald-100", "borders"),
        ("ring-emerald-200", "borders"),
        ("ring-emerald-300", "borders"),
        ("ring-emerald-400", "borders"),
        ("ring-emerald-50", "borders"),
        ("ring-emerald-500", "borders"),
        ("ring-emerald-600", "borders"),
        ("ring-emerald-700", "borders"),
        ("ring-emerald-800", "borders"),
        ("ring-emerald-900", "borders"),
        ("ring-fuchsia-100", "borders"),
        ("ring-fuchsia-200", "borders"),
        ("ring-fuchsia-300", "borders"),
        ("ring-fuchsia-400", "borders"),
        ("ring-fuchsia-50", "borders"),
        ("ring-fuchsia-500", "borders"),
        ("ring-fuchsia-600", "borders"),
        ("ring-fuchsia-700", "borders"),
        ("ring-fuchsia-800", "borders"),
        ("ring-fuchsia-900", "borders"),
        ("ring-gray-100", "borders"),
        ("ring-gray-200", "borders"),
        ("ring-gray-300", "borders"),
        ("ring-gray-400", "borders"),
        ("ring-gray-50", "borders"),
        ("ring-gray-500", "borders"),
        ("ring-gray-600", "borders"),
        ("ring-gray-700", "borders"),
        ("ring-gray-800", "borders"),
        ("ring-gray-900", "borders"),
        ("ring-green-100", "borders"),
        ("ring-green-200", "borders"),
        ("ring-green-300", "borders"),
        ("ring-green-400", "borders"),
        ("ring-green-50", "borders"),
        ("ring-green-500", "borders"),
        ("ring-green-600", "borders"),
        ("ring-green-700", "borders"),
        ("ring-green-800", "borders"),
        ("ring-green-900", "borders"),
        ("ring-indigo-100", "borders"),
        ("ring-indigo-200", "borders"),
        ("ring-indigo-300", "borders"),
        ("ring-indigo-400", "borders"),
        ("ring-indigo-50", "borders"),
        ("ring-indigo-500", "borders"),
        ("ring-indigo-600", "borders"),
        ("ring-indigo-700", "borders"),
        ("ring-indigo-800", "borders"),
        ("ring-indigo-900", "borders"),
        ("ring-inherit", "borders"),
        ("ring-inset", "borders"),
        ("ring-lime-100", "borders"),
        ("ring-lime-200", "borders"),
        ("ring-lime-300", "borders"),
        ("ring-lime-400", "borders"),
        ("ring-lime-50", "borders"),
        ("ring-lime-500", "borders"),
        ("ring-lime-600", "borders"),
        ("ring-lime-700", "borders"),
        ("ring-lime-800", "borders"),
        ("ring-lime-900", "borders"),
        ("ring-neutral-100", "borders"),
        ("ring-neutral-200", "borders"),
        ("ring-neutral-300", "borders"),
        ("ring-neutral-400", "borders"),
        ("ring-neutral-50", "borders"),
        ("ring-neutral-500", "borders"),
        ("ring-neutral-600", "borders"),
        ("ring-neutral-700", "borders"),
        ("ring-neutral-800", "borders"),
        ("ring-neutral-900", "borders"),
        ("ring-offset-0", "borders"),
        ("ring-offset-1", "borders"),
        ("ring-offset-2", "borders"),
        ("ring-offset-4", "borders"),
        ("ring-offset-8", "borders"),
        ("ring-offset-amber-100", "borders"),
        ("ring-offset-amber-200", "borders"),
        ("ring-offset-amber-300", "borders"),
        ("ring-offset-amber-400", "borders"),
        ("ring-offset-amber-50", "borders"),
        ("ring-offset-amber-500", "borders"),
        ("ring-offset-amber-600", "borders"),
        ("ring-offset-amber-700", "borders"),
        ("ring-offset-amber-800", "borders"),
        ("ring-offset-amber-900", "borders"),
        ("ring-offset-black", "borders"),
        ("ring-offset-blue-100", "borders"),
        ("ring-offset-blue-200", "borders"),
        ("ring-offset-blue-300", "borders"),
        ("ring-offset-blue-400", "borders"),
        ("ring-offset-blue-50", "borders"),
        ("ring-offset-blue-500", "borders"),
        ("ring-offset-blue-600", "borders"),
        ("ring-offset-blue-700", "borders"),
        ("ring-offset-blue-800", "borders"),
        ("ring-offset-blue-900", "borders"),
        ("ring-offset-current", "borders"),
        ("ring-offset-cyan-100", "borders"),
        ("ring-offset-cyan-200", "borders"),
        ("ring-offset-cyan-300", "borders"),
        ("ring-offset-cyan-400", "borders"),
        ("ring-offset-cyan-50", "borders"),
        ("ring-offset-cyan-500", "borders"),
        ("ring-offset-cyan-600", "borders"),
        ("ring-offset-cyan-700", "borders"),
        ("ring-offset-cyan-800", "borders"),
        ("ring-offset-cyan-900", "borders"),
        ("ring-offset-emerald-100", "borders"),
        ("ring-offset-emerald-200", "borders"),
        ("ring-offset-emerald-300", "borders"),
        ("ring-offset-emerald-400", "borders"),
        ("ring-offset-emerald-50", "borders"),
        ("ring-offset-emerald-500", "borders"),
        ("ring-offset-emerald-600", "borders"),
        ("ring-offset-emerald-700", "borders"),
        ("ring-offset-emerald-800", "borders"),
        ("ring-offset-emerald-900", "borders"),
        ("ring-offset-fuchsia-100", "borders"),
        ("ring-offset-fuchsia-200", "borders"),
        ("ring-offset-fuchsia-300", "borders"),
        ("ring-offset-fuchsia-400", "borders"),
        ("ring-offset-fuchsia-50", "borders"),
        ("ring-offset-fuchsia-500", "borders"),
        ("ring-offset-fuchsia-600", "borders"),
        ("ring-offset-fuchsia-700", "borders"),
        ("ring-offset-fuchsia-800", "borders"),
        ("ring-offset-fuchsia-900", "borders"),
        ("ring-offset-gray-100", "borders"),
        ("ring-offset-gray-200", "borders"),
        ("ring-offset-gray-300", "borders"),
        ("ring-offset-gray-400", "borders"),
        ("ring-offset-gray-50", "borders"),
        ("ring-offset-gray-500", "borders"),
        ("ring-offset-gray-600", "borders"),
        ("ring-offset-gray-700", "borders"),
        ("ring-offset-gray-800", "borders"),
        ("ring-offset-gray-900", "borders"),
        ("ring-offset-green-100", "borders"),
        ("ring-offset-green-200", "borders"),
        ("ring-offset-green-300", "borders"),
        ("ring-offset-green-400", "borders"),
        ("ring-offset-green-50", "borders"),
        ("ring-offset-green-500", "borders"),
        ("ring-offset-green-600", "borders"),
        ("ring-offset-green-700", "borders"),
        ("ring-offset-green-800", "borders"),
        ("ring-offset-green-900", "borders"),
        ("ring-offset-indigo-100", "borders"),
        ("ring-offset-indigo-200", "borders"),
        ("ring-offset-indigo-300", "borders"),
        ("ring-offset-indigo-400", "borders"),
        ("ring-offset-indigo-50", "borders"),
        ("ring-offset-indigo-500", "borders"),
        ("ring-offset-indigo-600", "borders"),
        ("ring-offset-indigo-700", "borders"),
        ("ring-offset-indigo-800", "borders"),
        ("ring-offset-indigo-900", "borders"),
        ("ring-offset-inherit", "borders"),
        ("ring-offset-lime-100", "borders"),
        ("ring-offset-lime-200", "borders"),
        ("ring-offset-lime-300", "borders"),
        ("ring-offset-lime-400", "borders"),
        ("ring-offset-lime-50", "borders"),
        ("ring-offset-lime-500", "borders"),
        ("ring-offset-lime-600", "borders"),
        ("ring-offset-lime-700", "borders"),
        ("ring-offset-lime-800", "borders"),
        ("ring-offset-lime-900", "borders"),
        ("ring-offset-neutral-100", "borders"),
        ("ring-offset-neutral-200", "borders"),
        ("ring-offset-neutral-300", "borders"),
        ("ring-offset-neutral-400", "borders"),
        ("ring-offset-neutral-50", "borders"),
        ("ring-offset-neutral-500", "borders"),
        ("ring-offset-neutral-600", "borders"),
        ("ring-offset-neutral-700", "borders"),
        ("ring-offset-neutral-800", "borders"),
        ("ring-offset-neutral-900", "borders"),
        ("ring-offset-orange-100", "borders"),
        ("ring-offset-orange-200", "borders"),
        ("ring-offset-orange-300", "borders"),
        ("ring-offset-orange-400", "borders"),
        ("ring-offset-orange-50", "borders"),
        ("ring-offset-orange-500", "borders"),
        ("ring-offset-orange-600", "borders"),
        ("ring-offset-orange-700", "borders"),
        ("ring-offset-orange-800", "borders"),
        ("ring-offset-orange-900", "borders"),
        ("ring-offset-pink-100", "borders"),
        ("ring-offset-pink-200", "borders"),
        ("ring-offset-pink-300", "borders"),
        ("ring-offset-pink-400", "borders"),
        ("ring-offset-pink-50", "borders"),
        ("ring-offset-pink-500", "borders"),
        ("ring-offset-pink-600", "borders"),
        ("ring-offset-pink-700", "borders"),
        ("ring-offset-pink-800", "borders"),
        ("ring-offset-pink-900", "borders"),
        ("ring-offset-purple-100", "borders"),
        ("ring-offset-purple-200", "borders"),
        ("ring-offset-purple-300", "borders"),
        ("ring-offset-purple-400", "borders"),
        ("ring-offset-purple-50", "borders"),
        ("ring-offset-purple-500", "borders"),
        ("ring-offset-purple-600", "borders"),
        ("ring-offset-purple-700", "borders"),
        ("ring-offset-purple-800", "borders"),
        ("ring-offset-purple-900", "borders"),
        ("ring-offset-red-100", "borders"),
        ("ring-offset-red-200", "borders"),
        ("ring-offset-red-300", "borders"),
        ("ring-offset-red-400", "borders"),
        ("ring-offset-red-50", "borders"),
        ("ring-offset-red-500", "borders"),
        ("ring-offset-red-600", "borders"),
        ("ring-offset-red-700", "borders"),
        ("ring-offset-red-800", "borders"),
        ("ring-offset-red-900", "borders"),
        ("ring-offset-rose-100", "borders"),
        ("ring-offset-rose-200", "borders"),
        ("ring-offset-rose-300", "borders"),
        ("ring-offset-rose-400", "borders"),
        ("ring-offset-rose-50", "borders"),
        ("ring-offset-rose-500", "borders"),
        ("ring-offset-rose-600", "borders"),
        ("ring-offset-rose-700", "borders"),
        ("ring-offset-rose-800", "borders"),
        ("ring-offset-rose-900", "borders"),
        ("ring-offset-sky-100", "borders"),
        ("ring-offset-sky-200", "borders"),
        ("ring-offset-sky-300", "borders"),
        ("ring-offset-sky-400", "borders"),
        ("ring-offset-sky-50", "borders"),
        ("ring-offset-sky-500", "borders"),
        ("ring-offset-sky-600", "borders"),
        ("ring-offset-sky-700", "borders"),
        ("ring-offset-sky-800", "borders"),
        ("ring-offset-sky-900", "borders"),
        ("ring-offset-slate-100", "borders"),
        ("ring-offset-slate-200", "borders"),
        ("ring-offset-slate-300", "borders"),
        ("ring-offset-slate-400", "borders"),
        ("ring-offset-slate-50", "borders"),
        ("ring-offset-slate-500", "borders"),
        ("ring-offset-slate-600", "borders"),
        ("ring-offset-slate-700", "borders"),
        ("ring-offset-slate-800", "borders"),
        ("ring-offset-slate-900", "borders"),
        ("ring-offset-stone-100", "borders"),
        ("ring-offset-stone-200", "borders"),
        ("ring-offset-stone-300", "borders"),
        ("ring-offset-stone-400", "borders"),
        ("ring-offset-stone-50", "borders"),
        ("ring-offset-stone-500", "borders"),
        ("ring-offset-stone-600", "borders"),
        ("ring-offset-stone-700", "borders"),
        ("ring-offset-stone-800", "borders"),
        ("ring-offset-stone-900", "borders"),
        ("ring-offset-teal-100", "borders"),
        ("ring-offset-teal-200", "borders"),
        ("ring-offset-teal-300", "borders"),
        ("ring-offset-teal-400", "borders"),
        ("ring-offset-teal-50", "borders"),
        ("ring-offset-teal-500", "borders"),
        ("ring-offset-teal-600", "borders"),
        ("ring-offset-teal-700", "borders"),
        ("ring-offset-teal-800", "borders"),
        ("ring-offset-teal-900", "borders"),
        ("ring-offset-transparent", "borders"),
        ("ring-offset-violet-100", "borders"),
        ("ring-offset-violet-200", "borders"),
        ("ring-offset-violet-300", "borders"),
        ("ring-offset-violet-400", "borders"),
        ("ring-offset-violet-50", "borders"),
        ("ring-offset-violet-500", "borders"),
        ("ring-offset-violet-600", "borders"),
        ("ring-offset-violet-700", "borders"),
        ("ring-offset-violet-800", "borders"),
        ("ring-offset-violet-900", "borders"),
        ("ring-offset-white", "borders"),
        ("ring-offset-yellow-100", "borders"),
        ("ring-offset-yellow-200", "borders"),
        ("ring-offset-yellow-300", "borders"),
        ("ring-offset-yellow-400", "borders"),
        ("ring-offset-yellow-50", "borders"),
        ("ring-offset-yellow-500", "borders"),
        ("ring-offset-yellow-600", "borders"),
        ("ring-offset-yellow-700", "borders"),
        ("ring-offset-yellow-800", "borders"),
        ("ring-offset-yellow-900", "borders"),
        ("ring-offset-zinc-100", "borders"),
        ("ring-offset-zinc-200", "borders"),
        ("ring-offset-zinc-300", "borders"),
        ("ring-offset-zinc-400", "borders"),
        ("ring-offset-zinc-50", "borders"),
        ("ring-offset-zinc-500", "borders"),
        ("ring-offset-zinc-600", "borders"),
        ("ring-offset-zinc-700", "borders"),
        ("ring-offset-zinc-800", "borders"),
        ("ring-offset-zinc-900", "borders"),
        ("ring-orange-100", "borders"),
        ("ring-orange-200", "borders"),
        ("ring-orange-300", "borders"),
        ("ring-orange-400", "borders"),
        ("ring-orange-50", "borders"),
        ("ring-orange-500", "borders"),
        ("ring-orange-600", "borders"),
        ("ring-orange-700", "borders"),
        ("ring-orange-800", "borders"),
        ("ring-orange-900", "borders"),
        ("ring-pink-100", "borders"),
        ("ring-pink-200", "borders"),
        ("ring-pink-300", "borders"),
        ("ring-pink-400", "borders"),
        ("ring-pink-50", "borders"),
        ("ring-pink-500", "borders"),
        ("ring-pink-600", "borders"),
        ("ring-pink-700", "borders"),
        ("ring-pink-800", "borders"),
        ("ring-pink-900", "borders"),
        ("ring-purple-100", "borders"),
        ("ring-purple-200", "borders"),
        ("ring-purple-300", "borders"),
        ("ring-purple-400", "borders"),
        ("ring-purple-50", "borders"),
        ("ring-purple-500", "borders"),
        ("ring-purple-600", "borders"),
        ("ring-purple-700", "borders"),
        ("ring-purple-800", "borders"),
        ("ring-purple-900", "borders"),
        ("ring-red-100", "borders"),
        ("ring-red-200", "borders"),
        ("ring-red-300", "borders"),
        ("ring-red-400", "borders"),
        ("ring-red-50", "borders"),
        ("ring-red-500", "borders"),
        ("ring-red-600", "borders"),
        ("ring-red-700", "borders"),
        ("ring-red-800", "borders"),
        ("ring-red-900", "borders"),
        ("ring-rose-100", "borders"),
        ("ring-rose-200", "borders"),
        ("ring-rose-300", "borders"),
        ("ring-rose-400", "borders"),
        ("ring-rose-50", "borders"),
        ("ring-rose-500", "borders"),
        ("ring-rose-600", "borders"),
        ("ring-rose-700", "borders"),
        ("ring-rose-800", "borders"),
        ("ring-rose-900", "borders"),
        ("ring-sky-100", "borders"),
        ("ring-sky-200", "borders"),
        ("ring-sky-300", "borders"),
        ("ring-sky-400", "borders"),
        ("ring-sky-50", "borders"),
        ("ring-sky-500", "borders"),
        ("ring-sky-600", "borders"),
        ("ring-sky-700", "borders"),
        ("ring-sky-800", "borders"),
        ("ring-sky-900", "borders"),
        ("ring-slate-100", "borders"),
        ("ring-slate-200", "borders"),
        ("ring-slate-300", "borders"),
        ("ring-slate-400", "borders"),
        ("ring-slate-50", "borders"),
        ("ring-slate-500", "borders"),
        ("ring-slate-600", "borders"),
        ("ring-slate-700", "borders"),
        ("ring-slate-800", "borders"),
        ("ring-slate-900", "borders"),
        ("ring-stone-100", "borders"),
        ("ring-stone-200", "borders"),
        ("ring-stone-300", "borders"),
        ("ring-stone-400", "borders"),
        ("ring-stone-50", "borders"),
        ("ring-stone-500", "borders"),
        ("ring-stone-600", "borders"),
        ("ring-stone-700", "borders"),
        ("ring-stone-800", "borders"),
        ("ring-stone-900", "borders"),
        ("ring-teal-100", "borders"),
        ("ring-teal-200", "borders"),
        ("ring-teal-300", "borders"),
        ("ring-teal-400", "borders"),
        ("ring-teal-50", "borders"),
        ("ring-teal-500", "borders"),
        ("ring-teal-600", "borders"),
        ("ring-teal-700", "borders"),
        ("ring-teal-800", "borders"),
        ("ring-teal-900", "borders"),
        ("ring-transparent", "borders"),
        ("ring-violet-100", "borders"),
        ("ring-violet-200", "borders"),
        ("ring-violet-300", "borders"),
        ("ring-violet-400", "borders"),
        ("ring-violet-50", "borders"),
        ("ring-violet-500", "borders"),
        ("ring-violet-600", "borders"),
        ("ring-violet-700", "borders"),
        ("ring-violet-800", "borders"),
        ("ring-violet-900", "borders"),
        ("ring-white", "borders"),
        ("ring-yellow-100", "borders"),
        ("ring-yellow-200", "borders"),
        ("ring-yellow-300", "borders"),
        ("ring-yellow-400", "borders"),
        ("ring-yellow-50", "borders"),
        ("ring-yellow-500", "borders"),
        ("ring-yellow-600", "borders"),
        ("ring-yellow-700", "borders"),
        ("ring-yellow-800", "borders"),
        ("ring-yellow-900", "borders"),
        ("ring-zinc-100", "borders"),
        ("ring-zinc-200", "borders"),
        ("ring-zinc-300", "borders"),
        ("ring-zinc-400", "borders"),
        ("ring-zinc-50", "borders"),
        ("ring-zinc-500", "borders"),
        ("ring-zinc-600", "borders"),
        ("ring-zinc-700", "borders"),
        ("ring-zinc-800", "borders"),
        ("ring-zinc-900", "borders"),
        ("rounded", "borders"),
        ("rounded-2xl", "borders"),
        ("rounded-3xl", "borders"),
        ("rounded-b", "borders"),
        ("rounded-b-2xl", "borders"),
        ("rounded-b-3xl", "borders"),
        ("rounded-b-full", "borders"),
        ("rounded-b-lg", "borders"),
        ("rounded-b-md", "borders"),
        ("rounded-b-none", "borders"),
        ("rounded-b-sm", "borders"),
        ("rounded-b-xl", "borders"),
        ("rounded-bl", "borders"),
        ("rounded-bl-2xl", "borders"),
        ("rounded-bl-3xl", "borders"),
        ("rounded-bl-full", "borders"),
        ("rounded-bl-lg", "borders"),
        ("rounded-bl-md", "borders"),
        ("rounded-bl-none", "borders"),
        ("rounded-bl-sm", "borders"),
        ("rounded-bl-xl", "borders"),
        ("rounded-br", "borders"),
        ("rounded-br-2xl", "borders"),
        ("rounded-br-3xl", "borders"),
        ("rounded-br-full", "borders"),
        ("rounded-br-lg", "borders"),
        ("rounded-br-md", "borders"),
        ("rounded-br-none", "borders"),
        ("rounded-br-sm", "borders"),
        ("rounded-br-xl", "borders"),
        ("rounded-full", "borders"),
        ("rounded-l", "borders"),
        ("rounded-l-2xl", "borders"),
        ("rounded-l-3xl", "borders"),
        ("rounded-l-full", "borders"),
        ("rounded-l-lg", "borders"),
        ("rounded-l-md", "borders"),
        ("rounded-l-none", "borders"),
        ("rounded-l-sm", "borders"),
        ("rounded-l-xl", "borders"),
        ("rounded-lg", "borders"),
        ("rounded-md", "borders"),
        ("rounded-none", "borders"),
        ("rounded-r", "borders"),
        ("rounded-r-2xl", "borders"),
        ("rounded-r-3xl", "borders"),
        ("rounded-r-full", "borders"),
        ("rounded-r-lg", "borders"),
        ("rounded-r-md", "borders"),
        ("rounded-r-none", "borders"),
        ("rounded-r-sm", "borders"),
        ("rounded-r-xl", "borders"),
        ("rounded-sm", "borders"),
        ("rounded-t", "borders"),
        ("rounded-t-2xl", "borders"),
        ("rounded-t-3xl", "borders"),
        ("rounded-t-full", "borders"),
        ("rounded-t-lg", "borders"),
        ("rounded-t-md", "borders"),
        ("rounded-t-none", "borders"),
        ("rounded-t-sm", "borders"),
        ("rounded-t-xl", "borders"),
        ("rounded-tl", "borders"),
        ("rounded-tl-2xl", "borders"),
        ("rounded-tl-3xl", "borders"),
        ("rounded-tl-full", "borders"),
        ("rounded-tl-lg", "borders"),
        ("rounded-tl-md", "borders"),
        ("rounded-tl-none", "borders"),
        ("rounded-tl-sm", "borders"),
        ("rounded-tl-xl", "borders"),
        ("rounded-tr", "borders"),
        ("rounded-tr-2xl", "borders"),
        ("rounded-tr-3xl", "borders"),
        ("rounded-tr-full", "borders"),
        ("rounded-tr-lg", "borders"),
        ("rounded-tr-md", "borders"),
        ("rounded-tr-none", "borders"),
        ("rounded-tr-sm", "borders"),
        ("rounded-tr-xl", "borders"),
        ("rounded-xl", "borders"),
        ("bg-blend-color", "effects"),
        ("bg-blend-color-burn", "effects"),
        ("bg-blend-color-dodge", "effects"),
        ("bg-blend-darken", "effects"),
        ("bg-blend-difference", "effects"),
        ("bg-blend-exclusion", "effects"),
        ("bg-blend-hard-light", "effects"),
        ("bg-blend-hue", "effects"),
        ("bg-blend-lighten", "effects"),
        ("bg-blend-luminosity", "effects"),
        ("bg-blend-multiply", "effects"),
        ("bg-blend-normal", "effects"),
        ("bg-blend-overlay", "effects"),
        ("bg-blend-saturation", "effects"),
        ("bg-blend-screen", "effects"),
        ("bg-blend-soft-light", "effects"),
        ("mix-blend-color", "effects"),
        ("mix-blend-color-burn", "effects"),
        ("mix-blend-color-dodge", "effects"),
        ("mix-blend-darken", "effects"),
        ("mix-blend-difference", "effects"),
        ("mix-blend-exclusion", "effects"),
        ("mix-blend-hard-light", "effects"),
        ("mix-blend-hue", "effects"),
        ("mix-blend-lighten", "effects"),
        ("mix-blend-luminosity", "effects"),
        ("mix-blend-multiply", "effects"),
        ("mix-blend-normal", "effects"),
        ("mix-blend-overlay", "effects"),
        ("mix-blend-saturation", "effects"),
        ("mix-blend-screen", "effects"),
        ("mix-blend-soft-light", "effects"),
        ("opacity-0", "effects"),
        ("opacity-10", "effects"),
        ("opacity-100", "effects"),
        ("opacity-20", "effects"),
        ("opacity-25", "effects"),
        ("opacity-30", "effects"),
        ("opacity-40", "effects"),
        ("opacity-5", "effects"),
        ("opacity-50", "effects"),
        ("opacity-60", "effects"),
        ("opacity-70", "effects"),
        ("opacity-75", "effects"),
        ("opacity-80", "effects"),
        ("opacity-90", "effects"),
        ("opacity-95", "effects"),
        ("shadow", "effects"),
        ("shadow-2xl", "effects"),
        ("shadow-amber-100", "effects"),
        ("shadow-amber-200", "effects"),
        ("shadow-amber-300", "effects"),
        ("shadow-amber-400", "effects"),
        ("shadow-amber-50", "effects"),
        ("shadow-amber-500", "effects"),
        ("shadow-amber-600", "effects"),
        ("shadow-amber-700", "effects"),
        ("shadow-amber-800", "effects"),
        ("shadow-amber-900", "effects"),
        ("shadow-black", "effects"),
        ("shadow-blue-100", "effects"),
        ("shadow-blue-200", "effects"),
        ("shadow-blue-300", "effects"),
        ("shadow-blue-400", "effects"),
        ("shadow-blue-50", "effects"),
        ("shadow-blue-500", "effects"),
        ("shadow-blue-600", "effects"),
        ("shadow-blue-700", "effects"),
        ("shadow-blue-800", "effects"),
        ("shadow-blue-900", "effects"),
        ("shadow-current", "effects"),
        ("shadow-cyan-100", "effects"),
        ("shadow-cyan-200", "effects"),
        ("shadow-cyan-300", "effects"),
        ("shadow-cyan-400", "effects"),
        ("shadow-cyan-50", "effects"),
        ("shadow-cyan-500", "effects"),
        ("shadow-cyan-600", "effects"),
        ("shadow-cyan-700", "effects"),
        ("shadow-cyan-800", "effects"),
        ("shadow-cyan-900", "effects"),
        ("shadow-emerald-100", "effects"),
        ("shadow-emerald-200", "effects"),
        ("shadow-emerald-300", "effects"),
        ("shadow-emerald-400", "effects"),
        ("shadow-emerald-50", "effects"),
        ("shadow-emerald-500", "effects"),
        ("shadow-emerald-600", "effects"),
        ("shadow-emerald-700", "effects"),
        ("shadow-emerald-800", "effects"),
        ("shadow-emerald-900", "effects"),
        ("shadow-fuchsia-100", "effects"),
        ("shadow-fuchsia-200", "effects"),
        ("shadow-fuchsia-300", "effects"),
        ("shadow-fuchsia-400", "effects"),
        ("shadow-fuchsia-50", "effects"),
        ("shadow-fuchsia-500", "effects"),
        ("shadow-fuchsia-600", "effects"),
        ("shadow-fuchsia-700", "effects"),
        ("shadow-fuchsia-800", "effects"),
        ("shadow-fuchsia-900", "effects"),
        ("shadow-gray-100", "effects"),
        ("shadow-gray-200", "effects"),
        ("shadow-gray-300", "effects"),
        ("shadow-gray-400", "effects"),
        ("shadow-gray-50", "effects"),
        ("shadow-gray-500", "effects"),
        ("shadow-gray-600", "effects"),
        ("shadow-gray-700", "effects"),
        ("shadow-gray-800", "effects"),
        ("shadow-gray-900", "effects"),
        ("shadow-green-100", "effects"),
        ("shadow-green-200", "effects"),
        ("shadow-green-300", "effects"),
        ("shadow-green-400", "effects"),
        ("shadow-green-50", "effects"),
        ("shadow-green-500", "effects"),
        ("shadow-green-600", "effects"),
        ("shadow-green-700", "effects"),
        ("shadow-green-800", "effects"),
        ("shadow-green-900", "effects"),
        ("shadow-indigo-100", "effects"),
        ("shadow-indigo-200", "effects"),
        ("shadow-indigo-300", "effects"),
        ("shadow-indigo-400", "effects"),
        ("shadow-indigo-50", "effects"),
        ("shadow-indigo-500", "effects"),
        ("shadow-indigo-600", "effects"),
        ("shadow-indigo-700", "effects"),
        ("shadow-indigo-800", "effects"),
        ("shadow-indigo-900", "effects"),
        ("shadow-inherit", "effects"),
        ("shadow-inner", "effects"),
        ("shadow-lg", "effects"),
        ("shadow-lime-100", "effects"),
        ("shadow-lime-200", "effects"),
        ("shadow-lime-300", "effects"),
        ("shadow-lime-400", "effects"),
        ("shadow-lime-50", "effects"),
        ("shadow-lime-500", "effects"),
        ("shadow-lime-600", "effects"),
        ("shadow-lime-700", "effects"),
        ("shadow-lime-800", "effects"),
        ("shadow-lime-900", "effects"),
        ("shadow-md", "effects"),
        ("shadow-neutral-100", "effects"),
        ("shadow-neutral-200", "effects"),
        ("shadow-neutral-300", "effects"),
        ("shadow-neutral-400", "effects"),
        ("shadow-neutral-50", "effects"),
        ("shadow-neutral-500", "effects"),
        ("shadow-neutral-600", "effects"),
        ("shadow-neutral-700", "effects"),
        ("shadow-neutral-800", "effects"),
        ("shadow-neutral-900", "effects"),
        ("shadow-none", "effects"),
        ("shadow-orange-100", "effects"),
        ("shadow-orange-200", "effects"),
        ("shadow-orange-300", "effects"),
        ("shadow-orange-400", "effects"),
        ("shadow-orange-50", "effects"),
        ("shadow-orange-500", "effects"),
        ("shadow-orange-600", "effects"),
        ("shadow-orange-700", "effects"),
        ("shadow-orange-800", "effects"),
        ("shadow-orange-900", "effects"),
        ("shadow-pink-100", "effects"),
        ("shadow-pink-200", "effects"),
        ("shadow-pink-300", "effects"),
        ("shadow-pink-400", "effects"),
        ("shadow-pink-50", "effects"),
        ("shadow-pink-500", "effects"),
        ("shadow-pink-600", "effects"),
        ("shadow-pink-700", "effects"),
        ("shadow-pink-800", "effects"),
        ("shadow-pink-900", "effects"),
        ("shadow-purple-100", "effects"),
        ("shadow-purple-200", "effects"),
        ("shadow-purple-300", "effects"),
        ("shadow-purple-400", "effects"),
        ("shadow-purple-50", "effects"),
        ("shadow-purple-500", "effects"),
        ("shadow-purple-600", "effects"),
        ("shadow-purple-700", "effects"),
        ("shadow-purple-800", "effects"),
        ("shadow-purple-900", "effects"),
        ("shadow-red-100", "effects"),
        ("shadow-red-200", "effects"),
        ("shadow-red-300", "effects"),
        ("shadow-red-400", "effects"),
        ("shadow-red-50", "effects"),
        ("shadow-red-500", "effects"),
        ("shadow-red-600", "effects"),
        ("shadow-red-700", "effects"),
        ("shadow-red-800", "effects"),
        ("shadow-red-900", "effects"),
        ("shadow-rose-100", "effects"),
        ("shadow-rose-200", "effects"),
        ("shadow-rose-300", "effects"),
        ("shadow-rose-400", "effects"),
        ("shadow-rose-50", "effects"),
        ("shadow-rose-500", "effects"),
        ("shadow-rose-600", "effects"),
        ("shadow-rose-700", "effects"),
        ("shadow-rose-800", "effects"),
        ("shadow-rose-900", "effects"),
        ("shadow-sky-100", "effects"),
        ("shadow-sky-200", "effects"),
        ("shadow-sky-300", "effects"),
        ("shadow-sky-400", "effects"),
        ("shadow-sky-50", "effects"),
        ("shadow-sky-500", "effects"),
        ("shadow-sky-600", "effects"),
        ("shadow-sky-700", "effects"),
        ("shadow-sky-800", "effects"),
        ("shadow-sky-900", "effects"),
        ("shadow-slate-100", "effects"),
        ("shadow-slate-200", "effects"),
        ("shadow-slate-300", "effects"),
        ("shadow-slate-400", "effects"),
        ("shadow-slate-50", "effects"),
        ("shadow-slate-500", "effects"),
        ("shadow-slate-600", "effects"),
        ("shadow-slate-700", "effects"),
        ("shadow-slate-800", "effects"),
        ("shadow-slate-900", "effects"),
        ("shadow-sm", "effects"),
        ("shadow-stone-100", "effects"),
        ("shadow-stone-200", "effects"),
        ("shadow-stone-300", "effects"),
        ("shadow-stone-400", "effects"),
        ("shadow-stone-50", "effects"),
        ("shadow-stone-500", "effects"),
        ("shadow-stone-600", "effects"),
        ("shadow-stone-700", "effects"),
        ("shadow-stone-800", "effects"),
        ("shadow-stone-900", "effects"),
        ("shadow-teal-100", "effects"),
        ("shadow-teal-200", "effects"),
        ("shadow-teal-300", "effects"),
        ("shadow-teal-400", "effects"),
        ("shadow-teal-50", "effects"),
        ("shadow-teal-500", "effects"),
        ("shadow-teal-600", "effects"),
        ("shadow-teal-700", "effects"),
        ("shadow-teal-800", "effects"),
        ("shadow-teal-900", "effects"),
        ("shadow-transparent", "effects"),
        ("shadow-violet-100", "effects"),
        ("shadow-violet-200", "effects"),
        ("shadow-violet-300", "effects"),
        ("shadow-violet-400", "effects"),
        ("shadow-violet-50", "effects"),
        ("shadow-violet-500", "effects"),
        ("shadow-violet-600", "effects"),
        ("shadow-violet-700", "effects"),
        ("shadow-violet-800", "effects"),
        ("shadow-violet-900", "effects"),
        ("shadow-white", "effects"),
        ("shadow-xl", "effects"),
        ("shadow-yellow-100", "effects"),
        ("shadow-yellow-200", "effects"),
        ("shadow-yellow-300", "effects"),
        ("shadow-yellow-400", "effects"),
        ("shadow-yellow-50", "effects"),
        ("shadow-yellow-500", "effects"),
        ("shadow-yellow-600", "effects"),
        ("shadow-yellow-700", "effects"),
        ("shadow-yellow-800", "effects"),
        ("shadow-yellow-900", "effects"),
        ("shadow-zinc-100", "effects"),
        ("shadow-zinc-200", "effects"),
        ("shadow-zinc-300", "effects"),
        ("shadow-zinc-400", "effects"),
        ("shadow-zinc-50", "effects"),
        ("shadow-zinc-500", "effects"),
        ("shadow-zinc-600", "effects"),
        ("shadow-zinc-700", "effects"),
        ("shadow-zinc-800", "effects"),
        ("shadow-zinc-900", "effects"),
        ("backdrop-blur", "filters"),
        ("backdrop-blur-2xl", "filters"),
        ("backdrop-blur-3xl", "filters"),
        ("backdrop-blur-lg", "filters"),
        ("backdrop-blur-md", "filters"),
        ("backdrop-blur-none", "filters"),
        ("backdrop-blur-sm", "filters"),
        ("backdrop-blur-xl", "filters"),
        ("backdrop-brightness-0", "filters"),
        ("backdrop-brightness-100", "filters"),
        ("backdrop-brightness-105", "filters"),
        ("backdrop-brightness-110", "filters"),
        ("backdrop-brightness-125", "filters"),
        ("backdrop-brightness-150", "filters"),
        ("backdrop-brightness-200", "filters"),
        ("backdrop-brightness-50", "filters"),
        ("backdrop-brightness-75", "filters"),
        ("backdrop-brightness-90", "filters"),
        ("backdrop-brightness-95", "filters"),
        ("backdrop-contrast-0", "filters"),
        ("backdrop-contrast-100", "filters"),
        ("backdrop-contrast-125", "filters"),
        ("backdrop-contrast-150", "filters"),
        ("backdrop-contrast-200", "filters"),
        ("backdrop-contrast-50", "filters"),
        ("backdrop-contrast-75", "filters"),
        ("backdrop-grayscale", "filters"),
        ("backdrop-grayscale-0", "filters"),
        ("backdrop-hue-rotate-0", "filters"),
        ("backdrop-hue-rotate-15", "filters"),
        ("backdrop-hue-rotate-180", "filters"),
        ("backdrop-hue-rotate-30", "filters"),
        ("backdrop-hue-rotate-60", "filters"),
        ("backdrop-hue-rotate-90", "filters"),
        ("backdrop-invert", "filters"),
        ("backdrop-invert-0", "filters"),
        ("backdrop-opacity-0", "filters"),
        ("backdrop-opacity-10", "filters"),
        ("backdrop-opacity-100", "filters"),
        ("backdrop-opacity-20", "filters"),
        ("backdrop-opacity-25", "filters"),
        ("backdrop-opacity-30", "filters"),
        ("backdrop-opacity-40", "filters"),
        ("backdrop-opacity-5", "filters"),
        ("backdrop-opacity-50", "filters"),
        ("backdrop-opacity-60", "filters"),
        ("backdrop-opacity-70", "filters"),
        ("backdrop-opacity-75", "filters"),
        ("backdrop-opacity-80", "filters"),
        ("backdrop-opacity-90", "filters"),
        ("backdrop-opacity-95", "filters"),
        ("backdrop-saturate-0", "filters"),
        ("backdrop-saturate-100", "filters"),
        ("backdrop-saturate-150", "filters"),
        ("backdrop-saturate-200", "filters"),
        ("backdrop-saturate-50", "filters"),
        ("backdrop-sepia", "filters"),
        ("backdrop-sepia-0", "filters"),
        ("blur", "filters"),
        ("blur-2xl", "filters"),
        ("blur-3xl", "filters"),
        ("blur-lg", "filters"),
        ("blur-md", "filters"),
        ("blur-none", "filters"),
        ("blur-sm", "filters"),
        ("blur-xl", "filters"),
        ("brightness-0", "filters"),
        ("brightness-100", "filters"),
        ("brightness-105", "filters"),
        ("brightness-110", "filters"),
        ("brightness-125", "filters"),
        ("brightness-150", "filters"),
        ("brightness-200", "filters"),
        ("brightness-50", "filters"),
        ("brightness-75", "filters"),
        ("brightness-90", "filters"),
        ("brightness-95", "filters"),
        ("contrast-0", "filters"),
        ("contrast-100", "filters"),
        ("contrast-125", "filters"),
        ("contrast-150", "filters"),
        ("contrast-200", "filters"),
        ("contrast-50", "filters"),
        ("contrast-75", "filters"),
        ("drop-shadow", "filters"),
        ("drop-shadow-2xl", "filters"),
        ("drop-shadow-lg", "filters"),
        ("drop-shadow-md", "filters"),
        ("drop-shadow-none", "filters"),
        ("drop-shadow-sm", "filters"),
        ("drop-shadow-xl", "filters"),
        ("grayscale", "filters"),
        ("grayscale-0", "filters"),
        ("hue-rotate-0", "filters"),
        ("hue-rotate-15", "filters"),
        ("hue-rotate-180", "filters"),
        ("hue-rotate-30", "filters"),
        ("hue-rotate-60", "filters"),
        ("hue-rotate-90", "filters"),
        ("invert", "filters"),
        ("invert-0", "filters"),
        ("saturate-0", "filters"),
        ("saturate-100", "filters"),
        ("saturate-150", "filters"),
        ("saturate-200", "filters"),
        ("saturate-50", "filters"),
        ("sepia", "filters"),
        ("sepia-0", "filters"),
        ("auto-cols-auto", "flex_and_grid"),
        ("auto-cols-fr", "flex_and_grid"),
        ("auto-cols-max", "flex_and_grid"),
        ("auto-cols-min", "flex_and_grid"),
        ("auto-rows-auto", "flex_and_grid"),
        ("auto-rows-fr", "flex_and_grid"),
        ("auto-rows-max", "flex_and_grid"),
        ("auto-rows-min", "flex_and_grid"),
        ("basis-0", "flex_and_grid"),
        ("basis-0.5", "flex_and_grid"),
        ("basis-1", "flex_and_grid"),
        ("basis-1.5", "flex_and_grid"),
        ("basis-1/12", "flex_and_grid"),
        ("basis-1/2", "flex_and_grid"),
        ("basis-1/3", "flex_and_grid"),
        ("basis-1/4", "flex_and_grid"),
        ("basis-1/5", "flex_and_grid"),
        ("basis-1/6", "flex_and_grid"),
        ("basis-10", "flex_and_grid"),
        ("basis-10/12", "flex_and_grid"),
        ("basis-11", "flex_and_grid"),
        ("basis-11/12", "flex_and_grid"),
        ("basis-12", "flex_and_grid"),
        ("basis-14", "flex_and_grid"),
        ("basis-16", "flex_and_grid"),
        ("basis-2", "flex_and_grid"),
        ("basis-2.5", "flex_and_grid"),
        ("basis-2/12", "flex_and_grid"),
        ("basis-2/3", "flex_and_grid"),
        ("basis-2/4", "flex_and_grid"),
        ("basis-2/5", "flex_and_grid"),
        ("basis-2/6", "flex_and_grid"),
        ("basis-20", "flex_and_grid"),
        ("basis-24", "flex_and_grid"),
        ("basis-28", "flex_and_grid"),
        ("basis-3", "flex_and_grid"),
        ("basis-3.5", "flex_and_grid"),
        ("basis-3/12", "flex_and_grid"),
        ("basis-3/4", "flex_and_grid"),
        ("basis-3/5", "flex_and_grid"),
        ("basis-3/6", "flex_and_grid"),
        ("basis-32", "flex_and_grid"),
        ("basis-36", "flex_and_grid"),
        ("basis-4", "flex_and_grid"),
        ("basis-4/12", "flex_and_grid"),
        ("basis-4/5", "flex_and_grid"),
        ("basis-4/6", "flex_and_grid"),
        ("basis-40", "flex_and_grid"),
        ("basis-44", "flex_and_grid"),
        ("basis-48", "flex_and_grid"),
        ("basis-5", "flex_and_grid"),
        ("basis-5/12", "flex_and_grid"),
        ("basis-5/6", "flex_and_grid"),
        ("basis-52", "flex_and_grid"),
        ("basis-56", "flex_and_grid"),
        ("basis-6", "flex_and_grid"),
        ("basis-6/12", "flex_and_grid"),
        ("basis-60", "flex_and_grid"),
        ("basis-64", "flex_and_grid"),
        ("basis-7", "flex_and_grid"),
        ("basis-7/12", "flex_and_grid"),
        ("basis-72", "flex_and_grid"),
        ("basis-8", "flex_and_grid"),
        ("basis-8/12", "flex_and_grid"),
        ("basis-80", "flex_and_grid"),
        ("basis-9", "flex_and_grid"),
        ("basis-9/12", "flex_and_grid"),
        ("basis-96", "flex_and_grid"),
        ("basis-auto", "flex_and_grid"),
        ("basis-full", "flex_and_grid"),
        ("basis-px", "flex_and_grid"),
        ("col-auto", "flex_and_grid"),
        ("col-end-1", "flex_and_grid"),
        ("col-end-10", "flex_and_grid"),
        ("col-end-11", "flex_and_grid"),
        ("col-end-12", "flex_and_grid"),
        ("col-end-13", "flex_and_grid"),
        ("col-end-2", "flex_and_grid"),
        ("col-end-3", "flex_and_grid"),
        ("col-end-4", "flex_and_grid"),
        ("col-end-5", "flex_and_grid"),
        ("col-end-6", "flex_and_grid"),
        ("col-end-7", "flex_and_grid"),
        ("col-end-8", "flex_and_grid"),
        ("col-end-9", "flex_and_grid"),
        ("col-end-auto", "flex_and_grid"),
        ("col-span-1", "flex_and_grid"),
        ("col-span-10", "flex_and_grid"),
        ("col-span-11", "flex_and_grid"),
        ("col-span-12", "flex_and_grid"),
        ("col-span-2", "flex_and_grid"),
        ("col-span-3", "flex_and_grid"),
        ("col-span-4", "flex_and_grid"),
        ("col-span-5", "flex_and_grid"),
        ("col-span-6", "flex_and_grid"),
        ("col-span-7", "flex_and_grid"),
        ("col-span-8", "flex_and_grid"),
        ("col-span-9", "flex_and_grid"),
        ("col-span-full", "flex_and_grid"),
        ("col-start-1", "flex_and_grid"),
        ("col-start-10", "flex_and_grid"),
        ("col-start-11", "flex_and_grid"),
        ("col-start-12", "flex_and_grid"),
        ("col-start-13", "flex_and_grid"),
        ("col-start-2", "flex_and_grid"),
        ("col-start-3", "flex_and_grid"),
        ("col-start-4", "flex_and_grid"),
        ("col-start-5", "flex_and_grid"),
        ("col-start-6", "flex_and_grid"),
        ("col-start-7", "flex_and_grid"),
        ("col-start-8", "flex_and_grid"),
        ("col-start-9", "flex_and_grid"),
        ("col-start-auto", "flex_and_grid"),
        ("content-around", "flex_and_grid"),
        ("content-between", "flex_and_grid"),
        ("content-center", "flex_and_grid"),
        ("content-end", "flex_and_grid"),
        ("content-evenly", "flex_and_grid"),
        ("content-start", "flex_and_grid"),
        ("flex-1", "flex_and_grid"),
        ("flex-auto", "flex_and_grid"),
        ("flex-col", "flex_and_grid"),
        ("flex-col-reverse", "flex_and_grid"),
        ("flex-initial", "flex_and_grid"),
        ("flex-none", "flex_and_grid"),
        ("flex-nowrap", "flex_and_grid"),
        ("flex-row", "flex_and_grid"),
        ("flex-row-reverse", "flex_and_grid"),
        ("flex-wrap", "flex_and_grid"),
        ("flex-wrap-reverse", "flex_and_grid"),
        ("gap-0", "flex_and_grid"),
        ("gap-0.5", "flex_and_grid"),
        ("gap-1", "flex_and_grid"),
        ("gap-1.5", "flex_and_grid"),
        ("gap-10", "flex_and_grid"),
        ("gap-11", "flex_and_grid"),
        ("gap-12", "flex_and_grid"),
        ("gap-14", "flex_and_grid"),
        ("gap-16", "flex_and_grid"),
        ("gap-2", "flex_and_grid"),
        ("gap-2.5", "flex_and_grid"),
        ("gap-20", "flex_and_grid"),
        ("gap-24", "flex_and_grid"),
        ("gap-28", "flex_and_grid"),
        ("gap-3", "flex_and_grid"),
        ("gap-3.5", "flex_and_grid"),
        ("gap-32", "flex_and_grid"),
        ("gap-36", "flex_and_grid"),
        ("gap-4", "flex_and_grid"),
        ("gap-40", "flex_and_grid"),
        ("gap-44", "flex_and_grid"),
        ("gap-48", "flex_and_grid"),
        ("gap-5", "flex_and_grid"),
        ("gap-52", "flex_and_grid"),
        ("gap-56", "flex_and_grid"),
        ("gap-6", "flex_and_grid"),
        ("gap-60", "flex_and_grid"),
        ("gap-64", "flex_and_grid"),
        ("gap-7", "flex_and_grid"),
        ("gap-72", "flex_and_grid"),
        ("gap-8", "flex_and_grid"),
        ("gap-80", "flex_and_grid"),
        ("gap-9", "flex_and_grid"),
        ("gap-96", "flex_and_grid"),
        ("gap-px", "flex_and_grid"),
        ("gap-x-0", "flex_and_grid"),
        ("gap-x-0.5", "flex_and_grid"),
        ("gap-x-1", "flex_and_grid"),
        ("gap-x-1.5", "flex_and_grid"),
        ("gap-x-10", "flex_and_grid"),
        ("gap-x-11", "flex_and_grid"),
        ("gap-x-12", "flex_and_grid"),
        ("gap-x-14", "flex_and_grid"),
        ("gap-x-16", "flex_and_grid"),
        ("gap-x-2", "flex_and_grid"),
        ("gap-x-2.5", "flex_and_grid"),
        ("gap-x-20", "flex_and_grid"),
        ("gap-x-24", "flex_and_grid"),
        ("gap-x-28", "flex_and_grid"),
        ("gap-x-3", "flex_and_grid"),
        ("gap-x-3.5", "flex_and_grid"),
        ("gap-x-32", "flex_and_grid"),
        ("gap-x-36", "flex_and_grid"),
        ("gap-x-4", "flex_and_grid"),
        ("gap-x-40", "flex_and_grid"),
        ("gap-x-44", "flex_and_grid"),
        ("gap-x-48", "flex_and_grid"),
        ("gap-x-5", "flex_and_grid"),
        ("gap-x-52", "flex_and_grid"),
        ("gap-x-56", "flex_and_grid"),
        ("gap-x-6", "flex_and_grid"),
        ("gap-x-60", "flex_and_grid"),
        ("gap-x-64", "flex_and_grid"),
        ("gap-x-7", "flex_and_grid"),
        ("gap-x-72", "flex_and_grid"),
        ("gap-x-8", "flex_and_grid"),
        ("gap-x-80", "flex_and_grid"),
        ("gap-x-9", "flex_and_grid"),
        ("gap-x-96", "flex_and_grid"),
        ("gap-x-px", "flex_and_grid"),
        ("gap-y-0", "flex_and_grid"),
        ("gap-y-0.5", "flex_and_grid"),
        ("gap-y-1", "flex_and_grid"),
        ("gap-y-1.5", "flex_and_grid"),
        ("gap-y-10", "flex_and_grid"),
        ("gap-y-11", "flex_and_grid"),
        ("gap-y-12", "flex_and_grid"),
        ("gap-y-14", "flex_and_grid"),
        ("gap-y-16", "flex_and_grid"),
        ("gap-y-2", "flex_and_grid"),
        ("gap-y-2.5", "flex_and_grid"),
        ("gap-y-20", "flex_and_grid"),
        ("gap-y-24", "flex_and_grid"),
        ("gap-y-28", "flex_and_grid"),
        ("gap-y-3", "flex_and_grid"),
        ("gap-y-3.5", "flex_and_grid"),
        ("gap-y-32", "flex_and_grid"),
        ("gap-y-36", "flex_and_grid"),
        ("gap-y-4", "flex_and_grid"),
        ("gap-y-40", "flex_and_grid"),
        ("gap-y-44", "flex_and_grid"),
        ("gap-y-48", "flex_and_grid"),
        ("gap-y-5", "flex_and_grid"),
        ("gap-y-52", "flex_and_grid"),
        ("gap-y-56", "flex_and_grid"),
        ("gap-y-6", "flex_and_grid"),
        ("gap-y-60", "flex_and_grid"),
        ("gap-y-64", "flex_and_grid"),
        ("gap-y-7", "flex_and_grid"),
        ("gap-y-72", "flex_and_grid"),
        ("gap-y-8", "flex_and_grid"),
        ("gap-y-80", "flex_and_grid"),
        ("gap-y-9", "flex_and_grid"),
        ("gap-y-96", "flex_and_grid"),
        ("gap-y-px", "flex_and_grid"),
        ("grid-cols-1", "flex_and_grid"),
        ("grid-cols-10", "flex_and_grid"),
        ("grid-cols-11", "flex_and_grid"),
        ("grid-cols-12", "flex_and_grid"),
        ("grid-cols-2", "flex_and_grid"),
        ("grid-cols-3", "flex_and_grid"),
        ("grid-cols-4", "flex_and_grid"),
        ("grid-cols-5", "flex_and_grid"),
        ("grid-cols-6", "flex_and_grid"),
        ("grid-cols-7", "flex_and_grid"),
        ("grid-cols-8", "flex_and_grid"),
        ("grid-cols-9", "flex_and_grid"),
        ("grid-cols-none", "flex_and_grid"),
        ("grid-flow-col", "flex_and_grid"),
        ("grid-flow-col-dense", "flex_and_grid"),
        ("grid-flow-row", "flex_and_grid"),
        ("grid-flow-row-dense", "flex_and_grid"),
        ("grid-rows-1", "flex_and_grid"),
        ("grid-rows-2", "flex_and_grid"),
        ("grid-rows-3", "flex_and_grid"),
        ("grid-rows-4", "flex_and_grid"),
        ("grid-rows-5", "flex_and_grid"),
        ("grid-rows-6", "flex_and_grid"),
        ("grid-rows-none", "flex_and_grid"),
        ("grow", "flex_and_grid"),
        ("grow-0", "flex_and_grid"),
        ("items-baseline", "flex_and_grid"),
        ("items-center", "flex_and_grid"),
        ("items-end", "flex_and_grid"),
        ("items-start", "flex_and_grid"),
        ("items-stretch", "flex_and_grid"),
        ("justify-around", "flex_and_grid"),
        ("justify-between", "flex_and_grid"),
        ("justify-center", "flex_and_grid"),
        ("justify-end", "flex_and_grid"),
        ("justify-evenly", "flex_and_grid"),
        ("justify-items-center", "flex_and_grid"),
        ("justify-items-end", "flex_and_grid"),
        ("justify-items-start", "flex_and_grid"),
        ("justify-items-stretch", "flex_and_grid"),
        ("justify-self-auto", "flex_and_grid"),
        ("justify-self-center", "flex_and_grid"),
        ("justify-self-end", "flex_and_grid"),
        ("justify-self-start", "flex_and_grid"),
        ("justify-self-stretch", "flex_and_grid"),
        ("justify-start", "flex_and_grid"),
        ("order-1", "flex_and_grid"),
        ("order-10", "flex_and_grid"),
        ("order-11", "flex_and_grid"),
        ("order-12", "flex_and_grid"),
        ("order-2", "flex_and_grid"),
        ("order-3", "flex_and_grid"),
        ("order-4", "flex_and_grid"),
        ("order-5", "flex_and_grid"),
        ("order-6", "flex_and_grid"),
        ("order-7", "flex_and_grid"),
        ("order-8", "flex_and_grid"),
        ("order-9", "flex_and_grid"),
        ("order-first", "flex_and_grid"),
        ("order-last", "flex_and_grid"),
        ("order-none", "flex_and_grid"),
        ("place-content-around", "flex_and_grid"),
        ("place-content-between", "flex_and_grid"),
        ("place-content-center", "flex_and_grid"),
        ("place-content-end", "flex_and_grid"),
        ("place-content-evenly", "flex_and_grid"),
        ("place-content-start", "flex_and_grid"),
        ("place-content-stretch", "flex_and_grid"),
        ("place-items-center", "flex_and_grid"),
        ("place-items-end", "flex_and_grid"),
        ("place-items-start", "flex_and_grid"),
        ("place-items-stretch", "flex_and_grid"),
        ("place-self-auto", "flex_and_grid"),
        ("place-self-center", "flex_and_grid"),
        ("place-self-end", "flex_and_grid"),
        ("place-self-start", "flex_and_grid"),
        ("place-self-stretch", "flex_and_grid"),
        ("row-auto", "flex_and_grid"),
        ("row-end-1", "flex_and_grid"),
        ("row-end-2", "flex_and_grid"),
        ("row-end-3", "flex_and_grid"),
        ("row-end-4", "flex_and_grid"),
        ("row-end-5", "flex_and_grid"),
        ("row-end-6", "flex_and_grid"),
        ("row-end-7", "flex_and_grid"),
        ("row-end-auto", "flex_and_grid"),
        ("row-span-1", "flex_and_grid"),
        ("row-span-2", "flex_and_grid"),
        ("row-span-3", "flex_and_grid"),
        ("row-span-4", "flex_and_grid"),
        ("row-span-5", "flex_and_grid"),
        ("row-span-6", "flex_and_grid"),
        ("row-span-full", "flex_and_grid"),
        ("row-start-1", "flex_and_grid"),
        ("row-start-2", "flex_and_grid"),
        ("row-start-3", "flex_and_grid"),
        ("row-start-4", "flex_and_grid"),
        ("row-start-5", "flex_and_grid"),
        ("row-start-6", "flex_and_grid"),
        ("row-start-7", "flex_and_grid"),
        ("row-start-auto", "flex_and_grid"),
        ("self-auto", "flex_and_grid"),
        ("self-baseline", "flex_and_grid"),
        ("self-center", "flex_and_grid"),
        ("self-end", "flex_and_grid"),
        ("self-start", "flex_and_grid"),
        ("self-stretch", "flex_and_grid"),
        ("shrink", "flex_and_grid"),
        ("shrink-0", "flex_and_grid"),
        ("accent-amber-100", "interactivity"),
        ("accent-amber-200", "interactivity"),
        ("accent-amber-300", "interactivity"),
        ("accent-amber-400", "interactivity"),
        ("accent-amber-50", "interactivity"),
        ("accent-amber-500", "interactivity"),
        ("accent-amber-600", "interactivity"),
        ("accent-amber-700", "interactivity"),
        ("accent-amber-800", "interactivity"),
        ("accent-amber-900", "interactivity"),
        ("accent-auto", "interactivity"),
        ("accent-black", "interactivity"),
        ("accent-blue-100", "interactivity"),
        ("accent-blue-200", "interactivity"),
        ("accent-blue-300", "interactivity"),
        ("accent-blue-400", "interactivity"),
        ("accent-blue-50", "interactivity"),
        ("accent-blue-500", "interactivity"),
        ("accent-blue-600", "interactivity"),
        ("accent-blue-700", "interactivity"),
        ("accent-blue-800", "interactivity"),
        ("accent-blue-900", "interactivity"),
        ("accent-current", "interactivity"),
        ("accent-cyan-100", "interactivity"),
        ("accent-cyan-200", "interactivity"),
        ("accent-cyan-300", "interactivity"),
        ("accent-cyan-400", "interactivity"),
        ("accent-cyan-50", "interactivity"),
        ("accent-cyan-500", "interactivity"),
        ("accent-cyan-600", "interactivity"),
        ("accent-cyan-700", "interactivity"),
        ("accent-cyan-800", "interactivity"),
        ("accent-cyan-900", "interactivity"),
        ("accent-emerald-100", "interactivity"),
        ("accent-emerald-200", "interactivity"),
        ("accent-emerald-300", "interactivity"),
        ("accent-emerald-400", "interactivity"),
        ("accent-emerald-50", "interactivity"),
        ("accent-emerald-500", "interactivity"),
        ("accent-emerald-600", "interactivity"),
        ("accent-emerald-700", "interactivity"),
        ("accent-emerald-800", "interactivity"),
        ("accent-emerald-900", "interactivity"),
        ("accent-fuchsia-100", "interactivity"),
        ("accent-fuchsia-200", "interactivity"),
        ("accent-fuchsia-300", "interactivity"),
        ("accent-fuchsia-400", "interactivity"),
        ("accent-fuchsia-50", "interactivity"),
        ("accent-fuchsia-500", "interactivity"),
        ("accent-fuchsia-600", "interactivity"),
        ("accent-fuchsia-700", "interactivity"),
        ("accent-fuchsia-800", "interactivity"),
        ("accent-fuchsia-900", "interactivity"),
        ("accent-gray-100", "interactivity"),
        ("accent-gray-200", "interactivity"),
        ("accent-gray-300", "interactivity"),
        ("accent-gray-400", "interactivity"),
        ("accent-gray-50", "interactivity"),
        ("accent-gray-500", "interactivity"),
        ("accent-gray-600", "interactivity"),
        ("accent-gray-700", "interactivity"),
        ("accent-gray-800", "interactivity"),
        ("accent-gray-900", "interactivity"),
        ("accent-green-100", "interactivity"),
        ("accent-green-200", "interactivity"),
        ("accent-green-300", "interactivity"),
        ("accent-green-400", "interactivity"),
        ("accent-green-50", "interactivity"),
        ("accent-green-500", "interactivity"),
        ("accent-green-600", "interactivity"),
        ("accent-green-700", "interactivity"),
        ("accent-green-800", "interactivity"),
        ("accent-green-900", "interactivity"),
        ("accent-indigo-100", "interactivity"),
        ("accent-indigo-200", "interactivity"),
        ("accent-indigo-300", "interactivity"),
        ("accent-indigo-400", "interactivity"),
        ("accent-indigo-50", "interactivity"),
        ("accent-indigo-500", "interactivity"),
        ("accent-indigo-600", "interactivity"),
        ("accent-indigo-700", "interactivity"),
        ("accent-indigo-800", "interactivity"),
        ("accent-indigo-900", "interactivity"),
        ("accent-inherit", "interactivity"),
        ("accent-lime-100", "interactivity"),
        ("accent-lime-200", "interactivity"),
        ("accent-lime-300", "interactivity"),
        ("accent-lime-400", "interactivity"),
        ("accent-lime-50", "interactivity"),
        ("accent-lime-500", "interactivity"),
        ("accent-lime-600", "interactivity"),
        ("accent-lime-700", "interactivity"),
        ("accent-lime-800", "interactivity"),
        ("accent-lime-900", "interactivity"),
        ("accent-neutral-100", "interactivity"),
        ("accent-neutral-200", "interactivity"),
        ("accent-neutral-300", "interactivity"),
        ("accent-neutral-400", "interactivity"),
        ("accent-neutral-50", "interactivity"),
        ("accent-neutral-500", "interactivity"),
        ("accent-neutral-600", "interactivity"),
        ("accent-neutral-700", "interactivity"),
        ("accent-neutral-800", "interactivity"),
        ("accent-neutral-900", "interactivity"),
        ("accent-orange-100", "interactivity"),
        ("accent-orange-200", "interactivity"),
        ("accent-orange-300", "interactivity"),
        ("accent-orange-400", "interactivity"),
        ("accent-orange-50", "interactivity"),
        ("accent-orange-500", "interactivity"),
        ("accent-orange-600", "interactivity"),
        ("accent-orange-700", "interactivity"),
        ("accent-orange-800", "interactivity"),
        ("accent-orange-900", "interactivity"),
        ("accent-pink-100", "interactivity"),
        ("accent-pink-200", "interactivity"),
        ("accent-pink-300", "interactivity"),
        ("accent-pink-400", "interactivity"),
        ("accent-pink-50", "interactivity"),
        ("accent-pink-500", "interactivity"),
        ("accent-pink-600", "interactivity"),
        ("accent-pink-700", "interactivity"),
        ("accent-pink-800", "interactivity"),
        ("accent-pink-900", "interactivity"),
        ("accent-purple-100", "interactivity"),
        ("accent-purple-200", "interactivity"),
        ("accent-purple-300", "interactivity"),
        ("accent-purple-400", "interactivity"),
        ("accent-purple-50", "interactivity"),
        ("accent-purple-500", "interactivity"),
        ("accent-purple-600", "interactivity"),
        ("accent-purple-700", "interactivity"),
        ("accent-purple-800", "interactivity"),
        ("accent-purple-900", "interactivity"),
        ("accent-red-100", "interactivity"),
        ("accent-red-200", "interactivity"),
        ("accent-red-300", "interactivity"),
        ("accent-red-400", "interactivity"),
        ("accent-red-50", "interactivity"),
        ("accent-red-500", "interactivity"),
        ("accent-red-600", "interactivity"),
        ("accent-red-700", "interactivity"),
        ("accent-red-800", "interactivity"),
        ("accent-red-900", "interactivity"),
        ("accent-rose-100", "interactivity"),
        ("accent-rose-200", "interactivity"),
        ("accent-rose-300", "interactivity"),
        ("accent-rose-400", "interactivity"),
        ("accent-rose-50", "interactivity"),
        ("accent-rose-500", "interactivity"),
        ("accent-rose-600", "interactivity"),
        ("accent-rose-700", "interactivity"),
        ("accent-rose-800", "interactivity"),
        ("accent-rose-900", "interactivity"),
        ("accent-sky-100", "interactivity"),
        ("accent-sky-200", "interactivity"),
        ("accent-sky-300", "interactivity"),
        ("accent-sky-400", "interactivity"),
        ("accent-sky-50", "interactivity"),
        ("accent-sky-500", "interactivity"),
        ("accent-sky-600", "interactivity"),
        ("accent-sky-700", "interactivity"),
        ("accent-sky-800", "interactivity"),
        ("accent-sky-900", "interactivity"),
        ("accent-slate-100", "interactivity"),
        ("accent-slate-200", "interactivity"),
        ("accent-slate-300", "interactivity"),
        ("accent-slate-400", "interactivity"),
        ("accent-slate-50", "interactivity"),
        ("accent-slate-500", "interactivity"),
        ("accent-slate-600", "interactivity"),
        ("accent-slate-700", "interactivity"),
        ("accent-slate-800", "interactivity"),
        ("accent-slate-900", "interactivity"),
        ("accent-stone-100", "interactivity"),
        ("accent-stone-200", "interactivity"),
        ("accent-stone-300", "interactivity"),
        ("accent-stone-400", "interactivity"),
        ("accent-stone-50", "interactivity"),
        ("accent-stone-500", "interactivity"),
        ("accent-stone-600", "interactivity"),
        ("accent-stone-700", "interactivity"),
        ("accent-stone-800", "interactivity"),
        ("accent-stone-900", "interactivity"),
        ("accent-teal-100", "interactivity"),
        ("accent-teal-200", "interactivity"),
        ("accent-teal-300", "interactivity"),
        ("accent-teal-400", "interactivity"),
        ("accent-teal-50", "interactivity"),
        ("accent-teal-500", "interactivity"),
        ("accent-teal-600", "interactivity"),
        ("accent-teal-700", "interactivity"),
        ("accent-teal-800", "interactivity"),
        ("accent-teal-900", "interactivity"),
        ("accent-transparent", "interactivity"),
        ("accent-violet-100", "interactivity"),
        ("accent-violet-200", "interactivity"),
        ("accent-violet-300", "interactivity"),
        ("accent-violet-400", "interactivity"),
        ("accent-violet-50", "interactivity"),
        ("accent-violet-500", "interactivity"),
        ("accent-violet-600", "interactivity"),
        ("accent-violet-700", "interactivity"),
        ("accent-violet-800", "interactivity"),
        ("accent-violet-900", "interactivity"),
        ("accent-white", "interactivity"),
        ("accent-yellow-100", "interactivity"),
        ("accent-yellow-200", "interactivity"),
        ("accent-yellow-300", "interactivity"),
        ("accent-yellow-400", "interactivity"),
        ("accent-yellow-50", "interactivity"),
        ("accent-yellow-500", "interactivity"),
        ("accent-yellow-600", "interactivity"),
        ("accent-yellow-700", "interactivity"),
        ("accent-yellow-800", "interactivity"),
        ("accent-yellow-900", "interactivity"),
        ("accent-zinc-100", "interactivity"),
        ("accent-zinc-200", "interactivity"),
        ("accent-zinc-300", "interactivity"),
        ("accent-zinc-400", "interactivity"),
        ("accent-zinc-50", "interactivity"),
        ("accent-zinc-500", "interactivity"),
        ("accent-zinc-600", "interactivity"),
        ("accent-zinc-700", "interactivity"),
        ("accent-zinc-800", "interactivity"),
        ("accent-zinc-900", "interactivity"),
        ("appearance-none", "interactivity"),
        ("caret-amber-100", "interactivity"),
        ("caret-amber-200", "interactivity"),
        ("caret-amber-300", "interactivity"),
        ("caret-amber-400", "interactivity"),
        ("caret-amber-50", "interactivity"),
        ("caret-amber-500", "interactivity"),
        ("caret-amber-600", "interactivity"),
        ("caret-amber-700", "interactivity"),
        ("caret-amber-800", "interactivity"),
        ("caret-amber-900", "interactivity"),
        ("caret-black", "interactivity"),
        ("caret-blue-100", "interactivity"),
        ("caret-blue-200", "interactivity"),
        ("caret-blue-300", "interactivity"),
        ("caret-blue-400", "interactivity"),
        ("caret-blue-50", "interactivity"),
        ("caret-blue-500", "interactivity"),
        ("caret-blue-600", "interactivity"),
        ("caret-blue-700", "interactivity"),
        ("caret-blue-800", "interactivity"),
        ("caret-blue-900", "interactivity"),
        ("caret-current", "interactivity"),
        ("caret-cyan-100", "interactivity"),
        ("caret-cyan-200", "interactivity"),
        ("caret-cyan-300", "interactivity"),
        ("caret-cyan-400", "interactivity"),
        ("caret-cyan-50", "interactivity"),
        ("caret-cyan-500", "interactivity"),
        ("caret-cyan-600", "interactivity"),
        ("caret-cyan-700", "interactivity"),
        ("caret-cyan-800", "interactivity"),
        ("caret-cyan-900", "interactivity"),
        ("caret-emerald-100", "interactivity"),
        ("caret-emerald-200", "interactivity"),
        ("caret-emerald-300", "interactivity"),
        ("caret-emerald-400", "interactivity"),
        ("caret-emerald-50", "interactivity"),
        ("caret-emerald-500", "interactivity"),
        ("caret-emerald-600", "interactivity"),
        ("caret-emerald-700", "interactivity"),
        ("caret-emerald-800", "interactivity"),
        ("caret-emerald-900", "interactivity"),
        ("caret-fuchsia-100", "interactivity"),
        ("caret-fuchsia-200", "interactivity"),
        ("caret-fuchsia-300", "interactivity"),
        ("caret-fuchsia-400", "interactivity"),
        ("caret-fuchsia-50", "interactivity"),
        ("caret-fuchsia-500", "interactivity"),
        ("caret-fuchsia-600", "interactivity"),
        ("caret-fuchsia-700", "interactivity"),
        ("caret-fuchsia-800", "interactivity"),
        ("caret-fuchsia-900", "interactivity"),
        ("caret-gray-100", "interactivity"),
        ("caret-gray-200", "interactivity"),
        ("caret-gray-300", "interactivity"),
        ("caret-gray-400", "interactivity"),
        ("caret-gray-50", "interactivity"),
        ("caret-gray-500", "interactivity"),
        ("caret-gray-600", "interactivity"),
        ("caret-gray-700", "interactivity"),
        ("caret-gray-800", "interactivity"),
        ("caret-gray-900", "interactivity"),
        ("caret-green-100", "interactivity"),
        ("caret-green-200", "interactivity"),
        ("caret-green-300", "interactivity"),
        ("caret-green-400", "interactivity"),
        ("caret-green-50", "interactivity"),
        ("caret-green-500", "interactivity"),
        ("caret-green-600", "interactivity"),
        ("caret-green-700", "interactivity"),
        ("caret-green-800", "interactivity"),
        ("caret-green-900", "interactivity"),
        ("caret-indigo-100", "interactivity"),
        ("caret-indigo-200", "interactivity"),
        ("caret-indigo-300", "interactivity"),
        ("caret-indigo-400", "interactivity"),
        ("caret-indigo-50", "interactivity"),
        ("caret-indigo-500", "interactivity"),
        ("caret-indigo-600", "interactivity"),
        ("caret-indigo-700", "interactivity"),
        ("caret-indigo-800", "interactivity"),
        ("caret-indigo-900", "interactivity"),
        ("caret-inherit", "interactivity"),
        ("caret-lime-100", "interactivity"),
        ("caret-lime-200", "interactivity"),
        ("caret-lime-300", "interactivity"),
        ("caret-lime-400", "interactivity"),
        ("caret-lime-50", "interactivity"),
        ("caret-lime-500", "interactivity"),
        ("caret-lime-600", "interactivity"),
        ("caret-lime-700", "interactivity"),
        ("caret-lime-800", "interactivity"),
        ("caret-lime-900", "interactivity"),
        ("caret-neutral-100", "interactivity"),
        ("caret-neutral-200", "interactivity"),
        ("caret-neutral-300", "interactivity"),
        ("caret-neutral-400", "interactivity"),
        ("caret-neutral-50", "interactivity"),
        ("caret-neutral-500", "interactivity"),
        ("caret-neutral-600", "interactivity"),
        ("caret-neutral-700", "interactivity"),
        ("caret-neutral-800", "interactivity"),
        ("caret-neutral-900", "interactivity"),
        ("caret-orange-100", "interactivity"),
        ("caret-orange-200", "interactivity"),
        ("caret-orange-300", "interactivity"),
        ("caret-orange-400", "interactivity"),
        ("caret-orange-50", "interactivity"),
        ("caret-orange-500", "interactivity"),
        ("caret-orange-600", "interactivity"),
        ("caret-orange-700", "interactivity"),
        ("caret-orange-800", "interactivity"),
        ("caret-orange-900", "interactivity"),
        ("caret-pink-100", "interactivity"),
        ("caret-pink-200", "interactivity"),
        ("caret-pink-300", "interactivity"),
        ("caret-pink-400", "interactivity"),
        ("caret-pink-50", "interactivity"),
        ("caret-pink-500", "interactivity"),
        ("caret-pink-600", "interactivity"),
        ("caret-pink-700", "interactivity"),
        ("caret-pink-800", "interactivity"),
        ("caret-pink-900", "interactivity"),
        ("caret-purple-100", "interactivity"),
        ("caret-purple-200", "interactivity"),
        ("caret-purple-300", "interactivity"),
        ("caret-purple-400", "interactivity"),
        ("caret-purple-50", "interactivity"),
        ("caret-purple-500", "interactivity"),
        ("caret-purple-600", "interactivity"),
        ("caret-purple-700", "interactivity"),
        ("caret-purple-800", "interactivity"),
        ("caret-purple-900", "interactivity"),
        ("caret-red-100", "interactivity"),
        ("caret-red-200", "interactivity"),
        ("caret-red-300", "interactivity"),
        ("caret-red-400", "interactivity"),
        ("caret-red-50", "interactivity"),
        ("caret-red-500", "interactivity"),
        ("caret-red-600", "interactivity"),
        ("caret-red-700", "interactivity"),
        ("caret-red-800", "interactivity"),
        ("caret-red-900", "interactivity"),
        ("caret-rose-100", "interactivity"),
        ("caret-rose-200", "interactivity"),
        ("caret-rose-300", "interactivity"),
        ("caret-rose-400", "interactivity"),
        ("caret-rose-50", "interactivity"),
        ("caret-rose-500", "interactivity"),
        ("caret-rose-600", "interactivity"),
        ("caret-rose-700", "interactivity"),
        ("caret-rose-800", "interactivity"),
        ("caret-rose-900", "interactivity"),
        ("caret-sky-100", "interactivity"),
        ("caret-sky-200", "interactivity"),
        ("caret-sky-300", "interactivity"),
        ("caret-sky-400", "interactivity"),
        ("caret-sky-50", "interactivity"),
        ("caret-sky-500", "interactivity"),
        ("caret-sky-600", "interactivity"),
        ("caret-sky-700", "interactivity"),
        ("caret-sky-800", "interactivity"),
        ("caret-sky-900", "interactivity"),
        ("caret-slate-100", "interactivity"),
        ("caret-slate-200", "interactivity"),
        ("caret-slate-300", "interactivity"),
        ("caret-slate-400", "interactivity"),
        ("caret-slate-50", "interactivity"),
        ("caret-slate-500", "interactivity"),
        ("caret-slate-600", "interactivity"),
        ("caret-slate-700", "interactivity"),
        ("caret-slate-800", "interactivity"),
        ("caret-slate-900", "interactivity"),
        ("caret-stone-100", "interactivity"),
        ("caret-stone-200", "interactivity"),
        ("caret-stone-300", "interactivity"),
        ("caret-stone-400", "interactivity"),
        ("caret-stone-50", "interactivity"),
        ("caret-stone-500", "interactivity"),
        ("caret-stone-600", "interactivity"),
        ("caret-stone-700", "interactivity"),
        ("caret-stone-800", "interactivity"),
        ("caret-stone-900", "interactivity"),
        ("caret-teal-100", "interactivity"),
        ("caret-teal-200", "interactivity"),
        ("caret-teal-300", "interactivity"),
        ("caret-teal-400", "interactivity"),
        ("caret-teal-50", "interactivity"),
        ("caret-teal-500", "interactivity"),
        ("caret-teal-600", "interactivity"),
        ("caret-teal-700", "interactivity"),
        ("caret-teal-800", "interactivity"),
        ("caret-teal-900", "interactivity"),
        ("caret-transparent", "interactivity"),
        ("caret-violet-100", "interactivity"),
        ("caret-violet-200", "interactivity"),
        ("caret-violet-300", "interactivity"),
        ("caret-violet-400", "interactivity"),
        ("caret-violet-50", "interactivity"),
        ("caret-violet-500", "interactivity"),
        ("caret-violet-600", "interactivity"),
        ("caret-violet-700", "interactivity"),
        ("caret-violet-800", "interactivity"),
        ("caret-violet-900", "interactivity"),
        ("caret-white", "interactivity"),
        ("caret-yellow-100", "interactivity"),
        ("caret-yellow-200", "interactivity"),
        ("caret-yellow-300", "interactivity"),
        ("caret-yellow-400", "interactivity"),
        ("caret-yellow-50", "interactivity"),
        ("caret-yellow-500", "interactivity"),
        ("caret-yellow-600", "interactivity"),
        ("caret-yellow-700", "interactivity"),
        ("caret-yellow-800", "interactivity"),
        ("caret-yellow-900", "interactivity"),
        ("caret-zinc-100", "interactivity"),
        ("caret-zinc-200", "interactivity"),
        ("caret-zinc-300", "interactivity"),
        ("caret-zinc-400", "interactivity"),
        ("caret-zinc-50", "interactivity"),
        ("caret-zinc-500", "interactivity"),
        ("caret-zinc-600", "interactivity"),
        ("caret-zinc-700", "interactivity"),
        ("caret-zinc-800", "interactivity"),
        ("caret-zinc-900", "interactivity"),
        ("cursor-alias", "interactivity"),
        ("cursor-all-scroll", "interactivity"),
        ("cursor-auto", "interactivity"),
        ("cursor-cell", "interactivity"),
        ("cursor-col-resize", "interactivity"),
        ("cursor-context-menu", "interactivity"),
        ("cursor-copy", "interactivity"),
        ("cursor-crosshair", "interactivity"),
        ("cursor-default", "interactivity"),
        ("cursor-e-resize", "interactivity"),
        ("cursor-ew-resize", "interactivity"),
        ("cursor-grab", "interactivity"),
        ("cursor-grabbing", "interactivity"),
        ("cursor-help", "interactivity"),
        ("cursor-move", "interactivity"),
        ("cursor-n-resize", "interactivity"),
        ("cursor-ne-resize", "interactivity"),
        ("cursor-nesw-resize", "interactivity"),
        ("cursor-no-drop", "interactivity"),
        ("cursor-none", "interactivity"),
        ("cursor-not-allowed", "interactivity"),
        ("cursor-ns-resize", "interactivity"),
        ("cursor-nw-resize", "interactivity"),
        ("cursor-nwse-resize", "interactivity"),
        ("cursor-pointer", "interactivity"),
        ("cursor-progress", "interactivity"),
        ("cursor-row-resize", "interactivity"),
        ("cursor-s-resize", "interactivity"),
        ("cursor-se-resize", "interactivity"),
        ("cursor-sw-resize", "interactivity"),
        ("cursor-text", "interactivity"),
        ("cursor-vertical-text", "interactivity"),
        ("cursor-w-resize", "interactivity"),
        ("cursor-wait", "interactivity"),
        ("cursor-zoom-in", "interactivity"),
        ("cursor-zoom-out", "interactivity"),
        ("pointer-events-auto", "interactivity"),
        ("pointer-events-none", "interactivity"),
        ("resize", "interactivity"),
        ("resize-none", "interactivity"),
        ("resize-x", "interactivity"),
        ("resize-y", "interactivity"),
        ("scroll-auto", "interactivity"),
        ("scroll-m-0", "interactivity"),
        ("scroll-m-0.5", "interactivity"),
        ("scroll-m-1", "interactivity"),
        ("scroll-m-1.5", "interactivity"),
        ("scroll-m-10", "interactivity"),
        ("scroll-m-11", "interactivity"),
        ("scroll-m-12", "interactivity"),
        ("scroll-m-14", "interactivity"),
        ("scroll-m-16", "interactivity"),
        ("scroll-m-2", "interactivity"),
        ("scroll-m-2.5", "interactivity"),
        ("scroll-m-20", "interactivity"),
        ("scroll-m-24", "interactivity"),
        ("scroll-m-28", "interactivity"),
        ("scroll-m-3", "interactivity"),
        ("scroll-m-3.5", "interactivity"),
        ("scroll-m-32", "interactivity"),
        ("scroll-m-36", "interactivity"),
        ("scroll-m-4", "interactivity"),
        ("scroll-m-40", "interactivity"),
        ("scroll-m-44", "interactivity"),
        ("scroll-m-48", "interactivity"),
        ("scroll-m-5", "interactivity"),
        ("scroll-m-52", "interactivity"),
        ("scroll-m-56", "interactivity"),
        ("scroll-m-6", "interactivity"),
        ("scroll-m-60", "interactivity"),
        ("scroll-m-64", "interactivity"),
        ("scroll-m-7", "interactivity"),
        ("scroll-m-72", "interactivity"),
        ("scroll-m-8", "interactivity"),
        ("scroll-m-80", "interactivity"),
        ("scroll-m-9", "interactivity"),
        ("scroll-m-96", "interactivity"),
        ("scroll-m-px", "interactivity"),
        ("scroll-mb-0", "interactivity"),
        ("scroll-mb-0.5", "interactivity"),
        ("scroll-mb-1", "interactivity"),
        ("scroll-mb-1.5", "interactivity"),
        ("scroll-mb-10", "interactivity"),
        ("scroll-mb-11", "interactivity"),
        ("scroll-mb-12", "interactivity"),
        ("scroll-mb-14", "interactivity"),
        ("scroll-mb-16", "interactivity"),
        ("scroll-mb-2", "interactivity"),
        ("scroll-mb-2.5", "interactivity"),
        ("scroll-mb-20", "interactivity"),
        ("scroll-mb-24", "interactivity"),
        ("scroll-mb-28", "interactivity"),
        ("scroll-mb-3", "interactivity"),
        ("scroll-mb-3.5", "interactivity"),
        ("scroll-mb-32", "interactivity"),
        ("scroll-mb-36", "interactivity"),
        ("scroll-mb-4", "interactivity"),
        ("scroll-mb-40", "interactivity"),
        ("scroll-mb-44", "interactivity"),
        ("scroll-mb-48", "interactivity"),
        ("scroll-mb-5", "interactivity"),
        ("scroll-mb-52", "interactivity"),
        ("scroll-mb-56", "interactivity"),
        ("scroll-mb-6", "interactivity"),
        ("scroll-mb-60", "interactivity"),
        ("scroll-mb-64", "interactivity"),
        ("scroll-mb-7", "interactivity"),
        ("scroll-mb-72", "interactivity"),
        ("scroll-mb-8", "interactivity"),
        ("scroll-mb-80", "interactivity"),
        ("scroll-mb-9", "interactivity"),
        ("scroll-mb-96", "interactivity"),
        ("scroll-mb-px", "interactivity"),
        ("scroll-ml-0", "interactivity"),
        ("scroll-ml-0.5", "interactivity"),
        ("scroll-ml-1", "interactivity"),
        ("scroll-ml-1.5", "interactivity"),
        ("scroll-ml-10", "interactivity"),
        ("scroll-ml-11", "interactivity"),
        ("scroll-ml-12", "interactivity"),
        ("scroll-ml-14", "interactivity"),
        ("scroll-ml-16", "interactivity"),
        ("scroll-ml-2", "interactivity"),
        ("scroll-ml-2.5", "interactivity"),
        ("scroll-ml-20", "interactivity"),
        ("scroll-ml-24", "interactivity"),
        ("scroll-ml-28", "interactivity"),
        ("scroll-ml-3", "interactivity"),
        ("scroll-ml-3.5", "interactivity"),
        ("scroll-ml-32", "interactivity"),
        ("scroll-ml-36", "interactivity"),
        ("scroll-ml-4", "interactivity"),
        ("scroll-ml-40", "interactivity"),
        ("scroll-ml-44", "interactivity"),
        ("scroll-ml-48", "interactivity"),
        ("scroll-ml-5", "interactivity"),
        ("scroll-ml-52", "interactivity"),
        ("scroll-ml-56", "interactivity"),
        ("scroll-ml-6", "interactivity"),
        ("scroll-ml-60", "interactivity"),
        ("scroll-ml-64", "interactivity"),
        ("scroll-ml-7", "interactivity"),
        ("scroll-ml-72", "interactivity"),
        ("scroll-ml-8", "interactivity"),
        ("scroll-ml-80", "interactivity"),
        ("scroll-ml-9", "interactivity"),
        ("scroll-ml-96", "interactivity"),
        ("scroll-ml-px", "interactivity"),
        ("scroll-mr-0", "interactivity"),
        ("scroll-mr-0.5", "interactivity"),
        ("scroll-mr-1", "interactivity"),
        ("scroll-mr-1.5", "interactivity"),
        ("scroll-mr-10", "interactivity"),
        ("scroll-mr-11", "interactivity"),
        ("scroll-mr-12", "interactivity"),
        ("scroll-mr-14", "interactivity"),
        ("scroll-mr-16", "interactivity"),
        ("scroll-mr-2", "interactivity"),
        ("scroll-mr-2.5", "interactivity"),
        ("scroll-mr-20", "interactivity"),
        ("scroll-mr-24", "interactivity"),
        ("scroll-mr-28", "interactivity"),
        ("scroll-mr-3", "interactivity"),
        ("scroll-mr-3.5", "interactivity"),
        ("scroll-mr-32", "interactivity"),
        ("scroll-mr-36", "interactivity"),
        ("scroll-mr-4", "interactivity"),
        ("scroll-mr-40", "interactivity"),
        ("scroll-mr-44", "interactivity"),
        ("scroll-mr-48", "interactivity"),
        ("scroll-mr-5", "interactivity"),
        ("scroll-mr-52", "interactivity"),
        ("scroll-mr-56", "interactivity"),
        ("scroll-mr-6", "interactivity"),
        ("scroll-mr-60", "interactivity"),
        ("scroll-mr-64", "interactivity"),
        ("scroll-mr-7", "interactivity"),
        ("scroll-mr-72", "interactivity"),
        ("scroll-mr-8", "interactivity"),
        ("scroll-mr-80", "interactivity"),
        ("scroll-mr-9", "interactivity"),
        ("scroll-mr-96", "interactivity"),
        ("scroll-mr-px", "interactivity"),
        ("scroll-mt-0", "interactivity"),
        ("scroll-mt-0.5", "interactivity"),
        ("scroll-mt-1", "interactivity"),
        ("scroll-mt-1.5", "interactivity"),
        ("scroll-mt-10", "interactivity"),
        ("scroll-mt-11", "interactivity"),
        ("scroll-mt-12", "interactivity"),
        ("scroll-mt-14", "interactivity"),
        ("scroll-mt-16", "interactivity"),
        ("scroll-mt-2", "interactivity"),
        ("scroll-mt-2.5", "interactivity"),
        ("scroll-mt-20", "interactivity"),
        ("scroll-mt-24", "interactivity"),
        ("scroll-mt-28", "interactivity"),
        ("scroll-mt-3", "interactivity"),
        ("scroll-mt-3.5", "interactivity"),
        ("scroll-mt-32", "interactivity"),
        ("scroll-mt-36", "interactivity"),
        ("scroll-mt-4", "interactivity"),
        ("scroll-mt-40", "interactivity"),
        ("scroll-mt-44", "interactivity"),
        ("scroll-mt-48", "interactivity"),
        ("scroll-mt-5", "interactivity"),
        ("scroll-mt-52", "interactivity"),
        ("scroll-mt-56", "interactivity"),
        ("scroll-mt-6", "interactivity"),
        ("scroll-mt-60", "interactivity"),
        ("scroll-mt-64", "interactivity"),
        ("scroll-mt-7", "interactivity"),
        ("scroll-mt-72", "interactivity"),
        ("scroll-mt-8", "interactivity"),
        ("scroll-mt-80", "interactivity"),
        ("scroll-mt-9", "interactivity"),
        ("scroll-mt-96", "interactivity"),
        ("scroll-mt-px", "interactivity"),
        ("scroll-mx-0", "interactivity"),
        ("scroll-mx-0.5", "interactivity"),
        ("scroll-mx-1", "interactivity"),
        ("scroll-mx-1.5", "interactivity"),
        ("scroll-mx-10", "interactivity"),
        ("scroll-mx-11", "interactivity"),
        ("scroll-mx-12", "interactivity"),
        ("scroll-mx-14", "interactivity"),
        ("scroll-mx-16", "interactivity"),
        ("scroll-mx-2", "interactivity"),
        ("scroll-mx-2.5", "interactivity"),
        ("scroll-mx-20", "interactivity"),
        ("scroll-mx-24", "interactivity"),
        ("scroll-mx-28", "interactivity"),
        ("scroll-mx-3", "interactivity"),
        ("scroll-mx-3.5", "interactivity"),
        ("scroll-mx-32", "interactivity"),
        ("scroll-mx-36", "interactivity"),
        ("scroll-mx-4", "interactivity"),
        ("scroll-mx-40", "interactivity"),
        ("scroll-mx-44", "interactivity"),
        ("scroll-mx-48", "interactivity"),
        ("scroll-mx-5", "interactivity"),
        ("scroll-mx-52", "interactivity"),
        ("scroll-mx-56", "interactivity"),
        ("scroll-mx-6", "interactivity"),
        ("scroll-mx-60", "interactivity"),
        ("scroll-mx-64", "interactivity"),
        ("scroll-mx-7", "interactivity"),
        ("scroll-mx-72", "interactivity"),
        ("scroll-mx-8", "interactivity"),
        ("scroll-mx-80", "interactivity"),
        ("scroll-mx-9", "interactivity"),
        ("scroll-mx-96", "interactivity"),
        ("scroll-mx-px", "interactivity"),
        ("scroll-my-0", "interactivity"),
        ("scroll-my-0.5", "interactivity"),
        ("scroll-my-1", "interactivity"),
        ("scroll-my-1.5", "interactivity"),
        ("scroll-my-10", "interactivity"),
        ("scroll-my-11", "interactivity"),
        ("scroll-my-12", "interactivity"),
        ("scroll-my-14", "interactivity"),
        ("scroll-my-16", "interactivity"),
        ("scroll-my-2", "interactivity"),
        ("scroll-my-2.5", "interactivity"),
        ("scroll-my-20", "interactivity"),
        ("scroll-my-24", "interactivity"),
        ("scroll-my-28", "interactivity"),
        ("scroll-my-3", "interactivity"),
        ("scroll-my-3.5", "interactivity"),
        ("scroll-my-32", "interactivity"),
        ("scroll-my-36", "interactivity"),
        ("scroll-my-4", "interactivity"),
        ("scroll-my-40", "interactivity"),
        ("scroll-my-44", "interactivity"),
        ("scroll-my-48", "interactivity"),
        ("scroll-my-5", "interactivity"),
        ("scroll-my-52", "interactivity"),
        ("scroll-my-56", "interactivity"),
        ("scroll-my-6", "interactivity"),
        ("scroll-my-60", "interactivity"),
        ("scroll-my-64", "interactivity"),
        ("scroll-my-7", "interactivity"),
        ("scroll-my-72", "interactivity"),
        ("scroll-my-8", "interactivity"),
        ("scroll-my-80", "interactivity"),
        ("scroll-my-9", "interactivity"),
        ("scroll-my-96", "interactivity"),
        ("scroll-my-px", "interactivity"),
        ("scroll-p-0", "interactivity"),
        ("scroll-p-0.5", "interactivity"),
        ("scroll-p-1", "interactivity"),
        ("scroll-p-1.5", "interactivity"),
        ("scroll-p-10", "interactivity"),
        ("scroll-p-11", "interactivity"),
        ("scroll-p-12", "interactivity"),
        ("scroll-p-14", "interactivity"),
        ("scroll-p-16", "interactivity"),
        ("scroll-p-2", "interactivity"),
        ("scroll-p-2.5", "interactivity"),
        ("scroll-p-20", "interactivity"),
        ("scroll-p-24", "interactivity"),
        ("scroll-p-28", "interactivity"),
        ("scroll-p-3", "interactivity"),
        ("scroll-p-3.5", "interactivity"),
        ("scroll-p-32", "interactivity"),
        ("scroll-p-36", "interactivity"),
        ("scroll-p-4", "interactivity"),
        ("scroll-p-40", "interactivity"),
        ("scroll-p-44", "interactivity"),
        ("scroll-p-48", "interactivity"),
        ("scroll-p-5", "interactivity"),
        ("scroll-p-52", "interactivity"),
        ("scroll-p-56", "interactivity"),
        ("scroll-p-6", "interactivity"),
        ("scroll-p-60", "interactivity"),
        ("scroll-p-64", "interactivity"),
        ("scroll-p-7", "interactivity"),
        ("scroll-p-72", "interactivity"),
        ("scroll-p-8", "interactivity"),
        ("scroll-p-80", "interactivity"),
        ("scroll-p-9", "interactivity"),
        ("scroll-p-96", "interactivity"),
        ("scroll-p-px", "interactivity"),
        ("scroll-pb-0", "interactivity"),
        ("scroll-pb-0.5", "interactivity"),
        ("scroll-pb-1", "interactivity"),
        ("scroll-pb-1.5", "interactivity"),
        ("scroll-pb-10", "interactivity"),
        ("scroll-pb-11", "interactivity"),
        ("scroll-pb-12", "interactivity"),
        ("scroll-pb-14", "interactivity"),
        ("scroll-pb-16", "interactivity"),
        ("scroll-pb-2", "interactivity"),
        ("scroll-pb-2.5", "interactivity"),
        ("scroll-pb-20", "interactivity"),
        ("scroll-pb-24", "interactivity"),
        ("scroll-pb-28", "interactivity"),
        ("scroll-pb-3", "interactivity"),
        ("scroll-pb-3.5", "interactivity"),
        ("scroll-pb-32", "interactivity"),
        ("scroll-pb-36", "interactivity"),
        ("scroll-pb-4", "interactivity"),
        ("scroll-pb-40", "interactivity"),
        ("scroll-pb-44", "interactivity"),
        ("scroll-pb-48", "interactivity"),
        ("scroll-pb-5", "interactivity"),
        ("scroll-pb-52", "interactivity"),
        ("scroll-pb-56", "interactivity"),
        ("scroll-pb-6", "interactivity"),
        ("scroll-pb-60", "interactivity"),
        ("scroll-pb-64", "interactivity"),
        ("scroll-pb-7", "interactivity"),
        ("scroll-pb-72", "interactivity"),
        ("scroll-pb-8", "interactivity"),
        ("scroll-pb-80", "interactivity"),
        ("scroll-pb-9", "interactivity"),
        ("scroll-pb-96", "interactivity"),
        ("scroll-pb-px", "interactivity"),
        ("scroll-pl-0", "interactivity"),
        ("scroll-pl-0.5", "interactivity"),
        ("scroll-pl-1", "interactivity"),
        ("scroll-pl-1.5", "interactivity"),
        ("scroll-pl-10", "interactivity"),
        ("scroll-pl-11", "interactivity"),
        ("scroll-pl-12", "interactivity"),
        ("scroll-pl-14", "interactivity"),
        ("scroll-pl-16", "interactivity"),
        ("scroll-pl-2", "interactivity"),
        ("scroll-pl-2.5", "interactivity"),
        ("scroll-pl-20", "interactivity"),
        ("scroll-pl-24", "interactivity"),
        ("scroll-pl-28", "interactivity"),
        ("scroll-pl-3", "interactivity"),
        ("scroll-pl-3.5", "interactivity"),
        ("scroll-pl-32", "interactivity"),
        ("scroll-pl-36", "interactivity"),
        ("scroll-pl-4", "interactivity"),
        ("scroll-pl-40", "interactivity"),
        ("scroll-pl-44", "interactivity"),
        ("scroll-pl-48", "interactivity"),
        ("scroll-pl-5", "interactivity"),
        ("scroll-pl-52", "interactivity"),
        ("scroll-pl-56", "interactivity"),
        ("scroll-pl-6", "interactivity"),
        ("scroll-pl-60", "interactivity"),
        ("scroll-pl-64", "interactivity"),
        ("scroll-pl-7", "interactivity"),
        ("scroll-pl-72", "interactivity"),
        ("scroll-pl-8", "interactivity"),
        ("scroll-pl-80", "interactivity"),
        ("scroll-pl-9", "interactivity"),
        ("scroll-pl-96", "interactivity"),
        ("scroll-pl-px", "interactivity"),
        ("scroll-pr-0", "interactivity"),
        ("scroll-pr-0.5", "interactivity"),
        ("scroll-pr-1", "interactivity"),
        ("scroll-pr-1.5", "interactivity"),
        ("scroll-pr-10", "interactivity"),
        ("scroll-pr-11", "interactivity"),
        ("scroll-pr-12", "interactivity"),
        ("scroll-pr-14", "interactivity"),
        ("scroll-pr-16", "interactivity"),
        ("scroll-pr-2", "interactivity"),
        ("scroll-pr-2.5", "interactivity"),
        ("scroll-pr-20", "interactivity"),
        ("scroll-pr-24", "interactivity"),
        ("scroll-pr-28", "interactivity"),
        ("scroll-pr-3", "interactivity"),
        ("scroll-pr-3.5", "interactivity"),
        ("scroll-pr-32", "interactivity"),
        ("scroll-pr-36", "interactivity"),
        ("scroll-pr-4", "interactivity"),
        ("scroll-pr-40", "interactivity"),
        ("scroll-pr-44", "interactivity"),
        ("scroll-pr-48", "interactivity"),
        ("scroll-pr-5", "interactivity"),
        ("scroll-pr-52", "interactivity"),
        ("scroll-pr-56", "interactivity"),
        ("scroll-pr-6", "interactivity"),
        ("scroll-pr-60", "interactivity"),
        ("scroll-pr-64", "interactivity"),
        ("scroll-pr-7", "interactivity"),
        ("scroll-pr-72", "interactivity"),
        ("scroll-pr-8", "interactivity"),
        ("scroll-pr-80", "interactivity"),
        ("scroll-pr-9", "interactivity"),
        ("scroll-pr-96", "interactivity"),
        ("scroll-pr-px", "interactivity"),
        ("scroll-pt-0", "interactivity"),
        ("scroll-pt-0.5", "interactivity"),
        ("scroll-pt-1", "interactivity"),
        ("scroll-pt-1.5", "interactivity"),
        ("scroll-pt-10", "interactivity"),
        ("scroll-pt-11", "interactivity"),
        ("scroll-pt-12", "interactivity"),
        ("scroll-pt-14", "interactivity"),
        ("scroll-pt-16", "interactivity"),
        ("scroll-pt-2", "interactivity"),
        ("scroll-pt-2.5", "interactivity"),
        ("scroll-pt-20", "interactivity"),
        ("scroll-pt-24", "interactivity"),
        ("scroll-pt-28", "interactivity"),
        ("scroll-pt-3", "interactivity"),
        ("scroll-pt-3.5", "interactivity"),
        ("scroll-pt-32", "interactivity"),
        ("scroll-pt-36", "interactivity"),
        ("scroll-pt-4", "interactivity"),
        ("scroll-pt-40", "interactivity"),
        ("scroll-pt-44", "interactivity"),
        ("scroll-pt-48", "interactivity"),
        ("scroll-pt-5", "interactivity"),
        ("scroll-pt-52", "interactivity"),
        ("scroll-pt-56", "interactivity"),
        ("scroll-pt-6", "interactivity"),
        ("scroll-pt-60", "interactivity"),
        ("scroll-pt-64", "interactivity"),
        ("scroll-pt-7", "interactivity"),
        ("scroll-pt-72", "interactivity"),
        ("scroll-pt-8", "interactivity"),
        ("scroll-pt-80", "interactivity"),
        ("scroll-pt-9", "interactivity"),
        ("scroll-pt-96", "interactivity"),
        ("scroll-pt-px", "interactivity"),
        ("scroll-px-0", "interactivity"),
        ("scroll-px-0.5", "interactivity"),
        ("scroll-px-1", "interactivity"),
        ("scroll-px-1.5", "interactivity"),
        ("scroll-px-10", "interactivity"),
        ("scroll-px-11", "interactivity"),
        ("scroll-px-12", "interactivity"),
        ("scroll-px-14", "interactivity"),
        ("scroll-px-16", "interactivity"),
        ("scroll-px-2", "interactivity"),
        ("scroll-px-2.5", "interactivity"),
        ("scroll-px-20", "interactivity"),
        ("scroll-px-24", "interactivity"),
        ("scroll-px-28", "interactivity"),
        ("scroll-px-3", "interactivity"),
        ("scroll-px-3.5", "interactivity"),
        ("scroll-px-32", "interactivity"),
        ("scroll-px-36", "interactivity"),
        ("scroll-px-4", "interactivity"),
        ("scroll-px-40", "interactivity"),
        ("scroll-px-44", "interactivity"),
        ("scroll-px-48", "interactivity"),
        ("scroll-px-5", "interactivity"),
        ("scroll-px-52", "interactivity"),
        ("scroll-px-56", "interactivity"),
        ("scroll-px-6", "interactivity"),
        ("scroll-px-60", "interactivity"),
        ("scroll-px-64", "interactivity"),
        ("scroll-px-7", "interactivity"),
        ("scroll-px-72", "interactivity"),
        ("scroll-px-8", "interactivity"),
        ("scroll-px-80", "interactivity"),
        ("scroll-px-9", "interactivity"),
        ("scroll-px-96", "interactivity"),
        ("scroll-px-px", "interactivity"),
        ("scroll-py-0", "interactivity"),
        ("scroll-py-0.5", "interactivity"),
        ("scroll-py-1", "interactivity"),
        ("scroll-py-1.5", "interactivity"),
        ("scroll-py-10", "interactivity"),
        ("scroll-py-11", "interactivity"),
        ("scroll-py-12", "interactivity"),
        ("scroll-py-14", "interactivity"),
        ("scroll-py-16", "interactivity"),
        ("scroll-py-2", "interactivity"),
        ("scroll-py-2.5", "interactivity"),
        ("scroll-py-20", "interactivity"),
        ("scroll-py-24", "interactivity"),
        ("scroll-py-28", "interactivity"),
        ("scroll-py-3", "interactivity"),
        ("scroll-py-3.5", "interactivity"),
        ("scroll-py-32", "interactivity"),
        ("scroll-py-36", "interactivity"),
        ("scroll-py-4", "interactivity"),
        ("scroll-py-40", "interactivity"),
        ("scroll-py-44", "interactivity"),
        ("scroll-py-48", "interactivity"),
        ("scroll-py-5", "interactivity"),
        ("scroll-py-52", "interactivity"),
        ("scroll-py-56", "interactivity"),
        ("scroll-py-6", "interactivity"),
        ("scroll-py-60", "interactivity"),
        ("scroll-py-64", "interactivity"),
        ("scroll-py-7", "interactivity"),
        ("scroll-py-72", "interactivity"),
        ("scroll-py-8", "interactivity"),
        ("scroll-py-80", "interactivity"),
        ("scroll-py-9", "interactivity"),
        ("scroll-py-96", "interactivity"),
        ("scroll-py-px", "interactivity"),
        ("scroll-smooth", "interactivity"),
        ("select-all", "interactivity"),
        ("select-auto", "interactivity"),
        ("select-none", "interactivity"),
        ("select-text", "interactivity"),
        ("snap-align-none", "interactivity"),
        ("snap-always", "interactivity"),
        ("snap-both", "interactivity"),
        ("snap-center", "interactivity"),
        ("snap-end", "interactivity"),
        ("snap-mandatory", "interactivity"),
        ("snap-none", "interactivity"),
        ("snap-normal", "interactivity"),
        ("snap-proximity", "interactivity"),
        ("snap-start", "interactivity"),
        ("snap-x", "interactivity"),
        ("snap-y", "interactivity"),
        ("touch-auto", "interactivity"),
        ("touch-manipulation", "interactivity"),
        ("touch-none", "interactivity"),
        ("touch-pan-down", "interactivity"),
        ("touch-pan-left", "interactivity"),
        ("touch-pan-right", "interactivity"),
        ("touch-pan-up", "interactivity"),
        ("touch-pan-x", "interactivity"),
        ("touch-pan-y", "interactivity"),
        ("touch-pinch-zoom", "interactivity"),
        ("will-change-auto", "interactivity"),
        ("will-change-contents", "interactivity"),
        ("will-change-scroll", "interactivity"),
        ("will-change-transform", "interactivity"),
        ("2xl ", "layout"),
        ("absolute", "layout"),
        ("aspect-auto", "layout"),
        ("aspect-square", "layout"),
        ("aspect-video", "layout"),
        ("block", "layout"),
        ("bottom-0", "layout"),
        ("bottom-0.5", "layout"),
        ("bottom-1", "layout"),
        ("bottom-1.5", "layout"),
        ("bottom-1/2", "layout"),
        ("bottom-1/3", "layout"),
        ("bottom-1/4", "layout"),
        ("bottom-10", "layout"),
        ("bottom-11", "layout"),
        ("bottom-12", "layout"),
        ("bottom-14", "layout"),
        ("bottom-16", "layout"),
        ("bottom-2", "layout"),
        ("bottom-2.5", "layout"),
        ("bottom-2/3", "layout"),
        ("bottom-2/4", "layout"),
        ("bottom-20", "layout"),
        ("bottom-24", "layout"),
        ("bottom-28", "layout"),
        ("bottom-3", "layout"),
        ("bottom-3.5", "layout"),
        ("bottom-3/4", "layout"),
        ("bottom-32", "layout"),
        ("bottom-36", "layout"),
        ("bottom-4", "layout"),
        ("bottom-40", "layout"),
        ("bottom-44", "layout"),
        ("bottom-48", "layout"),
        ("bottom-5", "layout"),
        ("bottom-52", "layout"),
        ("bottom-56", "layout"),
        ("bottom-6", "layout"),
        ("bottom-60", "layout"),
        ("bottom-64", "layout"),
        ("bottom-7", "layout"),
        ("bottom-72", "layout"),
        ("bottom-8", "layout"),
        ("bottom-80", "layout"),
        ("bottom-9", "layout"),
        ("bottom-96", "layout"),
        ("bottom-auto", "layout"),
        ("bottom-full", "layout"),
        ("bottom-px", "layout"),
        ("box-border", "layout"),
        ("box-content", "layout"),
        ("box-decoration-clone", "layout"),
        ("box-decoration-slice", "layout"),
        ("break-after-all", "layout"),
        ("break-after-auto", "layout"),
        ("break-after-avoid", "layout"),
        ("break-after-avoid-page", "layout"),
        ("break-after-column", "layout"),
        ("break-after-left", "layout"),
        ("break-after-page", "layout"),
        ("break-after-right", "layout"),
        ("break-before-all", "layout"),
        ("break-before-auto", "layout"),
        ("break-before-avoid", "layout"),
        ("break-before-avoid-page", "layout"),
        ("break-before-column", "layout"),
        ("break-before-left", "layout"),
        ("break-before-page", "layout"),
        ("break-before-right", "layout"),
        ("break-inside-auto", "layout"),
        ("break-inside-avoid", "layout"),
        ("break-inside-avoid-column", "layout"),
        ("break-inside-avoid-page", "layout"),
        ("clear-both", "layout"),
        ("clear-left", "layout"),
        ("clear-none", "layout"),
        ("clear-right", "layout"),
        ("columns-1", "layout"),
        ("columns-10", "layout"),
        ("columns-11", "layout"),
        ("columns-12", "layout"),
        ("columns-2", "layout"),
        ("columns-2xl", "layout"),
        ("columns-2xs", "layout"),
        ("columns-3", "layout"),
        ("columns-3xl", "layout"),
        ("columns-3xs", "layout"),
        ("columns-4", "layout"),
        ("columns-4xl", "layout"),
        ("columns-5", "layout"),
        ("columns-5xl", "layout"),
        ("columns-6", "layout"),
        ("columns-6xl", "layout"),
        ("columns-7", "layout"),
        ("columns-7xl", "layout"),
        ("columns-8", "layout"),
        ("columns-9", "layout"),
        ("columns-auto", "layout"),
        ("columns-lg", "layout"),
        ("columns-md", "layout"),
        ("columns-sm", "layout"),
        ("columns-xl", "layout"),
        ("columns-xs", "layout"),
        ("container", "layout"),
        ("contents", "layout"),
        ("fixed", "layout"),
        ("flex", "layout"),
        ("float-left", "layout"),
        ("float-none", "layout"),
        ("float-right", "layout"),
        ("flow-root", "layout"),
        ("grid", "layout"),
        ("hidden", "layout"),
        ("inline", "layout"),
        ("inline-block", "layout"),
        ("inline-flex", "layout"),
        ("inline-grid", "layout"),
        ("inline-table", "layout"),
        ("inset-0", "layout"),
        ("inset-0.5", "layout"),
        ("inset-1", "layout"),
        ("inset-1.5", "layout"),
        ("inset-1/2", "layout"),
        ("inset-1/3", "layout"),
        ("inset-1/4", "layout"),
        ("inset-10", "layout"),
        ("inset-11", "layout"),
        ("inset-12", "layout"),
        ("inset-14", "layout"),
        ("inset-16", "layout"),
        ("inset-2", "layout"),
        ("inset-2.5", "layout"),
        ("inset-2/3", "layout"),
        ("inset-2/4", "layout"),
        ("inset-20", "layout"),
        ("inset-24", "layout"),
        ("inset-28", "layout"),
        ("inset-3", "layout"),
        ("inset-3.5", "layout"),
        ("inset-3/4", "layout"),
        ("inset-32", "layout"),
        ("inset-36", "layout"),
        ("inset-4", "layout"),
        ("inset-40", "layout"),
        ("inset-44", "layout"),
        ("inset-48", "layout"),
        ("inset-5", "layout"),
        ("inset-52", "layout"),
        ("inset-56", "layout"),
        ("inset-6", "layout"),
        ("inset-60", "layout"),
        ("inset-64", "layout"),
        ("inset-7", "layout"),
        ("inset-72", "layout"),
        ("inset-8", "layout"),
        ("inset-80", "layout"),
        ("inset-9", "layout"),
        ("inset-96", "layout"),
        ("inset-auto", "layout"),
        ("inset-full", "layout"),
        ("inset-px", "layout"),
        ("inset-x-0", "layout"),
        ("inset-x-0.5", "layout"),
        ("inset-x-1", "layout"),
        ("inset-x-1.5", "layout"),
        ("inset-x-1/2", "layout"),
        ("inset-x-1/3", "layout"),
        ("inset-x-1/4", "layout"),
        ("inset-x-10", "layout"),
        ("inset-x-11", "layout"),
        ("inset-x-12", "layout"),
        ("inset-x-14", "layout"),
        ("inset-x-16", "layout"),
        ("inset-x-2", "layout"),
        ("inset-x-2.5", "layout"),
        ("inset-x-2/3", "layout"),
        ("inset-x-2/4", "layout"),
        ("inset-x-20", "layout"),
        ("inset-x-24", "layout"),
        ("inset-x-28", "layout"),
        ("inset-x-3", "layout"),
        ("inset-x-3.5", "layout"),
        ("inset-x-3/4", "layout"),
        ("inset-x-32", "layout"),
        ("inset-x-36", "layout"),
        ("inset-x-4", "layout"),
        ("inset-x-40", "layout"),
        ("inset-x-44", "layout"),
        ("inset-x-48", "layout"),
        ("inset-x-5", "layout"),
        ("inset-x-52", "layout"),
        ("inset-x-56", "layout"),
        ("inset-x-6", "layout"),
        ("inset-x-60", "layout"),
        ("inset-x-64", "layout"),
        ("inset-x-7", "layout"),
        ("inset-x-72", "layout"),
        ("inset-x-8", "layout"),
        ("inset-x-80", "layout"),
        ("inset-x-9", "layout"),
        ("inset-x-96", "layout"),
        ("inset-x-auto", "layout"),
        ("inset-x-full", "layout"),
        ("inset-x-px", "layout"),
        ("inset-y-0", "layout"),
        ("inset-y-0.5", "layout"),
        ("inset-y-1", "layout"),
        ("inset-y-1.5", "layout"),
        ("inset-y-1/2", "layout"),
        ("inset-y-1/3", "layout"),
        ("inset-y-1/4", "layout"),
        ("inset-y-10", "layout"),
        ("inset-y-11", "layout"),
        ("inset-y-12", "layout"),
        ("inset-y-14", "layout"),
        ("inset-y-16", "layout"),
        ("inset-y-2", "layout"),
        ("inset-y-2.5", "layout"),
        ("inset-y-2/3", "layout"),
        ("inset-y-2/4", "layout"),
        ("inset-y-20", "layout"),
        ("inset-y-24", "layout"),
        ("inset-y-28", "layout"),
        ("inset-y-3", "layout"),
        ("inset-y-3.5", "layout"),
        ("inset-y-3/4", "layout"),
        ("inset-y-32", "layout"),
        ("inset-y-36", "layout"),
        ("inset-y-4", "layout"),
        ("inset-y-40", "layout"),
        ("inset-y-44", "layout"),
        ("inset-y-48", "layout"),
        ("inset-y-5", "layout"),
        ("inset-y-52", "layout"),
        ("inset-y-56", "layout"),
        ("inset-y-6", "layout"),
        ("inset-y-60", "layout"),
        ("inset-y-64", "layout"),
        ("inset-y-7", "layout"),
        ("inset-y-72", "layout"),
        ("inset-y-8", "layout"),
        ("inset-y-80", "layout"),
        ("inset-y-9", "layout"),
        ("inset-y-96", "layout"),
        ("inset-y-auto", "layout"),
        ("inset-y-full", "layout"),
        ("inset-y-px", "layout"),
        ("invisible", "layout"),
        ("isolate", "layout"),
        ("isolation-auto", "layout"),
        ("left-0", "layout"),
        ("left-0.5", "layout"),
        ("left-1", "layout"),
        ("left-1.5", "layout"),
        ("left-1/2", "layout"),
        ("left-1/3", "layout"),
        ("left-1/4", "layout"),
        ("left-10", "layout"),
        ("left-11", "layout"),
        ("left-12", "layout"),
        ("left-14", "layout"),
        ("left-16", "layout"),
        ("left-2", "layout"),
        ("left-2.5", "layout"),
        ("left-2/3", "layout"),
        ("left-2/4", "layout"),
        ("left-20", "layout"),
        ("left-24", "layout"),
        ("left-28", "layout"),
        ("left-3", "layout"),
        ("left-3.5", "layout"),
        ("left-3/4", "layout"),
        ("left-32", "layout"),
        ("left-36", "layout"),
        ("left-4", "layout"),
        ("left-40", "layout"),
        ("left-44", "layout"),
        ("left-48", "layout"),
        ("left-5", "layout"),
        ("left-52", "layout"),
        ("left-56", "layout"),
        ("left-6", "layout"),
        ("left-60", "layout"),
        ("left-64", "layout"),
        ("left-7", "layout"),
        ("left-72", "layout"),
        ("left-8", "layout"),
        ("left-80", "layout"),
        ("left-9", "layout"),
        ("left-96", "layout"),
        ("left-auto", "layout"),
        ("left-full", "layout"),
        ("left-px", "layout"),
        ("lg ", "layout"),
        ("list-item", "layout"),
        ("md ", "layout"),
        ("object-bottom", "layout"),
        ("object-center", "layout"),
        ("object-contain", "layout"),
        ("object-cover", "layout"),
        ("object-fill", "layout"),
        ("object-left", "layout"),
        ("object-left-bottom", "layout"),
        ("object-left-top", "layout"),
        ("object-none", "layout"),
        ("object-right", "layout"),
        ("object-right-bottom", "layout"),
        ("object-right-top", "layout"),
        ("object-scale-down", "layout"),
        ("object-top", "layout"),
        ("overflow-auto", "layout"),
        ("overflow-clip", "layout"),
        ("overflow-hidden", "layout"),
        ("overflow-scroll", "layout"),
        ("overflow-visible", "layout"),
        ("overflow-x-auto", "layout"),
        ("overflow-x-clip", "layout"),
        ("overflow-x-hidden", "layout"),
        ("overflow-x-scroll", "layout"),
        ("overflow-x-visible", "layout"),
        ("overflow-y-auto", "layout"),
        ("overflow-y-clip", "layout"),
        ("overflow-y-hidden", "layout"),
        ("overflow-y-scroll", "layout"),
        ("overflow-y-visible", "layout"),
        ("overscroll-auto", "layout"),
        ("overscroll-contain", "layout"),
        ("overscroll-none", "layout"),
        ("overscroll-x-auto", "layout"),
        ("overscroll-x-contain", "layout"),
        ("overscroll-x-none", "layout"),
        ("overscroll-y-auto", "layout"),
        ("overscroll-y-contain", "layout"),
        ("overscroll-y-none", "layout"),
        ("relative", "layout"),
        ("right-0", "layout"),
        ("right-0.5", "layout"),
        ("right-1", "layout"),
        ("right-1.5", "layout"),
        ("right-1/2", "layout"),
        ("right-1/3", "layout"),
        ("right-1/4", "layout"),
        ("right-10", "layout"),
        ("right-11", "layout"),
        ("right-12", "layout"),
        ("right-14", "layout"),
        ("right-16", "layout"),
        ("right-2", "layout"),
        ("right-2.5", "layout"),
        ("right-2/3", "layout"),
        ("right-2/4", "layout"),
        ("right-20", "layout"),
        ("right-24", "layout"),
        ("right-28", "layout"),
        ("right-3", "layout"),
        ("right-3.5", "layout"),
        ("right-3/4", "layout"),
        ("right-32", "layout"),
        ("right-36", "layout"),
        ("right-4", "layout"),
        ("right-40", "layout"),
        ("right-44", "layout"),
        ("right-48", "layout"),
        ("right-5", "layout"),
        ("right-52", "layout"),
        ("right-56", "layout"),
        ("right-6", "layout"),
        ("right-60", "layout"),
        ("right-64", "layout"),
        ("right-7", "layout"),
        ("right-72", "layout"),
        ("right-8", "layout"),
        ("right-80", "layout"),
        ("right-9", "layout"),
        ("right-96", "layout"),
        ("right-auto", "layout"),
        ("right-full", "layout"),
        ("right-px", "layout"),
        ("sm ", "layout"),
        ("static", "layout"),
        ("sticky", "layout"),
        ("table", "layout"),
        ("table-caption", "layout"),
        ("table-cell", "layout"),
        ("table-column", "layout"),
        ("table-column-group", "layout"),
        ("table-footer-group", "layout"),
        ("table-header-group", "layout"),
        ("table-row", "layout"),
        ("table-row-group", "layout"),
        ("top-0", "layout"),
        ("top-0.5", "layout"),
        ("top-1", "layout"),
        ("top-1.5", "layout"),
        ("top-1/2", "layout"),
        ("top-1/3", "layout"),
        ("top-1/4", "layout"),
        ("top-10", "layout"),
        ("top-11", "layout"),
        ("top-12", "layout"),
        ("top-14", "layout"),
        ("top-16", "layout"),
        ("top-2", "layout"),
        ("top-2.5", "layout"),
        ("top-2/3", "layout"),
        ("top-2/4", "layout"),
        ("top-20", "layout"),
        ("top-24", "layout"),
        ("top-28", "layout"),
        ("top-3", "layout"),
        ("top-3.5", "layout"),
        ("top-3/4", "layout"),
        ("top-32", "layout"),
        ("top-36", "layout"),
        ("top-4", "layout"),
        ("top-40", "layout"),
        ("top-44", "layout"),
        ("top-48", "layout"),
        ("top-5", "layout"),
        ("top-52", "layout"),
        ("top-56", "layout"),
        ("top-6", "layout"),
        ("top-60", "layout"),
        ("top-64", "layout"),
        ("top-7", "layout"),
        ("top-72", "layout"),
        ("top-8", "layout"),
        ("top-80", "layout"),
        ("top-9", "layout"),
        ("top-96", "layout"),
        ("top-auto", "layout"),
        ("top-full", "layout"),
        ("top-px", "layout"),
        ("visible", "layout"),
        ("xl ", "layout"),
        ("z-0", "layout"),
        ("z-10", "layout"),
        ("z-20", "layout"),
        ("z-30", "layout"),
        ("z-40", "layout"),
        ("z-50", "layout"),
        ("z-auto", "layout"),
        ("h-0", "sizing"),
        ("h-0.5", "sizing"),
        ("h-1", "sizing"),
        ("h-1.5", "sizing"),
        ("h-1/2", "sizing"),
        ("h-1/3", "sizing"),
        ("h-1/4", "sizing"),
        ("h-1/5", "sizing"),
        ("h-1/6", "sizing"),
        ("h-10", "sizing"),
        ("h-11", "sizing"),
        ("h-12", "sizing"),
        ("h-14", "sizing"),
        ("h-16", "sizing"),
        ("h-2", "sizing"),
        ("h-2.5", "sizing"),
        ("h-2/3", "sizing"),
        ("h-2/4", "sizing"),
        ("h-2/5", "sizing"),
        ("h-2/6", "sizing"),
        ("h-20", "sizing"),
        ("h-24", "sizing"),
        ("h-28", "sizing"),
        ("h-3", "sizing"),
        ("h-3.5", "sizing"),
        ("h-3/4", "sizing"),
        ("h-3/5", "sizing"),
        ("h-3/6", "sizing"),
        ("h-32", "sizing"),
        ("h-36", "sizing"),
        ("h-4", "sizing"),
        ("h-4/5", "sizing"),
        ("h-4/6", "sizing"),
        ("h-40", "sizing"),
        ("h-44", "sizing"),
        ("h-48", "sizing"),
        ("h-5", "sizing"),
        ("h-5/6", "sizing"),
        ("h-52", "sizing"),
        ("h-56", "sizing"),
        ("h-6", "sizing"),
        ("h-60", "sizing"),
        ("h-64", "sizing"),
        ("h-7", "sizing"),
        ("h-72", "sizing"),
        ("h-8", "sizing"),
        ("h-80", "sizing"),
        ("h-9", "sizing"),
        ("h-96", "sizing"),
        ("h-auto", "sizing"),
        ("h-fit", "sizing"),
        ("h-full", "sizing"),
        ("h-max", "sizing"),
        ("h-min", "sizing"),
        ("h-px", "sizing"),
        ("h-screen", "sizing"),
        ("max-h-0", "sizing"),
        ("max-h-0.5", "sizing"),
        ("max-h-1", "sizing"),
        ("max-h-1.5", "sizing"),
        ("max-h-10", "sizing"),
        ("max-h-11", "sizing"),
        ("max-h-12", "sizing"),
        ("max-h-14", "sizing"),
        ("max-h-16", "sizing"),
        ("max-h-2", "sizing"),
        ("max-h-2.5", "sizing"),
        ("max-h-20", "sizing"),
        ("max-h-24", "sizing"),
        ("max-h-28", "sizing"),
        ("max-h-3", "sizing"),
        ("max-h-3.5", "sizing"),
        ("max-h-32", "sizing"),
        ("max-h-36", "sizing"),
        ("max-h-4", "sizing"),
        ("max-h-40", "sizing"),
        ("max-h-44", "sizing"),
        ("max-h-48", "sizing"),
        ("max-h-5", "sizing"),
        ("max-h-52", "sizing"),
        ("max-h-56", "sizing"),
        ("max-h-6", "sizing"),
        ("max-h-60", "sizing"),
        ("max-h-64", "sizing"),
        ("max-h-7", "sizing"),
        ("max-h-72", "sizing"),
        ("max-h-8", "sizing"),
        ("max-h-80", "sizing"),
        ("max-h-9", "sizing"),
        ("max-h-96", "sizing"),
        ("max-h-fit", "sizing"),
        ("max-h-full", "sizing"),
        ("max-h-max", "sizing"),
        ("max-h-min", "sizing"),
        ("max-h-px", "sizing"),
        ("max-h-screen", "sizing"),
        ("max-w-0", "sizing"),
        ("max-w-2xl", "sizing"),
        ("max-w-3xl", "sizing"),
        ("max-w-4xl", "sizing"),
        ("max-w-5xl", "sizing"),
        ("max-w-6xl", "sizing"),
        ("max-w-7xl", "sizing"),
        ("max-w-fit", "sizing"),
        ("max-w-full", "sizing"),
        ("max-w-lg", "sizing"),
        ("max-w-max", "sizing"),
        ("max-w-md", "sizing"),
        ("max-w-min", "sizing"),
        ("max-w-none", "sizing"),
        ("max-w-prose", "sizing"),
        ("max-w-screen-2xl", "sizing"),
        ("max-w-screen-lg", "sizing"),
        ("max-w-screen-md", "sizing"),
        ("max-w-screen-sm", "sizing"),
        ("max-w-screen-xl", "sizing"),
        ("max-w-sm", "sizing"),
        ("max-w-xl", "sizing"),
        ("max-w-xs", "sizing"),
        ("min-h-0", "sizing"),
        ("min-h-fit", "sizing"),
        ("min-h-full", "sizing"),
        ("min-h-max", "sizing"),
        ("min-h-min", "sizing"),
        ("min-h-screen", "sizing"),
        ("min-w-0", "sizing"),
        ("min-w-fit", "sizing"),
        ("min-w-full", "sizing"),
        ("min-w-max", "sizing"),
        ("min-w-min", "sizing"),
        ("w-0", "sizing"),
        ("w-0.5", "sizing"),
        ("w-1", "sizing"),
        ("w-1.5", "sizing"),
        ("w-1/12", "sizing"),
        ("w-1/2", "sizing"),
        ("w-1/3", "sizing"),
        ("w-1/4", "sizing"),
        ("w-1/5", "sizing"),
        ("w-1/6", "sizing"),
        ("w-10", "sizing"),
        ("w-10/12", "sizing"),
        ("w-11", "sizing"),
        ("w-11/12", "sizing"),
        ("w-12", "sizing"),
        ("w-14", "sizing"),
        ("w-16", "sizing"),
        ("w-2", "sizing"),
        ("w-2.5", "sizing"),
        ("w-2/12", "sizing"),
        ("w-2/3", "sizing"),
        ("w-2/4", "sizing"),
        ("w-2/5", "sizing"),
        ("w-2/6", "sizing"),
        ("w-20", "sizing"),
        ("w-24", "sizing"),
        ("w-28", "sizing"),
        ("w-3", "sizing"),
        ("w-3.5", "sizing"),
        ("w-3/12", "sizing"),
        ("w-3/4", "sizing"),
        ("w-3/5", "sizing"),
        ("w-3/6", "sizing"),
        ("w-32", "sizing"),
        ("w-36", "sizing"),
        ("w-4", "sizing"),
        ("w-4/12", "sizing"),
        ("w-4/5", "sizing"),
        ("w-4/6", "sizing"),
        ("w-40", "sizing"),
        ("w-44", "sizing"),
        ("w-48", "sizing"),
        ("w-5", "sizing"),
        ("w-5/12", "sizing"),
        ("w-5/6", "sizing"),
        ("w-52", "sizing"),
        ("w-56", "sizing"),
        ("w-6", "sizing"),
        ("w-6/12", "sizing"),
        ("w-60", "sizing"),
        ("w-64", "sizing"),
        ("w-7", "sizing"),
        ("w-7/12", "sizing"),
        ("w-72", "sizing"),
        ("w-8", "sizing"),
        ("w-8/12", "sizing"),
        ("w-80", "sizing"),
        ("w-9", "sizing"),
        ("w-9/12", "sizing"),
        ("w-96", "sizing"),
        ("w-auto", "sizing"),
        ("w-fit", "sizing"),
        ("w-full", "sizing"),
        ("w-max", "sizing"),
        ("w-min", "sizing"),
        ("w-px", "sizing"),
        ("w-screen", "sizing"),
        ("m-0", "spacing"),
        ("m-0.5", "spacing"),
        ("m-1", "spacing"),
        ("m-1.5", "spacing"),
        ("m-10", "spacing"),
        ("m-11", "spacing"),
        ("m-12", "spacing"),
        ("m-14", "spacing"),
        ("m-16", "spacing"),
        ("m-2", "spacing"),
        ("m-2.5", "spacing"),
        ("m-20", "spacing"),
        ("m-24", "spacing"),
        ("m-28", "spacing"),
        ("m-3", "spacing"),
        ("m-3.5", "spacing"),
        ("m-32", "spacing"),
        ("m-36", "spacing"),
        ("m-4", "spacing"),
        ("m-40", "spacing"),
        ("m-44", "spacing"),
        ("m-48", "spacing"),
        ("m-5", "spacing"),
        ("m-52", "spacing"),
        ("m-56", "spacing"),
        ("m-6", "spacing"),
        ("m-60", "spacing"),
        ("m-64", "spacing"),
        ("m-7", "spacing"),
        ("m-72", "spacing"),
        ("m-8", "spacing"),
        ("m-80", "spacing"),
        ("m-9", "spacing"),
        ("m-96", "spacing"),
        ("m-auto", "spacing"),
        ("m-px", "spacing"),
        ("mb-0", "spacing"),
        ("mb-0.5", "spacing"),
        ("mb-1", "spacing"),
        ("mb-1.5", "spacing"),
        ("mb-10", "spacing"),
        ("mb-11", "spacing"),
        ("mb-12", "spacing"),
        ("mb-14", "spacing"),
        ("mb-16", "spacing"),
        ("mb-2", "spacing"),
        ("mb-2.5", "spacing"),
        ("mb-20", "spacing"),
        ("mb-24", "spacing"),
        ("mb-28", "spacing"),
        ("mb-3", "spacing"),
        ("mb-3.5", "spacing"),
        ("mb-32", "spacing"),
        ("mb-36", "spacing"),
        ("mb-4", "spacing"),
        ("mb-40", "spacing"),
        ("mb-44", "spacing"),
        ("mb-48", "spacing"),
        ("mb-5", "spacing"),
        ("mb-52", "spacing"),
        ("mb-56", "spacing"),
        ("mb-6", "spacing"),
        ("mb-60", "spacing"),
        ("mb-64", "spacing"),
        ("mb-7", "spacing"),
        ("mb-72", "spacing"),
        ("mb-8", "spacing"),
        ("mb-80", "spacing"),
        ("mb-9", "spacing"),
        ("mb-96", "spacing"),
        ("mb-auto", "spacing"),
        ("mb-px", "spacing"),
        ("ml-0", "spacing"),
        ("ml-0.5", "spacing"),
        ("ml-1", "spacing"),
        ("ml-1.5", "spacing"),
        ("ml-10", "spacing"),
        ("ml-11", "spacing"),
        ("ml-12", "spacing"),
        ("ml-14", "spacing"),
        ("ml-16", "spacing"),
        ("ml-2", "spacing"),
        ("ml-2.5", "spacing"),
        ("ml-20", "spacing"),
        ("ml-24", "spacing"),
        ("ml-28", "spacing"),
        ("ml-3", "spacing"),
        ("ml-3.5", "spacing"),
        ("ml-32", "spacing"),
        ("ml-36", "spacing"),
        ("ml-4", "spacing"),
        ("ml-40", "spacing"),
        ("ml-44", "spacing"),
        ("ml-48", "spacing"),
        ("ml-5", "spacing"),
        ("ml-52", "spacing"),
        ("ml-56", "spacing"),
        ("ml-6", "spacing"),
        ("ml-60", "spacing"),
        ("ml-64", "spacing"),
        ("ml-7", "spacing"),
        ("ml-72", "spacing"),
        ("ml-8", "spacing"),
        ("ml-80", "spacing"),
        ("ml-9", "spacing"),
        ("ml-96", "spacing"),
        ("ml-auto", "spacing"),
        ("ml-px", "spacing"),
        ("mr-0", "spacing"),
        ("mr-0.5", "spacing"),
        ("mr-1", "spacing"),
        ("mr-1.5", "spacing"),
        ("mr-10", "spacing"),
        ("mr-11", "spacing"),
        ("mr-12", "spacing"),
        ("mr-14", "spacing"),
        ("mr-16", "spacing"),
        ("mr-2", "spacing"),
        ("mr-2.5", "spacing"),
        ("mr-20", "spacing"),
        ("mr-24", "spacing"),
        ("mr-28", "spacing"),
        ("mr-3", "spacing"),
        ("mr-3.5", "spacing"),
        ("mr-32", "spacing"),
        ("mr-36", "spacing"),
        ("mr-4", "spacing"),
        ("mr-40", "spacing"),
        ("mr-44", "spacing"),
        ("mr-48", "spacing"),
        ("mr-5", "spacing"),
        ("mr-52", "spacing"),
        ("mr-56", "spacing"),
        ("mr-6", "spacing"),
        ("mr-60", "spacing"),
        ("mr-64", "spacing"),
        ("mr-7", "spacing"),
        ("mr-72", "spacing"),
        ("mr-8", "spacing"),
        ("mr-80", "spacing"),
        ("mr-9", "spacing"),
        ("mr-96", "spacing"),
        ("mr-auto", "spacing"),
        ("mr-px", "spacing"),
        ("mt-0", "spacing"),
        ("mt-0.5", "spacing"),
        ("mt-1", "spacing"),
        ("mt-1.5", "spacing"),
        ("mt-10", "spacing"),
        ("mt-11", "spacing"),
        ("mt-12", "spacing"),
        ("mt-14", "spacing"),
        ("mt-16", "spacing"),
        ("mt-2", "spacing"),
        ("mt-2.5", "spacing"),
        ("mt-20", "spacing"),
        ("mt-24", "spacing"),
        ("mt-28", "spacing"),
        ("mt-3", "spacing"),
        ("mt-3.5", "spacing"),
        ("mt-32", "spacing"),
        ("mt-36", "spacing"),
        ("mt-4", "spacing"),
        ("mt-40", "spacing"),
        ("mt-44", "spacing"),
        ("mt-48", "spacing"),
        ("mt-5", "spacing"),
        ("mt-52", "spacing"),
        ("mt-56", "spacing"),
        ("mt-6", "spacing"),
        ("mt-60", "spacing"),
        ("mt-64", "spacing"),
        ("mt-7", "spacing"),
        ("mt-72", "spacing"),
        ("mt-8", "spacing"),
        ("mt-80", "spacing"),
        ("mt-9", "spacing"),
        ("mt-96", "spacing"),
        ("mt-auto", "spacing"),
        ("mt-px", "spacing"),
        ("mx-0", "spacing"),
        ("mx-0.5", "spacing"),
        ("mx-1", "spacing"),
        ("mx-1.5", "spacing"),
        ("mx-10", "spacing"),
        ("mx-11", "spacing"),
        ("mx-12", "spacing"),
        ("mx-14", "spacing"),
        ("mx-16", "spacing"),
        ("mx-2", "spacing"),
        ("mx-2.5", "spacing"),
        ("mx-20", "spacing"),
        ("mx-24", "spacing"),
        ("mx-28", "spacing"),
        ("mx-3", "spacing"),
        ("mx-3.5", "spacing"),
        ("mx-32", "spacing"),
        ("mx-36", "spacing"),
        ("mx-4", "spacing"),
        ("mx-40", "spacing"),
        ("mx-44", "spacing"),
        ("mx-48", "spacing"),
        ("mx-5", "spacing"),
        ("mx-52", "spacing"),
        ("mx-56", "spacing"),
        ("mx-6", "spacing"),
        ("mx-60", "spacing"),
        ("mx-64", "spacing"),
        ("mx-7", "spacing"),
        ("mx-72", "spacing"),
        ("mx-8", "spacing"),
        ("mx-80", "spacing"),
        ("mx-9", "spacing"),
        ("mx-96", "spacing"),
        ("mx-auto", "spacing"),
        ("mx-px", "spacing"),
        ("my-0", "spacing"),
        ("my-0.5", "spacing"),
        ("my-1", "spacing"),
        ("my-1.5", "spacing"),
        ("my-10", "spacing"),
        ("my-11", "spacing"),
        ("my-12", "spacing"),
        ("my-14", "spacing"),
        ("my-16", "spacing"),
        ("my-2", "spacing"),
        ("my-2.5", "spacing"),
        ("my-20", "spacing"),
        ("my-24", "spacing"),
        ("my-28", "spacing"),
        ("my-3", "spacing"),
        ("my-3.5", "spacing"),
        ("my-32", "spacing"),
        ("my-36", "spacing"),
        ("my-4", "spacing"),
        ("my-40", "spacing"),
        ("my-44", "spacing"),
        ("my-48", "spacing"),
        ("my-5", "spacing"),
        ("my-52", "spacing"),
        ("my-56", "spacing"),
        ("my-6", "spacing"),
        ("my-60", "spacing"),
        ("my-64", "spacing"),
        ("my-7", "spacing"),
        ("my-72", "spacing"),
        ("my-8", "spacing"),
        ("my-80", "spacing"),
        ("my-9", "spacing"),
        ("my-96", "spacing"),
        ("my-auto", "spacing"),
        ("my-px", "spacing"),
        ("p-0", "spacing"),
        ("p-0.5", "spacing"),
        ("p-1", "spacing"),
        ("p-1.5", "spacing"),
        ("p-10", "spacing"),
        ("p-11", "spacing"),
        ("p-12", "spacing"),
        ("p-14", "spacing"),
        ("p-16", "spacing"),
        ("p-2", "spacing"),
        ("p-2.5", "spacing"),
        ("p-20", "spacing"),
        ("p-24", "spacing"),
        ("p-28", "spacing"),
        ("p-3", "spacing"),
        ("p-3.5", "spacing"),
        ("p-32", "spacing"),
        ("p-36", "spacing"),
        ("p-4", "spacing"),
        ("p-40", "spacing"),
        ("p-44", "spacing"),
        ("p-48", "spacing"),
        ("p-5", "spacing"),
        ("p-52", "spacing"),
        ("p-56", "spacing"),
        ("p-6", "spacing"),
        ("p-60", "spacing"),
        ("p-64", "spacing"),
        ("p-7", "spacing"),
        ("p-72", "spacing"),
        ("p-8", "spacing"),
        ("p-80", "spacing"),
        ("p-9", "spacing"),
        ("p-96", "spacing"),
        ("p-px", "spacing"),
        ("pb-0", "spacing"),
        ("pb-0.5", "spacing"),
        ("pb-1", "spacing"),
        ("pb-1.5", "spacing"),
        ("pb-10", "spacing"),
        ("pb-11", "spacing"),
        ("pb-12", "spacing"),
        ("pb-14", "spacing"),
        ("pb-16", "spacing"),
        ("pb-2", "spacing"),
        ("pb-2.5", "spacing"),
        ("pb-20", "spacing"),
        ("pb-24", "spacing"),
        ("pb-28", "spacing"),
        ("pb-3", "spacing"),
        ("pb-3.5", "spacing"),
        ("pb-32", "spacing"),
        ("pb-36", "spacing"),
        ("pb-4", "spacing"),
        ("pb-40", "spacing"),
        ("pb-44", "spacing"),
        ("pb-48", "spacing"),
        ("pb-5", "spacing"),
        ("pb-52", "spacing"),
        ("pb-56", "spacing"),
        ("pb-6", "spacing"),
        ("pb-60", "spacing"),
        ("pb-64", "spacing"),
        ("pb-7", "spacing"),
        ("pb-72", "spacing"),
        ("pb-8", "spacing"),
        ("pb-80", "spacing"),
        ("pb-9", "spacing"),
        ("pb-96", "spacing"),
        ("pb-px", "spacing"),
        ("pl-0", "spacing"),
        ("pl-0.5", "spacing"),
        ("pl-1", "spacing"),
        ("pl-1.5", "spacing"),
        ("pl-10", "spacing"),
        ("pl-11", "spacing"),
        ("pl-12", "spacing"),
        ("pl-14", "spacing"),
        ("pl-16", "spacing"),
        ("pl-2", "spacing"),
        ("pl-2.5", "spacing"),
        ("pl-20", "spacing"),
        ("pl-24", "spacing"),
        ("pl-28", "spacing"),
        ("pl-3", "spacing"),
        ("pl-3.5", "spacing"),
        ("pl-32", "spacing"),
        ("pl-36", "spacing"),
        ("pl-4", "spacing"),
        ("pl-40", "spacing"),
        ("pl-44", "spacing"),
        ("pl-48", "spacing"),
        ("pl-5", "spacing"),
        ("pl-52", "spacing"),
        ("pl-56", "spacing"),
        ("pl-6", "spacing"),
        ("pl-60", "spacing"),
        ("pl-64", "spacing"),
        ("pl-7", "spacing"),
        ("pl-72", "spacing"),
        ("pl-8", "spacing"),
        ("pl-80", "spacing"),
        ("pl-9", "spacing"),
        ("pl-96", "spacing"),
        ("pl-px", "spacing"),
        ("pr-0", "spacing"),
        ("pr-0.5", "spacing"),
        ("pr-1", "spacing"),
        ("pr-1.5", "spacing"),
        ("pr-10", "spacing"),
        ("pr-11", "spacing"),
        ("pr-12", "spacing"),
        ("pr-14", "spacing"),
        ("pr-16", "spacing"),
        ("pr-2", "spacing"),
        ("pr-2.5", "spacing"),
        ("pr-20", "spacing"),
        ("pr-24", "spacing"),
        ("pr-28", "spacing"),
        ("pr-3", "spacing"),
        ("pr-3.5", "spacing"),
        ("pr-32", "spacing"),
        ("pr-36", "spacing"),
        ("pr-4", "spacing"),
        ("pr-40", "spacing"),
        ("pr-44", "spacing"),
        ("pr-48", "spacing"),
        ("pr-5", "spacing"),
        ("pr-52", "spacing"),
        ("pr-56", "spacing"),
        ("pr-6", "spacing"),
        ("pr-60", "spacing"),
        ("pr-64", "spacing"),
        ("pr-7", "spacing"),
        ("pr-72", "spacing"),
        ("pr-8", "spacing"),
        ("pr-80", "spacing"),
        ("pr-9", "spacing"),
        ("pr-96", "spacing"),
        ("pr-px", "spacing"),
        ("pt-0", "spacing"),
        ("pt-0.5", "spacing"),
        ("pt-1", "spacing"),
        ("pt-1.5", "spacing"),
        ("pt-10", "spacing"),
        ("pt-11", "spacing"),
        ("pt-12", "spacing"),
        ("pt-14", "spacing"),
        ("pt-16", "spacing"),
        ("pt-2", "spacing"),
        ("pt-2.5", "spacing"),
        ("pt-20", "spacing"),
        ("pt-24", "spacing"),
        ("pt-28", "spacing"),
        ("pt-3", "spacing"),
        ("pt-3.5", "spacing"),
        ("pt-32", "spacing"),
        ("pt-36", "spacing"),
        ("pt-4", "spacing"),
        ("pt-40", "spacing"),
        ("pt-44", "spacing"),
        ("pt-48", "spacing"),
        ("pt-5", "spacing"),
        ("pt-52", "spacing"),
        ("pt-56", "spacing"),
        ("pt-6", "spacing"),
        ("pt-60", "spacing"),
        ("pt-64", "spacing"),
        ("pt-7", "spacing"),
        ("pt-72", "spacing"),
        ("pt-8", "spacing"),
        ("pt-80", "spacing"),
        ("pt-9", "spacing"),
        ("pt-96", "spacing"),
        ("pt-px", "spacing"),
        ("px-0", "spacing"),
        ("px-0.5", "spacing"),
        ("px-1", "spacing"),
        ("px-1.5", "spacing"),
        ("px-10", "spacing"),
        ("px-11", "spacing"),
        ("px-12", "spacing"),
        ("px-14", "spacing"),
        ("px-16", "spacing"),
        ("px-2", "spacing"),
        ("px-2.5", "spacing"),
        ("px-20", "spacing"),
        ("px-24", "spacing"),
        ("px-28", "spacing"),
        ("px-3", "spacing"),
        ("px-3.5", "spacing"),
        ("px-32", "spacing"),
        ("px-36", "spacing"),
        ("px-4", "spacing"),
        ("px-40", "spacing"),
        ("px-44", "spacing"),
        ("px-48", "spacing"),
        ("px-5", "spacing"),
        ("px-52", "spacing"),
        ("px-56", "spacing"),
        ("px-6", "spacing"),
        ("px-60", "spacing"),
        ("px-64", "spacing"),
        ("px-7", "spacing"),
        ("px-72", "spacing"),
        ("px-8", "spacing"),
        ("px-80", "spacing"),
        ("px-9", "spacing"),
        ("px-96", "spacing"),
        ("px-px", "spacing"),
        ("py-0", "spacing"),
        ("py-0.5", "spacing"),
        ("py-1", "spacing"),
        ("py-1.5", "spacing"),
        ("py-10", "spacing"),
        ("py-11", "spacing"),
        ("py-12", "spacing"),
        ("py-14", "spacing"),
        ("py-16", "spacing"),
        ("py-2", "spacing"),
        ("py-2.5", "spacing"),
        ("py-20", "spacing"),
        ("py-24", "spacing"),
        ("py-28", "spacing"),
        ("py-3", "spacing"),
        ("py-3.5", "spacing"),
        ("py-32", "spacing"),
        ("py-36", "spacing"),
        ("py-4", "spacing"),
        ("py-40", "spacing"),
        ("py-44", "spacing"),
        ("py-48", "spacing"),
        ("py-5", "spacing"),
        ("py-52", "spacing"),
        ("py-56", "spacing"),
        ("py-6", "spacing"),
        ("py-60", "spacing"),
        ("py-64", "spacing"),
        ("py-7", "spacing"),
        ("py-72", "spacing"),
        ("py-8", "spacing"),
        ("py-80", "spacing"),
        ("py-9", "spacing"),
        ("py-96", "spacing"),
        ("py-px", "spacing"),
        ("space-x-0 ", "spacing"),
        ("space-x-0.5 ", "spacing"),
        ("space-x-1 ", "spacing"),
        ("space-x-1.5 ", "spacing"),
        ("space-x-10 ", "spacing"),
        ("space-x-11 ", "spacing"),
        ("space-x-12 ", "spacing"),
        ("space-x-14 ", "spacing"),
        ("space-x-16 ", "spacing"),
        ("space-x-2 ", "spacing"),
        ("space-x-2.5 ", "spacing"),
        ("space-x-20 ", "spacing"),
        ("space-x-24 ", "spacing"),
        ("space-x-28 ", "spacing"),
        ("space-x-3 ", "spacing"),
        ("space-x-3.5 ", "spacing"),
        ("space-x-32 ", "spacing"),
        ("space-x-36 ", "spacing"),
        ("space-x-4 ", "spacing"),
        ("space-x-40 ", "spacing"),
        ("space-x-44 ", "spacing"),
        ("space-x-48 ", "spacing"),
        ("space-x-5 ", "spacing"),
        ("space-x-52 ", "spacing"),
        ("space-x-56 ", "spacing"),
        ("space-x-6 ", "spacing"),
        ("space-x-60 ", "spacing"),
        ("space-x-64 ", "spacing"),
        ("space-x-7 ", "spacing"),
        ("space-x-72 ", "spacing"),
        ("space-x-8 ", "spacing"),
        ("space-x-80 ", "spacing"),
        ("space-x-9 ", "spacing"),
        ("space-x-96 ", "spacing"),
        ("space-x-px ", "spacing"),
        ("space-x-reverse ", "spacing"),
        ("space-y-0 ", "spacing"),
        ("space-y-0.5 ", "spacing"),
        ("space-y-1 ", "spacing"),
        ("space-y-1.5 ", "spacing"),
        ("space-y-10 ", "spacing"),
        ("space-y-11 ", "spacing"),
        ("space-y-12 ", "spacing"),
        ("space-y-14 ", "spacing"),
        ("space-y-16 ", "spacing"),
        ("space-y-2 ", "spacing"),
        ("space-y-2.5 ", "spacing"),
        ("space-y-20 ", "spacing"),
        ("space-y-24 ", "spacing"),
        ("space-y-28 ", "spacing"),
        ("space-y-3 ", "spacing"),
        ("space-y-3.5 ", "spacing"),
        ("space-y-32 ", "spacing"),
        ("space-y-36 ", "spacing"),
        ("space-y-4 ", "spacing"),
        ("space-y-40 ", "spacing"),
        ("space-y-44 ", "spacing"),
        ("space-y-48 ", "spacing"),
        ("space-y-5 ", "spacing"),
        ("space-y-52 ", "spacing"),
        ("space-y-56 ", "spacing"),
        ("space-y-6 ", "spacing"),
        ("space-y-60 ", "spacing"),
        ("space-y-64 ", "spacing"),
        ("space-y-7 ", "spacing"),
        ("space-y-72 ", "spacing"),
        ("space-y-8 ", "spacing"),
        ("space-y-80 ", "spacing"),
        ("space-y-9 ", "spacing"),
        ("space-y-96 ", "spacing"),
        ("space-y-px ", "spacing"),
        ("space-y-reverse ", "spacing"),
        ("fill-amber-100", "svg"),
        ("fill-amber-200", "svg"),
        ("fill-amber-300", "svg"),
        ("fill-amber-400", "svg"),
        ("fill-amber-50", "svg"),
        ("fill-amber-500", "svg"),
        ("fill-amber-600", "svg"),
        ("fill-amber-700", "svg"),
        ("fill-amber-800", "svg"),
        ("fill-amber-900", "svg"),
        ("fill-black", "svg"),
        ("fill-blue-100", "svg"),
        ("fill-blue-200", "svg"),
        ("fill-blue-300", "svg"),
        ("fill-blue-400", "svg"),
        ("fill-blue-50", "svg"),
        ("fill-blue-500", "svg"),
        ("fill-blue-600", "svg"),
        ("fill-blue-700", "svg"),
        ("fill-blue-800", "svg"),
        ("fill-blue-900", "svg"),
        ("fill-current", "svg"),
        ("fill-cyan-100", "svg"),
        ("fill-cyan-200", "svg"),
        ("fill-cyan-300", "svg"),
        ("fill-cyan-400", "svg"),
        ("fill-cyan-50", "svg"),
        ("fill-cyan-500", "svg"),
        ("fill-cyan-600", "svg"),
        ("fill-cyan-700", "svg"),
        ("fill-cyan-800", "svg"),
        ("fill-cyan-900", "svg"),
        ("fill-emerald-100", "svg"),
        ("fill-emerald-200", "svg"),
        ("fill-emerald-300", "svg"),
        ("fill-emerald-400", "svg"),
        ("fill-emerald-50", "svg"),
        ("fill-emerald-500", "svg"),
        ("fill-emerald-600", "svg"),
        ("fill-emerald-700", "svg"),
        ("fill-emerald-800", "svg"),
        ("fill-emerald-900", "svg"),
        ("fill-fuchsia-100", "svg"),
        ("fill-fuchsia-200", "svg"),
        ("fill-fuchsia-300", "svg"),
        ("fill-fuchsia-400", "svg"),
        ("fill-fuchsia-50", "svg"),
        ("fill-fuchsia-500", "svg"),
        ("fill-fuchsia-600", "svg"),
        ("fill-fuchsia-700", "svg"),
        ("fill-fuchsia-800", "svg"),
        ("fill-fuchsia-900", "svg"),
        ("fill-gray-100", "svg"),
        ("fill-gray-200", "svg"),
        ("fill-gray-300", "svg"),
        ("fill-gray-400", "svg"),
        ("fill-gray-50", "svg"),
        ("fill-gray-500", "svg"),
        ("fill-gray-600", "svg"),
        ("fill-gray-700", "svg"),
        ("fill-gray-800", "svg"),
        ("fill-gray-900", "svg"),
        ("fill-green-100", "svg"),
        ("fill-green-200", "svg"),
        ("fill-green-300", "svg"),
        ("fill-green-400", "svg"),
        ("fill-green-50", "svg"),
        ("fill-green-500", "svg"),
        ("fill-green-600", "svg"),
        ("fill-green-700", "svg"),
        ("fill-green-800", "svg"),
        ("fill-green-900", "svg"),
        ("fill-indigo-100", "svg"),
        ("fill-indigo-200", "svg"),
        ("fill-indigo-300", "svg"),
        ("fill-indigo-400", "svg"),
        ("fill-indigo-50", "svg"),
        ("fill-indigo-500", "svg"),
        ("fill-indigo-600", "svg"),
        ("fill-indigo-700", "svg"),
        ("fill-indigo-800", "svg"),
        ("fill-indigo-900", "svg"),
        ("fill-inherit", "svg"),
        ("fill-lime-100", "svg"),
        ("fill-lime-200", "svg"),
        ("fill-lime-300", "svg"),
        ("fill-lime-400", "svg"),
        ("fill-lime-50", "svg"),
        ("fill-lime-500", "svg"),
        ("fill-lime-600", "svg"),
        ("fill-lime-700", "svg"),
        ("fill-lime-800", "svg"),
        ("fill-lime-900", "svg"),
        ("fill-neutral-100", "svg"),
        ("fill-neutral-200", "svg"),
        ("fill-neutral-300", "svg"),
        ("fill-neutral-400", "svg"),
        ("fill-neutral-50", "svg"),
        ("fill-neutral-500", "svg"),
        ("fill-neutral-600", "svg"),
        ("fill-neutral-700", "svg"),
        ("fill-neutral-800", "svg"),
        ("fill-neutral-900", "svg"),
        ("fill-orange-100", "svg"),
        ("fill-orange-200", "svg"),
        ("fill-orange-300", "svg"),
        ("fill-orange-400", "svg"),
        ("fill-orange-50", "svg"),
        ("fill-orange-500", "svg"),
        ("fill-orange-600", "svg"),
        ("fill-orange-700", "svg"),
        ("fill-orange-800", "svg"),
        ("fill-orange-900", "svg"),
        ("fill-pink-100", "svg"),
        ("fill-pink-200", "svg"),
        ("fill-pink-300", "svg"),
        ("fill-pink-400", "svg"),
        ("fill-pink-50", "svg"),
        ("fill-pink-500", "svg"),
        ("fill-pink-600", "svg"),
        ("fill-pink-700", "svg"),
        ("fill-pink-800", "svg"),
        ("fill-pink-900", "svg"),
        ("fill-purple-100", "svg"),
        ("fill-purple-200", "svg"),
        ("fill-purple-300", "svg"),
        ("fill-purple-400", "svg"),
        ("fill-purple-50", "svg"),
        ("fill-purple-500", "svg"),
        ("fill-purple-600", "svg"),
        ("fill-purple-700", "svg"),
        ("fill-purple-800", "svg"),
        ("fill-purple-900", "svg"),
        ("fill-red-100", "svg"),
        ("fill-red-200", "svg"),
        ("fill-red-300", "svg"),
        ("fill-red-400", "svg"),
        ("fill-red-50", "svg"),
        ("fill-red-500", "svg"),
        ("fill-red-600", "svg"),
        ("fill-red-700", "svg"),
        ("fill-red-800", "svg"),
        ("fill-red-900", "svg"),
        ("fill-rose-100", "svg"),
        ("fill-rose-200", "svg"),
        ("fill-rose-300", "svg"),
        ("fill-rose-400", "svg"),
        ("fill-rose-50", "svg"),
        ("fill-rose-500", "svg"),
        ("fill-rose-600", "svg"),
        ("fill-rose-700", "svg"),
        ("fill-rose-800", "svg"),
        ("fill-rose-900", "svg"),
        ("fill-sky-100", "svg"),
        ("fill-sky-200", "svg"),
        ("fill-sky-300", "svg"),
        ("fill-sky-400", "svg"),
        ("fill-sky-50", "svg"),
        ("fill-sky-500", "svg"),
        ("fill-sky-600", "svg"),
        ("fill-sky-700", "svg"),
        ("fill-sky-800", "svg"),
        ("fill-sky-900", "svg"),
        ("fill-slate-100", "svg"),
        ("fill-slate-200", "svg"),
        ("fill-slate-300", "svg"),
        ("fill-slate-400", "svg"),
        ("fill-slate-50", "svg"),
        ("fill-slate-500", "svg"),
        ("fill-slate-600", "svg"),
        ("fill-slate-700", "svg"),
        ("fill-slate-800", "svg"),
        ("fill-slate-900", "svg"),
        ("fill-stone-100", "svg"),
        ("fill-stone-200", "svg"),
        ("fill-stone-300", "svg"),
        ("fill-stone-400", "svg"),
        ("fill-stone-50", "svg"),
        ("fill-stone-500", "svg"),
        ("fill-stone-600", "svg"),
        ("fill-stone-700", "svg"),
        ("fill-stone-800", "svg"),
        ("fill-stone-900", "svg"),
        ("fill-teal-100", "svg"),
        ("fill-teal-200", "svg"),
        ("fill-teal-300", "svg"),
        ("fill-teal-400", "svg"),
        ("fill-teal-50", "svg"),
        ("fill-teal-500", "svg"),
        ("fill-teal-600", "svg"),
        ("fill-teal-700", "svg"),
        ("fill-teal-800", "svg"),
        ("fill-teal-900", "svg"),
        ("fill-transparent", "svg"),
        ("fill-violet-100", "svg"),
        ("fill-violet-200", "svg"),
        ("fill-violet-300", "svg"),
        ("fill-violet-400", "svg"),
        ("fill-violet-50", "svg"),
        ("fill-violet-500", "svg"),
        ("fill-violet-600", "svg"),
        ("fill-violet-700", "svg"),
        ("fill-violet-800", "svg"),
        ("fill-violet-900", "svg"),
        ("fill-white", "svg"),
        ("fill-yellow-100", "svg"),
        ("fill-yellow-200", "svg"),
        ("fill-yellow-300", "svg"),
        ("fill-yellow-400", "svg"),
        ("fill-yellow-50", "svg"),
        ("fill-yellow-500", "svg"),
        ("fill-yellow-600", "svg"),
        ("fill-yellow-700", "svg"),
        ("fill-yellow-800", "svg"),
        ("fill-yellow-900", "svg"),
        ("fill-zinc-100", "svg"),
        ("fill-zinc-200", "svg"),
        ("fill-zinc-300", "svg"),
        ("fill-zinc-400", "svg"),
        ("fill-zinc-50", "svg"),
        ("fill-zinc-500", "svg"),
        ("fill-zinc-600", "svg"),
        ("fill-zinc-700", "svg"),
        ("fill-zinc-800", "svg"),
        ("fill-zinc-900", "svg"),
        ("stroke-0", "svg"),
        ("stroke-1", "svg"),
        ("stroke-2", "svg"),
        ("stroke-amber-100", "svg"),
        ("stroke-amber-200", "svg"),
        ("stroke-amber-300", "svg"),
        ("stroke-amber-400", "svg"),
        ("stroke-amber-50", "svg"),
        ("stroke-amber-500", "svg"),
        ("stroke-amber-600", "svg"),
        ("stroke-amber-700", "svg"),
        ("stroke-amber-800", "svg"),
        ("stroke-amber-900", "svg"),
        ("stroke-black", "svg"),
        ("stroke-blue-100", "svg"),
        ("stroke-blue-200", "svg"),
        ("stroke-blue-300", "svg"),
        ("stroke-blue-400", "svg"),
        ("stroke-blue-50", "svg"),
        ("stroke-blue-500", "svg"),
        ("stroke-blue-600", "svg"),
        ("stroke-blue-700", "svg"),
        ("stroke-blue-800", "svg"),
        ("stroke-blue-900", "svg"),
        ("stroke-current", "svg"),
        ("stroke-cyan-100", "svg"),
        ("stroke-cyan-200", "svg"),
        ("stroke-cyan-300", "svg"),
        ("stroke-cyan-400", "svg"),
        ("stroke-cyan-50", "svg"),
        ("stroke-cyan-500", "svg"),
        ("stroke-cyan-600", "svg"),
        ("stroke-cyan-700", "svg"),
        ("stroke-cyan-800", "svg"),
        ("stroke-cyan-900", "svg"),
        ("stroke-emerald-100", "svg"),
        ("stroke-emerald-200", "svg"),
        ("stroke-emerald-300", "svg"),
        ("stroke-emerald-400", "svg"),
        ("stroke-emerald-50", "svg"),
        ("stroke-emerald-500", "svg"),
        ("stroke-emerald-600", "svg"),
        ("stroke-emerald-700", "svg"),
        ("stroke-emerald-800", "svg"),
        ("stroke-emerald-900", "svg"),
        ("stroke-fuchsia-100", "svg"),
        ("stroke-fuchsia-200", "svg"),
        ("stroke-fuchsia-300", "svg"),
        ("stroke-fuchsia-400", "svg"),
        ("stroke-fuchsia-50", "svg"),
        ("stroke-fuchsia-500", "svg"),
        ("stroke-fuchsia-600", "svg"),
        ("stroke-fuchsia-700", "svg"),
        ("stroke-fuchsia-800", "svg"),
        ("stroke-fuchsia-900", "svg"),
        ("stroke-gray-100", "svg"),
        ("stroke-gray-200", "svg"),
        ("stroke-gray-300", "svg"),
        ("stroke-gray-400", "svg"),
        ("stroke-gray-50", "svg"),
        ("stroke-gray-500", "svg"),
        ("stroke-gray-600", "svg"),
        ("stroke-gray-700", "svg"),
        ("stroke-gray-800", "svg"),
        ("stroke-gray-900", "svg"),
        ("stroke-green-100", "svg"),
        ("stroke-green-200", "svg"),
        ("stroke-green-300", "svg"),
        ("stroke-green-400", "svg"),
        ("stroke-green-50", "svg"),
        ("stroke-green-500", "svg"),
        ("stroke-green-600", "svg"),
        ("stroke-green-700", "svg"),
        ("stroke-green-800", "svg"),
        ("stroke-green-900", "svg"),
        ("stroke-indigo-100", "svg"),
        ("stroke-indigo-200", "svg"),
        ("stroke-indigo-300", "svg"),
        ("stroke-indigo-400", "svg"),
        ("stroke-indigo-50", "svg"),
        ("stroke-indigo-500", "svg"),
        ("stroke-indigo-600", "svg"),
        ("stroke-indigo-700", "svg"),
        ("stroke-indigo-800", "svg"),
        ("stroke-indigo-900", "svg"),
        ("stroke-inherit", "svg"),
        ("stroke-lime-100", "svg"),
        ("stroke-lime-200", "svg"),
        ("stroke-lime-300", "svg"),
        ("stroke-lime-400", "svg"),
        ("stroke-lime-50", "svg"),
        ("stroke-lime-500", "svg"),
        ("stroke-lime-600", "svg"),
        ("stroke-lime-700", "svg"),
        ("stroke-lime-800", "svg"),
        ("stroke-lime-900", "svg"),
        ("stroke-neutral-100", "svg"),
        ("stroke-neutral-200", "svg"),
        ("stroke-neutral-300", "svg"),
        ("stroke-neutral-400", "svg"),
        ("stroke-neutral-50", "svg"),
        ("stroke-neutral-500", "svg"),
        ("stroke-neutral-600", "svg"),
        ("stroke-neutral-700", "svg"),
        ("stroke-neutral-800", "svg"),
        ("stroke-neutral-900", "svg"),
        ("stroke-orange-100", "svg"),
        ("stroke-orange-200", "svg"),
        ("stroke-orange-300", "svg"),
        ("stroke-orange-400", "svg"),
        ("stroke-orange-50", "svg"),
        ("stroke-orange-500", "svg"),
        ("stroke-orange-600", "svg"),
        ("stroke-orange-700", "svg"),
        ("stroke-orange-800", "svg"),
        ("stroke-orange-900", "svg"),
        ("stroke-pink-100", "svg"),
        ("stroke-pink-200", "svg"),
        ("stroke-pink-300", "svg"),
        ("stroke-pink-400", "svg"),
        ("stroke-pink-50", "svg"),
        ("stroke-pink-500", "svg"),
        ("stroke-pink-600", "svg"),
        ("stroke-pink-700", "svg"),
        ("stroke-pink-800", "svg"),
        ("stroke-pink-900", "svg"),
        ("stroke-purple-100", "svg"),
        ("stroke-purple-200", "svg"),
        ("stroke-purple-300", "svg"),
        ("stroke-purple-400", "svg"),
        ("stroke-purple-50", "svg"),
        ("stroke-purple-500", "svg"),
        ("stroke-purple-600", "svg"),
        ("stroke-purple-700", "svg"),
        ("stroke-purple-800", "svg"),
        ("stroke-purple-900", "svg"),
        ("stroke-red-100", "svg"),
        ("stroke-red-200", "svg"),
        ("stroke-red-300", "svg"),
        ("stroke-red-400", "svg"),
        ("stroke-red-50", "svg"),
        ("stroke-red-500", "svg"),
        ("stroke-red-600", "svg"),
        ("stroke-red-700", "svg"),
        ("stroke-red-800", "svg"),
        ("stroke-red-900", "svg"),
        ("stroke-rose-100", "svg"),
        ("stroke-rose-200", "svg"),
        ("stroke-rose-300", "svg"),
        ("stroke-rose-400", "svg"),
        ("stroke-rose-50", "svg"),
        ("stroke-rose-500", "svg"),
        ("stroke-rose-600", "svg"),
        ("stroke-rose-700", "svg"),
        ("stroke-rose-800", "svg"),
        ("stroke-rose-900", "svg"),
        ("stroke-sky-100", "svg"),
        ("stroke-sky-200", "svg"),
        ("stroke-sky-300", "svg"),
        ("stroke-sky-400", "svg"),
        ("stroke-sky-50", "svg"),
        ("stroke-sky-500", "svg"),
        ("stroke-sky-600", "svg"),
        ("stroke-sky-700", "svg"),
        ("stroke-sky-800", "svg"),
        ("stroke-sky-900", "svg"),
        ("stroke-slate-100", "svg"),
        ("stroke-slate-200", "svg"),
        ("stroke-slate-300", "svg"),
        ("stroke-slate-400", "svg"),
        ("stroke-slate-50", "svg"),
        ("stroke-slate-500", "svg"),
        ("stroke-slate-600", "svg"),
        ("stroke-slate-700", "svg"),
        ("stroke-slate-800", "svg"),
        ("stroke-slate-900", "svg"),
        ("stroke-stone-100", "svg"),
        ("stroke-stone-200", "svg"),
        ("stroke-stone-300", "svg"),
        ("stroke-stone-400", "svg"),
        ("stroke-stone-50", "svg"),
        ("stroke-stone-500", "svg"),
        ("stroke-stone-600", "svg"),
        ("stroke-stone-700", "svg"),
        ("stroke-stone-800", "svg"),
        ("stroke-stone-900", "svg"),
        ("stroke-teal-100", "svg"),
        ("stroke-teal-200", "svg"),
        ("stroke-teal-300", "svg"),
        ("stroke-teal-400", "svg"),
        ("stroke-teal-50", "svg"),
        ("stroke-teal-500", "svg"),
        ("stroke-teal-600", "svg"),
        ("stroke-teal-700", "svg"),
        ("stroke-teal-800", "svg"),
        ("stroke-teal-900", "svg"),
        ("stroke-transparent", "svg"),
        ("stroke-violet-100", "svg"),
        ("stroke-violet-200", "svg"),
        ("stroke-violet-300", "svg"),
        ("stroke-violet-400", "svg"),
        ("stroke-violet-50", "svg"),
        ("stroke-violet-500", "svg"),
        ("stroke-violet-600", "svg"),
        ("stroke-violet-700", "svg"),
        ("stroke-violet-800", "svg"),
        ("stroke-violet-900", "svg"),
        ("stroke-white", "svg"),
        ("stroke-yellow-100", "svg"),
        ("stroke-yellow-200", "svg"),
        ("stroke-yellow-300", "svg"),
        ("stroke-yellow-400", "svg"),
        ("stroke-yellow-50", "svg"),
        ("stroke-yellow-500", "svg"),
        ("stroke-yellow-600", "svg"),
        ("stroke-yellow-700", "svg"),
        ("stroke-yellow-800", "svg"),
        ("stroke-yellow-900", "svg"),
        ("stroke-zinc-100", "svg"),
        ("stroke-zinc-200", "svg"),
        ("stroke-zinc-300", "svg"),
        ("stroke-zinc-400", "svg"),
        ("stroke-zinc-50", "svg"),
        ("stroke-zinc-500", "svg"),
        ("stroke-zinc-600", "svg"),
        ("stroke-zinc-700", "svg"),
        ("stroke-zinc-800", "svg"),
        ("stroke-zinc-900", "svg"),
        ("border-collapse", "tables"),
        ("border-separate", "tables"),
        ("table-auto", "tables"),
        ("table-fixed", "tables"),
        ("origin-bottom", "transforms"),
        ("origin-bottom-left", "transforms"),
        ("origin-bottom-right", "transforms"),
        ("origin-center", "transforms"),
        ("origin-left", "transforms"),
        ("origin-right", "transforms"),
        ("origin-top", "transforms"),
        ("origin-top-left", "transforms"),
        ("origin-top-right", "transforms"),
        ("rotate-0", "transforms"),
        ("rotate-1", "transforms"),
        ("rotate-12", "transforms"),
        ("rotate-180", "transforms"),
        ("rotate-2", "transforms"),
        ("rotate-3", "transforms"),
        ("rotate-45", "transforms"),
        ("rotate-6", "transforms"),
        ("rotate-90", "transforms"),
        ("scale-0", "transforms"),
        ("scale-100", "transforms"),
        ("scale-105", "transforms"),
        ("scale-110", "transforms"),
        ("scale-125", "transforms"),
        ("scale-150", "transforms"),
        ("scale-50", "transforms"),
        ("scale-75", "transforms"),
        ("scale-90", "transforms"),
        ("scale-95", "transforms"),
        ("scale-x-0", "transforms"),
        ("scale-x-100", "transforms"),
        ("scale-x-105", "transforms"),
        ("scale-x-110", "transforms"),
        ("scale-x-125", "transforms"),
        ("scale-x-150", "transforms"),
        ("scale-x-50", "transforms"),
        ("scale-x-75", "transforms"),
        ("scale-x-90", "transforms"),
        ("scale-x-95", "transforms"),
        ("scale-y-0", "transforms"),
        ("scale-y-100", "transforms"),
        ("scale-y-105", "transforms"),
        ("scale-y-110", "transforms"),
        ("scale-y-125", "transforms"),
        ("scale-y-150", "transforms"),
        ("scale-y-50", "transforms"),
        ("scale-y-75", "transforms"),
        ("scale-y-90", "transforms"),
        ("scale-y-95", "transforms"),
        ("skew-x-0", "transforms"),
        ("skew-x-1", "transforms"),
        ("skew-x-12", "transforms"),
        ("skew-x-2", "transforms"),
        ("skew-x-3", "transforms"),
        ("skew-x-6", "transforms"),
        ("skew-y-0", "transforms"),
        ("skew-y-1", "transforms"),
        ("skew-y-12", "transforms"),
        ("skew-y-2", "transforms"),
        ("skew-y-3", "transforms"),
        ("skew-y-6", "transforms"),
        ("translate-x-0", "transforms"),
        ("translate-x-0.5", "transforms"),
        ("translate-x-1", "transforms"),
        ("translate-x-1.5", "transforms"),
        ("translate-x-1/2", "transforms"),
        ("translate-x-1/3", "transforms"),
        ("translate-x-1/4", "transforms"),
        ("translate-x-10", "transforms"),
        ("translate-x-11", "transforms"),
        ("translate-x-12", "transforms"),
        ("translate-x-14", "transforms"),
        ("translate-x-16", "transforms"),
        ("translate-x-2", "transforms"),
        ("translate-x-2.5", "transforms"),
        ("translate-x-2/3", "transforms"),
        ("translate-x-2/4", "transforms"),
        ("translate-x-20", "transforms"),
        ("translate-x-24", "transforms"),
        ("translate-x-28", "transforms"),
        ("translate-x-3", "transforms"),
        ("translate-x-3.5", "transforms"),
        ("translate-x-3/4", "transforms"),
        ("translate-x-32", "transforms"),
        ("translate-x-36", "transforms"),
        ("translate-x-4", "transforms"),
        ("translate-x-40", "transforms"),
        ("translate-x-44", "transforms"),
        ("translate-x-48", "transforms"),
        ("translate-x-5", "transforms"),
        ("translate-x-52", "transforms"),
        ("translate-x-56", "transforms"),
        ("translate-x-6", "transforms"),
        ("translate-x-60", "transforms"),
        ("translate-x-64", "transforms"),
        ("translate-x-7", "transforms"),
        ("translate-x-72", "transforms"),
        ("translate-x-8", "transforms"),
        ("translate-x-80", "transforms"),
        ("translate-x-9", "transforms"),
        ("translate-x-96", "transforms"),
        ("translate-x-full", "transforms"),
        ("translate-x-px", "transforms"),
        ("translate-y-0", "transforms"),
        ("translate-y-0.5", "transforms"),
        ("translate-y-1", "transforms"),
        ("translate-y-1.5", "transforms"),
        ("translate-y-1/2", "transforms"),
        ("translate-y-1/3", "transforms"),
        ("translate-y-1/4", "transforms"),
        ("translate-y-10", "transforms"),
        ("translate-y-11", "transforms"),
        ("translate-y-12", "transforms"),
        ("translate-y-14", "transforms"),
        ("translate-y-16", "transforms"),
        ("translate-y-2", "transforms"),
        ("translate-y-2.5", "transforms"),
        ("translate-y-2/3", "transforms"),
        ("translate-y-2/4", "transforms"),
        ("translate-y-20", "transforms"),
        ("translate-y-24", "transforms"),
        ("translate-y-28", "transforms"),
        ("translate-y-3", "transforms"),
        ("translate-y-3.5", "transforms"),
        ("translate-y-3/4", "transforms"),
        ("translate-y-32", "transforms"),
        ("translate-y-36", "transforms"),
        ("translate-y-4", "transforms"),
        ("translate-y-40", "transforms"),
        ("translate-y-44", "transforms"),
        ("translate-y-48", "transforms"),
        ("translate-y-5", "transforms"),
        ("translate-y-52", "transforms"),
        ("translate-y-56", "transforms"),
        ("translate-y-6", "transforms"),
        ("translate-y-60", "transforms"),
        ("translate-y-64", "transforms"),
        ("translate-y-7", "transforms"),
        ("translate-y-72", "transforms"),
        ("translate-y-8", "transforms"),
        ("translate-y-80", "transforms"),
        ("translate-y-9", "transforms"),
        ("translate-y-96", "transforms"),
        ("translate-y-full", "transforms"),
        ("translate-y-px", "transforms"),
        ("align-baseline", "typography"),
        ("align-bottom", "typography"),
        ("align-middle", "typography"),
        ("align-sub", "typography"),
        ("align-super", "typography"),
        ("align-text-bottom", "typography"),
        ("align-text-top", "typography"),
        ("align-top", "typography"),
        ("antialiased", "typography"),
        ("break-all", "typography"),
        ("break-normal", "typography"),
        ("break-words", "typography"),
        ("capitalize", "typography"),
        ("content-none", "typography"),
        ("decoration-0", "typography"),
        ("decoration-1", "typography"),
        ("decoration-2", "typography"),
        ("decoration-4", "typography"),
        ("decoration-8", "typography"),
        ("decoration-amber-100", "typography"),
        ("decoration-amber-200", "typography"),
        ("decoration-amber-300", "typography"),
        ("decoration-amber-400", "typography"),
        ("decoration-amber-50", "typography"),
        ("decoration-amber-500", "typography"),
        ("decoration-amber-600", "typography"),
        ("decoration-amber-700", "typography"),
        ("decoration-amber-800", "typography"),
        ("decoration-amber-900", "typography"),
        ("decoration-auto", "typography"),
        ("decoration-black", "typography"),
        ("decoration-blue-100", "typography"),
        ("decoration-blue-200", "typography"),
        ("decoration-blue-300", "typography"),
        ("decoration-blue-400", "typography"),
        ("decoration-blue-50", "typography"),
        ("decoration-blue-500", "typography"),
        ("decoration-blue-600", "typography"),
        ("decoration-blue-700", "typography"),
        ("decoration-blue-800", "typography"),
        ("decoration-blue-900", "typography"),
        ("decoration-current", "typography"),
        ("decoration-cyan-100", "typography"),
        ("decoration-cyan-200", "typography"),
        ("decoration-cyan-300", "typography"),
        ("decoration-cyan-400", "typography"),
        ("decoration-cyan-50", "typography"),
        ("decoration-cyan-500", "typography"),
        ("decoration-cyan-600", "typography"),
        ("decoration-cyan-700", "typography"),
        ("decoration-cyan-800", "typography"),
        ("decoration-cyan-900", "typography"),
        ("decoration-dashed", "typography"),
        ("decoration-dotted", "typography"),
        ("decoration-double", "typography"),
        ("decoration-emerald-100", "typography"),
        ("decoration-emerald-200", "typography"),
        ("decoration-emerald-300", "typography"),
        ("decoration-emerald-400", "typography"),
        ("decoration-emerald-50", "typography"),
        ("decoration-emerald-500", "typography"),
        ("decoration-emerald-600", "typography"),
        ("decoration-emerald-700", "typography"),
        ("decoration-emerald-800", "typography"),
        ("decoration-emerald-900", "typography"),
        ("decoration-from-font", "typography"),
        ("decoration-fuchsia-100", "typography"),
        ("decoration-fuchsia-200", "typography"),
        ("decoration-fuchsia-300", "typography"),
        ("decoration-fuchsia-400", "typography"),
        ("decoration-fuchsia-50", "typography"),
        ("decoration-fuchsia-500", "typography"),
        ("decoration-fuchsia-600", "typography"),
        ("decoration-fuchsia-700", "typography"),
        ("decoration-fuchsia-800", "typography"),
        ("decoration-fuchsia-900", "typography"),
        ("decoration-gray-100", "typography"),
        ("decoration-gray-200", "typography"),
        ("decoration-gray-300", "typography"),
        ("decoration-gray-400", "typography"),
        ("decoration-gray-50", "typography"),
        ("decoration-gray-500", "typography"),
        ("decoration-gray-600", "typography"),
        ("decoration-gray-700", "typography"),
        ("decoration-gray-800", "typography"),
        ("decoration-gray-900", "typography"),
        ("decoration-green-100", "typography"),
        ("decoration-green-200", "typography"),
        ("decoration-green-300", "typography"),
        ("decoration-green-400", "typography"),
        ("decoration-green-50", "typography"),
        ("decoration-green-500", "typography"),
        ("decoration-green-600", "typography"),
        ("decoration-green-700", "typography"),
        ("decoration-green-800", "typography"),
        ("decoration-green-900", "typography"),
        ("decoration-indigo-100", "typography"),
        ("decoration-indigo-200", "typography"),
        ("decoration-indigo-300", "typography"),
        ("decoration-indigo-400", "typography"),
        ("decoration-indigo-50", "typography"),
        ("decoration-indigo-500", "typography"),
        ("decoration-indigo-600", "typography"),
        ("decoration-indigo-700", "typography"),
        ("decoration-indigo-800", "typography"),
        ("decoration-indigo-900", "typography"),
        ("decoration-inherit", "typography"),
        ("decoration-lime-100", "typography"),
        ("decoration-lime-200", "typography"),
        ("decoration-lime-300", "typography"),
        ("decoration-lime-400", "typography"),
        ("decoration-lime-50", "typography"),
        ("decoration-lime-500", "typography"),
        ("decoration-lime-600", "typography"),
        ("decoration-lime-700", "typography"),
        ("decoration-lime-800", "typography"),
        ("decoration-lime-900", "typography"),
        ("decoration-neutral-100", "typography"),
        ("decoration-neutral-200", "typography"),
        ("decoration-neutral-300", "typography"),
        ("decoration-neutral-400", "typography"),
        ("decoration-neutral-50", "typography"),
        ("decoration-neutral-500", "typography"),
        ("decoration-neutral-600", "typography"),
        ("decoration-neutral-700", "typography"),
        ("decoration-neutral-800", "typography"),
        ("decoration-neutral-900", "typography"),
        ("decoration-orange-100", "typography"),
        ("decoration-orange-200", "typography"),
        ("decoration-orange-300", "typography"),
        ("decoration-orange-400", "typography"),
        ("decoration-orange-50", "typography"),
        ("decoration-orange-500", "typography"),
        ("decoration-orange-600", "typography"),
        ("decoration-orange-700", "typography"),
        ("decoration-orange-800", "typography"),
        ("decoration-orange-900", "typography"),
        ("decoration-pink-100", "typography"),
        ("decoration-pink-200", "typography"),
        ("decoration-pink-300", "typography"),
        ("decoration-pink-400", "typography"),
        ("decoration-pink-50", "typography"),
        ("decoration-pink-500", "typography"),
        ("decoration-pink-600", "typography"),
        ("decoration-pink-700", "typography"),
        ("decoration-pink-800", "typography"),
        ("decoration-pink-900", "typography"),
        ("decoration-purple-100", "typography"),
        ("decoration-purple-200", "typography"),
        ("decoration-purple-300", "typography"),
        ("decoration-purple-400", "typography"),
        ("decoration-purple-50", "typography"),
        ("decoration-purple-500", "typography"),
        ("decoration-purple-600", "typography"),
        ("decoration-purple-700", "typography"),
        ("decoration-purple-800", "typography"),
        ("decoration-purple-900", "typography"),
        ("decoration-red-100", "typography"),
        ("decoration-red-200", "typography"),
        ("decoration-red-300", "typography"),
        ("decoration-red-400", "typography"),
        ("decoration-red-50", "typography"),
        ("decoration-red-500", "typography"),
        ("decoration-red-600", "typography"),
        ("decoration-red-700", "typography"),
        ("decoration-red-800", "typography"),
        ("decoration-red-900", "typography"),
        ("decoration-rose-100", "typography"),
        ("decoration-rose-200", "typography"),
        ("decoration-rose-300", "typography"),
        ("decoration-rose-400", "typography"),
        ("decoration-rose-50", "typography"),
        ("decoration-rose-500", "typography"),
        ("decoration-rose-600", "typography"),
        ("decoration-rose-700", "typography"),
        ("decoration-rose-800", "typography"),
        ("decoration-rose-900", "typography"),
        ("decoration-sky-100", "typography"),
        ("decoration-sky-200", "typography"),
        ("decoration-sky-300", "typography"),
        ("decoration-sky-400", "typography"),
        ("decoration-sky-50", "typography"),
        ("decoration-sky-500", "typography"),
        ("decoration-sky-600", "typography"),
        ("decoration-sky-700", "typography"),
        ("decoration-sky-800", "typography"),
        ("decoration-sky-900", "typography"),
        ("decoration-slate-100", "typography"),
        ("decoration-slate-200", "typography"),
        ("decoration-slate-300", "typography"),
        ("decoration-slate-400", "typography"),
        ("decoration-slate-50", "typography"),
        ("decoration-slate-500", "typography"),
        ("decoration-slate-600", "typography"),
        ("decoration-slate-700", "typography"),
        ("decoration-slate-800", "typography"),
        ("decoration-slate-900", "typography"),
        ("decoration-solid", "typography"),
        ("decoration-stone-100", "typography"),
        ("decoration-stone-200", "typography"),
        ("decoration-stone-300", "typography"),
        ("decoration-stone-400", "typography"),
        ("decoration-stone-50", "typography"),
        ("decoration-stone-500", "typography"),
        ("decoration-stone-600", "typography"),
        ("decoration-stone-700", "typography"),
        ("decoration-stone-800", "typography"),
        ("decoration-stone-900", "typography"),
        ("decoration-teal-100", "typography"),
        ("decoration-teal-200", "typography"),
        ("decoration-teal-300", "typography"),
        ("decoration-teal-400", "typography"),
        ("decoration-teal-50", "typography"),
        ("decoration-teal-500", "typography"),
        ("decoration-teal-600", "typography"),
        ("decoration-teal-700", "typography"),
        ("decoration-teal-800", "typography"),
        ("decoration-teal-900", "typography"),
        ("decoration-transparent", "typography"),
        ("decoration-violet-100", "typography"),
        ("decoration-violet-200", "typography"),
        ("decoration-violet-300", "typography"),
        ("decoration-violet-400", "typography"),
        ("decoration-violet-50", "typography"),
        ("decoration-violet-500", "typography"),
        ("decoration-violet-600", "typography"),
        ("decoration-violet-700", "typography"),
        ("decoration-violet-800", "typography"),
        ("decoration-violet-900", "typography"),
        ("decoration-wavy", "typography"),
        ("decoration-white", "typography"),
        ("decoration-yellow-100", "typography"),
        ("decoration-yellow-200", "typography"),
        ("decoration-yellow-300", "typography"),
        ("decoration-yellow-400", "typography"),
        ("decoration-yellow-50", "typography"),
        ("decoration-yellow-500", "typography"),
        ("decoration-yellow-600", "typography"),
        ("decoration-yellow-700", "typography"),
        ("decoration-yellow-800", "typography"),
        ("decoration-yellow-900", "typography"),
        ("decoration-zinc-100", "typography"),
        ("decoration-zinc-200", "typography"),
        ("decoration-zinc-300", "typography"),
        ("decoration-zinc-400", "typography"),
        ("decoration-zinc-50", "typography"),
        ("decoration-zinc-500", "typography"),
        ("decoration-zinc-600", "typography"),
        ("decoration-zinc-700", "typography"),
        ("decoration-zinc-800", "typography"),
        ("decoration-zinc-900", "typography"),
        ("diagonal-fractions", "typography"),
        ("font-black", "typography"),
        ("font-bold", "typography"),
        ("font-extrabold", "typography"),
        ("font-extralight", "typography"),
        ("font-light", "typography"),
        ("font-medium", "typography"),
        ("font-mono", "typography"),
        ("font-normal", "typography"),
        ("font-sans", "typography"),
        ("font-semibold", "typography"),
        ("font-serif", "typography"),
        ("font-thin", "typography"),
        ("indent-0", "typography"),
        ("indent-0.5", "typography"),
        ("indent-1", "typography"),
        ("indent-1.5", "typography"),
        ("indent-10", "typography"),
        ("indent-11", "typography"),
        ("indent-12", "typography"),
        ("indent-14", "typography"),
        ("indent-16", "typography"),
        ("indent-2", "typography"),
        ("indent-2.5", "typography"),
        ("indent-20", "typography"),
        ("indent-24", "typography"),
        ("indent-28", "typography"),
        ("indent-3", "typography"),
        ("indent-3.5", "typography"),
        ("indent-32", "typography"),
        ("indent-36", "typography"),
        ("indent-4", "typography"),
        ("indent-40", "typography"),
        ("indent-44", "typography"),
        ("indent-48", "typography"),
        ("indent-5", "typography"),
        ("indent-52", "typography"),
        ("indent-56", "typography"),
        ("indent-6", "typography"),
        ("indent-60", "typography"),
        ("indent-64", "typography"),
        ("indent-7", "typography"),
        ("indent-72", "typography"),
        ("indent-8", "typography"),
        ("indent-80", "typography"),
        ("indent-9", "typography"),
        ("indent-96", "typography"),
        ("indent-px", "typography"),
        ("italic", "typography"),
        ("leading-10", "typography"),
        ("leading-3", "typography"),
        ("leading-4", "typography"),
        ("leading-5", "typography"),
        ("leading-6", "typography"),
        ("leading-7", "typography"),
        ("leading-8", "typography"),
        ("leading-9", "typography"),
        ("leading-loose", "typography"),
        ("leading-none", "typography"),
        ("leading-normal", "typography"),
        ("leading-relaxed", "typography"),
        ("leading-snug", "typography"),
        ("leading-tight", "typography"),
        ("line-through", "typography"),
        ("lining-nums", "typography"),
        ("list-decimal", "typography"),
        ("list-disc", "typography"),
        ("list-inside", "typography"),
        ("list-none", "typography"),
        ("list-outside", "typography"),
        ("lowercase", "typography"),
        ("no-underline", "typography"),
        ("normal-case", "typography"),
        ("normal-nums", "typography"),
        ("not-italic", "typography"),
        ("oldstyle-nums", "typography"),
        ("ordinal", "typography"),
        ("overline", "typography"),
        ("proportional-nums", "typography"),
        ("slashed-zero", "typography"),
        ("stacked-fractions", "typography"),
        ("subpixel-antialiased", "typography"),
        ("tabular-nums", "typography"),
        ("text-2xl", "typography"),
        ("text-3xl", "typography"),
        ("text-4xl", "typography"),
        ("text-5xl", "typography"),
        ("text-6xl", "typography"),
        ("text-7xl", "typography"),
        ("text-8xl", "typography"),
        ("text-9xl", "typography"),
        ("text-amber-100", "typography"),
        ("text-amber-200", "typography"),
        ("text-amber-300", "typography"),
        ("text-amber-400", "typography"),
        ("text-amber-50", "typography"),
        ("text-amber-500", "typography"),
        ("text-amber-600", "typography"),
        ("text-amber-700", "typography"),
        ("text-amber-800", "typography"),
        ("text-amber-900", "typography"),
        ("text-base", "typography"),
        ("text-black", "typography"),
        ("text-blue-100", "typography"),
        ("text-blue-200", "typography"),
        ("text-blue-300", "typography"),
        ("text-blue-400", "typography"),
        ("text-blue-50", "typography"),
        ("text-blue-500", "typography"),
        ("text-blue-600", "typography"),
        ("text-blue-700", "typography"),
        ("text-blue-800", "typography"),
        ("text-blue-900", "typography"),
        ("text-center", "typography"),
        ("text-clip", "typography"),
        ("text-current", "typography"),
        ("text-cyan-100", "typography"),
        ("text-cyan-200", "typography"),
        ("text-cyan-300", "typography"),
        ("text-cyan-400", "typography"),
        ("text-cyan-50", "typography"),
        ("text-cyan-500", "typography"),
        ("text-cyan-600", "typography"),
        ("text-cyan-700", "typography"),
        ("text-cyan-800", "typography"),
        ("text-cyan-900", "typography"),
        ("text-ellipsis", "typography"),
        ("text-emerald-100", "typography"),
        ("text-emerald-200", "typography"),
        ("text-emerald-300", "typography"),
        ("text-emerald-400", "typography"),
        ("text-emerald-50", "typography"),
        ("text-emerald-500", "typography"),
        ("text-emerald-600", "typography"),
        ("text-emerald-700", "typography"),
        ("text-emerald-800", "typography"),
        ("text-emerald-900", "typography"),
        ("text-fuchsia-100", "typography"),
        ("text-fuchsia-200", "typography"),
        ("text-fuchsia-300", "typography"),
        ("text-fuchsia-400", "typography"),
        ("text-fuchsia-50", "typography"),
        ("text-fuchsia-500", "typography"),
        ("text-fuchsia-600", "typography"),
        ("text-fuchsia-700", "typography"),
        ("text-fuchsia-800", "typography"),
        ("text-fuchsia-900", "typography"),
        ("text-gray-100", "typography"),
        ("text-gray-200", "typography"),
        ("text-gray-300", "typography"),
        ("text-gray-400", "typography"),
        ("text-gray-50", "typography"),
        ("text-gray-500", "typography"),
        ("text-gray-600", "typography"),
        ("text-gray-700", "typography"),
        ("text-gray-800", "typography"),
        ("text-gray-900", "typography"),
        ("text-green-100", "typography"),
        ("text-green-200", "typography"),
        ("text-green-300", "typography"),
        ("text-green-400", "typography"),
        ("text-green-50", "typography"),
        ("text-green-500", "typography"),
        ("text-green-600", "typography"),
        ("text-green-700", "typography"),
        ("text-green-800", "typography"),
        ("text-green-900", "typography"),
        ("text-indigo-100", "typography"),
        ("text-indigo-200", "typography"),
        ("text-indigo-300", "typography"),
        ("text-indigo-400", "typography"),
        ("text-indigo-50", "typography"),
        ("text-indigo-500", "typography"),
        ("text-indigo-600", "typography"),
        ("text-indigo-700", "typography"),
        ("text-indigo-800", "typography"),
        ("text-indigo-900", "typography"),
        ("text-inherit", "typography"),
        ("text-justify", "typography"),
        ("text-left", "typography"),
        ("text-lg", "typography"),
        ("text-lime-100", "typography"),
        ("text-lime-200", "typography"),
        ("text-lime-300", "typography"),
        ("text-lime-400", "typography"),
        ("text-lime-50", "typography"),
        ("text-lime-500", "typography"),
        ("text-lime-600", "typography"),
        ("text-lime-700", "typography"),
        ("text-lime-800", "typography"),
        ("text-lime-900", "typography"),
        ("text-neutral-100", "typography"),
        ("text-neutral-200", "typography"),
        ("text-neutral-300", "typography"),
        ("text-neutral-400", "typography"),
        ("text-neutral-50", "typography"),
        ("text-neutral-500", "typography"),
        ("text-neutral-600", "typography"),
        ("text-neutral-700", "typography"),
        ("text-neutral-800", "typography"),
        ("text-neutral-900", "typography"),
        ("text-orange-100", "typography"),
        ("text-orange-200", "typography"),
        ("text-orange-300", "typography"),
        ("text-orange-400", "typography"),
        ("text-orange-50", "typography"),
        ("text-orange-500", "typography"),
        ("text-orange-600", "typography"),
        ("text-orange-700", "typography"),
        ("text-orange-800", "typography"),
        ("text-orange-900", "typography"),
        ("text-pink-100", "typography"),
        ("text-pink-200", "typography"),
        ("text-pink-300", "typography"),
        ("text-pink-400", "typography"),
        ("text-pink-50", "typography"),
        ("text-pink-500", "typography"),
        ("text-pink-600", "typography"),
        ("text-pink-700", "typography"),
        ("text-pink-800", "typography"),
        ("text-pink-900", "typography"),
        ("text-purple-100", "typography"),
        ("text-purple-200", "typography"),
        ("text-purple-300", "typography"),
        ("text-purple-400", "typography"),
        ("text-purple-50", "typography"),
        ("text-purple-500", "typography"),
        ("text-purple-600", "typography"),
        ("text-purple-700", "typography"),
        ("text-purple-800", "typography"),
        ("text-purple-900", "typography"),
        ("text-red-100", "typography"),
        ("text-red-200", "typography"),
        ("text-red-300", "typography"),
        ("text-red-400", "typography"),
        ("text-red-50", "typography"),
        ("text-red-500", "typography"),
        ("text-red-600", "typography"),
        ("text-red-700", "typography"),
        ("text-red-800", "typography"),
        ("text-red-900", "typography"),
        ("text-right", "typography"),
        ("text-rose-100", "typography"),
        ("text-rose-200", "typography"),
        ("text-rose-300", "typography"),
        ("text-rose-400", "typography"),
        ("text-rose-50", "typography"),
        ("text-rose-500", "typography"),
        ("text-rose-600", "typography"),
        ("text-rose-700", "typography"),
        ("text-rose-800", "typography"),
        ("text-rose-900", "typography"),
        ("text-sky-100", "typography"),
        ("text-sky-200", "typography"),
        ("text-sky-300", "typography"),
        ("text-sky-400", "typography"),
        ("text-sky-50", "typography"),
        ("text-sky-500", "typography"),
        ("text-sky-600", "typography"),
        ("text-sky-700", "typography"),
        ("text-sky-800", "typography"),
        ("text-sky-900", "typography"),
        ("text-slate-100", "typography"),
        ("text-slate-200", "typography"),
        ("text-slate-300", "typography"),
        ("text-slate-400", "typography"),
        ("text-slate-50", "typography"),
        ("text-slate-500", "typography"),
        ("text-slate-600", "typography"),
        ("text-slate-700", "typography"),
        ("text-slate-800", "typography"),
        ("text-slate-900", "typography"),
        ("text-sm", "typography"),
        ("text-stone-100", "typography"),
        ("text-stone-200", "typography"),
        ("text-stone-300", "typography"),
        ("text-stone-400", "typography"),
        ("text-stone-50", "typography"),
        ("text-stone-500", "typography"),
        ("text-stone-600", "typography"),
        ("text-stone-700", "typography"),
        ("text-stone-800", "typography"),
        ("text-stone-900", "typography"),
        ("text-teal-100", "typography"),
        ("text-teal-200", "typography"),
        ("text-teal-300", "typography"),
        ("text-teal-400", "typography"),
        ("text-teal-50", "typography"),
        ("text-teal-500", "typography"),
        ("text-teal-600", "typography"),
        ("text-teal-700", "typography"),
        ("text-teal-800", "typography"),
        ("text-teal-900", "typography"),
        ("text-transparent", "typography"),
        ("text-violet-100", "typography"),
        ("text-violet-200", "typography"),
        ("text-violet-300", "typography"),
        ("text-violet-400", "typography"),
        ("text-violet-50", "typography"),
        ("text-violet-500", "typography"),
        ("text-violet-600", "typography"),
        ("text-violet-700", "typography"),
        ("text-violet-800", "typography"),
        ("text-violet-900", "typography"),
        ("text-white", "typography"),
        ("text-xl", "typography"),
        ("text-xs", "typography"),
        ("text-yellow-100", "typography"),
        ("text-yellow-200", "typography"),
        ("text-yellow-300", "typography"),
        ("text-yellow-400", "typography"),
        ("text-yellow-50", "typography"),
        ("text-yellow-500", "typography"),
        ("text-yellow-600", "typography"),
        ("text-yellow-700", "typography"),
        ("text-yellow-800", "typography"),
        ("text-yellow-900", "typography"),
        ("text-zinc-100", "typography"),
        ("text-zinc-200", "typography"),
        ("text-zinc-300", "typography"),
        ("text-zinc-400", "typography"),
        ("text-zinc-50", "typography"),
        ("text-zinc-500", "typography"),
        ("text-zinc-600", "typography"),
        ("text-zinc-700", "typography"),
        ("text-zinc-800", "typography"),
        ("text-zinc-900", "typography"),
        ("tracking-normal", "typography"),
        ("tracking-tight", "typography"),
        ("tracking-tighter", "typography"),
        ("tracking-wide", "typography"),
        ("tracking-wider", "typography"),
        ("tracking-widest", "typography"),
        ("truncate", "typography"),
        ("underline", "typography"),
        ("underline-offset-0", "typography"),
        ("underline-offset-1", "typography"),
        ("underline-offset-2", "typography"),
        ("underline-offset-4", "typography"),
        ("underline-offset-8", "typography"),
        ("underline-offset-auto", "typography"),
        ("uppercase", "typography"),
        ("whitespace-normal", "typography"),
        ("whitespace-nowrap", "typography"),
        ("whitespace-pre", "typography"),
        ("whitespace-pre-line", "typography"),
        ("whitespace-pre-wrap", "typography"),
    ])
}
