pub mod attribute;
pub mod instance;
pub mod object;

use crate::Result;
use nom::{bytes::complete::tag, IResult};
use std::{
    cmp::Ordering,
    collections::{BTreeSet, HashSet},
    fmt::{Debug, Display, Formatter, Write},
    hash::{Hash, Hasher},
};
use text_utils::indent;

/// Tailwind Parsed Result
pub type ParsedItem<'a> = IResult<&'a str, Box<dyn TailwindInstance>>;
/// Tailwind Parsed Result
pub type ParsedList<'a> = IResult<&'a str, HashSet<Box<dyn TailwindInstance>>>;

pub trait TailwindInstance {
    /// used to deduplication and marking
    fn id(&self) -> String;
    // const ID: &'static str;

    fn selectors(&self) -> String {
        format!(".{}", self.id())
    }
    fn attributes(&self) -> BTreeSet<CssAttribute> {
        BTreeSet::default()
    }

    fn write_css(&self, f: &mut (dyn Write)) -> Result<()> {
        writeln!(f, "{} {{", self.selectors())?;
        for item in self.attributes() {
            writeln!(f, "{}", indent(item.to_string(), 4))?
        }
        writeln!(f, "}}")?;
        Ok(())
    }
}

///
#[derive(Clone)]
pub struct CssAttribute {
    key: String,
    value: String,
}

/// Uncategorized tailwind property
#[derive(Debug)]
pub struct TailwindObject {
    pub id: String,
    pub attributes: BTreeSet<CssAttribute>,
}
