use std::ptr::copy_nonoverlapping;

use capnp::Word;

use crate::secrets_store_capnp::block;
use crate::{memguard::SecretBytes, secrets_store::cipher::Cipher};
use spectral::prelude::*;

fn fixture_test<T>(
  cipher: &T,
  seal_key_raw: &[u8],
  seal_nonce: &[u8],
  maybe_public_key: Option<&[u8]>,
  crypted_key: &[u8],
  messages: &[&[u8]],
) where
  T: Cipher,
{
  let seal_key = SecretBytes::from_secured(seal_key_raw);
  let private_kex = cipher.open_private_key(&seal_key, seal_nonce, crypted_key).unwrap();

  for message in messages {
    let words = Word::allocate_zeroed_vec(message.len());
    unsafe {
      copy_nonoverlapping(message.as_ptr(), words.as_ptr() as *mut u8, words.len() * 8);
    }
    let mut message_payload: &[u8] = Word::words_to_bytes(&words);
    let message_reader =
      capnp::serialize::read_message_from_flat_slice(&mut message_payload, capnp::message::ReaderOptions::new())
        .unwrap();
    let block_reader = message_reader.get_root::<block::Reader>().unwrap();
    let cryped_content = block_reader.get_content().unwrap();

    let id1 = "recipient1";

    let decrypted = cipher
      .decrypt(
        (id1, &private_kex),
        block_reader.get_headers().unwrap().get(0),
        cryped_content,
      )
      .unwrap();

    assert_that(&decrypted.borrow().as_str()).is_equal_to("Hello, secret");
  }

  if let Some(public_key) = maybe_public_key {
    let mut message = capnp::message::Builder::new_default();

    let id1 = "recipient1";

    let mut block = message.init_root::<block::Builder>();
    let headers = block.reborrow().init_headers(1);

    let private_data = SecretBytes::from("New Secret".to_string());

    let crypted_data = cipher
      .encrypt(&[(id1, public_key.into())], &private_data, headers.get(0))
      .unwrap();
    block.set_content(&crypted_data);

    let mut message_payload: &[u8] = &capnp::serialize::write_message_to_words(&message);

    let message_reader =
      capnp::serialize::read_message_from_flat_slice(&mut message_payload, capnp::message::ReaderOptions::new())
        .unwrap();
    let block_reader = message_reader.get_root::<block::Reader>().unwrap();
    let cryped_content = block_reader.get_content().unwrap();

    let decrypted = cipher
      .decrypt(
        (id1, &private_kex),
        block_reader.get_headers().unwrap().get(0),
        cryped_content,
      )
      .unwrap();

    assert_that(&decrypted.borrow().as_str()).is_equal_to("New Secret");
  }
}

mod rsa1 {
  // This was generated with openssl

  const SEAL_NONCE: &[u8] = &hex!("8e0fe638fee592a0f4e2b311");
  const SEAL_KEY: &[u8] = &hex!("0f698890f99ae4ad83c1ded1e3797e42f0a4854b1adda416ef4c4d94b20aede4");
  const PUBLIC_KEY: &[u8] = &hex!(
    "
    30820222300d06092a864886f70d01010105000382020f003082020a0282020100bd7121180e55a5
    c3ec1580dce5154243b288f53f2056fdf23af0f2d37df3bd74a33cfed48bb09214b057c9a4e7bef0
    f2a47e8001d7ec2bed512847ded30d945d94398fd3889c0db42f7ae3808b548da489197deeab2708
    a1abff88924a90a36f7d8da714f79436776e56c17b34001bf72abf7a6b9939eb98bbf1dd3f0cfe39
    bb2076edb68259eece98af9d937501355a478fb25354cd86974217fa6be6f304fe6eddc4a1f38a03
    c4252100c164cbc092b0c7ef09f295219cbb66413e4a199da60a5f486d0c489cc4cb5e46dedbd052
    b3b2c6cdbb238000db2a06bc12c9327a4a4dc88ff321baf5a6b47cbe6fbba041b3020510709f5cb4
    64770e24b934d67242cfe4519e5c9aad4bf7070cf3455d3b9e14035b6decdfca167b4ba7b0e52ea8
    dc5b8caeace94385bac6dab445e2089271b29c2c29e37b0309763d34e8e987b94baab236d27fad1f
    9ed66e5c3c5dbe4bbc68c43bfa25ef94c671278853c5087c2288a1e60fe11849a25967f9303d4170
    8d1ecf45b79c77f6230a4bd41619f3b2c72c90b7f35801a3c4371c5c0ad3edc7575bd5720b78c36e
    1a3f10d038ccc9d5cd44e996574daf98e00deb06cc03d7b53f5767b3c191f65d9cdaca249c51677f
    96562c76450542620db9ed659fc2fd8cce0d21246934fbbed23122bfddd9799ece8b9821e4ab43d1
    77f0f08a1e1e9aff717b3d76b23c72b8495bb2caa54d9d86030203010001
  "
  );
  const CRYPTED_KEY: &[u8] = &hex!(
    "
    6dbc5306f8faf060ee49d9dabbee97ea09ea99279d2a4115de5daea89dc45737865bda0274365cee
    6097c5af368dd5e490ef9f493aa9d855d38f7d2e44133b8fb386168c1d968ca0446dbc82343445d3
    5ac9f2fb56da120b503c78bdaa1099fba12255df1005fa60124337de29fb5363ebfd52721e67b5ec
    b9a4e1327e129e2b2a34b5154caf45fc8306cea79a1cfbac03e33e3631782f31e8b36ea96948fad0
    a5473b33b5bb6247e8ed73c023b41bf2f262ffa3c4bf236043341d7b4878d14bddec3e795b98d3e0
    29f6de45b9d658c9b4e93192f4e0f3136409730f57ff6b615a600a38a9d477c2c58b7c77b0a0637b
    7b3f61edd9f09527af98accc4d4d831e9072c52414c6e4924fc24fc31ac4cdf5ece308611d9ef956
    d89a10621838d9bf0aff796196b9422a04495d7e97dc7a707cda48697d5c847cab85c1144304ae2d
    26def8753321ff84378e9b9b87b9f533e55bc371e5e27412ee01eb47ca267f380d8ecce761a73d4c
    8a1334cfea13740b95c935fc10166ced3bc9b2a2bbfe17b7db2a0f8f56a71d7be4d1da83285fc6f9
    b2019f35c8ea7cc25a5c9207935008fcdfc80127533079d0d15cf53dd750acc41928bceafa2999c3
    2e933852ece6cde042850bc1f084f10cc6eba8ee54b0a0491aeff6c788598b63186d4e0e44626db0
    8d90642c9e10de23955c4c7f62c89f0c0d7e83a25c5bb506221e0f9d3eb8c1b028a6d2be4758e0f2
    b2464e3928423aaf5ef4eaa2fd8dbb98a3adfbb13f61f93f61de623e2282edcbc8d05a2fa0026513
    4f9d7993e73331b47ee49e923e6db38d2822110d490f28bddf96511334b67e8879182d66fa803ced
    9473bf96309439cce4d7b758c43fe3bacc5566956303781ce2f07c58ab45de18eb853db785fbf7ce
    9525b6020d06332e0b4b674b16e70abc4176f9432c694330a3dd769086da1f9d71d3fa3da4dc4dff
    be3c99719a6d0460c393a20f5092a9240f2ba69a7f122af28f0e2540d149ec244d5b178cd47cd9d6
    cad7e8749385ae089e76fca61311954b8beda5ba08ed10ecc3e1377e899ce9817e1d49dfb1f6e561
    4171f5a8792a7069d3d1b955943fdba946d50d36529d83ab712fc3283008564164c4be18d628fd9d
    71eebe5b950e443ff6ad1075c6edd62f4a17cb165f53db12dc2f59d37965220912ee4c1f70b59296
    31ff5b6e943705cb3db1ab8c3e80ed20f0375d4f7565bd12a2be59e505a3b47a8bb5f854d0955a6f
    aae71b283de2f8a36e1a3b3c4124f0b3b33c8c20ce238131045c904f90066b1470885d8f17ea2ff2
    082aab98ea15098e9f00e1ac3632efcff8732662ff4f48b70fdf7f0969aa30e4aa9a2cb7a1fb022a
    98d2b0e91737a3c5c8df847da9a180b3869ca705c20af325ec685802b529d69338c6dec066540c04
    4e2640048fc50c6ffdbb36c8db29ca8dc0f5664d30c3fe6456373efb469bba41a2772883a5cc254c
    300f4067cb57bcf1011e631fe73b7c33afd500f017afbd3a8dc779d6e48a913625c7259b72b659e3
    3b8463068b6dc683059ec4fcf09f49fba5c6e305ecd5bf855bde0924e94123e87444081b08451e22
    78a7065148c8fc5d401b9fea1be319f4075e269572e8b767208c36bf708dd09cc31f289b8a6e7da1
    8fc4adcfa3c6c62a8e068d60da0da9b3d89b5276f12b33f28b00e22ded3e95237de5d3f0424ee533
    8ea9ae64f637239260d39255012315b2ef7685706fa769e06211bf62ecb1bfd00e35528322254ca8
    77a7d755ce65f44cdc551aa8b9df07fb8e2c51d76ab115005843fc8cc90fe54fc761a41a1534c560
    f8de2fb3a11eeed6b068c5ed26bcb4f45c3fbb54eb007b238a26f282a2a48320431fce00070387e8
    8037814898960c80b2f02872a82b31d2d526c26e8bc2eeaaf7377b91dc6b8f3e65c650a1bdc0154e
    861aabc9b5b2778f3c857579ef4035d47044f781ae5a3a5cbd229e3c87918d260ecaebc7a958f254
    70073cf0159f43025f0ae6977bdce0cfc24d1202e39ac2b247e3052a5335611930c86c8f06476e61
    8cac43214fdbb0e7990e79518ea7071cec5fa89dccb2ea2d3f199b65dcedc2093f2e35eddefd22f6
    11811854cb00fca18f7e299440ead2375b65e7499f6f72567d58f4f8a21f61d6733173dc34c7249d
    50de0fc52f2bff627791171a6f826660295188e7cd7b2a034858652b2914b6a380327935a7e5094c
    e4d55d9715f4f0335a6f4424078b3a99c58f10f79ae364e60b1e90d6c382e390bf720a9b8f43d72b
    7cd54cb57c9c871cf41991fdabe47d192babe3942f9415312858ed81b1386acb6076883925ff62e7
    fb78eec9511d33605c507b1a3bf699dbbf66dfe782e6f320bdcae01584d1b87d27a387e8b564481d
    a09839583282af1201bb2b8f67f72806a089e10ae57d4531e04f459facf9264113ec7d11e107d116
    41880bd1c03134c9dd08dec8e83b625e0390f486c4cf39e93089139d0740ee91c83362c38d970769
    7ad8e7ff23020ed4740e656dc5c92170627d811b2bbc5242c1c909b965bc433b92261ac7bb71afbe
    060458fbd483afd007f1f7f12cd7db1e61e6974c452cf998ff513394d9bcd9a245e4b4fbc50942dc
    827d4d0c8a6979e9c14a0ba7fdd28ee51e7937d041759aed63bb27295ebdbe1f250bc76fc4711883
    336efea0ba5d999d7e91a0f29b7cb74e211891c35ed1cdf1f5279406306fef73127f53bb01fb7dba
    9eeb48590abcd9503212aa876ebe55031f51a4afc0f41d89a60aebd6b2847f6b0e81b2385a7b30e9
    40c14ca46a63e4886be3c346a98f239b587688574a8b0fad62462012dff86ffab45580f3461a3c5c
    7eb688106731bdb57813f96b8a8d8a46656823a04d2dcb5541ea491e4fa3eb20849b478937a74344
    8052793d2dd5849a74ea017802505942fe3a9808457878431c8234cdf2e875fae49ea302780abfd3
    a36ed5d06090dd7896b1d7ff6cd4004019727e6e1fc0010c47cabb75085240698a95e42de45bf571
    0b08f2db932c1278393027a34a8ecbe1703a1d6ca213bf5dae09cc73e62929ad0d2be7b45b3ec6c3
    ea291f6e9f1268af8c77af156e956b28a36b3ad127fc0d217ea38de67b21389e5908428e784ef656
    b063c05b8fd88f89b0df3c802cbc093b263686d64ebe59cc86af240f0a5a273e1c3950b702754993
    1b9f429c498cfcd86ec3e4a41007694e77ffc695f533e8106ec883b1ad82686b1cf2754c220025aa
    a57622ca78751ea0cfa0ae2f7dbbf1da1f78f41ca0f25a83f239899fc1365cd5140f612b1e7efa3a
    d9b830fe13ffab78a40ac4305a57a387b0380c7997cf2a44d9bc380e5a1b78b170a30fd3fb4ecbe6
    c95db24039
  "
  );
  const MESSAGES: &[&[u8]] = &[
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000883895210061acb85b869ca100000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    aeda7c2c45acfa0286a5f4a55f1df9faa030a2aa686b79b3ba1e3e7a93b2b711854055f0ae9454bf
    06b2c78493d8b0a07612e3efb5b448c5097600843585e745f382f625460311866628c1451c3dabb4
    928c3add5a13a547d1bc1f337a671857241105093c1d1e685bf260ee5fbb4f950e64f52fabe8a3c7
    96bbfb8d1e0b0af480d3e90f5262204d115779fdcd425ea4793a9c316f43df8aadebfd0f71687d8c
    5d4ef64bbe257ba5e263b6785e599b16d630bef8f398943a8ed4949bee38b7b74b124e6de5d7c19a
    2d76c160ecbbd15bcb15ea79c9e754d3eeb03d8f3740eee24d9803b83b487e1fe7794a7cfb7bc4d3
    c0fc7ca5ea51e3480a5eff9c81870ae30cbd595369eaf2156703d4310682ed7f806c1c14018953e8
    96aba290522cf1b6f4d16a525c253a5cf6c1e0098c8921dc9e6f87ddf5ac3d01b2d13d07f07e2393
    1b5a38194df40111491074308d2621acddc48e42ded5203b22727e572203df04903957f171801822
    0d25bc1d004f67062416b0cf9b884b7efefb4c605a633a52b5f1cfc1691020b0a1c66082a92d6911
    7aa47494c0386e4773371b026ae42543ace440973302ba2aabf30b5cff0c385ab36915721427b26b
    54071d712112f93936c9c35b8af4625e4f3e89e00edd280ceff449de93096a4441d3d4e549b7b220
    19d6c2487f6f121f724f3fc9e51d5cff96db7d508ca2e20307598e13640625a745cf7b6c6f67fbdc
    55ce1f98ae3fceb2d275d4e247c7c8796e4529dcae000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000ee802b39c2781055f9bfa85200000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    455533253dcb79246495edbbb57fe2e61bda76edf94c43a63ebfa764ee0a5e5f126c304cfcd4ccc7
    49e1df1b2993ee408d5fcb69ffde11f6670d2c08b7c5c3466c7e956c6a30a93967970a9fef8b19a0
    b833b99f0c0a911b159e56af823220e3a45226c96932343745ece7415dfe3d34803722d4e25769d4
    e01e98707a52fa8927f4f82ad2fd56adec66714cacac51d75831faf4b03cbece6d82afedfafa05c9
    216c747191af359e3c797a8c7ee1049d1546db160262ba2f5a932c41f14659102bb78d0c29d7ac9e
    66667ccedca1fa7e273c02400d20f05f6cd88e083c32e3c9a2af9484b5c23f26fd9e39807ebcba31
    7e09e50be3430d04669a237f51dd74554d3906dbff9ef7b2786e3e21a1012e23231f4c9c98e202f3
    9853f83817c45da150bcff141b3e525a1a8abef9e58ba8ed4d116efd0e1b82745d1ab7e64bdf5dca
    8b1d9a2cb9f0e15ba5e7ba7e535a9f3b5c1f151b572b2424be6c7e5a8e2a5d7d509c51dda6c7726d
    0ecd22e2ab93538259bfd072c5c79266e0ace7a1cf11ecf796790b650600a289acbec43d2e3f813b
    7212184b01d9cd9b5e28656ea2ec4621f2f06b19aa18d96cfd20b381b70aa83a56e85a3f508a481b
    7de03ae7759a95d16e2966bb78a0bc1a4a5b6163417b7342063e315ef099cdb5fa145719955dd21f
    04e0486c24d1a8405b2def65b90bf95705b7757376f9e787fceba667f1fc9a0ddce3932951a06681
    082d4a8d43f34672a769d7453b4d19e18c3cadda35000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000e2b04c9d308def5ed92ec35300000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    335a8a464db7efafd074eb5d9bea5d6d8fa2a2550bbfaf333ff0fae85c7e831a6fc04d261f1c53d3
    9e792c23a15b93121a686b21d2c1c4ab510db5e8c5e63c2e50dcf80a955b5fec67710448c2ffcdd0
    0c22a6d2173765ed5dd375b1423c62f9ac438eba5bad763c272d07c9f8f0911ef15a115e7d2313cf
    54e48b7d2575ed03eaf5f2225694b356f877ef51e65d807f329a416c9a097e3a5cedc9657e806fa0
    89e861aacdaed6072dfa7c91a110095c8aff06bae1a519b0d6be792efe9957dfc26d8343a7c288cb
    fa209b73d5887b6927ba60954f4a66bb11cfd1fd6d9fdfe50d6ba1b464a3458e3ef2ea0aae9df089
    98df623fd07d6f2259c60f4a331e495b5827185c6ae4e2fa8f147a40717dc0041f50ab8a1d296c34
    6155bcba5f4b3d64a11b513057c7a5ea491c41524343fcf6108cbfff5d62ae223e29c73ef121cf6d
    f6d0d737d3a91bcbf0fb634d6e2d1a7004636ea4e5ebb9bf0c37330046308f2751d310ccc482221d
    700a57d3b1653cda8a48c35072c6f072984dc02de9a698419c31754d3635845fc14043050f6eb41a
    2c782fdf68a3b14e7e96ade6ee313c323f7eeb187467fa5c3f21e67d70e33d5c65a2a23634911a05
    03b3def6f5b8a5933a03afed47b3a4594025a83827488dfedf33deda660ce6ab63a83e6d592a39f6
    d4db20930bcb868dc16e51962911bdeb0a087bb2eee76d19ef31dfbbaca82babc1f9a590935e7e69
    683d65dd9a87f4e2cec1958527e6c153a72758591c000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000afaf5dc16b51d348c0aa43de00000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    a6600c008c2b5d4ad670f8558df769bafafb546c4100781247674c8758adda467b5b713264d262b6
    717e0e176da0feb9054a1f12e0321c293bea0f357c3bf0906d59a5d6020540abdeb403da24ecfe8b
    4bdcc73e9a7bf14386f10bd2283000a9b55c5689f6b33bd839e69e92b927b2dc101d331af0e92c96
    607606133e20c53d3819fe627a091a81cfe38c10fe824822ed9892d5cf48dfa91bf26cbaaddd5a4d
    0da94608bb7a43578f81a9d0db75c7e530bbd844ec74b63a934d50a09630e81cc7475332b3407183
    ca9cb1679cf44b2c7bd5ea3566f03034c65caafbe7ec31d84bd487ffa78ab109324ecc2c3deb880b
    1082fdfe0baac4ced4a88e58777fa0371bb5fcc1e7fc58560f81bcc54ff8f437c3cfe12efd80db34
    909b1c61dc67bdad58a92ef90015aed6d64faa0245c5e8e78de88d67921e29ee2e213c1316ad0a6f
    d698491414e5b3559468a578e62558416b87fcdece13f546412a26734437ae0c7f8e5ba784d3636b
    4abfe1f53cf10ca90dae0b24ed08ebb2ff9078a3466321ea4449b6974d1af500a4eb39b918b23e97
    d61cf4bf68725e59d183279adb0f2de5d0ed443e058b46434fd11cb53a9c064d263e8d47c2c995f2
    581979e51a212f9bc9ae04183f127afe0a4943cd0a7532b8c680b657f522dd56438f25e33627b38b
    ba88520539ec2852959b2f60387b54bc537c0443ba862e024c1b958acb02baa9473a64ee438c0ebc
    f7d327d83169cb2cd57acdda3717c7220259865b9e000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000ae9996275972c61c9c30a3f000000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    7d838712f1d82ba9b19b97c889b960605baf06452215314ef0b0a6664e5866da0d6a970f7daf8dc1
    f08e77e8145b0a371b9f9d2f0eb303a1285ce52766a004285eddff3658bf3c2748dd97ce6ddde511
    60d7b3bc5a27dbf77b74ade8f484fc05328f2e435b951b52f981d09d3279fedd5e0b08558cee826d
    91a8b86c3b37b5e82049f5a0219d586d1b819b7b0b25ad14f1b389fed302c3f8d74c8326f879917c
    c1503da503454a319aa47fd3e22a3a3aafefcff2ba1ae1117b962b0215a4772d3a7a41e16022bc47
    8fc2b2213e2e91ee5393bf8e6cb640c7bbc4a507e9c706f3e4e5dbf11fb2372d78a4d27d231de546
    7ea2915b1dc313d94fef3c80c0c2643b05bf05fb1fefe08f011f44bd9f9c61fc14f0864da0b666bb
    c950165e5d1703ec1343abe3bdb01a64aae8ed2fe85f785b8c226e88b011b115dea7fadc1b52dd25
    abdb597ac18dc131d0f2dabacec55033d1713ad1106b996d449dfefb4c01c0cc1c3db34ab3e40850
    50abf78fec8401b0b541eb48d2c63bac94cc540e859158bf79cdf0419a9fdc91f7eb754e3e64bcd3
    4a3cb2af831e9aa1e8758de1d8188ce605779244884282908079f97e13ee843fd6f8fa2fe50ad2d8
    4ebe998ee85df8da852faac8aa4def9fb0604e2b44f08034393d328d49aecf83a674bad792ae68de
    982b345bcdf2d8e8444c57bae4d88d75d8b93449c0f64b59d3777eb326a44901da2b3d80cb13ce8b
    d4a6e8077c8f1c75d98147a31e144bc352416a8857000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000631d84b62110fea263b45df100000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    a4e3a33da3ccdc326ae1b6e1cafe95f6494056ca826162309192f541d3a145a5e89e1b5050630be4
    62cd1c60214e701359ede621f6686ebde4cc3140a5ca3512f0ff7893761d334fabfa7f27a1cabe67
    95bdb3a53b8ee299ae7843089051ed6a98f0bd89667b4a10d9220a04c10c5bf7ca4facfd25adc717
    e63c5331ca1ee5ad9ed67dc565747e0c4bc77caa64c3331d559c775f1aec8ece2915f5eb41c5d609
    eee65932636ba0e3f9986e9b248e8101e55936e7360fb8c4f1b793816cc12e0a1841766b7a2ca90a
    9123675bb0f6029a62ec67b1e5349f0c779bd4b5d7283ac07a68bb1a8d5b4aa227b1c6f2f48ea8ed
    eef254f10bafe674898a0bb41e858c3ace837a01447d1f8c4ce526b78acc32f1f3122b857293d358
    905d00ef328e6986ab6ee481da58d5a7602686efff226925dd95e034ca47e6128c8da51ec21301da
    23afc811a3d6085c476b101b3d2da14a8857d4ceca13175d97ab74431abff9fbda7ac1320bada29b
    e58dafe54faca46051a181f9ce8e0c14efcd1363fd22fa5ccf9d81074702e4c8d25ce76939a7d4d2
    960c4e486493bc40b8d2100543249d9fa31c3727f94e8c9ce75e135cb9956535743cfaa478c7b1fb
    f55e20d5ecd47a9668ff61052ddc193c687a505bd96468fbf1b3df80d97dfeacb2fd074ece8007a2
    21e49fb99bf1e9f626c66fd0c4e58e60655297419ea8a0e7dcaadcbd3972aebe2ce16772e7f5e928
    887d17d34312867b0580f728ee2c7ebce3ec709522000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    0000000000000000050000006200000009000000170000002b79193245c6a528abf6851400000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    5efe823052bd570c6073091b5fa9c61c81d983c2501989674a8c401f449ea613425be7d93360d852
    b6dd023d70afe42ffc10d178737899453664295e2a2e206f06611c503957009dcdfb17eff1d11758
    bb4f93c085c7c70d5a94439fdbd2bb6dd59c640b2751e19f63e9aa6b6603d9e0e3e869867c7374f0
    66b276db7c04933c2f329b550c7ba07b739c2a6b8336a881ea141dd2811e6b028de1efe46b4ae5b6
    d0393e51008b76a9ed40d47057e68735231b0d770045dfedacc3400c828887635aa677c3165c2156
    3b91ca7adddb896dc1ccb724772a7f6026384d6530df02bd4388e42dce747fbdfbf6e1e859b1f772
    8670bba3671d51cae83a3b32c225fe69991da32b5342d57e30edfbb8b99b36f6e439ec9d86374bad
    fdeca1d5f014c16e8b0de5d73ed6fd405508c0f3f67f6ea31a88295f565e835afdd7fd5fa4dbecd5
    8a571fec105bf8e94b6f2634d33fa99bb7c410141517bbea0c56170378daed5b5be903b19e665304
    4ffdd968ec17c7b6fcedd34bcfbc8bafd2474cd438481121626f77012c2a02e6012591b9fe40aed1
    8d69e89a491184f067346563695c5d631eb2798f4bd80f57c2c0311e91b072a3092808eae6ee25b9
    d89a8cf590a206183280e0b814edb98d538a5968c438965d421b668e5869dfe72faed4a33ef1642a
    3a67e5b42b17f4890608df7ff2883d83ca13583e81f27e40b8e4eb2adce8b09a1a780733131e9f7d
    9edf9685f5132a851c016788cbe5c01d227c01b071000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    0000000000000000050000006200000009000000170000007e2a8404d16ce6787e15639b00000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    869b64c17e59ecf321fdc29796b6163a37fb9c1190dc3338ea5e96e8ba5973426cafb0302fcb775f
    98e3022204c387fb7dbd1c5ad08e61bdc82134244bd0be81866c84d39aff94fd2a0aa3b8c4da1c77
    dddae13f181ebe1b2aefac5ece0d4f311ba4c906e01b6e894ea37d0d1db05a52f58c7ae44184a134
    8fd904dc8409f64155b3ce0a9255728e5d040a9eb7ff46ed2307cae368731803aede3aee44e77c18
    b663ac8827818c3cd011c34bee18d9cf0869ff8846acf28d218205c02258c7aeafdbb6e62f70e6dd
    56778ae8de4eee9415ae9fb5f3efaba489a8673d02630df2bce0204419ecb031b963be8f88aa2dde
    550bafb4b384139bebce87bf1df43a5ccbfcd5545395f876cc4013c2fc01e93621587f1b9f7a9354
    1d1813ad1c1cdef36e2efffd758496c8cad258cf3d31ab1d522b97a587d9f58ba2460a6edb0cc910
    ce8fc4ebd80f82c9a8829ddde51ef01ce8f62fead7db2b80a6aa65cdec965b3b0e5af4a75a66df36
    ddfd7b75e2d35b3e58994aad006b5de608a030f2a3dabe330670e99d8164a962b05a1df289a77705
    9c1379094962ecec41a98d3122dad852619002dc0a01b7e4f60488e719d48d6a0b898579f92438d4
    a178e49156bd71fd3ea94de7f72459c3c19311469bc79cc916fb031d7000a220597471033fb95078
    8d37ab92a1fcd5229e7937f3734f2cb4a83501d6eeb2777e324d727f96f465091d047e154042cf28
    f139280dfbfedef0c28e8cd1637425feeedd88f732000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    0000000000000000050000006200000009000000170000001ffc93069975b5c17552197000000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    2d9e6f31914d3154473ec4d42b1f9da350c1a654948118f0396d8d6ff0e59ea4b1c0d43ee8f5a389
    7678016d6f863ecca79c7d460e8e27eb04a02bc418fea6555affc9e025357c03e3815534f640d48d
    90f25a7ed21e8cd3a2e8759bef4ce0e476a6f305298647f75ae6911134b344e61f49f9f4923da5f5
    744c4e26cb783d45688b66ed78bc7a6224b054c8e2ef8b5d8d3c54ea58aa57ea119aa12a12859e54
    a343c4e742cc4cf424efa11b8b7b780761f10f907e46002735cd0de5ca0a47dfad283385df514274
    3faf2ab6058d9949e608f9077fa11bd716f762c68a1ce6de658bcd3af1e4f70e3ccd124f67471e30
    5857246826187a60212c289ce5092647a4ef839eb6099e7ad3063e7405e4b3cd5e4755cdf4ed730a
    815c2835e9f85ac862208896228c5d8d9632c16aeae84975714e28e437b708ba65f0a94534625e73
    fac9da4ab6d36f692a7b29413a31d1a9bd38613cc85b7b85b153342cfda143381bdc28b745035b0f
    76f754ae6acb47473d652e08046ccb1478c70b11d13546a4b2d7eebc8f67aebd1fd3fadad2992524
    ea07d7dcb3d5875426f930592490277e15ebc47079bf9b2354de0e5ac2811604b3f7de0337b562f0
    b8157ff658a72d71dbd8b7564a6525afaa638d0cb2e20701b9179621cf74b6738df5b10cb85b6967
    c2937389051ad55e408f06aca5fc08086e47d1be1ce8a7b29fda4c7271a9b252af5e7f8b6a427734
    6a33205f326d246abd4d5550a3344822e428e1f6a3000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000ec69156b5c34b72ded3148ec00000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    a71850a5efc3bbbad0d356cae74fcb7deb428a6a754af28b49ba1b344c34449bfad1476a9737c1de
    8ddb5abc49918286eea3199eeb615b8d6bc76c66c8185707b9d43472989ef87c0fc667761a2ec31c
    55c563eea1b7a45a5676b5213b06bc5d41e14ffb87b3c86bc40a77553d3d0b65264444c7b6b089fc
    a009d8347cd0351e6d73b836af8f86c412a1d7ce4eb6135b1474eb47b9cca4f9418b7d22e3814b19
    71b1d2c0497c3f122b3a40cb30f9c471ce1cbe0cba2ddd6a6c06f7064d0eb5df2fe0792f1c5b1887
    f979e71a53b3b2fade8ad685d428f29dee5fac70f2b12f0523570120433b23f8675eae72d8d9fbd2
    f90e88d0e59d7b9c0dd7c418c6ce012f993422ed5764b7e3638804183dbd3fac1ea32a323296a023
    a3b567142c8f8d2efd8345250c5648d48961bdb652eea609acae2293f8ef6dedca09495e405a5eb2
    9938ed830744c925791bd33597c27b57a598d019e533bfa38f3d58249cf1a47106097f50d8b0b980
    be5d8f6fa719014d1ce8a62f355a167d185c08c9e6057499f8775bd8c9f7621deaeac176bbe0625e
    fa4603ef93a4d54e4b767dbcbde5a907dd70bb092acf011c2d9e47bfff9200c7ff564321e6e2c5f4
    e70547d45a79b77e9989207150bc4fddf3f330135cd087006b74cf505d7ced9c59e7eea7832aa7e2
    3c8e5cc48582d49ed1bea7e2d74eba18e7d4cefbbc03c5b7106ac6ba8299779a7e078393eb68f81e
    01487c2e30bb6bc86deeae692a132b20afd7c87193000000
  "
    ),
  ];

  #[test]
  #[cfg(feature = "openssl")]
  fn test_openssl_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::OPEN_SSL_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      Some(PUBLIC_KEY),
      CRYPTED_KEY,
      MESSAGES,
    );
  }

  #[test]
  #[cfg(feature = "rust_crypto")]
  fn test_rust_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::RUST_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      Some(PUBLIC_KEY),
      CRYPTED_KEY,
      MESSAGES,
    );
  }
}

mod rsa2 {
  // This was generated with rust-crypto rsa

  const SEAL_NONCE: &[u8] = &hex!("ec52605bb40e51c144782320");
  const SEAL_KEY: &[u8] = &hex!("3ae9d60336c318d2da12c088cd89831b30115bf2e3a18b878633471b094677c5");
  const CRYPTED_KEY: &[u8] = &hex!(
    "
      3e3da1a861baa34cce0e97e497bdf4dc80cc01eeacef65eb9f1da709211b7a3a4db495a5b60de937
      c1e94d5f8ce8270f11dc82d88ee2a3f843f63d81cb3b20811ac1cc4741e3548a939605f261a2e8f3
      c3b5b66e2ddb0576de228259188f1b2a10e697e485a8ef60e606f85225d541a1cd9c9c16430bf386
      14ac316d80613f16fa14cb56b7db863919297834edc42439389c198c00123d825dd3b7d4687c55d4
      c3b11f604ecf5a8cb96026aecfdfc789ff43f808f52a08826b7d3ecadcc573579742f42b929546e2
      59710c3acce83850bee5a5fe161c0985066c7bbcd98d8c19c356fb16938d7161b3450a3994b3977a
      87792aa1e77947d5cee6000e5c85b4158bc70e43a6e1cb42e6d022d31a252bb669672f430f640a74
      51c06edc5d685fc2656e2a65e58699f5474f8b4f6d1d5491b1dc481a8f4119bc269f0d736ccecfb6
      a9e03e9d3a65de76b2dca47188a921feaa0a7635e69715bc70dd3809bae56d6a6cffca938523d3ab
      7f26fff06b82ee9df10b22a97839065f9b7f0fe7f15abe5834b7293218d38ab9cc9e2c63b1058070
      76939cd2912988f4e934c145f3e763f521e05fa7c7d5a06b723d87dd0c7225a926d73ab1f9142cff
      b1f6eb396fa4b1a07096bc10bcf640c34eae8e350f02ef5be3367fe4e415fa593706d1a0fd753106
      fe418cdd2c4cb7bba1c5d47b7411b3aafd2a4e9465cd96759692857763bc222c38f237b6491b4208
      146acd4681315d041a0389a12c445a6b90bcf5d98c1a8836ae99986c6b2ac5d59a6a8c6335caf15e
      d25fbac91dafa0619ba046fc0658f765591d7d29ccf60356d7f9352616e466445406239b6277c48d
      a5126b2c0b94f07a96d21241ac0faa854a9545b8862b81b09e29f660640ce75e0913d81a1355bbaf
      537e7be82035c4565278e73e799159196863776e28616432f2f4560faa962ed275ae5939c35c80b3
      1bdb58c10d6aa4330dca34d69ede390cbbdc5abcbaa12d3697a632176b9e7dd91a3ae14ddf7ed062
      3b293821aa8d7add8174a1ad59ec1f390f901837e5860948c1c659e76cc0a10d38bf0f7600d54545
      ffb306ebdaaba7dbd671d34fa824a3c948de160fee3d7555be77c7edc7d818c449cd2f66420940ef
      dc1a10e931181302ba7a943880a57bad1f0169732a4fee319af336dc7072eb00b393fb141a53fe23
      31277070efe78494f3cbee4ff1d427f280442f721866f9005401faf2e1ffbccbd238aba9c2e1a7e8
      22f636d2626f4566814c100c639da9c6ec6bd2844bfd99a5e287fbbf2f6f51d3c350788e70a4ca2d
      23db72943d4201c33d763e0821bc06a8257eff058e077b3f6d4058589c1ce58bdb72a442193a78b8
      9cb39fbf45434ea96cda0ac934e317a8bec7a65c78350fe1e0c056a08def56022c8b06a6e274efd6
      88dade8ef76053f585703eef22f9fdcb6b30aa458bfec5d53ad2a03336d6bc9050e70b7245228ddd
      6e12795f0049f134cdd2b2c0e6da2216be1c87f1da7c46c4471cd137c771cefda232c2598e4878b0
      1cd87af33553108e6b48de805bbe97f0f74298e6bcc4dafdd7f55b0de771ec53cdf30f54718728f3
      fa09f0d7361e84b76289d556b2610f2b655853018d84e652f2b0e39560fd5cae53fdec345a6b85bf
      4702148f1abf18960a439f2e75022bb993a49cb5bea9e52db74d33fb3d2f04cf1402876359be5c56
      97a2ea1e178f40a6277bb75b6fa8bb513af7e33c1a296ac4ebb1bc1a6e7963ac7bf9beacf7f2b21a
      b8be933863c83bc1765ce0731c4e10ff597c6cfff6798c02acf0333fade2ca2e5f7619349b3082a1
      d3e21f8aec6cd11a09dba7353927df9ebd4fa868368e31b3ce44c3fe34bab894208c177b741b8859
      0d51bd4d6af4d9163d7c35d72e3dc59681304288b229171befcc3f6a7e092324b0bcaf1d9c8396c2
      944a6eab2f76e7a04a91fc9fa9d2820ca0d0ef7b33dbbca7b3fdb60ceb1eec8f2e3cd2ba89c61217
      04cfcdaf64096b845b26b5594e3af6ee6839f718a57cdcf77e2b439217d4d81426761fcb9a908dcf
      426f3f8d356abfae78dd2312700d8f75f00197e1446ca6aa9a7eaa45405482c860af55c866e1dd73
      7c5cb651710556435252ac71b8e9494bf96438153b343704657adf83bd5f6a95a8fb56593fb0b2ee
      38f51b55eac21f06d1878f1498368fcac0d2bf9033645ee5bda7bbdf054e78df6fc0dba876819486
      6815f448090e75ace663e62e5ec64ba93e4d2a92c04758a88e4f3be6caf6f04e21cbb66287cc4485
      7e30c9afa4359f26f5ec701539585aa38931e845fbabb4784269c5ec222e070bd11094d3c7ff51e1
      87efd2718b4f04f10b0fedf2358d3827760b83d31c08ad6bbb4831c961167817ed8f95873ebe88fa
      e4f0ed14dca2a7429cc790ea7494488c83e6c9160a86070098a8ea28af5306db2f19053c5cc5daaa
      d51ce20f82efafb61979d7c2f62c92d7f52b22983df0179b04f5a0690e8e5e080b85bf19cad7e4c3
      ecdb92b380bcc9699c55e91ccd691f9a05e7b2b3329b22903b08914eef13003945970288563966a7
      25467fef776dc650fd9804fdc4f7af9bd476d6330bf4dc8da596cfd6a89884279ff0fe63e6318358
      ebffffcf2e6ffac712aecd533b0e9edfeb9a70c1b35442f6f8dda15fd61de21bfd5377ddd6026fd7
      1985b2a43024099e906c28153cf70492aba2f54b522618ea6f0529bddd70e621a691d7f83081cb06
      b3e364ff3847ab60253a970e673bdca99d822f1cc4d779fe3553e6c2c3ef8b42bb8c38a0c77508e6
      94a4502acd0935526b10b94c500d3003b877a77da5bae8c5173c9ef022cda25be42edeb2e4157880
      1bad1f7b656f98fbfbac118e5781ddfd69cf88752905fb2ef4b9f18938614db8ece201f2d2dbf186
      a1c22fe91b27ec12371b0f345d733c27411284b1eb7508a0cbdad62ded94ca1f9ddcfd0f0ef1b6fc
      d0f2ba35350e5b17add1ab17ba516d712da21bdcfc1f2e85b16f1ca6a10aef20669e268a9c307ebe
      4ba63bf8e34bff7aa18c5dc7caf58dfd7a37655abff14a883c1e793d3aa83278224b7faa7ad35b06
      e8dab532c0fe74d5fd85fcda2a883023821116a1449db4dfc98a8c1dfeb5fe7d5e574282805eef62
      a86f536a0b9f669ae534b04adccccbbfcc35c2c7fe0b6ceef100caf588c9265beec2dbe6c852a115
      227c47e9bba6a83359f1edd33155fdf9d340a19bdec7d897cd9241907d3e6ca7e3039085a2e3c342
      e135eb12a34f5590b6543b8614fef6e2a290be4358ebba3aff6ea2bbe60e46eac776de992ba361d8
      6d8a7fb3bdb13dce1f299ab5248707ddc17828e6b3998385c863649d0e6039d6ec303ef9bae9ee15
      7600da
    "
  );
  const MESSAGES: &[&[u8]] = &[
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000c2a19c60aa5abe627b43297b00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      759f120dab5a47b64a77933de43a102770e3c170a9ea8fc9144c03df419a6ac9f4db11b2d066a8c7
      01e49f45fa33d7a1bf8bcd15a9712d58c6cef6b77b25ad20bd6ebe4b7d253d1305848d4b1828c659
      a436ed7a66c195d949b9cb7b85910a39d94100013e6582701c903eb09405c082cc25057da846fb98
      20f5b2aa499e64f5d93f20d1f17f463cfb92dfd39593d1485b018102c47f42d06b1fdcae3318ac9c
      eba4233faadca93801bc2b1621548eefae5b2960e187348f88e227eb9ab80dad3c427b1ea7706456
      f50ec6e483e20839bbc0b899d715b3c6641275f878ea0be1a7931a61a3f5602bccd75b089bd8c635
      3e79989df9f466d2086806060318ec08901a7ddabcd3161bddc5e3b298d7e5570fe2ee72500130aa
      770389a216d04d07cc8fc05fe96a6074046bcf49f357023b29d8151c5f3e73ab17b3430144c799e9
      ba0a6cc283af72dc2e84762eadebc9b5006e1b15a79bc8622c4b0eef7de1585fef5c95d0b069988d
      3d832291be51c5159e32778500fed9508552d3f8f4317c1a2b3875f8854bf79e375e2f02700b71e0
      da6c02275bb5ff9691813ef297e58a0f22668c780c54feb0b67b46767c443ae5567a556cb393accb
      6358872435a2c29d8648e4c84fffe5a869698e39ac6bf3cb7d905a12206d530391e2b1c89be4bb4b
      fb4c6453d6ef3f71d46b3cade45fd8af48f4c6a741f698100c420e3a951e6a8f63f96511fc875bfb
      1ade04734b65c0b5ac846727411344d7055a956bf7000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      0000000000000000050000006200000009000000170000006e89ed9ef60c4bfc9dd89afb00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      1d2d9b761f0b392eec6399ab9ab96bfb17c911b266fa877bbf58897543bc1874494becc91d33b682
      469b3939b699580fa932cb42f6c45bc732d9dd96172f105280a22795d12a3d46dc33b446cadb115c
      15e07ac0f7cd6059a7836a4b633cc9a17b401342bc6bd19f8486a6f1e369375512be7da6582f0b06
      b28bad5836277dade39876c8623303aadf72aa046e9592e8d2132ce297f0047f3973f75f701cea7c
      4b89e05aaaaf5dc2ddf838b691695eed75c75e9b02072a9fe8bbf2c91a2c1c0ee7968edcb84d6723
      219155eb15fcb8a067a1808e02ecac24fe88f9fb363a3377b2089eaa7d768fdf94565350a9ff106e
      10fe3cd3a929de904ad4d7170c947dc10a885a95090bf1515cca91b5db6c11aeb1fca9d01ead1af1
      323301a5eae81d235754571d73ecc4c57c0d039b2efbaff0ac212547eefcede71a9fc3b0d63755dc
      4cb92a927c99aee9d0e1fe57dca394907ddf0d2225b0901d9c493dc5886ad7f083c3f165e38cca93
      f10c49bcbc8a3c7a0129f44d76fa2e0af29c02196605ea469c11943912dcd33f8a2d5953c975d469
      4f87177f76a7f7124be70cc262fe39154826cfc0242c8aadd0a594bbbc9630e8dd47a1afc4da551f
      203e5927bc3435d19238d42dd7d276b2bfb9b4bc40fe6959c3ca083e2a0025873ab363c4d3131f64
      01e2f6c318c5663ddfa1980b8e748de36c470357cdc04237238cf8dea22797b71c4146ef52a8a9ab
      5e17b0e5775b69ea25805a055ec3a4914ec1489679000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      00000000000000000500000062000000090000001700000039324c6f967f582daaabafe700000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      538869e0f12be3a1a7bda77b812022b7a02854749695fe0c0d4e127be690798f77fbb2baaf398b54
      d4023278502651b8c319106cfef706e77dfdb07f556b167525f437bac9aebd4c4a42e432360175f7
      2bf7028f5a1cdcac021d5a3c81e4682ba2180c917cdd86bd3e448a4bc92e040707dd4ad56003cbdf
      8f6f39b45a70da07eac86274f81121d4156b11a51da30a2f899ff88c577c4bd0efed868b9a98dca7
      a47c79243af44b5c4ace5322e3498e987a440980bac6fefcaaa080fbcab7d9a12574a6d3df577351
      aa0ecd18d19e3f49c0f6637c8db56f31428a755a7b4cc96c36f22a61f213772604d984cc97fde5d9
      3e446dcf3103ec920adc413ddcb0a3b7b785f9c34a0b5f6901d00ceb3c1f15290187c06a1d577f2d
      ff46d1da8913206950cf321e1d15f637c45e5ef2ed2388ba4f0769999ad9654e0c57c070e27320cd
      6e45966219542b1c34e5a7d9a1db79a2ccc46baa0e0279db12d552bd5118822c46d497d694332bf5
      eea090a5eb8f8b3daf8ee14de185c280def16fd2b5b40bb0fb2f32dad2e1f0a18794800c6608ff06
      61b3641c5992e57d97a6cb730df9c582521d1be437a13c8d396226371f813746085dc5836cb04a00
      92587b6559ac8a13c95665d51a0bc96627d0f083efb417867f111ecbdcc4bfe5fbb953aa32f4af32
      d7634a828cd059d15a7d4648d4e203721250090dd3506692a8f8bbf64f3669cee9b5adcdc3350232
      89fbf227de6c8f6f44222900db5c84b36db3c5b877000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      00000000000000000500000062000000090000001700000056bc70eabdfa8cea788213ca00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      0a5cc98fd70a464f04f44bfc06359fc821ecfcfa91a1b4e64470d2cfeb1979b06088e266163cc850
      b45ce6b3632f7e899a9e6ca528349634df88124b42ac3a64c7d7fb21069772777204bba1f5fdf2f5
      e4ed060503b4b17c58ae061e4dadde0df23bd080360e4ab56805ba9183ab8d3981dd1536be104234
      d4fee4ce0b350a044dd91058511e27499cdd9b6a5e48bc475163c1e2105e8ca0789b39124ba4674d
      e4686e3492746d6898b8c3a2902334c25318bbd3842eee83d5fd217deb89902699a86ceec09f50cc
      2f6b3af049329aabbdedf71764d9933e40539d0381efa07ede9350dd5399aaf3b121e85ee10662f2
      757d759f6da1aa9f21a5ec005a123b7dc40f060b9ceec5ac362c664e06016be4c44278b3346a022f
      da8d92c49cac1e5473aeff1aa3fe6336232d5374aa95cd59f9d110e278ec01702c8dfe85f821d025
      121eb7ea0362a33ebf87ebb73aadac432d86b7b162e73914ce7d5f2e4b471d89e33508f4b0e2a1d0
      4a3cf5b3bb03ad238be3ac44aa9f8088f100fc9ac4681b9bf58ac2324a6e3e3513e686a834dd914f
      40c905b98258d44b541b9ef3ca55c13c98c9c2111f68d5df1128af36d8d1b79b9dd4e10b2aea1a90
      ed0e581022cdacce76e24f9f5741bc2deb50d8971498b17782be0b3e282f720efd81f3c6391cd9b0
      ce12dbf6fa2cc59c294beeed283c59cf03513af62d6792955bb6f38ef72e8c4207a0e4fe800ff0af
      716022c9128b41ead0f92fc2899c4d7c5cf25263db000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000536e25dd0457aab29067e6ec00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      9ac79862577231a3fdd899bae178d2c78845646311d551908ded7b55bd562b44f73edb49372c9b3d
      1148b998900a9e48eadfcbb02d2c45c1345d0b78b7ccce1a469c4cfebb48b0dc91f8960721093e3d
      a559802126a6e053a5d985c3ddb1e0a19774928eff00b60f490cc3ef8df4bb26c79e8a906b22c308
      13ca44c69de1337aeb5b2704984b1b780499f59dad9850c9a572b9a4abcf23cd56f25a6d728f8629
      9ea3f20e2670d62cfcec2a57f8b9ff3a4964d731164f0a5ca6c4cb547c1155e5220455a2a0b8d88c
      73d4eb150f149d4fce46812f7afe776df1adde347e6f5949dac4655db7dae11f1548c90a605829ae
      c084b35289f181fc153162c56b9646099dd1ab692097cafe76c8466183082c03b06901f2e9c5dbb0
      3d6cac789f11651b2b3eb8d040c424c139e797405044afdd72cad84dee8db88ad6ff3e7c0254950b
      b4ee9f38896fa4f8da21790cc47c7a7bcc122753ad75159d46bb96cac3487bd3b85bff44e7b0a59a
      a88b82d3f07f110c9ec74a3c80c754f8b3303b0f6126ca03e099e6c02665a0ec706718480e1f941b
      9b2b0393c5a9a3e9fc2a53707c017a0bca137a6990b9327fe1c3beb9192eb0eb6f7c19ad240c7610
      fc541c5a144dfc0c013c0668cfa82c8bc8eed737c47e2901939c976907b1f84406358644c28a2cac
      1917dce48b59109603ac9f56dfc8f2bd5e91ab667cde8720c4c5902f8b097abb4a0808e2df27ac28
      e90e41641dcdffef1e6747075374a3b0b172a553c2000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000cbc96a23aabe40910128849f00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      4bfff100d287f7ac820881415b51a969aad6965ad7abae1baad522ec551d81de561b6f86583732a0
      ff8b46f25bf9de5c9fe6b0dc2f0e1e643eb2a5bb84297fb5adccb09e004a8a0d0a9979bd3a1f395e
      b527058a9a6cf0482aad5447356d578094dbb8faa2e0364cca9a0d543d93836d770fa22623ada491
      b7973879e11e554e44f7d034ad8c0ebd76e333b76622c7a6872269692df5f2c1fdd7577479a032f9
      904e769b4cb7bd6709b7590a10290f986d18728df4c0cd1b1861ece781305292fafaefcddac29d8d
      b6db09e77ccfe5d1a4fbf79e849787aa58426c7930a72140cda6494f0cb59b62a07b8520fdaa006b
      a64593c91ab9956cc95554f30d2d48635ac7fd9f6208ac84898a1be61559f033c2a84ed25b91b77e
      f014725278181904526ff5019e25858f5122b9b83fb067f31a52dd74da9c1358f360982d5d0daed4
      76b3dc9f92c55f62641d9c5fbdbc92d804f80e71dbf45d332fb7ff0a1fd291dce264f261caa0639e
      e18ffa470f46b9c942cfd34b8843aee6896a258babdb356e3d292ec67eae66fb399b2153188792dc
      20fd5190dad37d7f5fbc12818e6ac01691f3c9b4bf85231f30e9fe81b52258343c08440d219089a9
      38f6eb52b9e470f2b22448c6fbad57e8da5d14c378adfc4182b916a4834eb3d940f0db5105f3a488
      f90c164bb1c35b3d3fddfc57375b0c82ea7fd7c93dce448730f8064a7071a54660804c6b13e50725
      83d05fbfc28a1c650a18e85e6c5406db2304734e44000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      0000000000000000050000006200000009000000170000000ccd959481f2ac68202be96900000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      53de6bf521ad11e55ee416a7fd32f41aa7a3907862ba4927fc4d0baba3c0fc35abbed4f5eb5af92f
      8f18dffde651e4f7d1e8d52fbf8373890aa7aaeda6f82f46b2ee9cced3edc4358373de69af233b09
      1566805e1dc0a85124151f7889e6fde1fba0b1b666bf9eac0a81761d974d89462a550327a9d79e1e
      72f785ee275c98165f3d5d73f7a2b8fe24737b93b1399d7f0d205a7c49f5631176ed3616310a0be5
      5bb6e8be2638b09769976a2cde173c12893d799a7d0b0db57cbdee34168f66f795d917b41b356f29
      a487fb0d9484f6d15dbbb52624d9d92f6a92b0c5e846162ecd78f6e28e5a69131d813910985fe21e
      c9da85b809a5795d728c0897fdf04e4d6c81722bcedccf90761bfb53a38ccfb2a1b747fb2434021d
      27f54711d2bc09d16e6bf4ea19e4738078e2839d924955165cc511637ecfb9b37af0a10c94886518
      b34c2c7b3f85f838e8e0cb4e376fe088faeedf6cb85717c114bc14a4dc7b3ce21de38a253db28f15
      ca15da69ac6caa0a944264c6778ef5295ce22ed2708026f7aa0ee4aa79fab368bc1ee16cce23e741
      1e2c3ad08e126c0b69de2181a136ed14bc173bedd7e98def065fb2da0f9d7cd36403d24aa323766c
      3bf5fa34ac16cc5d413e9d67c8e749c0b3d6690fe56d958b29b31fb61d44b721763fb8da081a43c4
      8dfd4363b5d42941b6fdf982d5a196700d447cce306ceffff965db3373034f680c883be8e44c0ac1
      2e8aa5c743d42043e7d1bfa15d710cf8971506fb1a000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      00000000000000000500000062000000090000001700000044eeadc9425f1a7c88480f7100000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      3aedd6af106708ebf862bb90e5313f14470ba03ebb96bce3c1d40c5928868943085784a9df6ac238
      86a1225f588256f0deac84b8a22d12feaa51a41eba12c5110f325d71ed16a70f6bf2b1a382d09486
      31541eec772f6d2de35db4db159442af25bf3c4a32863f692a21fc2339098c0c0027176f78e172a8
      d2435059bfd3ceb43e450506ee54d6a5871b3393156d27c6ef90203577426968ad2f4b735bdc323c
      3ab8490375ab6056b691e242eaf2da43e98acb3e66f0708a54a0011beaa7dbf1b652189f7a405d7d
      dad41bbd43ff0de7e8c236ef5731c2ec8050941a8a0cb00a09d534079ab860e293f89b98fd195a9e
      9a3597619d937f412424575fc0af9fae6875b238fdb0ebc73b507dadbbeda8083d6935e3a0035ab0
      54c60e8ebe09205b43e4ed4ae636b222e7806e173f2e1705aa5a87023ca11eafef5e20a27fb1d549
      564256906752622b6ccd1bb4191ac9f669df3b718e28ef07799c93724feb3bd6d7d51236fe0f6bc8
      8a1a1f10ed120d0a5ec217ff7ee63bbdb7871d60d7f733e50eb485f0f467cb9a490da9260e0f3afa
      c689b65788007b8e1bccc8619abe70fff8863106a940bae8d1bd0ab5bfe18ef22c79fe0d9cc8527d
      ff9fce03615cff705f9a6152b1658ba1d333102b8bb04c464441df8e16d8797a0544d61dff5081c2
      6eb2695d9302ba40a0d44d4533106e659d10030fc2ddd0bc527858673e76f378f1817e72126d1dc2
      e02efec0d48af2435e91bf0bf9958fbd31a945b0c1000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      0000000000000000050000006200000009000000170000002db1b079afea1d9b0074a9c400000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      4fe9a72e37e98bde087eb9505ee651084b978b308f1c4a1a2f787e696994fd232963d10f9877c2b1
      23bc79929525821347bdcaa03916efdb87d89c5faa79fce8e61dd6d751b5e47ed572aec732c8d11b
      f9933d15f7a6e2e5866c1e86b1fd0d11b4952e308de119bf5f971cfd2cf5d33fc9bc2dd2a40867c7
      ffdf709347bf4e2057534822b81e35824cefd096d6fc081e87bc23142b7602a7b17ef1cd6f248cfe
      50fdabd6dc5e142b59ffd605c91c3c57c8ff8525ed70157abb4d20dc739b9618fea96768071424b7
      58048d4635b8139168929663596fa398bfa5a6c5e24fe6efed7c8bba2542887eda6a6cb09ea51752
      0abf2c9ac64a0ee9476224a03259edce159440a75e73b6ec08eebf6a8b890c9b24d0a31353ac6c22
      72f9eeafbf85ad4bfe83a36962e16a1cf306043e36a4c9d4f59b5466452d13e0f4f7611d1fb7969d
      c8dcd5f833392abc8be41bb7ab667b663cb5fca34370b3e9464396749ffce5bc589edce19bf4dcc2
      fb01e77e951709cc4b91718498342478195cb4822cefddd55074ae9a2c0a8faf93f3604450062416
      d8c79d1dffd31bb88fb51199a90fbd5b08de15117997cdd321ef69ceef8a83252127cb63856c5493
      ed6ed21ed02f3d2553ecc7cb99db8741fa1a6afdf6247b184546f6a13f61b2810f7b585bf0c1e12f
      5049b11e3bc899baed90aed50bc3e74c90152e113a452d737749aac2491118f24ed83ca754e1dc25
      48a6326688387b5bb48a15cde38579ae08b46f04b6000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000d0d688119c56eed120eb2c2200000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      6982ca2ff344231126d8dfd6ac3a579a1135890c0d0ba2fa3989cf94778934620cb5aa5455861336
      0e8d7adca99c11e0b197d6cc9686b706a56822e59d6eab5a312c9c879cd1942cd9885b1c05b61bdd
      d1bb76becb39847134ee51f8342cee8420fe238480d6be236f4d9a969eba2f46655bbbeadafabf9a
      9a62b8918577e848d8b2c4aa5aeb437e6f21ceeb736d195d29151be916c97e46885fc806e041958a
      770b5c25d731b67bed0bce56c81b92903289bc81598cd29e132569dfece3a6fdc06cafb477f889f3
      7fe66fa588e5fed99af08b027234bbc6e8860f041e8b87cc4defc67322736e13ca3d2edf9767a842
      d02f4b6bc90f6abb6fe5e9fafe6374dfdb5d831f8bd357984863484fb6583c10e2a141a81a8c0916
      f1a50e3533ac45f63ffad1af18c00847c14e9bbce62ab4cbefae174de49ec88fdb92f41ec5672631
      4715912ef11a4c5e2a63272fabae0624e3254ad37eff9c0fc02b29b2741b3ec62177e78bbf6a00fb
      1db76850fefee07ceea19d077e41983e24c8ba44c0b7d36ff4a40aaa50772840792b048d3e1d7141
      fef90de538db6a1f5fd9527f52e490784b4ce81ba7c5e05f12bf4cd39749df03031baaaabcf7a53a
      9ed290c674b42be93e2863e1b0fb385575c124f9a073cc7eee07f22f0a05c6216fea9b4d6e5508e8
      021609abec744af44f89de865d00fe6d8b94d27e5780b53290f5be3cb4f1890996212e4e1005c9f1
      0fbe83992ae03ffb573052c46d60361caf13f508d8000000
    "
    ),
  ];

  #[test]
  #[cfg(feature = "openssl")]
  fn test_openssl_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::OPEN_SSL_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      None,
      CRYPTED_KEY,
      MESSAGES,
    );
  }

  #[test]
  #[cfg(feature = "rust_crypto")]
  fn test_rust_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::RUST_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      None,
      CRYPTED_KEY,
      MESSAGES,
    );
  }
}

mod rsa3 {
  // This was generated with rust-crypto rsa version 0.4

  const SEAL_NONCE: &[u8] = &hex!("cae362f8e470c3037718e2a1");
  const SEAL_KEY: &[u8] = &hex!("0667d8b8994fdd468c01d50f2f3158730bbd43f4fb761ed0e9dcf00c33de5222");
  const PUBLIC_KEY: &[u8] = &hex!(
    "
    30820220300b06092a864886f70d0101010382020f003082020a0282020100cd590bd0613cd3df5c
    05fb78acb1a80c44c979fe3eca543c1e609b18875dccccd1da0883c231cbae895c2c079793565570
    4fe6fbc4a1fca09f35fa96bb3d65ad2d2c3ea9b79db53481dcf6c49304d6243ce711bea4abb6bb83
    9489e9dc53214a5265a073dfde221e0200ce58ab9574c76d84c0a0e5c01f2302aaae7f85f4e1eaf9
    38704428403892eedd4369a289bcab16713fb458a9c9b21140bc10790f3030313e5d46a3a80aeb81
    d0e7f4c9c640605357ebe43f46449c07dfdabc7d711b4f2ebaf38dcf291267823e784c0d7300593c
    58591bcb10b5946651b9fd37302ed6268158aaefcb601d85c928db73448a97b05259c9b0a86fc366
    47248783e468fcd68901de05d620332ad7def284d556cf78eeadf9fbe023ac4128b74b3c7c7c2a9c
    baca6d3ed7ff33c8a15fd561dfec5038d8efc98cb9367132c08a749ed655b9de95b70a78c58325e5
    c341a9cd693dde17b603f9186c3289a763992cdb4e8827c321e261859e47d998c768a27deddcb5d8
    4bf17744e6d9e97ac06338f69c6de682a77676f951bdc592ef4f24937f3822d217f3128dbaf717de
    a8ed45aebf31fc55aa55954e1cb42a7f7d7c27a06d3595c847fc3d7d7009392f52889325a0552ed1
    49a8c4048ad0aee0ee4749e96269d148c35401cb35b135f77ababe3b7017bb21f89e60c31f5df4e5
    359b1b0bb12dfb56502be657876cedb1e9b8ff8cad72c90203010001
  "
  );
  const CRYPTED_KEY: &[u8] = &hex!(
    "
    7421f4270b363844584e4094f3eeeeb15be50d8275dde0edbbcc5f35ea90db2dddea59ab7e12890d
    adfc2d3a99885aeb2bade0bad5170e595a5a75cbfc9a84da61e74bc8e6d44dd0f02d659ca5b60cc3
    f8efa66a068910c5a4bff41b66701c8850e109152cb9abc0812215fb45ffd36e30944fcfed83de79
    a7b005cb9d99864403c07cd74f12bda5bbefdf57cf52638c70fe291322f80add9fa6ea237f599739
    c5b10e36c927f43f1ad4278f78966692c5f543da9b73f1a8449928e340733881d7d7a0385a4392c5
    e4a83cf0ebf844b34d5b8e8beec2328fd6da418e8edfbcb27a67da585921a6118d26e54c384146cd
    2a4b5b7057a6238b61e377b00cbf99ad3f0f5eb8364885acfbb26622e3f9660bfac5993edf22865e
    f14b20377209a3988dd6d5f95b0895da3012387dbcd002429f891fa14cee3c8c4b66ded8ebe9baa9
    bae7d48a8f3f11590250261976aa07e3ef8e0507b922e2b3d946fcbf5746e4a4502e0e447d87c982
    14dc42854f76806d885e4a0570e893349d541867ed845d504901fb04607eb6967721f6ac7dfad4a3
    276b314f847edf142b62f9aeaa6cafb642576970599291be929e19f4c6513b36c1ffe8bb154df033
    e78f62a02746629113c68f1e66c1ace39ce63e9bb973997a7146e8ecb463cc930f917c417be65bf6
    f355063553b4f8f9f9fbc699f9ab188ae5483d5e12cd0b417b7df55b7dce14cfa5119440739eab88
    0d8ec3e458dae7154e0255d79d43e6b249bbe749e61eb12e04c78443171c29ce255538712c2aaebc
    b8451bb2774817beb1d132900ff87167f43106ac1a68d2e121b83b4428d1116b98c5ca8ad528fa57
    a31d5234acd7bc04077df36fd861a704c4ac4a4c12e683b64b84a645e0bf81cb8bc60f34ca343bd5
    ce2730b8c7d769c45306967c85e98bcd9d1185288a5f02f9201b1f0dd6227a3c04cda6dd0f5553ed
    7ad3b2b8eaec2df5ef4a034ee9f851cd5bbb4160aad2daf44c8cce46dd289b219d519718318889f1
    0bcb9a9a4f35778c001b664c1d5afc776f84d2aab513dc78c7ed17cb1f71feb15299d9a5c5fa32c1
    139e51590beb35fb686302e89c47e8742253bb58d246293523c3ddfd7e2e0e52e717c7132717d747
    5a51f132046d212b8693af7caf4870431a8c52a02c3f793c3a138cd1f2adab713b2545ccdc89b102
    862ef98be5c66026c6c17a3a78322971c5416ffe4cc9706ffa9357294d924a4b03080d9b4f056dbc
    8d578914055064bb41840ac109eb792c3f143a01b408759fd155017ef9990e2c1993cffc01464dd1
    16f6e59ccfc9ad9159cec200b1a82a4d92300bb200ac018396215824b586ed97c8a236dbb3884205
    f07945c2efa0dafdf6633737fa3bbb17af30024cf7a9f6d1c859e933afe7c73da50f844308bb8ff2
    5d6fcea1f18b0547a170a81fa78f550e0a25289f6a5a3b4bf146c0e6f49dfb2c6d40ef7117412647
    3a6a49ff3fdd0325c177a740c7f85202ed351b5df323bdff23e79c980316f54e87435de7f46667c7
    7d5c6aea79fde86f44eb5dcefa54da17809602111cdc4f50b1274a1e001c8bfabeb48086140561a9
    55539982ed2daa89e502f10e3ddc7ad8f651b317cd7ee35a603be85fa33c075fe083c8add662c54b
    fae7c4caa666aad5b9704720382cc20d8e27c1d7c3a99ffe94e539fa86d7fdffefdce5bee077d0ca
    e78e731a9d9c5b8f915a06054c31bd49eac82a5cbdaa0131defd38c9ac376e49615ef02605da239c
    def8c95de04f05a9d760767a8bacf0f8e431fc6dfad379dabe3142c770779f475b739af97832fa26
    cbf7f2b2824eb4234f393931e02eb89b684496111b55bcfe7fdb257241c8e2673172d7baedc75236
    a2a1f59744357f4f613809d4e5abd473fe50d32eaf6865750c9dd97208158a0a124ec7625bb2da79
    371d8ca7d2103fbf4ebd7fa138198b6242258106f4584e079a67d9b4e3965bdf86d180661c193110
    d1ea45590355304abc26d6ecf029d6ca739dcff165420021cf935c4d0ce1d542af78687c2023e1cb
    13523dbdf99728f0affe2887d8853385d5ac769be5e796d9b57f8baf15f82a659fd1304aefe4d568
    c811d6b4ee54fa3638ed28b0b8f8d2f07e9fd32bb7effaa07802123f10a643fd5ea1561ac5875b2a
    94dc15fa0366d4574abbf0ae56f1162174687a53e93ed701fc0eed303948ab4bf3fcd644aaf0e63b
    68c1d5197e1d68c1c44e4f74de92b440d9d561828ec696711413bf9753295f4a87a7b09beb67ac97
    f9130a40a274b0d2bc860afd6f7c28bc05674aca2e801e6d418ac0f46392f10747028abfb5646693
    109ed63f8821a53e807aed4328bef9382618def1bab2bb596d6df4d96b35ac353e9a7603264c0725
    5c5d232e60cd2c69eec40286e5fd4d5ab5bcd5a447cfbd4714d39d302ad02a24c4869ace0557fa9a
    1e43bcc311561ee13746d50b1fc2da47dfedb88b42e1f1d25158c7644fff34f3e8d755efbd6e27c8
    b243ce67e143ffc5ebd79d0b37fcb51411842e170707a0c290b42a452f12685c8c3379374119d238
    d3e6f7ddfedecadf5fc95c54d9c7a57d5ec677ca8f093abd499a47c92bb7c3169c290c4a0312861d
    773533ed1abf97126c3ebc84c1d1e53e8698a93ca7dc64e1450c0f230f0477592a2ea0cebd8deddf
    24468ddcf7c574015c161524dc57f12890b6d53e604b1963e8dd8304329029d5bb13ebb44da74ac1
    799892dfd18c1b738ff40d8af8ecfb1445301115b56184b1adba7921f052570d9fc5f0ddc2d09331
    da5953f9a5f34e802d8c8324a906c448f23df3718f51836b3ca1e6c66f36a89243df05624a013065
    d500da512969dac92cb8c417bb01ff9c3195d5f8b9828712cdf3d42e85ba4357e7319032c042774d
    d0a5947b306ca7b4591704d0e722e1f10e0b3afb8488fc64bbd127125213a225f85ac05ef5ff4c98
    ae0c552e4b099419e4378e8d17f01444d7ef5de161182e73abd2572b332ac9af5937559d95705741
    6b2867c5fd27587d81b31d9d6c65023f9a9fce046ff46de6bd0d44fafafff5031b76ebac2d771ad9
    685b5460f40ee375b3965ca8c583d383322ebd500906f96ddbbc28edcf0bfe7a78cb0f2ce8beaa90
    7f0324f6ce2c34e483987690887c3b404ab864c8e033351ec04db51e0ff3fde8b2d3c77654ab29ff
    bafa38672775a4e5d476bfe539f416d094e502b63c39166c6d17c037cead2faf1f74e233643e1846
    f9cd7df46f3cf876108d26cd8230fce147a5b3a810de67ea8f90aa243d1c9be816577d7b62bfcc3b
    3f3d18bc7361399aa71693ad3c1b9d7c4aba2c0f8b00c8a3e6fdb3ea7e44872b0bd54478a726aeb0
    36831895e822
  "
  );
  const MESSAGES: &[&[u8]] = &[
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    00000000000000000500000062000000090000001700000053e3e800bcee95089806aa8c00000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    92731a5adf38430f2dde951023ab67cadd05afc5fd54055f952ea6bda32a5465f5e3592064cb4dc3
    025420710a1550b3494fc290cd5008a4ca29d60b8dbe75838bce7043deec6ae9e8b5b1b3771ff8db
    4bfd29d77bda84a3d3be43d2b8793ef9a8cc76c7d38c905ad1691c7e6682bed5a8b54bf84cb4c5c5
    47fce0c169d2f2d38a968a77e7721b8f8fdfd2a470993d620b12cc4b9cf605d1f8c469337549d806
    e6c38bac9b44d6a87c6eea04dbbee2db19a472fda027ff91ec15342d1e806e8dc6caa3442ddb94df
    e637b1c3f67387ec4eaf453b3be40439f61c607e2f95acd802a3759d05d829aa6104222183edeb8a
    2dbd2e87d6c48a445de9a53cd840df2a1d7c7a8bf7cbacc8e00673cf3762417c6799287d348f5e5d
    bb122754b9535b8d72ac1ab07b763e3ecb6925f56cb24ded7078ef934ba5b5d76f44bc989ef4f318
    ab8d27fc266647cce51e637e97881c35c894f9e8a556b9230be67ca1f3aeb303b63aa55c7baa10f1
    f2200c4f8a33d12a21ab86bf717432ae918db632d1d9e8aef6e051aac3db35d33f78b6fa5f356d7e
    449eab03983805cea34d6c920ac990ebeba6d99d776858efce2112a6b5e56f11f08367b96fecbfff
    efca5feb6f6440b5e3da7b45172e519edb086d2f898bf683699a4fa0250b43209b0ad6d01366c7f8
    cef96aca851e8fa0a0e7fbee27fa3bb9800dcadba207849e7a3d200b3c4895b4cad589d9e88a0d10
    73a42e32e818c82ea1563ded598b5ace52fa28b800000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    00000000000000000500000062000000090000001700000037acba4e1b6228a379cc95ec00000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    7a8a56e7a2bcb71065e46c71b9e867e8e8827dec820835424fea436f608433b75f651ad62309435c
    97803696e361d0044011f029762b2a12ed53fc9464a48b5c82f4c3607973500c4bdca2cfa0d5e7e5
    5a18720ce57f609a25468de0388fececf8817952a2649b0fef293040ea410ba38d23d5b007d94f4b
    4956ac4ec9cf3187529c93ae65aa009871de2e04f9992564647cb3266774279f859f26ec1b5fd362
    89faf0449f81448cbd148c727072bf75ab3f3c9c483c0b2132c44159082c835ffd5c4036a0b3fb68
    1ff229c4333492c6fd67498251ff72e31fbf708c5282df0928180390b4fa439c4928fac0de597484
    0373c52ad2fe687a65a9a9c45e39829004ffe9e3e9cd8858d0f6ece0cfcd4d3172bf0440af2db0e4
    cf32bd5a8852ebc0f5da0acfee8bd4a18dcbe6d197c997fe5a0190481bef040487e44f48946afdd8
    2e655893af5270cc480563e91d4be60c166a1a1c42eab04de77f8915403fb9e43649994a1a8835ac
    01fbf244cad30b5ab40b2cc0f4e2ba3dc71fb866d7052512f6408cbd54ff8eb7cb8091bf7b0a73fc
    39acb57b459dea70902df70ee3e9bb7558757dbcfb00e04aafdfcd04761033650c7b50fa30ada421
    a853fac26f34a9d05394ece5027ab8a2e3ca8602b31d6eb39c7ffa8ddc10992f3b71f1867cc729fe
    8f4b1397bc4f2bd4e7a29c74d1cd6bea36124e728a1fa4d2b486fdcc08c6ebc0f146a0cdc9ceab5a
    de9df27d5e79f33d636e47717c43b8b5989a9aebf4000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000cccbd6e3cddefa51db110d2e00000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    c5fd91477d482acf1b5b5ec3949f1f73b2953ee17ee9d7dea49a99e97c8d16f152e78d9055471e80
    7834b696e473a7a7348bdf55b2b867c144b29ac0034f92d4b9c9d1d9783f164db9e5e634054151a3
    ba4db639d05375661f65ae30ca09cfc3ce9cf7a0dc52bd8212ed7b29d6d571c871d9d8d8b4d88946
    dfcdee4dbca7a08d60e38cfa58986795e7d94676ad55c04ea8a0ceef0f1e8696b30d0c2b9db10307
    429fd5c21b4eb02f6020a7f5b664dcc9dee0cda4dc8ae6cea5175e404591c11859161e23fbfcabf8
    dfbe4537eefa4570130d2b41928cbc876b953ba488fa16af186e3294986c26ef6b8aac1e2f821acb
    a618dc25a73ce436b42e9399bbbf80c00334214839c98f50c6a92cb5c168bb52c54bd2df917f0f00
    9be0bea99cd736d85a54e0f287d6e728eb4422fb02291a17794b20a838c86b4a51a94186493b9f8f
    b4a8b6aa41d719889a1ac3e1ef24cc1434a45a744cd72e872ea080ce9e92cf7f4cd398ad2ef061dc
    305e9f4ad8374ae6ff6697d89ecb6879523ac3950ff9e4c956af83dc87c33678fddc5f9a70e854d5
    dd6b4b67f40a3c451c1828d4b2552143723947b7db738c5cf335d958532a8895c6bf679705ef9f7d
    b0674ef23641e1b4e88b758bdf4b4cb3afd2b643ad72c63051b2f45bb40a79e9cc1361ef331eb71d
    19b14a4ae92275a9d5091709cb9ec8ea7b8d4c4ec2f414c5ee7aecb76df4e1db71a7cac41d66eb96
    2e5d74d82b030ea8fee14263fa67ec33ac90344470000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000eded5278253180327b453cf000000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    581cfb9e8fe09a033f3c00b1570d74919149ef833c4d3776765c1cb8bd6b3b357929f250fe9586c3
    f1e96ef32f64f960bfb316469c6fa37d0e95476daf27aeb07543a252337157b3ef7f1477c81fbb32
    1bce6a92dcb8f9845643a6ed73208a71361675345de79cd904aba4a84dcee73bb76085705fe61bf0
    f2f0e0a9f200a3e61966a01bd3ced2e06578dde20d2342bbf6568342b931edfd3ab58b9909a9cfab
    340d8efcc0b31c095332153a22e405e9e80566695ca89f07126e994eb3ffe099e9f2c89fa09e5f41
    0f57fb9a470ff29511a773ba7bacdbd29f59aa512971405ab08b016610efb316b67d3303d6d4cdc4
    2b0cfe17b1aa50e6006d05227a6d51111b243dbc392bd13c4947d340373a028efc19a3bb43cf49e3
    62af57b70e5f2a7db84bbf9e0016f5cc762118edde414f6d7994c64d1f3663eb9e07635a57ffb98a
    85e9832c831a7f5be3cfaac30f86271255aa2e6527138e8bec3d3042e05f934e7dd72b8107760db4
    667686533958ee9aebf03a3a56562b5251f2c02c0e70494b065dc60f31ed0a51a2888e710cedfe3b
    3c668f8837b2b37dda55853996e18350e406dbe4e034d3f5776a11aecde963a838b94fdfd2ffcfcf
    82e2225981848d23d538f0fb2ffa39de2de7f52009dc8effe046078d695509edfbc7eda80d3ec854
    279cc3719ab6fd4f735feb1acc90ba5cb16145a50739f95a9580f1a15721cb025f4f18b6e5e76a90
    daa3685a205f654755a1c440a399388dd0245e1a45000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000bb734be36aa8534dd5151a3100000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    3c05c2b8c24a755c70abae48bf0a2ac22032e484c6353b2747501525efae67bf062e89001071b5b6
    b29725c5bbe3feb89e2e60984fd05733abaee69e8eec3df1d89bb397b5f8a4192b4c180db42fde71
    2a0ebfda6ee0b2c1af8deec2745502dbec3cd981f7dcbf3cb58b7010404a617953650b4d04a6ccfe
    47aa1a3a986eb60ece80175a535430e4178ce2e3ae9c75fc228c104cc8fd4501b376c1eef0a4cd6c
    a7443ad4a56da3f106345810866f00d8b43f29891ad9d5aa3b3182a28e238aa3180d8853d46443ad
    3db999dba4cd40131343639131d92e76b1a61de19a8f90e6fcddccb117b6664b31ac3167d2e34792
    03f7c478fc3f1abb4a582c4462d63463488b600e52d075c036b340eeffd8e1f181c152764931cbd3
    8d6edac0842affa8061e141656fcc470adb23ea39380b8ac2c6a029c8183c2901f07a305af4c7a4c
    094828f0accd10627c7b296311a938c2b2f31b26866d4e1ec9488b35bbf0b8244b01f690817fb647
    198d65194ec468a1f0aec3be83659b3100242068db5eb11e5464ebc5d8f155fe1f784650ea0dc65c
    45d436512db41a84e1bb73466639c1953f490b0370e58fa61a066a68b6a0700363554934dffcb781
    ae19da995201d925e105ce1ad6373f65200f0bc4936c9f2002a1615e02fece27669c8003ddfd62f4
    536428815122c636ccd42a3cc4eb8d50e7b95ca32cf1caf108c04eb7658cfd280f40c888775ca663
    49a1cbb11745c8fffb4b21cdfacafa9af4a40a5030000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000d8bd10e19a47316ad227b26600000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    8a376b430e2622060f4f33d441364d4a4e80720ac6c100da244544e8c66cf804e7ccdf09da0d6b6c
    50d74b76ed463058b465e2202295703c9e54bf61ef75371600b07b6ff2b18fd1551a92130e5aa086
    85194a0caf898f8c717645df15596be8e4d7dca2827057c5b3bda0cc01cd5a5b8f20eab6dc44b625
    517b87fc92474f74fba7ad4804f602edf27bf3f3b01f0eb8cae7bea99f1f2d20b873e856132a7f39
    cdc59268704de6a8af3d9df2d3aabff863fda560b391f75dfa4f5d64fdac001cd11c144a2be65b9a
    ac2af497ef223f1d81ab8b7514e642aee9f39f0b4f1bd9f7324efa5b42c3505174e380216109c146
    b2974cb4391e93eca7c87c4eb5242a143a7b7a1ae4ef7d7204516c59b90d12a3588ad28f4da88d08
    a952816a840e40bea4cd3a44c913fd1214b800ec226d5acbc00f7056f1a16db4c4d66e6a57157f04
    e54957252bcfbb8ec5753ed5233cfd9ce3eb0b241a32ca1c229510740394f76003fbb5c417c2f384
    169b2283d32ec071249548a757f8c8800a67ad39c407dd6043f99ab8c5502e5438e660a552f7475b
    fd03ddf0e46613e9842fe574adf7c1f96e7513d6f6d58a996d3cf3522a880a86fc79d8a8bb07c116
    6bf2a5b475b667f73a511394a6f74fa1774752946ee2ddd6f373836b913bf7d562a1baf7cb821a5c
    1fbbe8391d7c4c4ec1cd618d5d2430ee86850759be898481ea8fd74a4c4a1826907e6ebadcf71b74
    5d1593d4d9cb54cabff83e9bd030309f769ecf0f09000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    0000000000000000050000006200000009000000170000007a978e846d4359030170e7d700000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    3a207a34d335b57bcbd303b77080221defbc66bc20ddebda64486428e07857167fb16349e9111fea
    590400b127741aaf2f8f5b6b4eb4accbb7881df0fff3af1a4322446ddd7cf6d538dca6414322b8d3
    eb8ddc8faf18bb84e380731e77b8c03ffc127548c2c60dc7fb9402e378a7eabfb3489e109f29a906
    58db9bd7ee1c8f0c5441143abd00047bd830fd152d0396579928d36bd2d2f22aee20a60d543887ea
    30847dd56ce541a451b39f5fb7f6992858dfc2b0f3e4a771fd55a79ea78ab8fff316b68ff5639ce1
    7be2260a032d8616f60aab3b1d0fd973f9aeb7b859397d1ad315ec9be6af99807187b9e49a727c28
    5a95dee0bf84d7a8f0439525df80a556b3a54ff5c000da46ff6f4310231dd823ea114776fd84caa9
    0761f83f4d8d24a130854b4aefef91b8a361ee26fc1f3e1bfa4666211068c4cd7c22bd1dafe307be
    7fa626b3ef8c42d84a99d753172c2c8143d44f3468abbde1c5c96bfb2efad52ce6da1ec796ce49ba
    8c3477307b71d42a64b0604e768734d2ef1ae952c30614a198954afb1748a64106588c33f8070209
    401cf598580f79f8f264605044cd0164ee9aff80a7fc95f8672f5f581ad058345591539c2fbf3a0c
    2aeee994f035f3b1fc81b4070a199731f239a082378e2ab882c70deef02d1d1c996464a0a8a4acf8
    ee8befc5797d370db1f2bbcc74968b9780612cd9d863e0c1dbb84dd39790ef7ec246d3727087f77b
    896303f248fd4b1fd2a8cb725479adf4d68d5b4888000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000f4376c944e463c7f84e6473700000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    1e4c69c5592ec511c6fb774df41d77d8b8423e6ac5cc23be737d9aa57ceedf688025ec06a594accc
    309c6d4f2fd6ddee5aab29299400cf04dfe0b6f755c0bd2e241aa13aa895cb59fbcba260a64b6f57
    65055d1f0be6f620636cbf4af4c06d480fa601f79b8506df5f1445b4e51351ccdc73efdc34932481
    8bdea9c6ac2c6d710721d20fa1c62f2d1a1adc7c948099b9262f66c2b69cae82f0c7f6b6b08ff416
    2a60f13e82d13b66d03f4cfe1bceaae1a9e4acb0db2b406079e8f87b16900eb78af05b515ba206b1
    37dab81f4eef38d562288c6d0675a51db99b7c8aab2320f7b68b890543e87df5b12a7e5e7f9ca9d8
    957e201a4cac3790ae5240c843d31c3465e63b310c50b109e1800f37284c59a1b8b2f49d1f29226c
    e550a52b869978ced45f991cbc85071cd97456c904f126802dc95e814d8adc6893dc7c9d4646ff5a
    4eb3bcb124a0a99aba67fd3413b57e333ab49e56b2356e803b620f18a106e31afae13e4f3ebeec27
    e9d3ef41ffb2c864cfdb8cdb83c8ee19053af830b0461ced80db7f370fcd133781eaa11c11384005
    654759a2c4419a9de85abb42ea3a3062e7d5d9428010ee81f2aa44c8430af37dd7ecb9dd6287ae60
    fb5dabd59bcd7b75f564a39ad3200fa63f7951ff40410b96f61bd736b5b055356b8f4a1c3d212f28
    4e5a37ccfbc9725ee7591f0018b67f4fb61902be9402b2eeadc103d2ee8a9c3dda42f56dec936d3f
    d95ef2f0c803a3651172ded070b84923cb3ece2629000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    0000000000000000050000006200000009000000170000009daae7600066b2b7b912192400000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    46c2cbcb410c5c65f5a10e072312ff00c7e8ccb49741d8a3df359bf53a6297c1e7d4a6a691ab7c1c
    e1f2594d9bf4fd8a67dc584ec3990e9fcdb8fd60d8719e0ddfca7164a62e6dfebb2bfff62c133c27
    998d3eddb002c1aaf12d64fde7aae1825596edaa8ea3a6027091a2b9c09992e85bd2a6500a2a0d79
    2220038a6063aa680305354c3571e1a745187a470a3700777ad1f2fde73983d9f82d9972bf6a0991
    17c954f01067545988c8622352047fb6b8b0cb076245dc15e77542995f258dec6afe4ee792a0c911
    47d07b526e31e2ffba01664c90d9a9cd92f0d63ce0ac197a1568ef85c2ea99708919ec72cce9d629
    68d45f7c54e1279d9afdd00ffbda7095222a64248149807efc8c2a37d994a9d4f1130c48ce55fa9b
    9ac0207ce32e01cbe612549e2ac93b333228e806f774bb288bed03325e3ed8fa3224cd214bfaa734
    85c2756ffe1a61a8dda34719e93c32fc12ee3f1a66e55c5999f10defe476b249cb72d2baf1651091
    86e439e4e1da0c62fafd49659a97be4e622c58621fb476e52d431c3fa14e64c18128418ddc137001
    cb331378ccdfe126d97043c2e364bf1de075106fe4b742ac78d2f92a6d0ce30d1070e41990094993
    5c847bc74395fdf5e03ec1a29b768c1d0a46b3cd72b80c4612a3b4df5e055b4e674d53ae8ab0738e
    947a52952667e896cfd52a3bedab4da9c8556f837376a6024b8480f32826bea8624ea48410004806
    8957fe8c36e6a398fd1f818d4b3e3c16824490e915000000
  "
    ),
    &hex!(
      "
    00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
    000000000000000005000000620000000900000017000000ce8faf6d34dee018567f2d7900000000
    0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
    6b7eb1baa12aa4511721e79ae68ba49572fb8bed7e3ac7af28c36daafd6e5ba78f46f5a41792cec4
    0d5013707e57661980b15405470abc6a4760392f51e11b5789a9c2e5940cbf02e99e5e60fb69f513
    2891905696f77285d53228ceeba426e1ed42d8e8c83aa9702af70bbfb69d6a496bb97fb9311953c9
    cded4c828b6e4d994791eda5cd77d51da0435b67ec08c99c6cb8ac192bb6c6c596e519f5b14ab3fb
    7efaf7d4ad922a9c1869702ed314f5d07af923b8cc4ca879d30766a5707658f1170ae199e72c723c
    9c4264c22a6c912ab65e22f6a5043ebd774d65da2c6cd0f5d7bdfef7d18b30f4af0310105556b50e
    1bd901b5c2b9506c0de3b2bdbd87a216c3f59da9ae4cacdbb09688e9b68353ffd6b1f3481112ece5
    4c4e6e3c699cb1a681f1b882dcdc5b577a5ea406e71e9d7e513069a8229b95a2b05bc2de66aa0ee3
    24da843a9eb88298ad04cbc15ba540a0c51af8be80817dca8eab426c4d2fedf3ab4b93d19ac8bf85
    457ec1f117ae50c86af0be106251005866071b3d0ef2acc09205b9c4088dc2b85a09ca83554e1f0b
    9f4acc330ee6716620085cad97adebde3e3f41758af89d381155cc13529f12f34bb04d508fac1cf9
    726a412497c74d76b077124d4c24698ecb228809595b8f9b7c50665bfd8639661bf8ae1005b00ebf
    bd8eddd1388527ad7a1ff3fa8aef7320c992d2daf8097008e8a2e9682883dbddf8404b39dd56e74e
    5a6d5eb242d527e81a3b0d1c58a501c971ce7ae862000000
  "
    ),
  ];

  #[test]
  #[cfg(feature = "openssl")]
  fn test_openssl_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::OPEN_SSL_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      Some(PUBLIC_KEY),
      CRYPTED_KEY,
      MESSAGES,
    );
  }

  #[test]
  #[cfg(feature = "rust_crypto")]
  fn test_rust_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::RUST_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      Some(PUBLIC_KEY),
      CRYPTED_KEY,
      MESSAGES,
    );
  }
}
