use std::ptr::copy_nonoverlapping;

use capnp::Word;

use crate::secrets_store_capnp::block;
use crate::{memguard::SecretBytes, secrets_store::cipher::Cipher};
use spectral::prelude::*;

fn fixture_test<T>(cipher: &T, seal_key_raw: &[u8], seal_nonce: &[u8], crypted_key: &[u8], messages: &[&[u8]])
where
  T: Cipher,
{
  let seal_key = SecretBytes::from_secured(seal_key_raw);
  let private_kex = cipher.open_private_key(&seal_key, seal_nonce, crypted_key).unwrap();

  for message in messages {
    let words = Word::allocate_zeroed_vec(message.len());
    unsafe {
      copy_nonoverlapping(message.as_ptr(), words.as_ptr() as *mut u8, words.len() * 8);
    }
    let mut message_payload: &[u8] = Word::words_to_bytes(&words);
    let message_reader =
      capnp::serialize::read_message_from_flat_slice(&mut message_payload, capnp::message::ReaderOptions::new())
        .unwrap();
    let block_reader = message_reader.get_root::<block::Reader>().unwrap();
    let cryped_content = block_reader.get_content().unwrap();

    let id1 = "recipient1";

    let decrypted = cipher
      .decrypt(
        (id1, &private_kex),
        block_reader.get_headers().unwrap().get(0),
        cryped_content,
      )
      .unwrap();

    assert_that(&decrypted.borrow().as_str()).is_equal_to("Hello, secret");
  }
}

mod rsa1 {
  // This was generated with openssl

  const SEAL_NONCE: &[u8] = &hex!("da7351cbe5a9ee4f5a34abaf");
  const SEAL_KEY: &[u8] = &hex!("715391cd86a36ffb465abfa21c9cafa5864ffdcb02e720d90aeeeafa639666a8");
  const CRYPTED_KEY: &[u8] = &hex!(
    "
      35459ade12c1b85498bc10208e2eaa712c9672303375fb444d9b50c4e2f0f3a771fc2543df7f8677
      7a959936bdbf877f0d880b29614d829cc405ca37ec52b275572c11b571def36efbbd49d08f27d9df
      2787169905a26ce278d159033ca3eb6c3d7cf8d7001bc8c634998ffce4bd4f26ce0b2250a17419af
      b65c3b73d14847de96b9c03b74b622d10d03afa55341b90ea0343902424320d1ae1cb7ca01e32fc5
      b9998e31740a2991e5ab635406882fc1605aabd6bde68793f3af7f8c842984bb299e16fc3ddb9d55
      5edb4c139368305abb984097d27539acf05f3acdce37da8740b239dd2d35d44892b0e066b553344d
      7a11d6c01846bc10963bddc6bf9e934916d5a9f9cf1995d370ccc57da1f6ceaf37135716524e856a
      acceb2cbd57664508318450b3203fe70989ebc48b8760481efaa1492fe32b2d8531c9dc73fa9efbd
      47df2122076f82b0a65dd5b70a37f01126dc0d7de8f73c785ffe9286033f6e0712dba170965f3366
      da3859b8fe885ff86dc2fc383957faacf2f01891d9fa955b396f1c6ec9b1977dc378e07952d093dd
      536f93ff401f4a2075fe92d69640f360b4c407f338d7b204df011e24288424ef463179ac2b350637
      30934255e025166cf1096bd8e1cf406e41ef41608235a38e9532a56f62f078424de62c3463a3f906
      dab8af90c1e6693f55010ea4900cc342e3087cfd3fff702d36fe7cb7747bda298278cfcf8ec17aef
      a47bf98826bb19965a668d34cf099287a6793b3b21a66e4d93e5bf8d0d19176851a10445cdaf315e
      9a13096d2e540e4b99ae2f68d20f8ecd022874402c62ff9e304fe57a1d75c0ad4d70c7d0070dc457
      fa6b9881230a646628ffa1d6072f2a864ce517823fc0d7c94e8d6e4f8c672a9b882d651aad0bb372
      d8426f6a3d9b60cbfec75784e3a3c6dc4a788225aa1f6879b4dfe4ab73f805f7571e34ed9ca420c1
      2e3f6d2584522292ab8612a62f5dc076576899a008a911ec9ef8f3d55608cefe7f6f95797036c825
      ee5ea5f2e5806ebda53bdcb16220ee9d06ea7f168d33efdf4ddf1ad650816d3e38f82fce675c1dac
      1f330bfca1b3e366e49b231978c536e6a6fb5091b75849ced296ef8cd4285b894192c794c1fcbcb2
      8e57ca3a46481287f839cb73d2a9a55eabd2542d3115defdb48810e6efaf10ba8b5e3aff1e4ee07e
      2de295e1df5692e5d6e3dc08babdf0dc0006d79effadaab7a6e486a69e74d552fa9d9c358c15062f
      443ff9570a89947cf6711b0780ea2effafb22acc3f69cd963f0b6dca0729748cb5d36163b3621102
      bcea8097d3878dfe5b9e3306237170c253874a0b0b18b46eee620ecd363fe1737af64fa5f31d04ec
      9bbee8ff48d2b20093764f4cab05bf1308dd7c00c940e895dc22aa183f452e49a1aeb5fb2a8b18cd
      855c2ad20ff2ecf6bfa015bf9ae578145970316c229cb9fbd74862425d4b902c88b4d1fda3e8378d
      caa880c4719e642c983337df55060188a900bfca58ee8517ecbdae9a3e020d76ba14676143258a2d
      78ff32858ff3fe3470761e86673b751e6950141b7c56f48f8edf561cb7de2823732d6ecab69b58e2
      7302e6bc66af38a0e3b4aa8e80638c6aa745744d8d248780e8353af2da5619ad7ac2527390a6f897
      56fdd9aeec97231451675d4e5e4ee7fbdd1ebc85765058972afcba332f0a8d0aba7e05cf0df2f801
      adb0c9062e200eb44268669cdc2a841f1eef2b00f89b7802b817a278c8141cb9259f53cea54b799e
      f8aec883966d1f7befba35b2002e3bad87b480f0250829b1964c31781ce4edacf0d70dc92e06640c
      07f4c07ca3289c8b1caf66e56efcfc7b1af62424717c128bf13b585885aafcdad984514490717eb2
      9bbe185914b4f055379daa582e9d68e247e1ca1b18b84eda4b4cc73ed7b43b90b9911869a188afd4
      6a5cf3c5184f6b367c2a70389325a3d72d73a5b04af867b8a1cdd55558da413cb5f13adf4563e4e7
      5ceb720e5fb9e20adc70e2469264a9b2e75b833124fcad499818ec3d82d23075017e5548ed12a079
      4f75595b161aed14ac4b49aa09d9737a8b195fcfc19b4d0e9aa351e2256ae199655a246cdf0edbbb
      edbcbbc7d4dc891750434307a8a5eee7d526e2f4b3b74ead0c09b0e0f953e66c32f0f1fb5b1d2d40
      a1ffb90e57453dd2965773d5c1cb99d837671fedc3a37b7c45b1c13b2f5548c7455a8081ef16011c
      a7731b9d76d794341665f01e8b4f17380972c3e6bb271575821ddfbc83324b3ca5f0107ecf6ba14b
      446a9386d905ae6520cfbf745012ad3200001dad75fa34a47229a9fc1ce950cfc551aa4e187ae567
      6cd799d4b005f7d35fa391fa5a2cc5eefa7fc3bcb1051cdb4f680f718b51daaa93aa78830b58b61f
      8b9d6eb73ddf94439bb0433a101ec3d28b1250a014f1dac263253529886cef4959ec620e3335d9b0
      8d22ba9abe3fda304007a64527800345106c0cd60c2b96d1d9fb03034524d621988d9aa919c0650a
      50ab505e020e0cee4420c48749523817ffd994d674621c11a3c7fbd8980e5bb686fe43e6ef4c2999
      a1203c440fb9ebd06e0cf14fdda59bac949a873131035f1ea633b24e52e4806f8aee64ea44321542
      6c1491c5813ffe3e8fee21f8674954587e69cbbd0d86b8727662cbc37d180b2d6b0c8bc1f176b0e9
      d38247ae5b49f5cbb5ab321f79ab0dab9972a57eaac0820e45dce74c9a29254bf3373bd26d6383ee
      e43ea354af1fc6a3314e876904f23defe29c6057b287ff51c063f296e0db6debe4bdb7b799843bbb
      fdbb8e6fd6d9f05f65406f36dd5255143d7c59329966076ee5c018c6451f9864d3f1e86418e50d6c
      95bae6f60fd4df5a30ac0c8af8ba3753680ef82365fbde7ad6ea9cde2c81b9336a1826650ab52f1f
      bea4f15cfeb605e8830e7f689370df082c0be89d6ae5a9b3ac7e5b7eca9a4172ea95bd06084ccb28
      d5f16717bc374d6196777425a551cf76d5de74677289af2f949380395e5cf9c3f6f7e1759cb2e89f
      e3cf51ab750aafe4c9419e065a51ecd758be00113e4d04150c77cca12a19acbd334d2e03a4ec2b96
      adba8beccfe10866ccaad53da39b6cbe907612bfa3f27e0e7e10bd3516b971233c396e62eadd14f2
      491ec21f3a0478793a9cec35b6c053629e7875d06bef1fa0986a3e45e35fa8e46453b468d8a08ad5
      c29539c1180f6ea2db22510aefe35b4913e1ad499873f9c343be18ece0b73fbc2ce7a956a95e7ead
      a400d9c36476341c37b8f817a9ce5c993205436b8630733a6bbabba529755eb732e341a1ff8b2383
      d6cf446f56b62498bc6dd79606596cf0c7db48f7e1f15f17cc09c8f16e0f18a7d3e842de337a8939
      635944
    "
  );
  const MESSAGES: &[&[u8]] = &[
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      00000000000000000500000062000000090000001700000064b1af18aef6de7363a20f1a00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      26895eacaecf7d5d07476e5d4573aa2f6931bbe45289378526e675cefd956f7ca040a3d3336abf2e
      3d78cbb1a9dbd83e8b587d3a1e28ca66c92a91139dd3da000d9ceb767cd6a5de280f0c58f8383da9
      cd541e9769a3b9e00bda97e9697bf7c80aea37f77ccd65ba0df69637911d1529da515893979b5ee0
      e977c207d8993c3ab404727121966e923002dafcd626ee60bc44c67b0113534e5a2270aae581d725
      67f2747aa42886c719491ec4d86c78dd0cd167c30dd25ddbcc50755f48a1f72cd54017f5cc433fdf
      432e00a326c618b3022d9a8ed01c907668125ab5f475397f7660207e75d85a8ac9e8f2c3b3fa87e5
      97e9c550577dabc05d508f186ea4ea14851bc35c155588575eff13a30f176f35489e2e4dd1b17870
      b243fa7a99f544ca60c61e38cd9998b8a77e18d89967dfbe6480a4884c96f3c53129b97a830c5590
      edcd08a2b1362c406bc55000b9856624f31b54415961d80daca7b98884a88b11c733b57d0268a6b5
      f6fd30b6ff26dd96e9c0c3c63cfb3f9de22f171dc290e83274ebaac94bdf07f6f9e0ff26549e5b3e
      2a682e01dd927c42b2cdfb3b5d6ed12fa656ee8208cdbcff27a71a183746c94c65fbea90938ffaad
      2fd1825275dc92948f7232d72ced4e701c2df4aab68f28ba4417851056620e1e452385aff271ce6c
      3cf12c3e7ec6a2c9f4ab1baf405144627c06642674df02f9d32f75eb11b86c5816b1241e50651b64
      b83f9df87e8d408fc56192350d49b92a3aee92be6c000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000bd7680c81432f1e9c119151b00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      955d095c1b64a77203dd1286799dacc8cd136300e83b5ff7b51498b8faf5fdd369280adf0e363907
      2b0adf74a3fa65ff5dc43f2b8608811b2f62a45f0a5611166b064cbf85feb1cfe488a7dd3e41fb65
      b0bbfc7a61155e27e49f4b1216db3e91c5fb3ab06e24df7f2ec889b1f4ced2275cb81f7a76a38c2f
      fe667d472292ddaee88b2d3403ba874ed2d65cc23ea86af2412522171e510c027909d52ac5f8c2d2
      ec852c2999898702c7e3668985fb9c319e5affc932ebfebdd06bff6b55e2d6570969186285367853
      c4c7a7284e4d8c87565e617bb4ea83d295989f7686de6c1569f3da21cfd6e8dad8085db3970467f1
      6984e6f3a302405098156f3ec8e92956d02ba734c1791e07c5f1f9f677c3a0e742c9c5468a7fac95
      7127e00d7b251b8938731eac080e21681e5536d64410a1b6bbfbb78159c98db5cf2c1d5db7be5a54
      85fb21e2ddf780ffd95a577cc4052ddcf6d2c15d42ac576b340ded1737e6e97e67a64661ff7301f0
      2b99635c828bce867d309b619718dbbbeff9c69e9f2eeba3c0b4cde86c4d89b8daad5adf62c75d7c
      7db812652e6be1d7474c2dfa966e1d7925220d7f0752ab12e626bcac4ee25a8453261dc4d384c53c
      3ce5c6a125fad4cd294986356f995daa90ebba9e35f977fbeaec24a0c716418c10de5c7e885a14c4
      cc0c0ace9bf6fb2489c2ba03071fec178c4aa6ea731eb690504d1bc4889fcc7980f8e66667cd673f
      0bcdbb396d45f9855f2ea2eb81008db0c5e76a31dc000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000900d8ea564448db2b6fc2eea00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      85eeaebc5a7b32d77a04135a60d52ee878e974d8f2f42ef34c28a7131a4a31c580300151cc41e69e
      e758cd172df60dcc99f11d638ea7a30a6dd4b51c081ed9e66fde15de1e7dfb8a348bea0f6775453d
      827c661dcae9f4eed3a3baa78570ac2ddc050548de25d825aed59df1603d4e6a4440f37e384083dd
      d71290906062c1a3dd9376bcf9cf36c411e64a5c2f9b267c3b9e0ff11ab5b4cdf56054a2c8116867
      506cc27a15fb7f0b25e058c5fd8ec7bdfcd0c911dc29ae009e46a7d6b40192fd2c62a5097fcd67e0
      e3c83b4d4c0d9f797e622f75d6011f4246de3ca6f04e8d6b57c67ff0d0b7b05ac593c46480905e68
      1d33728dcabc574205655a673d3144ea4ef7e9d2eba078ac0ebbb47e50a50278cf0462531ff45b84
      95ca9162c79135fce82948bd5f712e6f33ca908ed280d95d4ff3e756db317f68f06af3450fe6915b
      bf87a06d3f5ef08e3c83c14a1d8e28382821bc7b28c6fcd9599081f3c3b2f295fe1095428b2acbc3
      1e0769e8a863519a3c723f605be58736b7247e9dfdba94e33d431c8a2324a86d7b10574482bd8f4a
      40d34745db8e5b1fb4e7e51f5cfd22ab3bcab9c9fd4297635c26793a0a78f4d656e0755261a3b926
      b995e2dffa075f5e5ed0a88ec7df8e07f8210c31a66b2160b3fb4d2782ace9125f07a27c5bb68997
      16a2cf0ebcd9b5b0b9eed528dcd4dc14cf8668f6ad140b8259e428303250feab2a4d739297b676b0
      0867abd850b5083908973f440f0c85328b09619c2e000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000c20aee77a5c7631d3784e32b00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      91e5de11ffe4a02e2930a26bccffb8ab5904c989d708aff84dcfb64f8196cabbf00bca3bb6ce8ccc
      6b4898be6f6b8520ff898b85c1c91d54376b2fd890e330fa5e937c54b4a332851aebe67c807219c3
      10d3efcef655cf6e244c75dd0560625cb8b018818da6153ad0788af547e0301da153ad976c14f70d
      81bfacf2dd6eda91b918922edc00ba2999a97e431312b29f2c52f9b55f70245a73b75c3347c81889
      0d7efd38506bee71c9ab174d3e8195b1bb7ed968cb87f238a2b1725e32defcb35f5678a3dc89a959
      df933d9391d37aa215080a2befc7e27642f57213df6c5909ff8bb167c323f1f8665dbf515f3d7f73
      07642d9354582a9d90f4418a0bf6eb5be2c4acf3b12bc632968a5a7618affa00492d3447a6b9b02f
      55e45faf8fc4616cadcc1fab0d7d58fb70888e7c036c9ca6f627b1c6e285963e3c36ed13a56638a0
      d77f1558d56edce1d2b83801a685fc20b361cfdb8e531de37344035dca56c704d7c633d0adb4d535
      136d084bca360f02f8b82526fc02b9d2f5b53caf19639906dcfdcee43bdbca03eb845a18d87854ac
      9a0c187eb1b0136d56962ef0a7ea93e710a44fd3431c200ef62aaacb4fade0ea3bd8b30eba6c5527
      76ebd0d68b9f6415595b227b3614790018eb18d6f0d5c46b89c7aaaf16f06ed656ba3dd4a4cdec72
      9eb48f7369720b9ba86f86ffe9ab16606efd1eeb693d9ff10981728da92db153b8a771564a9e0dcb
      d1e1bca834e964e60f1abe09a8a80f834ccfddf770000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000c0e84365e3491fd8056024c500000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      2f8f35345483a5b055de339d942c641ae2675555d4cb6ba4c216d0b1e8e25d34e005f1769581c0d8
      75a487080963aa94ea6983b92d6c854189f00025e8fdb1bbd0cf27b4725d80cb671c91f3dbd2b683
      58a36e1f97b26b0e57ecb1636174c9a49754dd870980b74eb6cc370844ca2d64a8e4da2e85bee87b
      cf3d0e46c37748d90601c36fbc7c53487d3408ea6b972cd32324bf4f536ac03c1ecca8ba4d7a2600
      5475ca9c0e744490222004388d92007d1323e8304cbdd84de952c4afcb9d08d5e9af14bed167cf9f
      142556c78bd03cc1963a53cada65464a6076683772e47e99727ee4b590b30760d6e6a3400dafd0eb
      54b2d144a69cad5ee6f54eb7c5d2b877bdfed6ab78c6efdfc3dccee506dce39268971a3921f10c33
      92eae3379b009f75d01f408588d424fd495fc202f85f44145263b562a6530a65f8cab0da2460d29e
      866279405f6dab3d67c0ae5220d1cf0f6bafdeef7272cea48b96457cf242206944367c38198c9569
      d89769e1103e22297bf023b29cd8e2b35e9b5213d1122e0dd2f14ea9d7074b794fb5f32b8463bef3
      f8bf65aefc38f8b909e4b12916d94ebcf6dd3a9416ee803c347fb9f4829ddcaf472e97a827bb53b3
      353a3fd6484dad111730b86a376644f9cdfa4a271e06d45eb09b416f5696e53d80119f1ad195b4cc
      70e27f21bbb9e1be1fa723959dbfbada1c14dae01fe53d145541fcff8b1b763b0642a8c3d5460eca
      80e11482074a18d2e2f7fea0ba54c9c0bbff3f4203000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000cdfa023bb2b12431752ed04e00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      4ce4b47606c63619f9cb3f0161ac90d944f8e7afb089b3c416cd0bed6e29963754f0f5ad5ce88bee
      f516de7da836fdf679b39b90c2ef9b98a7c4c5080c21fd273c9f4bd26bc02a7dae3449077833691a
      f7d2e60f7016983c21e797b0951ad59670a5fe1d1b0de8d97894200ecf8a21bc427afde93ad9dc65
      500e05eff8e7be7ddcc6cdcf60df1c7d8f49ba4f6e7d6942c2a34392e1b3ee48b5e59ab125e49c7e
      8bcf9944c423e0173d875c2fbc01d2a9f3ae7df1bed6c1bd5ed6b2f5809183f8c7d52811a55d656f
      98a2f47a1808230d6aed890f24b1636ff80f59ad4a8fcf6487960c89f6cc4eeff0a5ddc6320f6155
      da73af2ceb7f6de1236797de34981763d7cedbf09f722d83556cf470a9aaaaea382d9159a15e931e
      42566e53264f56f419204b7db01c79df98cb273537977f402cbcc8ee3686e159aec52625834acd85
      3973adf789c365c810a54fc1aca995b76df749d23687ae6b624b696d57f7beadae7ee8d6438fb3c4
      7f1ed24cb982a300ef7086414a904d6c17708f1743c568461d4be67c43158ed04c75425c62eba4f3
      a9458dedfad1de2412c85e80de5a1fb965dd0e48b0d37d8d39fa502fb9c658fef03516c27cb160cf
      91d49284b5d427bfa3d578138aad67f0b9f1ef214ab1d7095cac9974c1f0616917503803d4f21b8e
      fcbbf63ca12e4c15682c5744f2b5a21ed3269bb668c1f9132cac7de4fcff07b61ebbe0624df84d7d
      549e33f2414460c565f54e9beb3a5eb5efc437dc94000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      00000000000000000500000062000000090000001700000049ded1839b42c18a2053d27100000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      2b446ac3ad5d6026e9bbab7e4c96971717a9a598e2449f56b7be9695c62ca72496692f5778dc9606
      720d267d9afe035afb595bb67a06efb9d19f165c8c040c22a48a0293957e4d694571061b9767d389
      5d2a6b9f28aa53d6bd48e5788ce6b2fd4f48620d066ad1c811e5051a88572cd669c42b607bbd8da1
      2aeb39926f812ddd6b8c237312beee370d4d861abbe1c32135cbf8c72e100c7e669cf3339a7bd664
      f1374a332d20dd64d28d15ca9109ed5e67b2e5b743e4af67bdfaeedadd7a5f99890fcdc854803384
      d4394fc57813bb7b452221d96cde8509e3e49225acc660386f18913a23d303f98bda1f52f1ce8cc1
      4a15383e67f037aece9255ce15993326bf143a78d707a8da25ef2b12af130b2bb3e3cba5bd12ce31
      502737667117519cf44faebfc918f47a6d3c60be01eddaaf28f58dafb49f57b08ca2e1ade6617c05
      8bfd7986398345bc3021258f22b24626c8c8acad3b359d90b2dd75aa000d5fdb26b12478d732273f
      821f4eddfa422cdd13b26ee7192e2f5cc7b5c479c88d2af57de1633a7f8fd398e1a02c0db22ba1f5
      8c08c5c9ea7e8b21f6eb24389a17af6fc3ce826e10f48f67c6a10205807a7abf25a83d8b60028d5b
      950c71538dcddc8aa6bbe07b17856020ecfa3b3597781fe807653bde1070e7bdab88112a8b1c5db3
      dcdc6d695c9a391ff0d28d33f186aeb3a66bdb41f08259dc408b50a0c30fe99f2af51cd84e8cd60c
      96d1b706203cf38f09a7a243ce12ac0adcdef89c29000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000369a327582d851352c3c897c00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      14852f1b4d159bd885cebbadae9fade546845ee5f794e5e892e97373e493a0c9a2795e9e81ccca08
      32e663728d1c1a52081150f8cbaf1881dcf82af590baf32ded7fc51a3c001d1c1c7867048b624df0
      1aac0f3bc100755d8ad531cbafa50850a3ac8240cd24f264d603e0e8def7c5e677524f243529c99b
      9f7a2dd9d81337117d1f18409ff12910d861b46544f5d29ac546560056bd91c01108e6987c2d9692
      211ff17a0ee2fb4a0528fc83cf6a00f02cc9b785274aabe5a443c49a28b75f55c18c228dbb621508
      9415ae4711580962a9c6735d8d275f28974db835047b60611af7a98c9feedfea46ef0f8049263acf
      d73ed7fa6df410b16648672cc78cc5009462a6e031cb3ecfeb4ccd53471864d2dfc9c486b9ac10ab
      af1921b2fdbad50a269ebe13f771b28161c26eed20647661b6d5b2ad3e8c7ff71ae09675353688f3
      12f678d6c256c70085cc6180faefaab18f0be619cb299bfb07e6a823469d24e854be0a1a91369fe3
      346b05647359bbaf0f2d711c083fe3a130af1749f7027ad7752312dac202a5daba302dbd0f9379f0
      bd9167b7592e9e9dc59b5c0b600de0e0b17ebe368998d779f092ce5e7b9760eab53d466d1fb12f11
      b323b82212201cdb5853066f4cad920d426eeaa942431e631581f7e8495f308fd8b6e5458e6ce14f
      7a1952246f1dd5f4730e94bbab9c44256cb583d33c913050f991b0fdd202cacc925ef5432beee60c
      ab97899bbc9e6fa507d361770be02c85c7a9c15df9000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000ea7e9c5525e8a70d1db65b0100000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      5577f22c9e3d404da64296e317495b8f6f91f7eb9ce2f6cac0f7b59e8eb79129239761958f3b7cd0
      dba127c0037521fabe55407699889596b9d946c5e6f24b668859faba407aa18fb6c9fc678b7e5bee
      16787d5761e70e3e2cace23bba02e3cb8afa41eb9bbddd152feb6ca385335ebe964b51cfc6755eba
      29144fb9760a1d78762d74a6d747dd7cfd366e74b36ef0bf8549d5c4ed4afbbe8e81a897b9edbee6
      bde673a8bba1ccbaf3bc3ceb256e5b90c34a97f0eb3652a7fdbb0d92bf775851945250e70d093178
      6ddc6ece3c5869f67a54b4546edc0ac8ea7758b8d319559cac01405a22f88519fd0a363b85993794
      1ccde84d18b7ff92852fdbc17a756c105a6fbbc8ac8634a4c954e37644bcdbbd5e795c7cee3b9387
      98619103b63b36ee3872daf157f57e8b683d0b6d58b1c6532e75c902330bf4f3b30497c3985e2f94
      041e583fca5b2f4a10f7cbc1bc150117d5a9119f8f878d01cdce3ee3c6fef00022626fc9c303f7bd
      76851203769f0ec7f4247666c4ed3c8ff7f79f791ec3bdf98bac497eff7ff808f155e07f75f071e4
      88719d76d27eb980b5968cb0fc3a3012cb6dc2898ea816790c6595185feadaa5f64706491bdda3bf
      3bd839c37b99b5d486513e455be6b0a01292ecfb7e3a9f8813fa55dc53f368119d4e28f026536dfd
      56d3d3513d8d7f35dfea5974728fbf2fcfdc4c49e5960040b809f59d5a536675ea843be433e643ef
      3f4f983554bd8015d89b8c48af1e83b70dfc6a15e3000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      0000000000000000050000006200000009000000170000003d47bfa1cde4caff98ca13c200000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      60335e7b73f56158c187f70bf768c8952ecb7aca48fa6bff8cfeda4f83892a37cc9fea346ea5a754
      08e9ced737e4a84609256f19adbd5e05213b5e9cb4bbf99a371679ca9d198f1e5642f29da6db3af4
      7d61dff2552d798ea5d2c352a26470c09510b3a29f3b606c2e7e2a4c110094987cf306a20f3c8633
      8a4656d642389e39c7161a79bf00275251a07c875f76c25e3c2de2f9158e55da3a7a202f9cde3742
      88d22d71cf89d276b7e9371c9e9855753615aa13bcd92153341793b5d7da9c344f70bb8c96c9aece
      f95e965644dbd3ec7a585ddfc8cba688cf7d5b3cc6535e0f9c9f66a0a47a26b3050030fe74c8d0e6
      cee1f2fa88b4ec31cea77d038bc3e911d320a1cf4b49f2be980a750db23ac89ab1710cd64e1706fd
      9288e7437b035642728c5dab701c7284ebd5bd3a0701b38e0d01a34cc455259258730a9a61617c99
      8d19535feee3c56917c96a66e0cf955e777db57ab4cae49b982581a949e95df89823d327e1d5b3a1
      a2f5e6833e40d7ec6a2d2043c532ea6a730f1ef90674371211344a518e676e8fe6b9a7ab568322e0
      a95d7aa8e3128fb092d05beebc90fa6a0d4c1613f804ca04a62f068d3438741e7f2c6da07dd7ba3a
      ee35a1b95ec919b28da68a1c8b0ad58fe311d6ff43a5ab2d789d2d100bd5046df10b7ddb1fd67423
      0e16cb62f91be6db670a393d301a9caa564b00182f07a56034a263fd2f7431dd09da9e5b4cf1fccb
      59b9101123677d5465b6303f4a17bf8d6ff15f93c5000000
    "
    ),
  ];

  #[test]
  #[cfg(feature = "openssl")]
  fn test_openssl_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::OPEN_SSL_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      CRYPTED_KEY,
      MESSAGES,
    );
  }

  #[test]
  #[cfg(feature = "rust_crypto")]
  fn test_rust_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::RUST_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      CRYPTED_KEY,
      MESSAGES,
    );
  }
}

mod rsa2 {
  // This was generated with rust-crypto rsa

  const SEAL_NONCE: &[u8] = &hex!("ec52605bb40e51c144782320");
  const SEAL_KEY: &[u8] = &hex!("3ae9d60336c318d2da12c088cd89831b30115bf2e3a18b878633471b094677c5");
  const CRYPTED_KEY: &[u8] = &hex!(
    "
      3e3da1a861baa34cce0e97e497bdf4dc80cc01eeacef65eb9f1da709211b7a3a4db495a5b60de937
      c1e94d5f8ce8270f11dc82d88ee2a3f843f63d81cb3b20811ac1cc4741e3548a939605f261a2e8f3
      c3b5b66e2ddb0576de228259188f1b2a10e697e485a8ef60e606f85225d541a1cd9c9c16430bf386
      14ac316d80613f16fa14cb56b7db863919297834edc42439389c198c00123d825dd3b7d4687c55d4
      c3b11f604ecf5a8cb96026aecfdfc789ff43f808f52a08826b7d3ecadcc573579742f42b929546e2
      59710c3acce83850bee5a5fe161c0985066c7bbcd98d8c19c356fb16938d7161b3450a3994b3977a
      87792aa1e77947d5cee6000e5c85b4158bc70e43a6e1cb42e6d022d31a252bb669672f430f640a74
      51c06edc5d685fc2656e2a65e58699f5474f8b4f6d1d5491b1dc481a8f4119bc269f0d736ccecfb6
      a9e03e9d3a65de76b2dca47188a921feaa0a7635e69715bc70dd3809bae56d6a6cffca938523d3ab
      7f26fff06b82ee9df10b22a97839065f9b7f0fe7f15abe5834b7293218d38ab9cc9e2c63b1058070
      76939cd2912988f4e934c145f3e763f521e05fa7c7d5a06b723d87dd0c7225a926d73ab1f9142cff
      b1f6eb396fa4b1a07096bc10bcf640c34eae8e350f02ef5be3367fe4e415fa593706d1a0fd753106
      fe418cdd2c4cb7bba1c5d47b7411b3aafd2a4e9465cd96759692857763bc222c38f237b6491b4208
      146acd4681315d041a0389a12c445a6b90bcf5d98c1a8836ae99986c6b2ac5d59a6a8c6335caf15e
      d25fbac91dafa0619ba046fc0658f765591d7d29ccf60356d7f9352616e466445406239b6277c48d
      a5126b2c0b94f07a96d21241ac0faa854a9545b8862b81b09e29f660640ce75e0913d81a1355bbaf
      537e7be82035c4565278e73e799159196863776e28616432f2f4560faa962ed275ae5939c35c80b3
      1bdb58c10d6aa4330dca34d69ede390cbbdc5abcbaa12d3697a632176b9e7dd91a3ae14ddf7ed062
      3b293821aa8d7add8174a1ad59ec1f390f901837e5860948c1c659e76cc0a10d38bf0f7600d54545
      ffb306ebdaaba7dbd671d34fa824a3c948de160fee3d7555be77c7edc7d818c449cd2f66420940ef
      dc1a10e931181302ba7a943880a57bad1f0169732a4fee319af336dc7072eb00b393fb141a53fe23
      31277070efe78494f3cbee4ff1d427f280442f721866f9005401faf2e1ffbccbd238aba9c2e1a7e8
      22f636d2626f4566814c100c639da9c6ec6bd2844bfd99a5e287fbbf2f6f51d3c350788e70a4ca2d
      23db72943d4201c33d763e0821bc06a8257eff058e077b3f6d4058589c1ce58bdb72a442193a78b8
      9cb39fbf45434ea96cda0ac934e317a8bec7a65c78350fe1e0c056a08def56022c8b06a6e274efd6
      88dade8ef76053f585703eef22f9fdcb6b30aa458bfec5d53ad2a03336d6bc9050e70b7245228ddd
      6e12795f0049f134cdd2b2c0e6da2216be1c87f1da7c46c4471cd137c771cefda232c2598e4878b0
      1cd87af33553108e6b48de805bbe97f0f74298e6bcc4dafdd7f55b0de771ec53cdf30f54718728f3
      fa09f0d7361e84b76289d556b2610f2b655853018d84e652f2b0e39560fd5cae53fdec345a6b85bf
      4702148f1abf18960a439f2e75022bb993a49cb5bea9e52db74d33fb3d2f04cf1402876359be5c56
      97a2ea1e178f40a6277bb75b6fa8bb513af7e33c1a296ac4ebb1bc1a6e7963ac7bf9beacf7f2b21a
      b8be933863c83bc1765ce0731c4e10ff597c6cfff6798c02acf0333fade2ca2e5f7619349b3082a1
      d3e21f8aec6cd11a09dba7353927df9ebd4fa868368e31b3ce44c3fe34bab894208c177b741b8859
      0d51bd4d6af4d9163d7c35d72e3dc59681304288b229171befcc3f6a7e092324b0bcaf1d9c8396c2
      944a6eab2f76e7a04a91fc9fa9d2820ca0d0ef7b33dbbca7b3fdb60ceb1eec8f2e3cd2ba89c61217
      04cfcdaf64096b845b26b5594e3af6ee6839f718a57cdcf77e2b439217d4d81426761fcb9a908dcf
      426f3f8d356abfae78dd2312700d8f75f00197e1446ca6aa9a7eaa45405482c860af55c866e1dd73
      7c5cb651710556435252ac71b8e9494bf96438153b343704657adf83bd5f6a95a8fb56593fb0b2ee
      38f51b55eac21f06d1878f1498368fcac0d2bf9033645ee5bda7bbdf054e78df6fc0dba876819486
      6815f448090e75ace663e62e5ec64ba93e4d2a92c04758a88e4f3be6caf6f04e21cbb66287cc4485
      7e30c9afa4359f26f5ec701539585aa38931e845fbabb4784269c5ec222e070bd11094d3c7ff51e1
      87efd2718b4f04f10b0fedf2358d3827760b83d31c08ad6bbb4831c961167817ed8f95873ebe88fa
      e4f0ed14dca2a7429cc790ea7494488c83e6c9160a86070098a8ea28af5306db2f19053c5cc5daaa
      d51ce20f82efafb61979d7c2f62c92d7f52b22983df0179b04f5a0690e8e5e080b85bf19cad7e4c3
      ecdb92b380bcc9699c55e91ccd691f9a05e7b2b3329b22903b08914eef13003945970288563966a7
      25467fef776dc650fd9804fdc4f7af9bd476d6330bf4dc8da596cfd6a89884279ff0fe63e6318358
      ebffffcf2e6ffac712aecd533b0e9edfeb9a70c1b35442f6f8dda15fd61de21bfd5377ddd6026fd7
      1985b2a43024099e906c28153cf70492aba2f54b522618ea6f0529bddd70e621a691d7f83081cb06
      b3e364ff3847ab60253a970e673bdca99d822f1cc4d779fe3553e6c2c3ef8b42bb8c38a0c77508e6
      94a4502acd0935526b10b94c500d3003b877a77da5bae8c5173c9ef022cda25be42edeb2e4157880
      1bad1f7b656f98fbfbac118e5781ddfd69cf88752905fb2ef4b9f18938614db8ece201f2d2dbf186
      a1c22fe91b27ec12371b0f345d733c27411284b1eb7508a0cbdad62ded94ca1f9ddcfd0f0ef1b6fc
      d0f2ba35350e5b17add1ab17ba516d712da21bdcfc1f2e85b16f1ca6a10aef20669e268a9c307ebe
      4ba63bf8e34bff7aa18c5dc7caf58dfd7a37655abff14a883c1e793d3aa83278224b7faa7ad35b06
      e8dab532c0fe74d5fd85fcda2a883023821116a1449db4dfc98a8c1dfeb5fe7d5e574282805eef62
      a86f536a0b9f669ae534b04adccccbbfcc35c2c7fe0b6ceef100caf588c9265beec2dbe6c852a115
      227c47e9bba6a83359f1edd33155fdf9d340a19bdec7d897cd9241907d3e6ca7e3039085a2e3c342
      e135eb12a34f5590b6543b8614fef6e2a290be4358ebba3aff6ea2bbe60e46eac776de992ba361d8
      6d8a7fb3bdb13dce1f299ab5248707ddc17828e6b3998385c863649d0e6039d6ec303ef9bae9ee15
      7600da
    "
  );
  const MESSAGES: &[&[u8]] = &[
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000c2a19c60aa5abe627b43297b00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      759f120dab5a47b64a77933de43a102770e3c170a9ea8fc9144c03df419a6ac9f4db11b2d066a8c7
      01e49f45fa33d7a1bf8bcd15a9712d58c6cef6b77b25ad20bd6ebe4b7d253d1305848d4b1828c659
      a436ed7a66c195d949b9cb7b85910a39d94100013e6582701c903eb09405c082cc25057da846fb98
      20f5b2aa499e64f5d93f20d1f17f463cfb92dfd39593d1485b018102c47f42d06b1fdcae3318ac9c
      eba4233faadca93801bc2b1621548eefae5b2960e187348f88e227eb9ab80dad3c427b1ea7706456
      f50ec6e483e20839bbc0b899d715b3c6641275f878ea0be1a7931a61a3f5602bccd75b089bd8c635
      3e79989df9f466d2086806060318ec08901a7ddabcd3161bddc5e3b298d7e5570fe2ee72500130aa
      770389a216d04d07cc8fc05fe96a6074046bcf49f357023b29d8151c5f3e73ab17b3430144c799e9
      ba0a6cc283af72dc2e84762eadebc9b5006e1b15a79bc8622c4b0eef7de1585fef5c95d0b069988d
      3d832291be51c5159e32778500fed9508552d3f8f4317c1a2b3875f8854bf79e375e2f02700b71e0
      da6c02275bb5ff9691813ef297e58a0f22668c780c54feb0b67b46767c443ae5567a556cb393accb
      6358872435a2c29d8648e4c84fffe5a869698e39ac6bf3cb7d905a12206d530391e2b1c89be4bb4b
      fb4c6453d6ef3f71d46b3cade45fd8af48f4c6a741f698100c420e3a951e6a8f63f96511fc875bfb
      1ade04734b65c0b5ac846727411344d7055a956bf7000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      0000000000000000050000006200000009000000170000006e89ed9ef60c4bfc9dd89afb00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      1d2d9b761f0b392eec6399ab9ab96bfb17c911b266fa877bbf58897543bc1874494becc91d33b682
      469b3939b699580fa932cb42f6c45bc732d9dd96172f105280a22795d12a3d46dc33b446cadb115c
      15e07ac0f7cd6059a7836a4b633cc9a17b401342bc6bd19f8486a6f1e369375512be7da6582f0b06
      b28bad5836277dade39876c8623303aadf72aa046e9592e8d2132ce297f0047f3973f75f701cea7c
      4b89e05aaaaf5dc2ddf838b691695eed75c75e9b02072a9fe8bbf2c91a2c1c0ee7968edcb84d6723
      219155eb15fcb8a067a1808e02ecac24fe88f9fb363a3377b2089eaa7d768fdf94565350a9ff106e
      10fe3cd3a929de904ad4d7170c947dc10a885a95090bf1515cca91b5db6c11aeb1fca9d01ead1af1
      323301a5eae81d235754571d73ecc4c57c0d039b2efbaff0ac212547eefcede71a9fc3b0d63755dc
      4cb92a927c99aee9d0e1fe57dca394907ddf0d2225b0901d9c493dc5886ad7f083c3f165e38cca93
      f10c49bcbc8a3c7a0129f44d76fa2e0af29c02196605ea469c11943912dcd33f8a2d5953c975d469
      4f87177f76a7f7124be70cc262fe39154826cfc0242c8aadd0a594bbbc9630e8dd47a1afc4da551f
      203e5927bc3435d19238d42dd7d276b2bfb9b4bc40fe6959c3ca083e2a0025873ab363c4d3131f64
      01e2f6c318c5663ddfa1980b8e748de36c470357cdc04237238cf8dea22797b71c4146ef52a8a9ab
      5e17b0e5775b69ea25805a055ec3a4914ec1489679000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      00000000000000000500000062000000090000001700000039324c6f967f582daaabafe700000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      538869e0f12be3a1a7bda77b812022b7a02854749695fe0c0d4e127be690798f77fbb2baaf398b54
      d4023278502651b8c319106cfef706e77dfdb07f556b167525f437bac9aebd4c4a42e432360175f7
      2bf7028f5a1cdcac021d5a3c81e4682ba2180c917cdd86bd3e448a4bc92e040707dd4ad56003cbdf
      8f6f39b45a70da07eac86274f81121d4156b11a51da30a2f899ff88c577c4bd0efed868b9a98dca7
      a47c79243af44b5c4ace5322e3498e987a440980bac6fefcaaa080fbcab7d9a12574a6d3df577351
      aa0ecd18d19e3f49c0f6637c8db56f31428a755a7b4cc96c36f22a61f213772604d984cc97fde5d9
      3e446dcf3103ec920adc413ddcb0a3b7b785f9c34a0b5f6901d00ceb3c1f15290187c06a1d577f2d
      ff46d1da8913206950cf321e1d15f637c45e5ef2ed2388ba4f0769999ad9654e0c57c070e27320cd
      6e45966219542b1c34e5a7d9a1db79a2ccc46baa0e0279db12d552bd5118822c46d497d694332bf5
      eea090a5eb8f8b3daf8ee14de185c280def16fd2b5b40bb0fb2f32dad2e1f0a18794800c6608ff06
      61b3641c5992e57d97a6cb730df9c582521d1be437a13c8d396226371f813746085dc5836cb04a00
      92587b6559ac8a13c95665d51a0bc96627d0f083efb417867f111ecbdcc4bfe5fbb953aa32f4af32
      d7634a828cd059d15a7d4648d4e203721250090dd3506692a8f8bbf64f3669cee9b5adcdc3350232
      89fbf227de6c8f6f44222900db5c84b36db3c5b877000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      00000000000000000500000062000000090000001700000056bc70eabdfa8cea788213ca00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      0a5cc98fd70a464f04f44bfc06359fc821ecfcfa91a1b4e64470d2cfeb1979b06088e266163cc850
      b45ce6b3632f7e899a9e6ca528349634df88124b42ac3a64c7d7fb21069772777204bba1f5fdf2f5
      e4ed060503b4b17c58ae061e4dadde0df23bd080360e4ab56805ba9183ab8d3981dd1536be104234
      d4fee4ce0b350a044dd91058511e27499cdd9b6a5e48bc475163c1e2105e8ca0789b39124ba4674d
      e4686e3492746d6898b8c3a2902334c25318bbd3842eee83d5fd217deb89902699a86ceec09f50cc
      2f6b3af049329aabbdedf71764d9933e40539d0381efa07ede9350dd5399aaf3b121e85ee10662f2
      757d759f6da1aa9f21a5ec005a123b7dc40f060b9ceec5ac362c664e06016be4c44278b3346a022f
      da8d92c49cac1e5473aeff1aa3fe6336232d5374aa95cd59f9d110e278ec01702c8dfe85f821d025
      121eb7ea0362a33ebf87ebb73aadac432d86b7b162e73914ce7d5f2e4b471d89e33508f4b0e2a1d0
      4a3cf5b3bb03ad238be3ac44aa9f8088f100fc9ac4681b9bf58ac2324a6e3e3513e686a834dd914f
      40c905b98258d44b541b9ef3ca55c13c98c9c2111f68d5df1128af36d8d1b79b9dd4e10b2aea1a90
      ed0e581022cdacce76e24f9f5741bc2deb50d8971498b17782be0b3e282f720efd81f3c6391cd9b0
      ce12dbf6fa2cc59c294beeed283c59cf03513af62d6792955bb6f38ef72e8c4207a0e4fe800ff0af
      716022c9128b41ead0f92fc2899c4d7c5cf25263db000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000536e25dd0457aab29067e6ec00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      9ac79862577231a3fdd899bae178d2c78845646311d551908ded7b55bd562b44f73edb49372c9b3d
      1148b998900a9e48eadfcbb02d2c45c1345d0b78b7ccce1a469c4cfebb48b0dc91f8960721093e3d
      a559802126a6e053a5d985c3ddb1e0a19774928eff00b60f490cc3ef8df4bb26c79e8a906b22c308
      13ca44c69de1337aeb5b2704984b1b780499f59dad9850c9a572b9a4abcf23cd56f25a6d728f8629
      9ea3f20e2670d62cfcec2a57f8b9ff3a4964d731164f0a5ca6c4cb547c1155e5220455a2a0b8d88c
      73d4eb150f149d4fce46812f7afe776df1adde347e6f5949dac4655db7dae11f1548c90a605829ae
      c084b35289f181fc153162c56b9646099dd1ab692097cafe76c8466183082c03b06901f2e9c5dbb0
      3d6cac789f11651b2b3eb8d040c424c139e797405044afdd72cad84dee8db88ad6ff3e7c0254950b
      b4ee9f38896fa4f8da21790cc47c7a7bcc122753ad75159d46bb96cac3487bd3b85bff44e7b0a59a
      a88b82d3f07f110c9ec74a3c80c754f8b3303b0f6126ca03e099e6c02665a0ec706718480e1f941b
      9b2b0393c5a9a3e9fc2a53707c017a0bca137a6990b9327fe1c3beb9192eb0eb6f7c19ad240c7610
      fc541c5a144dfc0c013c0668cfa82c8bc8eed737c47e2901939c976907b1f84406358644c28a2cac
      1917dce48b59109603ac9f56dfc8f2bd5e91ab667cde8720c4c5902f8b097abb4a0808e2df27ac28
      e90e41641dcdffef1e6747075374a3b0b172a553c2000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000cbc96a23aabe40910128849f00000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      4bfff100d287f7ac820881415b51a969aad6965ad7abae1baad522ec551d81de561b6f86583732a0
      ff8b46f25bf9de5c9fe6b0dc2f0e1e643eb2a5bb84297fb5adccb09e004a8a0d0a9979bd3a1f395e
      b527058a9a6cf0482aad5447356d578094dbb8faa2e0364cca9a0d543d93836d770fa22623ada491
      b7973879e11e554e44f7d034ad8c0ebd76e333b76622c7a6872269692df5f2c1fdd7577479a032f9
      904e769b4cb7bd6709b7590a10290f986d18728df4c0cd1b1861ece781305292fafaefcddac29d8d
      b6db09e77ccfe5d1a4fbf79e849787aa58426c7930a72140cda6494f0cb59b62a07b8520fdaa006b
      a64593c91ab9956cc95554f30d2d48635ac7fd9f6208ac84898a1be61559f033c2a84ed25b91b77e
      f014725278181904526ff5019e25858f5122b9b83fb067f31a52dd74da9c1358f360982d5d0daed4
      76b3dc9f92c55f62641d9c5fbdbc92d804f80e71dbf45d332fb7ff0a1fd291dce264f261caa0639e
      e18ffa470f46b9c942cfd34b8843aee6896a258babdb356e3d292ec67eae66fb399b2153188792dc
      20fd5190dad37d7f5fbc12818e6ac01691f3c9b4bf85231f30e9fe81b52258343c08440d219089a9
      38f6eb52b9e470f2b22448c6fbad57e8da5d14c378adfc4182b916a4834eb3d940f0db5105f3a488
      f90c164bb1c35b3d3fddfc57375b0c82ea7fd7c93dce448730f8064a7071a54660804c6b13e50725
      83d05fbfc28a1c650a18e85e6c5406db2304734e44000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      0000000000000000050000006200000009000000170000000ccd959481f2ac68202be96900000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      53de6bf521ad11e55ee416a7fd32f41aa7a3907862ba4927fc4d0baba3c0fc35abbed4f5eb5af92f
      8f18dffde651e4f7d1e8d52fbf8373890aa7aaeda6f82f46b2ee9cced3edc4358373de69af233b09
      1566805e1dc0a85124151f7889e6fde1fba0b1b666bf9eac0a81761d974d89462a550327a9d79e1e
      72f785ee275c98165f3d5d73f7a2b8fe24737b93b1399d7f0d205a7c49f5631176ed3616310a0be5
      5bb6e8be2638b09769976a2cde173c12893d799a7d0b0db57cbdee34168f66f795d917b41b356f29
      a487fb0d9484f6d15dbbb52624d9d92f6a92b0c5e846162ecd78f6e28e5a69131d813910985fe21e
      c9da85b809a5795d728c0897fdf04e4d6c81722bcedccf90761bfb53a38ccfb2a1b747fb2434021d
      27f54711d2bc09d16e6bf4ea19e4738078e2839d924955165cc511637ecfb9b37af0a10c94886518
      b34c2c7b3f85f838e8e0cb4e376fe088faeedf6cb85717c114bc14a4dc7b3ce21de38a253db28f15
      ca15da69ac6caa0a944264c6778ef5295ce22ed2708026f7aa0ee4aa79fab368bc1ee16cce23e741
      1e2c3ad08e126c0b69de2181a136ed14bc173bedd7e98def065fb2da0f9d7cd36403d24aa323766c
      3bf5fa34ac16cc5d413e9d67c8e749c0b3d6690fe56d958b29b31fb61d44b721763fb8da081a43c4
      8dfd4363b5d42941b6fdf982d5a196700d447cce306ceffff965db3373034f680c883be8e44c0ac1
      2e8aa5c743d42043e7d1bfa15d710cf8971506fb1a000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      00000000000000000500000062000000090000001700000044eeadc9425f1a7c88480f7100000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      3aedd6af106708ebf862bb90e5313f14470ba03ebb96bce3c1d40c5928868943085784a9df6ac238
      86a1225f588256f0deac84b8a22d12feaa51a41eba12c5110f325d71ed16a70f6bf2b1a382d09486
      31541eec772f6d2de35db4db159442af25bf3c4a32863f692a21fc2339098c0c0027176f78e172a8
      d2435059bfd3ceb43e450506ee54d6a5871b3393156d27c6ef90203577426968ad2f4b735bdc323c
      3ab8490375ab6056b691e242eaf2da43e98acb3e66f0708a54a0011beaa7dbf1b652189f7a405d7d
      dad41bbd43ff0de7e8c236ef5731c2ec8050941a8a0cb00a09d534079ab860e293f89b98fd195a9e
      9a3597619d937f412424575fc0af9fae6875b238fdb0ebc73b507dadbbeda8083d6935e3a0035ab0
      54c60e8ebe09205b43e4ed4ae636b222e7806e173f2e1705aa5a87023ca11eafef5e20a27fb1d549
      564256906752622b6ccd1bb4191ac9f669df3b718e28ef07799c93724feb3bd6d7d51236fe0f6bc8
      8a1a1f10ed120d0a5ec217ff7ee63bbdb7871d60d7f733e50eb485f0f467cb9a490da9260e0f3afa
      c689b65788007b8e1bccc8619abe70fff8863106a940bae8d1bd0ab5bfe18ef22c79fe0d9cc8527d
      ff9fce03615cff705f9a6152b1658ba1d333102b8bb04c464441df8e16d8797a0544d61dff5081c2
      6eb2695d9302ba40a0d44d4533106e659d10030fc2ddd0bc527858673e76f378f1817e72126d1dc2
      e02efec0d48af2435e91bf0bf9958fbd31a945b0c1000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      0000000000000000050000006200000009000000170000002db1b079afea1d9b0074a9c400000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      4fe9a72e37e98bde087eb9505ee651084b978b308f1c4a1a2f787e696994fd232963d10f9877c2b1
      23bc79929525821347bdcaa03916efdb87d89c5faa79fce8e61dd6d751b5e47ed572aec732c8d11b
      f9933d15f7a6e2e5866c1e86b1fd0d11b4952e308de119bf5f971cfd2cf5d33fc9bc2dd2a40867c7
      ffdf709347bf4e2057534822b81e35824cefd096d6fc081e87bc23142b7602a7b17ef1cd6f248cfe
      50fdabd6dc5e142b59ffd605c91c3c57c8ff8525ed70157abb4d20dc739b9618fea96768071424b7
      58048d4635b8139168929663596fa398bfa5a6c5e24fe6efed7c8bba2542887eda6a6cb09ea51752
      0abf2c9ac64a0ee9476224a03259edce159440a75e73b6ec08eebf6a8b890c9b24d0a31353ac6c22
      72f9eeafbf85ad4bfe83a36962e16a1cf306043e36a4c9d4f59b5466452d13e0f4f7611d1fb7969d
      c8dcd5f833392abc8be41bb7ab667b663cb5fca34370b3e9464396749ffce5bc589edce19bf4dcc2
      fb01e77e951709cc4b91718498342478195cb4822cefddd55074ae9a2c0a8faf93f3604450062416
      d8c79d1dffd31bb88fb51199a90fbd5b08de15117997cdd321ef69ceef8a83252127cb63856c5493
      ed6ed21ed02f3d2553ecc7cb99db8741fa1a6afdf6247b184546f6a13f61b2810f7b585bf0c1e12f
      5049b11e3bc899baed90aed50bc3e74c90152e113a452d737749aac2491118f24ed83ca754e1dc25
      48a6326688387b5bb48a15cde38579ae08b46f04b6000000
    "
    ),
    &hex!(
      "
      00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
      000000000000000005000000620000000900000017000000d0d688119c56eed120eb2c2200000000
      0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
      6982ca2ff344231126d8dfd6ac3a579a1135890c0d0ba2fa3989cf94778934620cb5aa5455861336
      0e8d7adca99c11e0b197d6cc9686b706a56822e59d6eab5a312c9c879cd1942cd9885b1c05b61bdd
      d1bb76becb39847134ee51f8342cee8420fe238480d6be236f4d9a969eba2f46655bbbeadafabf9a
      9a62b8918577e848d8b2c4aa5aeb437e6f21ceeb736d195d29151be916c97e46885fc806e041958a
      770b5c25d731b67bed0bce56c81b92903289bc81598cd29e132569dfece3a6fdc06cafb477f889f3
      7fe66fa588e5fed99af08b027234bbc6e8860f041e8b87cc4defc67322736e13ca3d2edf9767a842
      d02f4b6bc90f6abb6fe5e9fafe6374dfdb5d831f8bd357984863484fb6583c10e2a141a81a8c0916
      f1a50e3533ac45f63ffad1af18c00847c14e9bbce62ab4cbefae174de49ec88fdb92f41ec5672631
      4715912ef11a4c5e2a63272fabae0624e3254ad37eff9c0fc02b29b2741b3ec62177e78bbf6a00fb
      1db76850fefee07ceea19d077e41983e24c8ba44c0b7d36ff4a40aaa50772840792b048d3e1d7141
      fef90de538db6a1f5fd9527f52e490784b4ce81ba7c5e05f12bf4cd39749df03031baaaabcf7a53a
      9ed290c674b42be93e2863e1b0fb385575c124f9a073cc7eee07f22f0a05c6216fea9b4d6e5508e8
      021609abec744af44f89de865d00fe6d8b94d27e5780b53290f5be3cb4f1890996212e4e1005c9f1
      0fbe83992ae03ffb573052c46d60361caf13f508d8000000
    "
    ),
  ];

  #[test]
  #[cfg(feature = "openssl")]
  fn test_openssl_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::OPEN_SSL_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      CRYPTED_KEY,
      MESSAGES,
    );
  }

  #[test]
  #[cfg(feature = "rust_crypto")]
  fn test_rust_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::RUST_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      CRYPTED_KEY,
      MESSAGES,
    );
  }
}

mod rsa3 {
  // This was generated with rust-crypto rsa version 0.4

  const SEAL_NONCE: &[u8] = &hex!("0bb52a6ef616a784554b87c4");
  const SEAL_KEY: &[u8] = &hex!("844a1e675f642f333422da75fdbb5684416c16b2abb7f09986cb95ddfa3304b6");
  const CRYPTED_KEY: &[u8] = &hex!(
    "
  2af66207fc977850c857f95bc5e617a3fff0b8b9533b8f99126daa84b5e2e9622407dbb39d1c24a4
  e43bbfc14c24d1a8c5315dbab5f60add2d38f700254c7a02c54b545755c3632843489fa69fa813bd
  931377101066f31fe4c95f20afcfb82caf9f0f3948fa4d03233b98ce32d6754fa07e5f164e00e626
  eb873d1aba75f5c6088c2e927c5b96c6fcd6fe2caf8a255d74ddda23b6e1c5b408cc7978a7d40cec
  ab423629852463cca1e6712e77284140843b1a65e1bed26ff5aa85490562cc227db28c146725edd2
  cbf608fe030510a0a94cba9ae1523d8854a6bcf61a0bb7b985ef12f142d3419d0b98262042965150
  995556afbbb44d5985d9765e7278132393bea0ba2031c0e6b1d3f1d4862589161f0cf92633d5791c
  ee95bba92771adf6ab231ce7979574e9a266486173cc715641289d1ebd307e4b3783719662f192b6
  7f048c07e61630e7615604b398299744f5fb476ca1f90955e4672bfe9fbea586b9930241345e0494
  5c2bad41896bf01d400cad14d888b0fc6460736e1a4cab9dfbc146d9600838ad69091b0867ea7d10
  0035274acfac1e688282841bc357fe60d9425327ccfc36b178b8136665ce91a1ed9483d99ec37d6c
  ae2903dc45ea7b93f182534f32a6c96bd5f1a0368b9854650279363d768d44542be2cc174cef3ad7
  33657f99fb57ebbd81ed084529aa7619fcdc0893216e1a5d69387340f1cf064186e09c8d603c0f43
  718f94ef4e48e77180e97f6c5ca1f979894098b00e2e095abb1725abbedc99597764d24b78e23c5e
  bae53f6d81ee04590045b0fad780a9a6eba0888748f6746ccb6eee1da9e09c099ce32376cb81b0de
  df29ff799c2b80a0aa2b0070fc72a754687d7f1bfbb55a07d46461db91e3d4c7098c1cf7a6cbc825
  2945e68ca94f1afd1c11d05fd93e68c1cb89ffd0b00ac1adef5387d1002176aeba60df7514cfbf1b
  4e16a42aa295a49262f5cc642d6f3677fe186de67544b715ce650275e0942edd3d4c36c9bc7c843f
  2cfb39c41dd9320f593345f80d40753cbd4b79904d140cf75d32dad1a83cd8ee891289f722997c98
  5629adb98765180aef5e409a0736af0d2ada48f958701e90ba58cb64158e0aaad8a26464174494ca
  e93d8343583d13e75db353764783eff9dae1bfa1e07739dd815f5e89187f8e97dcddb1bb0be0d12e
  841b9d04531c8d690cd8749f797175e90bae24be7da205cc7ae963743f7443ec432a39ba21d235bc
  b1a3a8d47cec78d862dbc8b33dd79e3160b936349043c73e24aad4a26acb0894e64806282d106f1c
  029766df8cb39116dd740ba99ce6e4a1f348dad2070c2e8cd617493045aedfa5321511ab17132613
  a2a09e8ea981094a42e256fe6d3b2dbb3d09504506b7c9f938ebbd0911c86a4035e624e6254b7505
  61c5fd5a87d468c04bf3445037c79e5d96d231fa82779f5b5be6ead2964e325bb733da01cc6fae98
  cfa7fbb2c210a1a7522eddf22704a47469da00de4c9ca650999915d2a6202417555461271799bb69
  baa114862e7d2f19ad993a11dfc70f519991526854e6cc257a5e4f87b552f7cb2baf3458ac92e73e
  8558b882c8772a118ffa695d151fd675f2333deebcf9c47d763382b11904358021678c87d600f2f5
  ca4a56dc778c5194f1ace9ef96afe15537ddb2a875e9c8dbf86bbee280043ddbeab7e7ab496c5b5f
  9a6fab34657888dfe836b7c2a261bf9bc4fa04d3b30292f6011ef92b8816bfcb9dca480ccdefecbf
  b5a21141bbdde82876d3a94f70b90044aed55033b17a307d07e1a3df9c0e2c3a306517543734efd7
  ecc08181f546f1f53c751e96ae9cf2e7cf94479623bc0b2d30865d06f4d1286d3f08ef235cb3f282
  ff9dc9a3cd7e570475919fee336a64db94b55fd9d0b50be48b3ba9be591f4c54e304c4c65d7e11e2
  8bb4c4b5d7877b4aeccfe9eefea5efccb4309e0936a7be2e70bc6357bd7552d14c2c0571ee5d2f02
  76e99862a316cae1b3b8749a7918565f9e6f51241395ec681649e3f6a13cde5fd4e3c9e712b4070e
  6307fb4b66be5b18c90c488fcfe59033f1cf3ee53045a2dc3596a82e785a3413b5d542198deff677
  7cb414e10eeda15243ec1e2f6ada1192eb2cde2c5b572605e7cf3d46e9ef4689fe46cea1a9740486
  5040f90009abe144307822322bf2a982d88860b3b5f0b7df70ac4a9ae404aaeff03db0778c434f0d
  037d6cda44332912d3b2455c0bb8a167b2b9dd1c0f2c81f803730ce6a544c8ef8758d412d523b709
  d7000fa86782a5191a6c09a3186f23505151fb391facd1e4f0142369c5dd4f4ea4d9807538bbf992
  f8bb96824cb0a2c87067b372a856fcf71dd62f63d811e8f0f5f7bf2e75cde67f31ad8d8eae9db90e
  a8f96ead2c4be8b760ff7ac1e1b725a734aad78ba2e4e1f17d6b2c0fade1f73b7b1926ff5e2e4881
  ab17e975ecb51c450b7f1c9f2dc31704cd39b3467d8d379c6876c0476c2058a7a011c79fafe9ee36
  4ef03cea5c802779a391dea8d914b3b744385aa7b26ca4e7361507f64df037b455167300b101aad4
  74fb699e7ac79dc138f12e5afc2ea61ae63662b3fb9b115d5ec3c18ad00092ea3cbf7e7799b7d970
  99425b30c6bf2fe97a9e2c1755d4a04b877ac686099641048b5c925dac6c3dbd9f543d299b87e5fd
  c54707191710b616ebfb0fcf2ed74f2c35d653196c4db2dfb01b3f19266e3a57ac1af52eb4a2df03
  1532ecfbecdd9131a4bcc1a6a5919e65002863dc2dcf0c20c1a1da0ce662fc6caf7b0051046e591c
  2ae46d4324ebc3fa9345bcce9a85e7aaf248b3fb7a16759f931dc1d0fe099d9aafaf59df153f0704
  404004a09c3af922752f591998355c075d8c5d88e8eb4d698b42efc37b0e5dda9696b1f502a9ed79
  a59067c436ddae362242dc0ba8de7073327296b00cc56eefc1efb284175a177b8e853ed4188c2ee7
  6f3cd8052bb4930f1ba4d8bb0ec126e2f02ff605617e097cd8df7e2c7c35020f70c2e27f007eab0a
  4dad3a2d8aff9bc8c5557bc5a8035d94311094a32392f9312bb7ad1786e8587cd263c462280585ed
  31762a5180d8a04eb9e20c4068b1367ab61c6a62b2c4f1c38df96e22291392d738deb562ed22d8e9
  eabf82bc76552991b052e3812b668fccec105ae7cca1272f9d4baff12851449321d24f6a24c11786
  2957881505bb2569cd1cefb4e58a11f078e383d8ced0147103c9e737006d0367cae60ac0792aecca
  ce57bb8c7009b91875e19b2ed584a798f48ca1ddc660a280c18c95b6bb3fb47390e408162c71c00d
  7721c8749e65c26b8a899f25949d4498a49f322af16968000e98ac138175297f8f7222e681399882
  3d7f941c6f
"
  );
  const MESSAGES: &[&[u8]] = &[
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  00000000000000000500000062000000090000001700000051c3657afeff939b1c3b036300000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  00fac6b37def279f28a9ae0440ce777c0ff252736b61221e38f9213bbbe0266065e81b9292d7de42
  5f7e0999f7cf3e62652a7725dd3ae95e7febfa5d51803a6e1f2acb785a329b9ab8f063c604b1c51e
  5b38f367ee5dcf2061e31bdd47818211e2fff2c82e04ef0b49d89afb733ce698c178ffc63bc563b1
  c9c09d77b13c1f85fb0eddda34e64abbb9db4155c3661958aaf3ed63311b5b99231304b77c907e1e
  64feebc3cccdc07f7dbe56a45152d69d0296c49fb65e4b0f95056e3268912297ceab844c8a002af4
  6708f3b083167e8c540eb4f15dfc21b1aace376d88af21ce6e5e6319989d56f7d5001b2c411661ec
  b7dfd1c5a41dd49da1138eed005c8ceb953bf52f6b025e78289349e92218026fa03c5b2eef3195da
  9c9d77a4746856f327edcbec4bbafd1902ee68166780552ee8abd0f0a9c1515bf04af458f5ed4153
  2b4f5a8ac4e9ece615652889b68a17cfdb048cb109fe27477a1f1992307577195bedd1a89ef1fef0
  9d3a21a69d4947c7f9e07834ff819c46161f998d7172407ca8cca688da24484ff2e4a9c2f24971a8
  89ecf28b556feb1fa9c92f2ec5c0c1242ebf7c7df8e6645cb57a15fdbdfd5af6053d3213160b671c
  b2735f20d9f0691f91e0ea8d671eae69dce491da19369ba4a97cd4d1f2b03938533d32d6ccbac809
  c91e4bac2d411bdec751f06b7e5a62db17f4a03324fc1690047ff166194632fbd3eea13a2dd57516
  7f20adb905f78e15ece192521db8cf7798ede96834000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  000000000000000005000000620000000900000017000000145869a35dcc65a085937d2e00000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  5159240f4e743ae1121d03360090c17d685bcc0eb3d9e024cf28ba041cd80006ad1d9cbadc92e707
  2071ff7ce8fe34b578776aefaa7cb9de14d4ebfa844414b781dfef00540e9337fa03d461498a504a
  49310706a79ff33bcb1fc05c1cb68b310ccfa53f21af08012a2a5457f66caf5ab23262a59e6111f2
  2551a305dd1af624a643f5b931c061f3b45c735d8a84e5822198d863dc4d1115f43a2a0570dc21b7
  a66d4dd6841416175c58d34ccb36062996fd798f23cd9407a657f6e33a2175945991ddaf4f9b4f15
  2cc6880b3c9eeed38188240baab74150d1c040a75add0bd9e620476c441b04652e603b1d235f9b32
  6a4c5f66a1445050f44531473404f6178936fa589025522204b7206dc5f8070ac983229bb67e2e9e
  cded30602bc97289334f16a5b1ebab6f652741c0f9e2f04b48f86bb6fa055b801077f6c30525d397
  f340dc2f9747db84ae4c3adc1bcd11c3ed2b6e86b2c46437bbee19131d34284067696083c2f4419f
  f15a13ffbe245c09347d457176af2385b5b011e5f943f9bdcf7074b83ca1336f9e8fb56968d30d9f
  fa631f4c91e26c3cdab861c3e4bf071246d95ab3e1b97b275d6a45789b70802f0dea89034592278e
  435fc5b5462788c0a9a8b89525a55f2f8d2f01b01f957a4e61776c1b6106c6ad20e09b265acd6d53
  f6094e33bc7f16127d83a2c1a1954516ed35c1329cbf27db2c8c9cb8c9a6386bc533a6357e372096
  07d69b8f0441d7b1e2fa3590f233682b56d7e714b2000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  000000000000000005000000620000000900000017000000746d17cd1e8d2052c5f5e0b100000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  049fcd8f2e8c0c0cab311c6d269fb399598268e07af3c0be810c998da2c44954e78010bdec577082
  47cc8267f4b7835ba1f39d29598fbf5fb17ddd79913dd4f606e52a2b15fec8fbd960dc34f25cc684
  442cc5783e95fe07809edcac652b326a8f8497bb4d26918654d03ab466c9bcd9f8feb3ed03d1351e
  58a18aeb7be22062bee82807deda940460c1ef686f04441653ec06cc4db711a02e4f904c046f4f03
  a65288b1fbbb010934254b3235b2600b42880d2bfde67fc7d4459911b1d35a8f3a8d98ee85a08967
  14369afbd0434eb1ed2c8f8a1e50bc257cd45126365dfad28a7fab52eb49a8f52dd731c8a8d0d970
  aa2228550d29dc6c20c0337d4b40f0b37dd326ec50f61a0cbf6961f75ba5a4279495765469e3e399
  1c5cb68cbca3197d99a85b82a3e8ef62c24b87cac9c76b5df9a3255ffe638114ac9c21a52c27cb16
  f3d1e6b7bc5e1188630779af2820bd3d704e0b76f2f5301a935e4a3dd4c04470d1149a1b2e7376f7
  60488d1a0966c706fe4a07ee70558dae342533db88710251682f25fc4e430ed69ac3c27878c2b528
  af445743addb579a46d56dc02b6ef07f4ba584f990fb9cbb6566d9304440e45a80fcdbaa6afc2791
  d98cadf5bc87b82298b05665b497ecc734d7362d8689345a7832501ec4693db5e01ab91966fe014b
  8295f1ae23a6913466a0bc0e4c303eec6ac2ed2c6897ba4141e3464c2e9f72e78607b24ebb81e939
  068dda668317a681d150eedbca760863f54514688e000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  0000000000000000050000006200000009000000170000001343b3615cf090c0e830decb00000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  5de7bc972dee5e246f11f6995f88b516b62daa7dbf9aa2b7b8ba79e8a37286c2adf214fc17bc8055
  21a6146faedfbcfa074c22aecb4f3810855d9431f53ecc83ec1ac3706d2de6d5ffc074080eb264bb
  71cf7badbec3c0ed97f9f3758b79bbfd823158ce166e1dfdee20b11cf7a1f09cf2e8d1f212621a3c
  b4d4b6a3e2590bdb76ade7d042dd8b5d356186af2a2e1584ad427d48fa07287b8b99cdd881872e3d
  6cd592f052f98240ca13430c74cbf57861dafedb9e321f142916614f2d9b210077d0fad6f863f253
  fdd05ec72e4cbf23234e3526050724e30e1ac76e5be59a41bf8f38fcead5146d88b4109d6fc37843
  1f57203729edb8aa571df98b528ae2e35980d3c5543a4cc0766ff624474bbf14f55dafc6746dcf1a
  a156640e7a1bb6b2f2507c53f50ad89bfd868e7a7697f018856fa9a9dba766995641ed9ed9dd32b4
  f22f4b37c415f1cc2376d53cdab8f934654f180ae9d82ccb591d54cde9566a634883c2c0e76b6059
  ab365d2d50791195b2ff56c88e2c6d0870600edbff3b16dc3f785510f3977da6895fbf00e2716712
  a69e76a4cf2d91ed00f9b963217086ff198eb140627684724b3ea16f1c0fb7b3ac0519808fee7b18
  c968b35b2dc0f5bf9dd5b058dbe87e75788e3c6033ef4d820d455511ed0a60d7e643e0d26cc233b6
  97483145dfe135b93258e735efc19ab290343303b91c23e27ee61767b24f354b2a2fd3a73598dffc
  3fc761743bcf9dff60f8914d4f786f06b53c72ff11000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  000000000000000005000000620000000900000017000000d57187e995c4944b2cf9cff600000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  18b50f13f7e08311c185f4975d4df2009a1c30714ab18ea9eb616415a03beb0f1d9647d832da210d
  4c18059a8647bd3cb2f18c91860346dd0e80760b72c4841059663ec85eccaf6f3eb49a4c8f97956c
  a1d8a3d6c988541d9636dc6ffe224e99e81543e27a546d4fc2a86102bdc59fd013987b8206d4de04
  82af54878fa47a0dfef54e4ae1b9bcd5c18196f297e478a8cc3770dffc13e6e8c93c9be997d3704d
  893ccc3d83ec5a1e5a6b2ad693059ae7cd06d9def8dc9970259693f58f80deed3fdcf77defe2aa90
  672ea5851f79ae54224d9e3055f469dbe095afe04d481b72e883c96768ae8c3515290acd6e483472
  161170983fa672bc3199f69e97a5b59aec412937331c57391546f24de9ca3c69339a69727ae1c9af
  e325360bf81b9825003fd0af9282146fea288af1fc33efecc5bc3f85574c7e2b18b665ca7cbab4e7
  6342982fe9e1e9750f9b54437e4721a066ca2bfcb6b01cc2de666fa8bbe02351b4092a13d1203b78
  d1915ebcb3be65e2cc338de07b5dc46d8c400759c008b364e235259e3acc69dc930eee60614fb662
  e253ed1f42cd2a7d80a181b74765b98f9b72a44400364e7ffea1bea549ba81af7271a389385b5167
  087977138943d927a7c984177ef983627e4e616279fb2b87fb2e45870ba978da2514d57cfc495fd7
  a29473c7541b1c2d04ed2abd508f925b7130480a86c5afe1fff04fd0a4ee9b4e1998aeaa7b273924
  0de580f7ca6a0d7d5e6bcc6021db7a847ce0778bd6000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  000000000000000005000000620000000900000017000000ec521c1742091002ef846a7f00000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  297988eda54e75cf9d5952aaebd41e6bbb2a55b91f212d4f4f5e1df41b603df9c0ed77b31a1889a2
  5573d84f19402609594f07c0dcf2cc1bbeff1b8f1f4374421b532d1382139a0d5c4716f9db6d9563
  2d954a8a20b529f6929fd3fd1b98e55f3c76a59028faaea97308a3a51ad0951ba890767b85603931
  51efadfecd3783a3611ce8d893a54cb19792141974bac11cecc936717cbf417a2dfa54debe754126
  4bedc500e8f35c59343b2c730349d34eb5a8947450e2e5293931dc1d85bc9fba7d0840cb4ece8045
  a467511cc8e34fda8437943556cd1c4c4bf2998411e6e06d0feefe12a9b41f237395db7f844cfc6a
  f959278acd227e35277d58ec7866878f5c137474d055b75d045b088fe1be8dba3cc7ae5bf60bbeb5
  59d6f6569d5e04fb1234604322242512e82745b02f9133c0d2ca1ea8807c7c3935ec8c9b1daa2ecb
  aa0b79e681cdabf57dc3860f27ba0fc31c1c93503b56982dfcc282b9dfb43df96dbeb8ec3015d29b
  afbb5ffece137446fc832df40cfb51c1c3ec06b0ba96b85738789f2d77f4cce20f3cc87115edea8d
  9d0cbc22ec8c324cf496b02bc8b4145d46cf35bd4056422df6a38ef19f245fedc1e1187729355df2
  f7ade982d3828ac59b64c17f86d81ab72615925962971bcab84693aef927ddd6badb962e4b30567a
  c3c70ae9a00fc57a9035ee045becd9bc8f8030e850eca28ae3becc1ce0c9faea9b502203328e45b1
  3b8032d4119ec065164fe7ab89544c0863b5eccb4f000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  000000000000000005000000620000000900000017000000af45d326f7008c992ac3c92100000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  5f4aacd4ca48b728f3ffd8889f0989eaacad91db6efe31fb039b9e40b585bec2e4ce8736093bc23c
  8db0caf0b4b0fdb8cde1c2c57f1b1337f276ec75780d561d4d2713adbbccafdb6b2671c3a45773a4
  2b2299742c0c61b575303e2fdb0c92af6af190e819e77df31e53c6d3fe89c2edd20dbd4a52af34e9
  80f0d556930f3bcae9af591044730dbe811122c28bbc8af98d2a36868ff2ecfe81924e8a6d667cd1
  cb6ea1b40a5d9ba7165e848a2fd19b5681c4e151a0b93e6e2048aeb1b3283d79535ab9f5e8a1dfb0
  5fbe32d556fe4b24c412e33eed6031c23a9cf9a35ab754d586d7490764b312dfbd413f2af90cffea
  06a2943f1e8618e5a1d89e20edec302de6a13e32e1efbc529b45403d5a75c8953c9d8162db4bdddb
  1b7ff659df083a08eb3fe29b1c7b1b420511ce0685adeda3d3ef76374bf7a0adaaa548d2d7348fae
  d4f51be065bf97d7e88fca360e8553a95b51dbf432418f9a5ea5f685afb26802f5beabd12c249aa0
  b54ea1597cb47ba4ab0b8bb8ce6bbd5b142f9586b739af6f38e64e2ccef74b46d72a4a1376349283
  1c5744a09ff593a04f1e1195aa94c9453ec78835b7d552044215dfdc298e7be05cd96a0eab32b4ca
  d2b4ffd47d92c6b7b50f36b13c56a37e156b681b7b35b51d1956176a7eb1d28134d33ca9ffbe8025
  db348597ba0357e7f38da57bf01895ad54441082a2699f6ac8820f17294b7addd1c1bbea8174e6c0
  30dd858b57f3418b8250b67ecde9f827f44f2d6ffc000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  00000000000000000500000062000000090000001700000046bb73dc645823da6d152f6a00000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  132ce6ff725b2bb9cfa92e2578400ae128269cbffb0d86c5b1bb5c40a42263636af77612454bf905
  6bb343510205c8141aa01d91cfabd782eb699478bc193e2a029f7aaaf806b05a04a0a38a2be5711a
  c2346a18b4466e7b909959b5e40094bd307a80064b7f5cc4f2317792eb04f9dbaf6bbb3d3fca947a
  341c3f56bbf07ae68a8d923be4e8c8d10c5e0e22a70c49f300fbdf063b25329797b8fc1eef3ccf42
  7d0e86add793a9d6177e7e56ea171718ea40e44ef15946c07bb6318f1179818dcca00e970069fc97
  237da12efc0cfa18b860d0b7ee494f8bda4c92b086b1673fd03d46faa1196108a2a94acaa35d9e63
  8b93b5d62d42cb2991456291dfee824f523a6c8a0ff0e083daa39247078bb9d3dcffc4bc331bb98d
  2eb1a94d3e46ca411601f749ae42233397123dbb31660d34ba4ac25c0f6a8fc6a978ca1adf64d006
  6c3b3e08e44995a2d6c9dbfee15d249e363c2bf29dbe2a343766b371fe6691db440b0be23d66104f
  cf43427bb04ba5453f994c225a8a569b6f0bcd8bff4e9e8cb85ecbc5f1c6ceb2e57895449c83e191
  461ff7f77accc207a77d798a97a78850c035cd9e3ef722313406d2d2cc655bd2d9419b60203ad6bd
  7bea68d57ecaaefc1bd304304385d60702da22a3ae11ecd4ad2e8defd4305e5d34b8c0d85709e863
  9dfbcf535143592c15f2a6f4a03fcc8b95ca30898cf9bd87db6fcd0fb5bb64ff1f4503e15548c0ff
  1a3b1b1614932a7e73818a64dc4e8743983fa0ad6e000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  000000000000000005000000620000000900000017000000d51251fad51a174d98c18e9400000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  aa672e709f70ab4c1c75825a6107d96418d26d083cc0f43718a3e8f0f2c73bfe888abe2607a0d521
  fe9908aca0d4932a10037828115985aa2519f94577ed42a08e2f2468faaf73b4ac58199c5ebf819d
  5c7965ea60baf51a33a852c191fd676a9895626c09811fba6b78feb03c5e327f5ad7554d22857ca1
  b149ff70d59a5bbb96d7831c34a8280526972ab4942e99540f49a63989e00f16e08d89e4087d0e1b
  e7980364080e7b52de02ea6abb905675bda55cfa5c945935faf060c7aea27188cb064d670a587813
  c1b09d4184c2270f82f778be822ea14fea34c156ee7d8969d9d3e90b7f01243bf9f3526e6ccaa75c
  843db421aecef11f3a8fdaacc62cc9e45ae2d6d658add855e6b676cb586a29f3281c8ca09fa5b18c
  2ed72be3012265632869159a330c714c28f505e24f953476516bdc5537e368347ecbf6af6845f3a1
  f97bc1a20b0ec291b5ad45e6ce9375df99f37c04eb1d5744a2cc03fbdfe032a9b4a271b51717d3ae
  fb97c590fd07312365ea53469aa6bb30c843d77d41419f660bca412ed545a5bdabde2c09e8503126
  408395b5386f2ba29a5e8aa39e244fb368b7ba6f39970232b52555aff9982d4ef3395eaa5359d374
  1738a4334bd7baa39ac917d6d51854ad3c11295e42767a260a3b4ea282e079c578498209d62ae341
  f5bdfbc5d62405b02a41fb124006e03294f8f839b0990663d64cabc1a752a2167e3be229f359b2ab
  14cb51611fe48bf5d7e42b859ef27e464891e7ca31000000
"
    ),
    &hex!(
      "
  00000000520000000000000000000200050000001f0000002d010000ea0000000400000001000200
  0000000000000000050000006200000009000000170000009d1a4612ce27ca163771a1df00000000
  0400000000000200050000005a0000000900000002100000726563697069656e7431000000000000
  3f7627c0ae2ad2ffc926df6a7956bbde18f01acbb101b8ba01c8bd34ef8702d487d6d88b50a2d3d3
  522413ecd209e5cd0c0c12acbaa024c44c4a0fca9d7031454ad1ee6154ece250b63fa97da1556397
  c40efe46c8db5e8c0f39a77ba0800a37a700b7854d01cc315863d6990d3a6f3c768068ed61add8c7
  236d226e852250f02f7755604d163bf814ddbf877b1bce624061abb80438477e21f210afa00b7d90
  236dba6ad506977a17d2d1b1bb398b8c99629f5baea8122207fc44d5c8a2e5b4bb43f0e2435a2d9f
  6796b1700e31d39b9cc64b84c910eb2abbd5daee1b2aa1b538fe5fa64bffe69700259435f0bdf88c
  79ec468030d7316363a613c0f3f2e8bf04e8ce87d3bf83700e60969fab287b8c20432e3dea53c495
  3194bbafecf0eae9bf3ae4ffc547f0fcce4b7b9753926c6a486ffde6f6917801d9867c11d5aef045
  913caf9ba9f16988a1a33917b18908fea7f2b7d66481733ffc1a7ba21b17384fa5a506bfd2c3089b
  79c93253f964d9b583f4066e34a486a379a2bdb19a13d484217918ec7b3461b5c38e969d3153680f
  de3c07b262b4a41bc667ba8ed46793e28162a64362f5726331213059c3c3af498759d6eac99b5363
  761df14fb7b430f7d3c2bf8aaabadb6c551b6ee7aecef21059b4d78cbf450d34dafec17351cc9a42
  9b19560fd11a0035169394ae0dd687a6cd701d7d28497dde0a3cebcfc423f6dd8b28f3a130cf3696
  5818ddb03a08a109c65a3de34c894dff812ccdb2d5000000
"
    ),
  ];

  #[test]
  #[cfg(feature = "openssl")]
  fn test_openssl_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::OPEN_SSL_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      CRYPTED_KEY,
      MESSAGES,
    );
  }

  #[test]
  #[cfg(feature = "rust_crypto")]
  fn test_rust_rsa_aes_gcm_fixture() {
    super::fixture_test(
      &crate::secrets_store::cipher::RUST_RSA_AES_GCM,
      SEAL_KEY,
      SEAL_NONCE,
      CRYPTED_KEY,
      MESSAGES,
    );
  }
}
