use std::os::unix::net::UnixDatagram;
use std::str::from_utf8;
use std::io::Write;
use std::fs::remove_file;
use std::path::PathBuf;
use gethostname::gethostname;
use chrono::{DateTime, Utc, Local, FixedOffset, TimeZone};
use tai64::Tai64N;
use structopt::StructOpt;
use problem::prelude::*;
use problem::result::Result as PResult;
use log::*;
use syslogio::{facility_by_id, severity_by_id, SyslogHeader, write_syslog_message};

/// Read syslog messages from a UNIX domain datagram socket and format it to standard output.
#[derive(Debug, StructOpt)]
struct Cli {
    /// Verbose mode (-v for INFO, -vv for DEBUG)
    #[structopt(short = "v", long, parse(from_occurrences))]
    pub verbose: isize,

    /// Silence mode (-s for no WARN, -ss for no ERROR)
    #[structopt(short = "s", long, parse(from_occurrences))]
    silent: isize,

    /// Size of the message buffer
    #[structopt(short = "b", long, default_value = "196609")]
    buffer_size: usize,

    /// Use this hostname instead of system provided
    #[structopt(short = "H", long)]
    hostname: Option<String>,

    /// Take timestamp from beginning of the datagram; otherwise from syslog message if in RFC3339
    /// format or current time
    /// (-T for Tai64N (binary, 12 bytes), -TT for RFC3339 with nanoseconds followed by single space, 36 bytes)
    #[structopt(short = "T", long, parse(from_occurrences))]
    datagram_timestamp: usize,

    /// Syslog socket
    #[structopt(default_value = "/dev/log")]
    syslog_socket: PathBuf,

    #[structopt(subcommand)]
    output: Output,
}

#[derive(Debug, StructOpt)]
enum Output {
    /// Output syslog messages in human readable format
    Human,
    /// Output syslog messages in format suitable for forwarding to a syslog server
    Syslog {
        /// Format of timestamp
        /// (default for syslog, -T for RFC3339 with milliseconds)
        #[structopt(short = "T", long, parse(from_occurrences))]
        timestamp_format: usize,

        /// Escape sequence to replace new line control character
        #[structopt(short = "n", long, default_value = "#012")]
        newline_escape: String,
    },
}

#[derive(Debug, Clone, Copy)]
enum DatagramTimestamp {
    Tai64N,
    Rfc3339,
}

impl DatagramTimestamp {
    fn from_occurrences(occurrences: usize) -> PResult<Option<DatagramTimestamp>> {
        Ok(match occurrences {
            0 => None,
            1 => Some(DatagramTimestamp::Tai64N),
            2 => Some(DatagramTimestamp::Rfc3339),
            o => problem!("No timestamp format for occurrences: {}", o)?,
        })
    }

    fn len(&self) -> usize {
        match self {
            DatagramTimestamp::Tai64N => 12,
            DatagramTimestamp::Rfc3339 => 36,
        }
    }

    fn parse<'b>(&self, dgram: &'b[u8]) -> PResult<(DateTime<FixedOffset>, &'b[u8])> {
        if dgram.len() < self.len() {
            return problem!("Datagram too short to contain datagram timestamp")
        }
        let (ts_buf, dgram_buf) = dgram.split_at(self.len());

        Ok((match self {
            DatagramTimestamp::Tai64N => {
                let Tai64N(tai64, nanos) = Tai64N::from_slice(ts_buf).problem_while("parsing Tai64N timestamp")?;
                Utc.timestamp_opt(tai64.to_unix(), nanos).single().ok_or_else(|| "Tai64N could not be converted to a valid date and time")?.into()
            },
            DatagramTimestamp::Rfc3339 => {
                let (ts_buf, sep) = ts_buf.split_at(self.len() - 1);
                if sep != b" " {
                    problem!("Datagram timestamp is not space separated form the message")?;
                }
                DateTime::parse_from_rfc3339(&String::from_utf8_lossy(ts_buf)).problem_while("parsing RFC3339 timestamp")?
            },
        }, dgram_buf))
    }
}

fn main() -> FinalResult {
    let args = Cli::from_args();
    let verbosity = args.verbose + 1 - args.silent;
    stderrlog::new()
        .module("syslogio")
        .module(module_path!())
        .module("problem")
        .quiet(verbosity < 0)
        .verbosity(verbosity as usize)
        .timestamp(stderrlog::Timestamp::Microsecond)
        .init()
        .unwrap();

    let datagram_timestamp = DatagramTimestamp::from_occurrences(args.datagram_timestamp)?;

    info!("Binding to syslog socket: {:?}", args.syslog_socket);
    remove_file(&args.syslog_socket).ok();
    let socket = UnixDatagram::bind(&args.syslog_socket)?;

    let mut buf = Vec::new();
    buf.resize(args.buffer_size, 0);
    let buf = buf.as_mut();

    let hostname = args.hostname.map(Ok).unwrap_or_else(|| gethostname().into_string()).map_err(|_| "Hostname is not UTF-8 compatible string")?;
    let mut stdout = std::io::stdout();

    info!("Receiving messages from syslog socket");
    loop {
        let dgram_len = socket.recv(buf)?;
        if dgram_len == buf.len() {
            warn!("Truncated message");
        }
        let dgram = &buf[..dgram_len];

        let (timestamp, message) = if let Some(ts) = &datagram_timestamp {
            match ts.parse(dgram).problem_while("parsing datagram timestamp").ok_or_log_error() {
                None => continue,
                Some((timestamp, message)) => {
                    debug!("datagram timestamp: {}", timestamp.format("%Y-%m-%dT%H:%M:%S%.6f%:z"));
                    (timestamp, message)
                }
            }
        } else {
            (Local::now().into(), dgram)
        };

        let (mut syslog, message) = SyslogHeader::parse(&message, Some(timestamp)).unwrap_or_else(|| {
            error!("Failed to parse syslog message: {}", String::from_utf8_lossy(message));
            (SyslogHeader::new(timestamp, "syslog", "crit", "syslog-unix").unwrap(), b"failed to parse syslog message")
        });

        trace!("{:?}: {}", syslog, String::from_utf8_lossy(message));

        // some programs append new line at the end of the log message
        let message = message.strip_suffix(b"\n").unwrap_or(message);

        match &args.output {
            Output::Human => {
                write!(stdout, "{}.{}: {} {}: ",
                    facility_by_id(syslog.facility).unwrap_or("(??)"),
                    severity_by_id(syslog.severity).unwrap_or("(??)"),
                    timestamp.format("%Y-%m-%d %H:%M:%S%.6f"),
                    from_utf8(syslog.tag.unwrap_or(b"syslog")).ok().unwrap_or("??"),
                )?;

                for (no, line) in message.split(|c| *c == b'\n').enumerate() {
                    if no != 0 {
                        stdout.write_all(b"\n\t")?;
                    }
                    stdout.write_all(line)?;
                }
                stdout.write_all(&[b'\n'])?;
                stdout.flush()?;
            }
            Output::Syslog { timestamp_format, newline_escape } => {
                syslog.hostname(hostname.as_bytes());
                syslog.write(&mut stdout, *timestamp_format > 0)?;
                write_syslog_message(&mut stdout, message, newline_escape.as_bytes())?;
                stdout.write_all(&[b'\n'])?;
                stdout.flush()?;
            }
        }
        debug!("message written");
    }
}
