use std::path::PathBuf;
use std::io::{Write, BufRead, BufReader};
use std::time::Duration;
use std::num::ParseFloatError;
use structopt::StructOpt;
use multistream_batch::channel::multi_buf_batch::{MultiBufBatchChannel, Command};
use regex::Regex;
use chrono::offset::Local;
use problem::prelude::*;
use log::*;
use syslogio::{facility_by_name, severity_by_name};
use syslogio::{SyslogUnix, SyslogHeader};

/// Read log message from standard input and format it as syslog message to a UNIX domain socket,
/// while reconstructing log messages spanning multiple lines as one message.
#[derive(Debug, StructOpt)]
struct Cli {
    /// Verbose mode (-v for INFO, -vv for DEBUG)
    #[structopt(short = "v", long, parse(from_occurrences))]
    pub verbose: isize,

    /// Silence mode (-s for no WARN, -ss for no ERROR)
    #[structopt(short = "s", long, parse(from_occurrences))]
    silent: isize,

    /// Size of the message buffer
    #[structopt(short = "b", long, default_value = "196609")]
    buffer_size: usize,

    /// Regex to match stream ID to demultiplex ordered streams of log lines by
    #[structopt(long, short = "i")]
    stream_id_pattern: Option<String>,

    /// Negate the pattern
    #[structopt(long, short = "n")]
    negate: bool,

    /// Match last line of single message instead of first
    #[structopt(long, short = "l")]
    match_last: bool,

    /// Strip matched pattern from line
    #[structopt(long, short = "S")]
    strip_pattern: bool,

    /// Maximum number of lines a single message can collect before flushing
    #[structopt(long, default_value = "2000", short = "M")]
    max_lines: usize,

    /// Maximum time duration in seconds a single message will be collecting lines for before flushing
    #[structopt(long, default_value = "0.2", short = "D", parse(try_from_str = duration_from_secs_str))]
    max_duration_ms: Duration,

    /// Path to syslog socket
    #[structopt(long, short = "u", default_value = "/dev/log")]
    syslog_socket: PathBuf,

    /// Log with syslog facility
    #[structopt(long, short = "f", default_value = "user")]
    facility: String,

    /// Log with syslog severity
    #[structopt(long, short = "r", default_value = "notice")]
    severity: String,

    /// Format of timestamp
    /// (default for syslog, -T for RFC3339 with milliseconds)
    #[structopt(short = "T", long, parse(from_occurrences))]
    timestamp_format: usize,

    /// Name of program producing the logs (syslog tag)
    program: String,

    /// Regex to match first or last line of a message
    message_pattern: String,
}

fn duration_from_secs_str(val: &str) -> Result<Duration, ParseFloatError> {
    Ok(Duration::from_secs_f64(val.parse::<f64>()?))
}

fn main() -> FinalResult {
    let args = Cli::from_args();
    let verbosity = args.verbose + 1 - args.silent;
    stderrlog::new()
        .module("syslogio")
        .module(module_path!())
        .module("problem")
        .quiet(verbosity < 0)
        .verbosity(verbosity as usize)
        .timestamp(stderrlog::Timestamp::Microsecond)
        .init()
        .unwrap();

    let facility = facility_by_name(&args.facility).problem_while("getting syslog facility number")?;
    let severity = severity_by_name(&args.severity).problem_while("getting syslog severity number")?;

    let stream_id_regex = args.stream_id_pattern.map(|pattern| Regex::new(&pattern).or_failed_to("compile regex for stream-id-pattern"));
    let pattern = Regex::new(&args.message_pattern).or_failed_to("compile regex for pattern");
    let negate = args.negate;
    let match_last = args.match_last;
    let strip_pattern = args.strip_pattern;

    let mut buf = Vec::new();
    buf.resize(args.buffer_size, 0);
    info!("Connecting to syslog socket: {:?}", args.syslog_socket);
    let mut syslog_unix = SyslogUnix::new(&mut buf, &args.syslog_socket).problem_while("Connecting with syslog socket")?;

    use Command::*;
    let mut mbatch = MultiBufBatchChannel::with_producer_thread(args.max_lines, args.max_duration_ms, args.max_lines * 2, move |sender| {
        info!("Reading messages from standard input");
        for line in BufReader::new(std::io::stdin()).lines().or_failed_to("read lines from standard input") {
            let timestamp = Local::now();

            let (stream_id, line) = if let Some(stream_id_regex) = stream_id_regex.as_ref() {
                let stream_id = stream_id_regex.find(&line).map(|m| m.as_str().to_owned());
                let line = stream_id_regex.replace(&line, "").into_owned();
                (stream_id, line)
            } else {
                (None, line)
            };

            let matched = pattern.is_match(&line);
            let line = if matched && strip_pattern {
                pattern.replace(&line, "").into_owned()
            } else {
                line
            };

            let matched = if negate { !matched } else { matched };

            if let Some(stream_id) = &stream_id {
                debug!("[{:?}/{}] {}", stream_id, if matched { "\u{2714}" } else { "\u{2715}" }, line);
            } else {
                debug!("[{}] {}", if matched { "\u{2714}" } else { "\u{2715}" }, line);
            }

            if match_last {
                sender.send(Append(stream_id.clone(), (timestamp, line))).unwrap();
                if matched {
                    sender.send(Flush(stream_id)).unwrap();
                }
            } else {
                if matched {
                    sender.send(Flush(stream_id.clone())).unwrap();
                }
                sender.send(Append(stream_id, (timestamp, line))).unwrap();
            }
        }
    });

    let mut messages_sent = 0u64;

    loop {
        match mbatch.next() {
            Ok((stream_id, mut lines)) => {
                // Use timestamp of the first line of the message
                let (timestamp, head) = lines.next().unwrap();

                let syslog = SyslogHeader::new_raw(timestamp.into(), facility, severity, args.program.as_bytes());
                if syslog_unix.with_datagram_send(&syslog, args.timestamp_format > 0, |dgram| {
                    in_context_of("writing message", || {
                        if let Some(stream_id) = &stream_id {
                            dgram.write_all(stream_id.as_bytes())?;
                        }

                        for (i, line) in std::iter::once(head).chain(lines.map(|(_timestamp, line)| line)).enumerate() {
                            if let Some(stream_id) = &stream_id {
                                trace!("[{:?}/{}] {}", stream_id, i, line);
                            } else {
                                trace!("[{}] {}", i, line);
                            }
                            if i > 0 {
                                dgram.write_all(b"\n")?;
                            }
                            dgram.write_all(line.as_bytes())?;
                        }
                        Ok(())
                    })
                }).ok_or_log_error().is_some() {
                    messages_sent += 1;
                }
            }
            Err(_) => break,
        }
    }

    info!("Sent {} messages; closing syslog socket", messages_sent);
    syslog_unix.close().problem_while("closing syslog socket")?;
    Ok(())
}
