use std::path::PathBuf;
use std::io::{stdin, Write, BufRead, BufReader};
use structopt::StructOpt;
use problem::prelude::*;
use chrono::offset::Local;
use syslogio::{facility_by_name, severity_by_name};
use syslogio::{SyslogUnix, SyslogHeader};
use log::*;

/// Read log message from standard input and format it as syslog message to a UNIX domain socket.
#[derive(Debug, StructOpt)]
struct Cli {
    /// Verbose mode (-v for INFO, -vv for DEBUG)
    #[structopt(short = "v", long, parse(from_occurrences))]
    pub verbose: isize,

    /// Silence mode (-s for no WARN, -ss for no ERROR)
    #[structopt(short = "s", long, parse(from_occurrences))]
    silent: isize,

    /// Size of the message buffer
    #[structopt(short = "b", long, default_value = "196609")]
    buffer_size: usize,

    /// Path to syslog socket
    #[structopt(long, short = "u", default_value = "/dev/log")]
    syslog_socket: PathBuf,

    /// Log with syslog facility
    #[structopt(long, short = "f", default_value = "user")]
    facility: String,

    /// Log with syslog severity
    #[structopt(long, short = "r", default_value = "notice")]
    severity: String,

    /// Format of timestamp
    /// (default for syslog, -T for RFC3339 with milliseconds)
    #[structopt(short = "T", long, parse(from_occurrences))]
    timestamp_format: usize,

    /// Read standard input to the end as a single message; otherwise read message per line
    #[structopt(short = "1", long)]
    one: bool,

    /// Name of program producing the logs (syslog tag)
    program: String,

    /// Message to write to syslog; otherwise read from standard input
    message: Option<String>,
}

fn main() -> FinalResult {
    let args = Cli::from_args();
    let verbosity = args.verbose + 1 - args.silent;
    stderrlog::new()
        .module("syslogio")
        .module(module_path!())
        .module("problem")
        .quiet(verbosity < 0)
        .verbosity(verbosity as usize)
        .timestamp(stderrlog::Timestamp::Microsecond)
        .init()
        .unwrap();

    let facility = facility_by_name(&args.facility).problem_while("getting syslog facility number")?;
    let severity = severity_by_name(&args.severity).problem_while("getting syslog severity number")?;

    let mut buf = Vec::new();
    buf.resize(args.buffer_size, 0);

    let mut messages_sent = 0u64;

    info!("Connecting to syslog socket: {:?}", args.syslog_socket);
    let mut syslog_unix = SyslogUnix::new(&mut buf, &args.syslog_socket).problem_while("Connecting with syslog socket")?;

    if args.one || args.message.is_some() {
        let syslog = SyslogHeader::new_raw(Local::now().into(), facility, severity, args.program.as_bytes());
        syslog_unix.with_datagram_send(&syslog, args.timestamp_format > 0, |dgram| {
            if args.one {
                info!("Reading one message from standard input");
                std::io::copy(&mut stdin(), dgram).problem_while("copying syslog message from standard input")?;
                messages_sent += 1;
            } else {
                let message = args.message.unwrap();
                dgram.write_all(message.as_bytes()).problem_while_with(|| format!("writing message: {}", message))?;
                messages_sent += 1;
            }
            Ok(())
        })?;
    } else {
        info!("Reading messages from standard input; one per line");
        for line in BufReader::new(stdin()).lines().or_failed_to("read lines from standard input") {
            let syslog = SyslogHeader::new_raw(Local::now().into(), facility, severity, args.program.as_bytes());
            if syslog_unix.with_datagram_send(&syslog, args.timestamp_format > 0, |dgram| {
                dgram.write_all(line.as_bytes()).problem_while_with(|| format!("writing message: {}", line))
            }).ok_or_log_error().is_some() {
                messages_sent += 1;
            }
        }
    }

    info!("Sent {} messages; closing syslog socket", messages_sent);
    syslog_unix.close().problem_while("closing syslog socket")?;
    Ok(())
}
