use std::os::unix::net::UnixDatagram;
use std::str::from_utf8;
use std::error::Error;
use std::fmt::{self, Display};
use std::net::Shutdown;
use problem::prelude::*;
use problem::result::Result as PResult;
use std::path::Path;
use std::io::{Write, Error as IoError};
use chrono::offset::Local;
use chrono::{DateTime, FixedOffset};
use itertools::Itertools;
use log::*;

#[derive(Debug)]
pub struct SyslogPriorityNameError;

impl Display for SyslogPriorityNameError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "invalid syslog facility or severity name")
    }
}
impl Error for SyslogPriorityNameError { }

const FACILITY: [&str; 24] = ["kern", "user", "mail", "daemon", "auth", "syslog", "lpr", "news", "uucp", "cron", "authpriv", "ftp", "ntp", "audit", "alert", "clockd", "local0", "local1", "local2", "local3", "local4", "local5", "local6", "local7"];

pub fn facility_by_id(facility: u8) -> Option<&'static str> {
    FACILITY.get(facility as usize).map(|s| *s)
}

pub fn facility_by_name(facility: &str) -> Result<u8, SyslogPriorityNameError> {
    FACILITY
        .iter()
        .position(|name| *name == facility)
        .map(|pos| pos as u8)
        .ok_or(SyslogPriorityNameError)
}

const SEVERITY: [&str; 8] = ["emerg", "alert", "crit", "err", "warning", "notice", "info", "debug"];

pub fn severity_by_id(severity: u8) -> Option<&'static str> {
    SEVERITY.get(severity as usize).map(|s| *s)
}

pub fn severity_by_name(facility: &str) -> Result<u8, SyslogPriorityNameError> {
    SEVERITY
        .iter()
        .position(|name| *name == facility)
        .map(|pos| pos as u8)
        .ok_or(SyslogPriorityNameError)
}

#[derive(Debug)]
pub struct SyslogHeader<'s> {
    pub facility: u8,
    pub severity: u8,
    pub timestamp: DateTime<FixedOffset>,
    pub timestamp_rfc3339: bool,
    pub hostname: Option<&'s [u8]>,
    pub tag: &'s [u8],
}

// escape new lines to pereserve message boundries
pub fn write_syslog_message(out: &mut impl Write, message: &[u8], escape: &[u8]) -> Result<(), IoError> {
    for (no, line) in message.split(|c| *c == b'\n').enumerate() {
        if no != 0 {
            out.write_all(escape)?;
        }
        out.write_all(line)?;
    }
    Ok(())
}

impl<'s> SyslogHeader<'s> {
    pub fn new(timestamp: DateTime<FixedOffset>, facility: &str, severity: &str, tag: &'s str) -> Result<SyslogHeader<'s>, SyslogPriorityNameError> {
        Ok(SyslogHeader {
            facility: facility_by_name(facility)?,
            severity: severity_by_name(severity)?,
            timestamp,
            timestamp_rfc3339: false,
            hostname: None,
            tag: tag.as_bytes(),
        })
    }

    pub fn new_raw(timestamp: DateTime<FixedOffset>, facility: u8, severity: u8, tag: &'s [u8]) -> SyslogHeader<'s> {
        SyslogHeader {
            facility: facility,
            severity: severity,
            timestamp,
            timestamp_rfc3339: false,
            hostname: None,
            tag,
        }
    }

    pub fn hostname(&mut self, hostname: &'s[u8]) {
        self.hostname = Some(hostname);
    }

    pub fn timestamp_rfc3339(&mut self) {
        self.timestamp_rfc3339 = true;
    }

    pub fn write(&self, out: &mut impl Write) -> Result<(), IoError> {
        write!(out, "<{}>{} ",
            ((self.facility as i32) << 3) + (self.severity as i32),
            if self.timestamp_rfc3339 {
                self.timestamp.format("%Y-%m-%dT%H:%M:%S%.6f%:z")
            } else {
                self.timestamp.format("%b %e %T")
            }
        )?;
        if let Some(hostname) = self.hostname {
            write_syslog_message(out, hostname, b"")?;
            out.write_all(b" ")?;
        }
        write_syslog_message(out, self.tag, b"")?;
        out.write_all(b": ")?;

        Ok(())
    }

    pub fn parse(message: &[u8], timestamp: Option<DateTime<FixedOffset>>) -> Option<(SyslogHeader, &[u8])> {
        let mut pri_iter = message
            .strip_prefix(b"<")?
            .splitn(2, |c| *c == b'>');

        let pri = pri_iter.next()?;

        let pri: u8 = from_utf8(pri).ok()?.parse().ok()?;
        let facility = pri >> 3;
        let severity = pri & 0x7;

        let message = pri_iter.next()?;

        if message.len() <= 16 + 2 {
            return None
        }

        let mut timestamp_rfc3339 = None;

        let message = if message[3] == b' ' && message[6] == b' ' && message[15] == b' ' {
            // Apr 14 15:31:38
            message.split_at(16).1
        } else if let Some((date, message)) = message.splitn(2, |c| *c == b' ').collect_tuple() {
            // 2022-04-14T15:34:45.327514+00:00
            let date = String::from_utf8_lossy(date);
            timestamp_rfc3339 = DateTime::parse_from_rfc3339(&date).ok();
            message
        } else {
            return None
        };

        let end = message.windows(2).position(|c| *c == [b':', b' '])?;

        let (tag, message) = message.split_at(end);
        let message = &message[2..];

        if timestamp_rfc3339.is_some() {
            debug!("using RFC3339 timestamp from syslog header");
        }

        Some((SyslogHeader {
            facility,
            severity,
            timestamp: timestamp_rfc3339.or(timestamp).unwrap_or_else(|| Local::now().into()),
            timestamp_rfc3339: timestamp_rfc3339.is_some(),
            hostname: None,
            tag,
        }, message))
    }
}

#[derive(Debug)]
pub struct SyslogUnix<'b> {
    socket: UnixDatagram,
    buf: &'b mut [u8],
}

impl<'b> SyslogUnix<'b> {
    pub fn new(buf: &'b mut [u8], path: &Path) -> PResult<SyslogUnix<'b>> {
        let socket = UnixDatagram::unbound().problem_while("creating syslog socket")?;
        socket.connect(path).problem_while("connecting to syslog socket")?;

        Ok(SyslogUnix {
            socket,
            buf,
        })
    }

    pub fn close(self) -> Result<(), IoError> {
        self.socket.shutdown(Shutdown::Both)?;
        Ok(())
    }

    pub fn datagram(&mut self, syslog: &SyslogHeader) -> Result<SyslogDatagram, IoError> {
        let mut sd = SyslogDatagram {
            buf_len: self.buf.len(),
            datagram: self.buf,
        };

        syslog.write(&mut sd)?;
        Ok(sd)
    }

    pub fn with_datagram_send<O>(&mut self, syslog: &SyslogHeader, f: impl FnOnce(&mut SyslogDatagram) -> PResult<O> ) -> PResult<O> {
        let mut dgram = self.datagram(syslog).problem_while("writing syslog header")?;

        let ret = f(&mut dgram)?;

        let len = dgram.bytes_written();
        self.send(len).problem_while("sending message to syslog socket")?;
        Ok(ret)
    }

    pub fn send(&mut self, datagram_len: usize) -> Result<usize, IoError> {
        let datagram = &self.buf[..datagram_len];
        trace!("{}", String::from_utf8_lossy(datagram));
        let len = self.socket.send(datagram)?;
        debug!("sent {} bytes datagram", len);
        Ok(len)
    }
}

#[derive(Debug)]
pub struct SyslogDatagram<'b> {
    buf_len: usize,
    datagram: &'b mut [u8],
}

impl<'b> Write for SyslogDatagram<'b> {
    fn write(&mut self, buf: &[u8]) -> Result<usize, IoError> {
        self.datagram.write(buf)
    }

    fn flush(&mut self) -> Result<(), IoError> {
        self.datagram.flush()
    }
}

impl<'b> SyslogDatagram<'b> {
    pub fn bytes_written(self) -> usize {
        self.buf_len - self.datagram.len()
    }
}
