/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

use std::path::Path;

use crate::{error::SyRes, Priority, LogStat, LogFacility};

use super::syslog_stream::SyslogStream;

pub trait SyslogStd
{
    /// Creates new instance
    fn openlog(ident: Option<&str>, logstat: LogStat, facility: LogFacility) -> SyRes<Self> where Self: Sized;

    /// Updates logmask
    fn setlogmask(&self, logmask: i32) -> SyRes<i32>;

    /// Closes connection to the syslog server
    fn closelog(&self) -> SyRes<()>;

    /// Sends message to syslog
    fn syslog(&self, pri: Priority, fmt: String);

    /// Sends message to syslog
    fn vsyslog<S: AsRef<str>>(&self, pri: Priority, fmt: S);
}

pub trait SyslogExt
{
    /// Creates new instance and attempts to connect to provided path
    fn openlog_custom<P: AsRef<Path>>(ident: Option<&str>, logstat: LogStat, facility: LogFacility, sock_path: P) -> SyRes<Self> where Self: Sized;

    /// Changes identity (appname)
    fn change_identity<I: AsRef<str>>(&self, ident: I) -> SyRes<()>;

    /// Creates a streamable syslog channel
    fn make_stream(&self, pri: Priority) -> Box<dyn SyslogStream>;
}
