/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

use std::{str, fmt::{self, Write, Arguments}, sync::Arc, path::Path};

// ATOMIC ACCESS
#[cfg(feature = "use_atomic_block")]
use super::mutex::{Mutex};
use super::{syslog_sync_internal::SyncSyslogInternal, syslog_stream::SyslogStream};
#[cfg(not(feature = "use_atomic_block"))]
use std::sync::{Mutex};

use crate::{map_error_code};
use crate::common::*;
use crate::error::{SyRes, SyslogError, SyslogErrCode};

pub use super::syslog_trait::{SyslogStd, SyslogExt};

/// A common instance which describes the syslog state
pub struct Syslog
{   
    /// A giantly locked syslog data
    inner: Arc<Mutex<SyncSyslogInternal>>,
}

unsafe impl Send for Syslog {}
unsafe impl Sync for Syslog {}

impl Drop for Syslog 
{
    fn drop(&mut self) 
    {
        match self.inner.lock()
        {
            Ok(mut inner) => 
                inner.disconnectlog(),
            Err(e) => 
                eprintln!("~syslog socket poisoned! '{}'", e),
        }
    }
}

/// A public implementations
impl SyslogStd for Syslog
{
    /// As in a libc, this function initializes the syslog instance. The 
    /// main difference with realization in C is it returns the instance
    /// to program used this crate. This structure implements the [Send] 
    /// and [Sync] so it does not require any additional synchonization.
    ///
    /// # Arguments
    /// 
    /// * `ident` - a identification of the sender. If not set, the crate
    ///             will determine automatically!
    /// * `logstat` - sets up the syslog behaviour. Use [LogStat] 
    /// 
    /// * `facility` - a syslog facility. Use [LogFacility]
    /// 
    /// # Returns
    ///
    /// * A [SyRes] with instance or Err()
    ///
    /// # Example
    /// 
    /// ```
    ///  Syslog::openlog(
    ///        Some("test1"), 
    ///         LogStat::LOG_NDELAY | LogStat::LOG_PID, 
    ///         LogFacility::LOG_DAEMON);
    /// ```  
    fn openlog(ident: Option<&str>, logstat: LogStat, facility: LogFacility) -> SyRes<Self>
    {
        let mut syslog = 
            SyncSyslogInternal::new(ident, logstat, facility, None);
       
        if logstat.contains(LogStat::LOG_NDELAY) == true
        {
            syslog.connectlog()?;
        }

        return Ok( 
            Self
            {
                inner: Arc::new(Mutex::new(syslog)),
            }
        );
    }

    /// Sets the logmask to filter out the syslog calls.
    /// 
    /// See macroses [LOG_MASK] and [LOG_UPTO] to generate mask
    ///
    /// # Example
    ///
    /// LOG_MASK!(Priority::LOG_EMERG) | LOG_MASK!(Priority::LOG_ERROR)
    ///
    /// or
    ///
    /// ~(LOG_MASK!(Priority::LOG_INFO))
    /// LOG_UPTO!(Priority::LOG_ERROR) 
    fn setlogmask(&self, logmask: i32) -> SyRes<i32>
    {
        let pri = 
            self.inner
                .lock()
                .map_err(|e| 
                    map_error_code!(SyslogErrCode::MutexPoisoned, "{}", e)
                )?
                .set_logmask(logmask);

        return Ok(pri);
    }

    /// Similar to libc, closelog() will close the log 
    fn closelog(&self) -> SyRes<()>
    {        
        self.inner
            .lock()
            .map_err(|e| 
                map_error_code!(SyslogErrCode::MutexPoisoned, "{}", e)
            )?
            .disconnectlog();

        return Ok(());
    }

    /// Similar to libc, syslog() sends data to syslog server.
    /// 
    /// # Arguments
    ///
    /// * `pri` - a priority [Priority]
    ///
    /// * `fmt` - a string message. In C exists a functions with
    ///     variable argumets amount. In Rust you should create your
    ///     own macros like format!() or use format!() 
    fn syslog(&self, pri: Priority, fmt: String)
    {
        self.vsyslog(pri, fmt);
    }

    /// Similar to syslog() and created for the compatability. 
    fn vsyslog<S: AsRef<str>>(&self, pri: Priority, fmt: S)
    {
        match self.inner.lock()
        {
            Ok(mut inner) => inner.vsyslog1(pri.bits(), fmt),
            Err(_) => {}
        }
    }
}

// --- NON STANDART API
impl SyslogExt for Syslog
{
        /// NON STANDARD FUNCTION
    /// 
    /// This function acting like `openlog()` but allows to open connection to 
    /// the arbitrary object.
    /// 
    /// # Arguments
    /// 
    /// * @see `openlog()`
    /// 
    /// * `sock_path` - [AsRef] [Path] a path to the unix datagram socket 
    fn openlog_custom<P>(ident: Option<&str>, logstat: LogStat, facility: LogFacility, sock_path: P) -> SyRes<Self>
    where P: AsRef<Path>
    {
        let mut syslog = 
            SyncSyslogInternal::new(ident, logstat, facility, Some(sock_path.as_ref()));
       
        if logstat.contains(LogStat::LOG_NDELAY) == true
        {
            syslog.connectlog()?;
        }

        return Ok( 
            Self
            {
                inner: Arc::new(Mutex::new(syslog)),
            }
        );
    }

    /// This function can be used to update the facility name, for example
    /// after fork().
    /// 
    /// # Arguments
    /// 
    /// * `ident` - a new identity (up to 48 UTF8 chars) 
    fn change_identity<I: AsRef<str>>(&self, ident: I) -> SyRes<()>
    {
        self.inner
            .lock()
            .map_err(|e| 
                map_error_code!(SyslogErrCode::MutexPoisoned, "{}", e)
            )?
            .set_logtag(ident);

        return Ok(());
    }

    /// Creates an instance which is implements [core::fmt::Write].
    /// 
    /// # Arguments
    /// 
    /// * `pri` - a priority [Priority] (can be updated)
    /// 
    /// # Returns
    /// 
    /// [StreamableSyslog] 
    fn make_stream(&self, pri: Priority) -> Box<dyn SyslogStream>
    {
        return Box::new(
            StreamableSyslog
            {
                inner: self.inner.clone(),
                pri: pri
            }
        );   
    }
}

/// An implementation which allows to [write!] to syslog.
/// 
/// Will block awaiting while inner mutex is locked.
/// You need to make sure that [SyslogStream] is stored as
/// mutable instance.
struct StreamableSyslog
{
    /// Syslog instance
    inner: Arc<Mutex<SyncSyslogInternal>>,

    /// Priority
    pri: Priority
}

impl SyslogStream for StreamableSyslog
{
    /// Updates the pri [Priority] 
    fn update_pri(&mut self, new_pri: Priority) -> Priority
    {
        let prev = self.pri;
        self.pri = new_pri;

        return prev;
    }
}

impl Write for StreamableSyslog
{
    fn write_str(&mut self, s: &str) -> fmt::Result 
    {
        self.inner
            .lock()
            .map_err(|_|
                fmt::Error
            )?
            .vsyslog1(self.pri.bits(), s);

        return Ok(());
    }

    fn write_fmt(self: &mut Self, args: Arguments<'_>) -> fmt::Result
    {
        if let Some(s) = args.as_str() 
        {
            return self.write_str(s);
        } 
        else 
        {
            return self.write_str(&args.to_string());
        }
    }
}


#[test]
fn test_single_message()
{
    /*use std::sync::Arc;
    use std::thread;
    use std::time::Duration;
    use super::{LOG_MASK};*/

    let log = 
            Syslog::openlog(
                Some("test1"), 
                LogStat::LOG_CONS | LogStat::LOG_NDELAY | LogStat::LOG_PID, 
                LogFacility::LOG_DAEMON);

    assert_eq!(log.is_ok(), true, "{}", log.err().unwrap());

    let log = log.unwrap();

    log.syslog(Priority::LOG_DEBUG, format!("test_set_logmask() проверка BOM"));

    let _ = log.closelog();

    return;
}

#[test]
fn test_multithreading()
{
    use std::sync::Arc;
    use std::thread;
    use std::time::{Instant, Duration};

    let log = 
            Syslog::openlog(
                Some("test1"), 
                LogStat::LOG_CONS | LogStat::LOG_NDELAY | LogStat::LOG_PID, 
                LogFacility::LOG_DAEMON);

    assert_eq!(log.is_ok(), true, "{}", log.err().unwrap());

    let log = Arc::new(log.unwrap());
    let c1_log = log.clone();
    let c2_log = log.clone();

    thread::spawn(move|| {
        for i in 0..5
        {
            thread::sleep(Duration::from_nanos(200));
            let now = Instant::now();
            c1_log.syslog(Priority::LOG_DEBUG, format!("a message from thread 1 #{}[]", i));
            let elapsed = now.elapsed();
            println!("t1: {:?}", elapsed);
        }
    });

    thread::spawn(move|| {
        for i in 0..5
        {
            thread::sleep(Duration::from_nanos(201));
            let now = Instant::now();
            c2_log.syslog(Priority::LOG_DEBUG, format!("сообщение от треда 2 №{}ХЪ", i));
            let elapsed = now.elapsed();
            println!("t2: {:?}", elapsed);
        }
    });

    let now = Instant::now();
    log.syslog(Priority::LOG_DEBUG, format!("A message from main, сообщение от главнюка"));
    let elapsed = now.elapsed();
    println!("main: {:?}", elapsed);

    thread::sleep(Duration::from_secs(2));

    let _ = log.closelog();

    return;
}
