/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */


use std::fmt;

/// Error code
#[derive(Clone, Copy, Debug, PartialEq)]
pub enum SyslogErrCode
{
    /// A message which can be output to stderr
    InternalError,

    /// Only for syslog_sync_queue.rs and returned only when syslog thread is stopped
    /// and someone is trying to do some operation
    #[cfg(feature = "use_sync_queue")]
    SyslogThreadNotAvailable,

    /// Only for syslog_sync_queue.rs and returned only when syslog thread is failed
    /// to send back data
    #[cfg(feature = "use_sync_queue")]
    UnboundedChannelError,

    /// Mutex poisoned, can not contunue
    MutexPoisoned,
}

/// A syslog crate error
pub struct SyslogError 
{
    errcode: SyslogErrCode,
    message: String,
}

impl SyslogError
{
    /// Creates new error
    pub(crate)
    fn new(errcode: SyslogErrCode, msg: String) -> Self
    {
        return SyslogError{errcode: errcode, message: msg};
    }

    /// Retuns a clone of [SyslogErrCode]
    pub 
    fn get_errcode(&self) -> SyslogErrCode
    {
        return self.errcode;
    }

    /// Moves out the error description
    pub 
    fn into_inner(self) -> String
    {
        return self.message;
    }
}

impl fmt::Display for SyslogError 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "{}", self.message)
    }
}
impl fmt::Debug for SyslogError 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "{}", self.message)
    }
}

pub type SyRes<T> = Result<T, SyslogError>;

#[macro_export]
macro_rules! throw_error 
{
    ($($arg:tt)*) => (
        return std::result::Result::Err(SyslogError::new(SyslogErrCode::InternalError, format!($($arg)*)))
    )
}

#[macro_export]
macro_rules! map_error 
{
    ($($arg:tt)*) => (
        return SyslogError::new(SyslogErrCode::InternalError, format!($($arg)*))
    )
}

#[macro_export]
macro_rules! throw_error_code
{
    ($code:expr, $($arg:tt)*) => (
        return std::result::Result::Err(SyslogError::new($code, format!($($arg)*)))
    )
}

#[macro_export]
macro_rules! map_error_code
{
    ($code:expr, $($arg:tt)*) => (
        return SyslogError::new($code, format!($($arg)*))
    )
}
