/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */

use std::path::Path;

use async_trait::async_trait;

use crate::{error::SyRes, Priority, LogStat, LogFacility};

#[async_trait]
pub trait SyslogStd: Send
{
    /// Creates new instance
    async 
    fn openlog(ident: Option<&str>, logstat: LogStat, facility: LogFacility) -> SyRes<Self> where Self: Sized;

    /// Updates logmask
    async fn setlogmask(&self, logmask: i32) -> i32;

    /// Closes connection to the syslog server
    async fn closelog(&self);

    /// Sends message to syslog
    async fn syslog(&self, pri: Priority, fmt: String);

    /// Sends message to syslog
    async fn vsyslog<S: AsRef<str> + Send>(&self, pri: Priority, fmt: S);
}

#[async_trait]
pub trait SyslogExt
{
    /// Creates new instance and attempts to connect to provided path
    async
    fn openlog_custom<P: AsRef<Path> + Send>(ident: Option<&str>, logstat: LogStat, facility: LogFacility, sock_path: P) -> SyRes<Self> where Self: Sized;

    /// Changes identity (appname)
    async fn change_identity<I: AsRef<str> + Send>(&self, ident: I);
}
