/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */


//! syslog-rs
//! 
//! Since v 0.2.4 this project is relicensed with MPLv2.0.
//! The contributors and authors agreed to change license:
//! Aleksandr Morozov
//! RELKOM s.r.o
//! 
//! An implementation of the syslog from glibc/libc like it was designed in
//! in both system libraries. The API is almost compatible with what is in
//! libc/glibc.
//! 
//! Supported GNU/Linux and *BSD (incl OSX, but not guaranteed)
//! 
//! Files:
//! - syslog_sync.rs - contains the thread-safe realization of the syslog (sync).
//!     Thread safe. The Atomic and crossbeam backoff are used for sync.
//! - syslog_async.rs - contains the async realization of the syslog (async)
//!     Thread safe. Tokio mutex are used for sync.
//! - syslog_sync_queue.rs - constains the sync realization, with asynchronious 
//!     processing. The internal sych of crossbeam are used to push data to queue.
//! - syslog_sync_internal.rs - a `use_sync` and `use_sync_queue` common code.
//! - unsafe_cell.rs - a file contains a Cell which can be used to share the syslog
//!     instance. See examples.
//! - portable.rs - all system level code which is portable
//! - common.rs - a common items mostly exported
//! - socket.rs - contains socket realization
//! - async_socket.rs - contains socket realization
//! - error.rs - an error wrapper and mapper
//! 
//! Features: 
//! - feature = "use_async" for asynchronious code (use syslog_rs::sy_async::{Syslog};)
//! - feature = "use_sync" for synchronious code (use syslog_rs::sy_sync::{Syslog};)
//! - feature = "use_sync_queue" for synchronious with async processing (use syslog_rs::sy_async_queue::{Syslog};)
//!
//! All features can be used simultaniously.
//! # Usage
//! 
//! syslog-rs = {version = "0.1", default-features = false, features = ["use_sync"]}
//! 
//! By default, all 3 features are enabled.

//-- REEXPORT --
#[macro_use] 
extern crate bitflags;
#[cfg(any(feature = "use_sync", feature = "use_sync_queue"))]
extern crate crossbeam;
extern crate chrono;
extern crate nix;
#[macro_use] 
extern crate lazy_static;

//-- ASYNC --
#[cfg(feature = "use_async")]
extern crate async_recursion;
#[cfg(feature = "use_async")]
extern crate tokio;

//-- SYNC --
#[cfg(feature = "use_sync")]
pub mod sync;
#[cfg(feature = "use_async")]
pub mod a_sync;

//-- NON EXPORT
mod portable;


//-- PUBLIC 
pub mod unsafe_cell;

#[macro_use] 
pub mod common;
#[macro_use] 
pub mod error;

//#[cfg(any(feature = "use_sync", feature = "use_sync_queue"))]
//pub use sync::syslog_trait::SyslogCommon;
#[cfg(feature = "use_sync")]
pub use sync::syslog_sync as sy_sync;
#[cfg(feature = "use_async")]
pub use a_sync::syslog_async as sy_async;
#[cfg(feature = "use_sync_queue")]
pub use sync::syslog_sync_queue as sy_sync_queue;

pub use common::*;
pub use unsafe_cell::*;
