/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */


use tokio::time::{sleep, Duration};
use tokio::sync::Mutex;

use async_recursion::async_recursion;

use chrono::offset::Local;

use nix::libc;

#[cfg(any(
    target_os = "freebsd",
    target_os = "dragonfly",
    target_os = "openbsd",
    target_os = "netbsd",
    target_os = "macos"
))]
use chrono::SecondsFormat;

use super::async_socket::*;
use crate::portable;
use crate::common::*;
use crate::error::{SyRes};

// https://docs.rs/tokio/1.6.1/tokio/net/struct.UnixDatagram.html#method.unbound
// https://docs.rs/tokio/1.6.1/tokio/sync/struct.Mutex.html

/// Internal structure with syslog setup
struct SyslogInternal
{
    /// A identification i.e program name, thread name
    logtag: Option<String>, 

    /// Defines how syslog operates
    logstat: LogStat,

    /// Holds the facility 
    facility: LogFacility,

    /// A logmask
    logmask: i32,

    /// A stream
    stream: SyslogSocket,
}

unsafe impl Sync for SyslogInternal{}
unsafe impl Send for SyslogInternal{}

// Drop is called when no more references are left.
impl Drop for SyslogInternal 
{
    fn drop(&mut self) 
    {
        self.disconnectlog();
    }
}

impl SyslogInternal
{
    fn new(
        logtag: Option<&str>, 
        logstat: LogStat,
        facility: LogFacility,
        logmask: i32,
        stream: SyslogSocket
    ) -> SyRes<Self>
    {
        let logtag = 
            match logtag
            {
                Some(r) => Some(r.to_string()),
                None => None,
            };

        let mut inst = 
            Self
            {
                logtag,
                logstat,
                facility,
                logmask,
                stream
            };

        if inst.logstat.intersects(LogStat::LOG_NDELAY) == true
        {
            inst.connectlog()?;
        }

        return Ok(inst);
    }
}

impl SyslogInternal
{
    pub(crate) 
    fn set_logmask(&mut self, logmask: i32) -> i32
    {
        let oldmask = self.logmask;

        if logmask != 0
        {
            self.logmask = logmask;
        }

        return oldmask;
    }

    pub(crate)  
    fn send_to_stderr(&self, msg: &mut [u8])
    {
        if self.logstat.intersects(LogStat::LOG_PERROR) == true
        {
            let mut newline = String::from("\n");
            send_to_stderr(libc::STDERR_FILENO, msg, &mut newline);
        }
    }

    pub(crate) 
    fn is_logmasked(&self, pri: i32) -> bool
    {
        if ((1 << (pri & LogMask::LOG_PRIMASK)) & self.logmask) == 0
        {
            return true;
        }

        return false;
    }

    pub(crate) 
    fn set_logtag<L: AsRef<str>>(&mut self, logtag: L)
    {
        self.logtag = 
            Some(truncate_n(logtag.as_ref(), RFC_MAX_APP_NAME));
    }

    /// Disconnects the unix stream from syslog.
    /// Should be called only when lock is acuired
    fn disconnectlog(&mut self)
    {
        if self.stream.is_none() == false
        {
            self.stream.shutdown();

            self.stream = SyslogSocket::none();
        }
    }

    /// Connects unix stream to the syslog and sets up the properties of
    /// the unix stream.
    /// Should be called only when lock is acuired
    fn connectlog(&mut self) -> SyRes<()>
    {
        let stream = SyslogSocket::connect()?;

        self.stream = stream;

        return Ok(());
    }
    
    /// An internal function which is called by the syslog or vsyslog.
    /// A glibc implementation RFC3164
    #[cfg(target_os = "linux")]
    #[async_recursion]
    async 
    fn vsyslog1(&mut self, mut pri: i32, fmt: &str)
    {
        // check for invalid bits
        match check_invalid_bits(&mut pri)
        {
            Ok(_) => {},
            Err(_e) => self.vsyslog1(get_internal_log(), fmt).await
        }

        // check priority against setlogmask
        if self.is_logmasked(pri) == true
        {
            return;
        }

        // set default facility if not specified in pri
        if (pri & LOG_FACMASK) == 0
        {
            pri |= self.facility.bits();
        }

        /*let mut hostname_buf = [0u8; MAXHOSTNAMELEN];
        let hostname = 
            match nix::unistd::gethostname(&mut hostname_buf)
            {
                Ok(r) =>
                {
                    match r.to_str()
                    {
                        Ok(r) => r,
                        Err(_e) => NILVALUE
                    }
                },
                Err(_e) => NILVALUE,
            };*/

        // get timedate
        let timedate = Local::now().format("%h %e %T").to_string();

        // get appname
        if self.logtag.is_none() == true
        {
            match portable::p_getprogname()
            {
                Some(r) => self.set_logtag(r),
                None => self.set_logtag("")
            }
        }

        let progname = self.logtag.as_ref().unwrap();

        
        let msg_final = 
            if fmt.ends_with("\n") == true
            {
                truncate(fmt)
            }
            else
            {
                fmt
            };

        // message based on RFC 3164
        let msg_pri = 
            [
                b"<", pri.to_string().as_bytes(), b">"
            ].concat();

        let msg_header = 
            [ 
                // timedate
                timedate.as_bytes(), 
                // hostname
               // " ".as_bytes(), hostname.as_bytes(), 
            ].concat();

        let mut msg = 
            [
                // appname
                b" ", progname.as_bytes(), 
                // PID
                b"[", portable::get_pid().to_string().as_str().as_bytes(), b"]:",
                // msg
                b" ", /*b"\xEF\xBB\xBF",*/ msg_final.as_bytes()
            ].concat();

        drop(progname);

        // output to stderr if required
        self.send_to_stderr(&mut msg);
        
        
        let fullmsg = 
            [
                msg_pri.as_slice(), 
                msg_header.as_slice(), 
                msg.as_slice()
            ].concat();

        if self.stream.is_none() == true
        {
            // open connection
            match self.connectlog()
            {
                Ok(_) => {},
                Err(e) =>
                {
                    self.send_to_stderr(unsafe { e.into_inner().as_bytes_mut() } );
                    return;
                }
            }
        }

        // There are two possible scenarios when send may fail:
        // 1. syslog temporary unavailable
        // 2. syslog out of buffer space
        // If we are connected to priv socket then in case of 1 we reopen connection
        //      and retry once.
        // If we are connected to unpriv then in case of 2 repeatedly retrying to send
        //      until syslog socket buffer space will be cleared

        loop
        {
            match self.stream.send(&fullmsg).await
            {
                Ok(_) => return,
                Err(err) =>
                {   
                    if let Some(libc::ENOBUFS) = err.raw_os_error()
                    {
                        // scenario 2
                        if self.stream.is_priv() == true
                        {
                            break;
                        }

                        sleep(Duration::from_micros(1)).await;
                    }
                    else
                    {
                        // scenario 1
                        self.disconnectlog();
                        match self.connectlog()
                        {
                            Ok(_) => {},
                            Err(_e) => break,
                        }

                        // if resend will fail then probably the scn 2 will take place
                    }   
                }
            }
        } // loop


        // If program reached this point then transmission over socket failed.
        // Try to output message to console

        if self.logstat.intersects(LogStat::LOG_CONS)
        {
            let fd = 
                unsafe {
                    libc::open(
                        PATH_CONSOLE.as_ptr(), 
                        libc::O_WRONLY | libc::O_NONBLOCK | libc::O_CLOEXEC, 
                        0
                    )
                };

            if fd >= 0
            {
                let mut without_pri = [msg_header.as_slice(), msg.as_slice()].concat();
                let mut newline = String::from("\r\n");
                send_to_stderr(fd, without_pri.as_mut_slice(),&mut newline);

                unsafe {libc::close(fd)};
            }
        }
    }

    /// An internal function which is called by the syslog or vsyslog.
    /// A glibc implementation RFC5424
    #[cfg(any(
        target_os = "freebsd",
        target_os = "dragonfly",
        target_os = "openbsd",
        target_os = "netbsd",
        target_os = "macos"
    ))]
    #[async_recursion]
    async 
    fn vsyslog1(&mut self, mut pri: i32, fmt: &str)
    {
        // check for invalid bits
        match check_invalid_bits(&mut pri)
        {
            Ok(_) => {},
            Err(_e) => self.vsyslog1(get_internal_log(), fmt.as_ref()).await
        }

        // check priority against setlogmask
        if self.is_logmasked(pri) == true
        {
            return;
        }

        // set default facility if not specified in pri
        if (pri & LOG_FACMASK) == 0
        {
            pri |= self.facility.bits();
        }

        let mut hostname_buf = [0u8; MAXHOSTNAMELEN];
        let hostname = 
            match nix::unistd::gethostname(&mut hostname_buf)
            {
                Ok(r) =>
                {
                    match r.to_str()
                    {
                        Ok(r) => r,
                        Err(_e) => NILVALUE
                    }
                },
                Err(_e) => NILVALUE,
            };

        // get timedate
        let timedate = 
            Local::now().to_rfc3339_opts(SecondsFormat::Secs, false);

        // get appname
        if self.logtag.is_none() == true
        {
            match portable::p_getprogname()
            {
                Some(r) => self.set_logtag(r),
                None => self.set_logtag("")
            }
        }

        let progname = self.logtag.as_ref().unwrap();
        

        let msg_final = 
            if fmt.ends_with("\n") == true
            {
                truncate(fmt)
            }
            else
            {
                fmt
            };

        // message based on RFC 5424
        let msg_pri = 
            [
                b"<", pri.to_string().as_bytes(), b">1"
            ].concat();

        let msg_header = 
            [ 
                // timedate
                b" ", timedate.as_bytes(), 
                // hostname
                b" ", hostname.as_bytes(), 
            ].concat();
        
        let mut msg = 
            [
                // appname
                b" ", progname.as_bytes(), 
                // PID
                b" ", portable::get_pid().to_string().as_str().as_bytes(),
                // message ID
                b" ", NILVALUE.as_bytes(), 
                // structured data
                b" ", NILVALUE.as_bytes(), 
                // msg
                b" ", /*b"\xEF\xBB\xBF",*/ msg_final.as_bytes()
            ].concat();
        
        // output to stderr if required
        self.send_to_stderr(&mut msg);
        
        
        let fullmsg = 
            [
                msg_pri.as_slice(), 
                msg_header.as_slice(), 
                msg.as_slice()
            ].concat();

        if self.stream.is_none() == true
        {
            // open connection
            match self.connectlog()
            {
                Ok(_) => {},
                Err(e) =>
                {
                    self.send_to_stderr(unsafe { e.into_inner().as_bytes_mut() } );
                    return;
                }
            }
        }

        // There are two possible scenarios when send may fail:
        // 1. syslog temporary unavailable
        // 2. syslog out of buffer space
        // If we are connected to priv socket then in case of 1 we reopen connection
        //      and retry once.
        // If we are connected to unpriv then in case of 2 repeatedly retrying to send
        //      until syslog socket buffer space will be cleared

        loop
        {
            match self.stream.send(&fullmsg).await
            {
                Ok(_) => return,
                Err(err) =>
                {   
                    if let Some(libc::ENOBUFS) = err.raw_os_error()
                    {
                        // scenario 2
                        if self.stream.is_priv() == true
                        {
                            break;
                        }

                        sleep(Duration::from_micros(1)).await;
                    }
                    else
                    {
                        // scenario 1

                        self.disconnectlog();
                        match self.connectlog()
                        {
                            Ok(_) => {},
                            Err(_e) => break,
                        }

                        // if resend will fail then probably the scn 2 will take place
                    }   
                }
            }
        } // loop


        // If program reached this point then transmission over socket failed.
        // Try to output message to console

        if self.logstat.intersects(LogStat::LOG_CONS)
        {
            let fd = 
                unsafe 
                {
                    libc::open(
                        PATH_CONSOLE.as_ptr(), 
                        libc::O_WRONLY | libc::O_NONBLOCK | libc::O_CLOEXEC, 
                        0
                    )
                };

            if fd >= 0
            {
                let mut without_pri = [msg_header.as_slice(), msg.as_slice()].concat();
                let mut newline = String::from("\r\n");
                send_to_stderr(fd, without_pri.as_mut_slice(),&mut newline);

                unsafe {libc::close(fd)};
            }
        }

    }
}

/// A common instance which describes the syslog state
pub struct Syslog
{   
    /// A giant lock to synchronize the access to assets of the [SyslogInternal]
    lock: Mutex<SyslogInternal>,
}

unsafe impl Send for Syslog {}
unsafe impl Sync for Syslog {}

/// Private realization. It is assumed that functions which are called,
/// are called after 'lock' is locked.

/// A public implementations
impl Syslog
{
    /// As in a libc, this function initializes the syslog instance. The 
    /// main difference with realization in C is it returns the instance
    /// to program used this crate. This structure implements the [Send] 
    /// and [Sync] so it does not require any additional synchonization.
    ///
    /// # Arguments
    /// 
    /// * `ident` - a identification of the sender. If not set, the crate
    ///             will determine automatically!
    /// * `logstat` - sets up the syslog behaviour. Use [LogStat] 
    /// 
    /// * `facility` - a syslog facility. Use [LogFacility]
    /// 
    /// # Returns
    ///
    /// * A [SyRes] with instance or Err()
    ///
    /// # Example
    /// 
    /// ```
    ///  Syslog::openlog(
    ///        Some("test1"), 
    ///         LogStat::LOG_NDELAY | LogStat::LOG_PID, 
    ///         LogFacility::LOG_DAEMON);
    /// ```
    pub async 
    fn openlog(
        ident: Option<&str>, 
        logstat: LogStat, 
        facility: LogFacility
    ) -> SyRes<Self>
    {

        let inner = 
            SyslogInternal::new(
                ident,
                logstat,
                facility,
                0xff,
                SyslogSocket::none()
            )?;

        let ret = 
            Self
            {
                lock: Mutex::new(inner),
            };

        return Ok(ret);
    }

    /// Sets the logmask to filter out the syslog calls.
    /// 
    /// See macroses [LOG_MASK] and [LOG_UPTO] to generate mask
    ///
    /// # Example
    ///
    /// LOG_MASK!(Priority::LOG_EMERG) | LOG_MASK!(Priority::LOG_ERROR)
    ///
    /// or
    ///
    /// ~(LOG_MASK!(Priority::LOG_INFO))
    /// LOG_UPTO!(Priority::LOG_ERROR)
    pub async 
    fn setlogmask(&self, logmask: i32) -> i32
    {
        return 
            self.lock
                .lock()
                .await
                .set_logmask(logmask);
    }

    /// Similar to libc, closelog() will close the log
    pub async 
    fn closelog(&self)
    {
        self.lock.lock().await.disconnectlog();
    }

    /// Similar to libc, syslog() sends data to syslog server.
    /// 
    /// # Arguments
    ///
    /// * `pri` - a priority [Priority]
    ///
    /// * `fmt` - a string message. In C exists a functions with
    ///     variable argumets amount. In Rust you should create your
    ///     own macros like format!() or use format!()]
    pub async 
    fn syslog(&self, pri: Priority, fmt: String)
    {
        self.lock
            .lock()
            .await
            .vsyslog1(pri.bits(), fmt.as_str())
            .await;
    }

    /// Similar to syslog() and created for the compatability.
    pub async 
    fn vsyslog<S: AsRef<str>>(&self, pri: Priority, fmt: S)
    {
        self.lock
            .lock()
            .await
            .vsyslog1(pri.bits(), fmt.as_ref())
            .await;
    }

    // --- NON STANDART API

    /// This function can be used to update the facility name, for example
    /// after fork().
    /// 
    /// # Arguments
    /// 
    /// * `ident` - a new identity (up to 48 UTF8 chars)
    pub async 
    fn change_identity<I: AsRef<str>>(&self, ident: I)
    {
        self.lock.lock().await.set_logtag(ident);
    }
}

#[tokio::test(flavor = "multi_thread", worker_threads = 1)]
async fn test_multithreading()
{
    use std::sync::Arc;
    use tokio::time::{sleep, Duration};
    use std::time::{Instant};

    let log = 
            Syslog::openlog(
                Some("test1"), 
                LogStat::LOG_CONS | LogStat::LOG_NDELAY | LogStat::LOG_PID, 
                LogFacility::LOG_DAEMON).await;

    assert_eq!(log.is_ok(), true, "{}", log.err().unwrap());

    let log = Arc::new(log.unwrap());
    let c1_log = log.clone();
    let c2_log = log.clone();

    tokio::spawn( async move 
        {
            for i in 0..5
            {
                let cc_c1_log = c1_log.clone();
                sleep(Duration::from_nanos(200)).await;
                tokio::spawn( async move 
                {
                    let now = Instant::now();
                    cc_c1_log.syslog(Priority::LOG_DEBUG, format!("a message from thread 1 #{}[]", i)).await;
                    let elapsed = now.elapsed();
                    println!("t1: {:?}", elapsed);
                });
            }
        }
    );

    tokio::spawn(async move 
        {
            for i in 0..5
            {
                let cc_c2_log = c2_log.clone();
                sleep(Duration::from_nanos(201)).await;
                tokio::spawn( async move 
                {
                    let now = Instant::now();
                    cc_c2_log.syslog(Priority::LOG_DEBUG, format!("сообщение от треда 2 №{}ХЪ", i)).await;
                    let elapsed = now.elapsed();
                    println!("t2: {:?}", elapsed);
                });
            }
        });

    let now = Instant::now();
    log.syslog(Priority::LOG_DEBUG, format!("A message from main, сообщение от главнюка")).await;
    let elapsed = now.elapsed();
    println!("main: {:?}", elapsed);

    sleep(Duration::from_secs(2)).await;

    log.closelog().await;

    sleep(Duration::from_nanos(201)).await;

    return;
}
