/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */


use std::os::unix::net::UnixDatagram;
use std::path::Path;
use std::thread::sleep;
use std::time::Duration;

use chrono::offset::Local;

use nix::libc;

#[cfg(any(
    target_os = "freebsd",
    target_os = "dragonfly",
    target_os = "openbsd",
    target_os = "netbsd",
    target_os = "macos"
))]
use chrono::SecondsFormat;

use crate::portable;
use crate::common::*;
use crate::error::{SyRes/* , SyslogError*/};

use super::socket::*;

/// A common instance which describes the syslog state
pub struct SyncSyslogInternal
{   
    /// A identification i.e program name, thread name
    logtag: Option<String>, 

    /// Defines how syslog operates
    logstat: LogStat,

    /// Holds the facility 
    facility: LogFacility,

    /// A logmask
    logmask: i32,

    /// A stream
    stream: Box<dyn SyslogTap + Send>,
}

impl Drop for SyncSyslogInternal 
{
    fn drop(&mut self) 
    {
        self.disconnectlog();
    }
}

/// Private realization. It is assumed that functions which are called,
/// are called after 'lock' is locked.
impl SyncSyslogInternal
{
    /// Creates new instance of [SyslogOption].
    ///
    /// # Arguments
    ///
    /// * `ident` - An optional argument which takes ref to str. If none, the
    ///     ident will be set later. Yje ident will be trucated to 48 UTF8
    ///     chars.
    /// 
    /// * `logstat` - A [LogStat] flags separated by '|'
    /// 
    /// * `facility` - A [LogFacility] flag
    ///
    /// # Returns
    ///
    /// * The instance of the [SyslogOption]. Never fails.
    pub(crate) 
    fn new(ident: Option<&str>, logstat: LogStat, facility: LogFacility, opt_path: Option<&Path>) -> Self
    {
        // check if log_facility is invalid
        let log_facility =
            if facility.is_empty() == false && 
                (facility & !LogMask::LOG_FACMASK).is_empty() == true
            {
                facility
            }
            else
            {
                // default facility code
                LogFacility::LOG_USER
            };

        let ident = 
            match ident
            {
                Some(r) => Some(truncate_n(r, RFC_MAX_APP_NAME)),
                None => None,
            };
                
        return 
            Self
            {
                logtag: ident,
                logstat: logstat, 
                facility: log_facility,
                logmask: 0xff,
                stream: Tap::<UnixDatagram>::new(opt_path),
            };
    }

    pub(crate) 
    fn new_path<P>(ident: Option<&str>, logstat: LogStat, facility: LogFacility, sock_path: P) -> Self
    where P: AsRef<Path>
    {
        return Self::new(ident, logstat, facility, Some(sock_path.as_ref()));
    }

    fn send_to_stderr(&self, msg: &mut [u8])
    {
        if self.logstat.intersects(LogStat::LOG_PERROR) == true
        {
            let mut newline = String::from("\n");
            send_to_stderr(libc::STDERR_FILENO, msg, &mut newline);
        }
    }

    fn is_logmasked(&self, pri: i32) -> bool
    {
        return ((1 << (pri & LogMask::LOG_PRIMASK)) & self.logmask) == 0;
    }

    pub(crate) 
    fn set_logmask(&mut self, logmask: i32) -> i32
    {
        let oldmask = self.logmask;

        if logmask != 0
        {
            self.logmask = logmask;
        }

        return oldmask;
    }

    pub(crate) 
    fn set_logtag<L: AsRef<str>>(&mut self, logtag: L)
    {
        self.logtag = Some(truncate_n(logtag.as_ref(), RFC_MAX_APP_NAME));
    }

    /// Disconnects the unix stream from syslog.
    /// Should be called only when lock is acuired
    pub(crate) 
    fn disconnectlog(&mut self)
    {
        let _ = self.stream.disconnectlog();
    }

    /// Connects unix stream to the syslog and sets up the properties of
    /// the unix stream.
    /// Should be called only when lock is acuired
    pub(crate) 
    fn connectlog(&mut self) -> SyRes<()>
    {
        return self.stream.connectlog();
    }

    /// An internal function which is called by the syslog or vsyslog.
    /// A glibc implementation RFC3164
    #[cfg(target_os = "linux")]
    pub(crate) 
    fn vsyslog1<S: AsRef<str>>(&mut self, mut pri: i32, fmt: S)
    {
        // check for invalid bits
        match check_invalid_bits(&mut pri)
        {
            Ok(_) => {},
            Err(_e) => self.vsyslog1(get_internal_log(), fmt.as_ref())
        }

        // check priority against setlogmask
        if self.is_logmasked(pri) == true
        {
            return;
        }

        // set default facility if not specified in pri
        if (pri & LOG_FACMASK) == 0
        {
            pri |= self.facility.bits();
        }

        // get timedate
        let timedate = Local::now().format("%h %e %T").to_string();

        // get appname
        if self.logtag.is_none() == true
        {
            match portable::p_getprogname()
            {
                Some(r) => 
                {
                    self.logtag = 
                        Some(truncate_n(r.as_str(), RFC_MAX_APP_NAME));
                },
                None => 
                {
                    self.logtag = 
                        Some(truncate_n("unknown", RFC_MAX_APP_NAME));
                }
            }
        }

        let progname = self.logtag.as_ref().unwrap();
        

        let msg = fmt.as_ref();
        let msg_final = 
            if msg.ends_with("\n") == true
            {
                truncate(msg)
            }
            else
            {
                msg
            };

        // message based on RFC 3164
        let msg_pri = 
            [
                b"<", pri.to_string().as_bytes(), b">"
            ].concat();

        let msg_header = 
            [ 
                // timedate
                timedate.as_bytes(), 
                // hostname
               // " ".as_bytes(), hostname.as_bytes(), 
            ].concat();

        let mut msg = 
            [
                // appname
                b" ", progname.as_bytes(), 
                // PID
                b"[", portable::get_pid().to_string().as_str().as_bytes(), b"]:",
                // msg
                b" ", /*b"\xEF\xBB\xBF",*/ msg_final.as_bytes()
            ].concat();

        // output to stderr if required
        self.send_to_stderr(&mut msg);
        
        
        let fullmsg = 
            [
                msg_pri.as_slice(), 
                msg_header.as_slice(), 
                msg.as_slice()
            ].concat();

        if self.stream.is_connected() == false
        {
            // open connection
            match self.connectlog()
            {
                Ok(_) => {},
                Err(e) =>
                {
                    self.send_to_stderr(unsafe { e.into_inner().as_bytes_mut() } );
                    return;
                }
            }
        }

        // There are two possible scenarios when send may fail:
        // 1. syslog temporary unavailable
        // 2. syslog out of buffer space
        // If we are connected to priv socket then in case of 1 we reopen connection
        //      and retry once.
        // If we are connected to unpriv then in case of 2 repeatedly retrying to send
        //      until syslog socket buffer space will be cleared

        loop
        {
            match self.stream.send(&fullmsg)
            {
                Ok(_) => return,
                Err(err) =>
                {   
                    if let Some(libc::ENOBUFS) = err.raw_os_error()
                    {
                        // scenario 2
                        if self.stream.is_priv() == true
                        {
                            break;
                        }

                        sleep(Duration::from_micros(1));
                    }
                    else
                    {
                        // scenario 1

                        self.disconnectlog();
                        match self.connectlog()
                        {
                            Ok(_) => {},
                            Err(_e) => break,
                        }

                        // if resend will fail then probably the scn 2 will take place
                    }   
                }
            }
        } // loop


        // If program reached this point then transmission over socket failed.
        // Try to output message to console

        if self.logstat.intersects(LogStat::LOG_CONS)
        {
            let fd = unsafe {
                libc::open(
                    PATH_CONSOLE.as_ptr(), 
                    libc::O_WRONLY | libc::O_NONBLOCK | libc::O_CLOEXEC, 
                    0
                )
            };

            if fd >= 0
            {
                let mut without_pri = [msg_header.as_slice(), msg.as_slice()].concat();
                let mut newline = String::from("\r\n");
                send_to_stderr(fd, without_pri.as_mut_slice(),&mut newline);

                unsafe {libc::close(fd)};
            }
        }
    }

    /// An internal function which is called by the syslog or vsyslog.
    /// A glibc implementation RFC5424
    #[cfg(any(
        target_os = "freebsd",
        target_os = "dragonfly",
        target_os = "openbsd",
        target_os = "netbsd",
        target_os = "macos"
    ))]
    pub(crate) fn vsyslog1<S: AsRef<str>>(&self, mut pri: i32, fmt: S)
    {
        // check for invalid bits
        match check_invalid_bits(&mut pri)
        {
            Ok(_) => {},
            Err(_e) => self.vsyslog1(get_internal_log(), fmt.as_ref())
        }

        // check priority against setlogmask
        if self.is_logmasked(pri) == true
        {
            return;
        }

        // set default facility if not specified in pri
        if (pri & LOG_FACMASK) == 0
        {
            pri |= self.facility.bits();
        }

        let mut hostname_buf = [0u8; MAXHOSTNAMELEN];
        let hostname = 
            match nix::unistd::gethostname(&mut hostname_buf)
            {
                Ok(r) =>
                {
                    match r.to_str()
                    {
                        Ok(r) => r,
                        Err(_e) => NILVALUE
                    }
                },
                Err(_e) => NILVALUE,
            };

        // get timedate
        let timedate = 
            Local::now().to_rfc3339_opts(SecondsFormat::Secs, false);

        // get appname
        if self.logtag.is_none() == true
        {
            match portable::p_getprogname()
            {
                Some(r) => 
                {
                    self.logtag = 
                        Some(truncate_n(r.as_str(), RFC_MAX_APP_NAME));
                },
                None => 
                {
                    self.logtag = Some(NILVALUE.to_string());
                }
            }
        }

        let progname = self.logtag.as_ref().unwrap();
        

        let msg = fmt.as_ref();
        let msg_final = 
            if msg.ends_with("\n") == true
            {
                truncate(msg)
            }
            else
            {
                msg
            };

        // message based on RFC 5424
        let msg_pri = 
            [
                b"<", pri.to_string().as_bytes(), b">1"
            ].concat();

        let msg_header = 
            [ 
                // timedate
                b" ", timedate.as_bytes(), 
                // hostname
                b" ", hostname.as_bytes(), 
            ].concat();
        
        let mut msg = 
            [
                // appname
                b" ", progname.as_bytes(), 
                // PID
                b" ", portable::get_pid().to_string().as_str().as_bytes(),
                // message ID
                b" ", NILVALUE.as_bytes(), 
                // structured data
                b" ", NILVALUE.as_bytes(), 
                // msg
                b" ", /*b"\xEF\xBB\xBF",*/ msg_final.as_bytes()
            ].concat();

        // output to stderr if required
        self.send_to_stderr(&mut msg);
        
        
        let fullmsg = [msg_pri.as_slice(), msg_header.as_slice(), msg.as_slice()].concat();

        if self.stream.is_connected() == false
        {
            // open connection
            match self.connectlog()
            {
                Ok(_) => {},
                Err(e) =>
                {
                    self.send_to_stderr(unsafe { e.into_inner().as_bytes_mut() } );
                    return;
                }
            }
        }

        // There are two possible scenarios when send may fail:
        // 1. syslog temporary unavailable
        // 2. syslog out of buffer space
        // If we are connected to priv socket then in case of 1 we reopen connection
        //      and retry once.
        // If we are connected to unpriv then in case of 2 repeatedly retrying to send
        //      until syslog socket buffer space will be cleared

        loop
        {
            match self.stream.send(&fullmsg)
            {
                Ok(_) => return,
                Err(err) =>
                {   
                    if let Some(libc::ENOBUFS) = err.raw_os_error()
                    {
                        // scenario 2
                        if self.stream.is_priv() == true
                        {
                            break;
                        }

                        sleep(Duration::from_micros(1));
                    }
                    else
                    {
                        // scenario 1
                        self.disconnectlog();
                        match self.connectlog()
                        {
                            Ok(_) => {},
                            Err(_e) => break,
                        }

                        // if resend will fail then probably the scn 2 will take place
                    }   
                }
            }
        } // loop


        // If program reached this point then transmission over socket failed.
        // Try to output message to console

        if self.logstat.intersects(LogStat::LOG_CONS)
        {
            let fd = unsafe {
                libc::open(
                    PATH_CONSOLE.as_ptr(), 
                    libc::O_WRONLY | libc::O_NONBLOCK | libc::O_CLOEXEC, 
                    0
                )
            };

            if fd >= 0
            {
                let mut without_pri = [msg_header.as_slice(), msg.as_slice()].concat();
                let mut newline = String::from("\r\n");
                send_to_stderr(fd, without_pri.as_mut_slice(),&mut newline);

                unsafe {libc::close(fd)};
            }
        }

    }
}

#[test]
fn test_bit_operations()
{
    let correct = 
        SyncSyslogInternal::new(Some("test1"), LogStat::LOG_PID, LogFacility::LOG_DAEMON, None);

    assert_eq!(correct.facility, LogFacility::LOG_DAEMON);
    assert_eq!((correct.facility & !LogFacility::LOG_DAEMON), LogFacility::empty());
}

#[test]
fn test_bit_operations2()
{
    let _correct = 
        SyncSyslogInternal::new(Some("test2"), LogStat::LOG_PID, LogFacility::LOG_DAEMON, None);

    let mut pri = Priority::LOG_ALERT.bits();

    let res = check_invalid_bits(&mut pri);

    assert_eq!(res.is_ok(), true);
    assert_eq!(pri, Priority::LOG_ALERT.bits());
}

#[test]
fn test_set_priority()
{
    use crate::LOG_MASK;

    let mut correct = 
        SyncSyslogInternal::new(Some("test1"), LogStat::LOG_PID, LogFacility::LOG_DAEMON, None);

    let ret = correct.set_logmask(LOG_MASK!(Priority::LOG_ERR));

    assert_eq!(ret, 0xff);

    let ret = correct.set_logmask(LOG_MASK!(Priority::LOG_ERR));

    assert_eq!(ret, LOG_MASK!(Priority::LOG_ERR));

    let ret = correct.is_logmasked(Priority::LOG_WARNING.bits());
    assert_eq!(ret, true);

    let ret = correct.is_logmasked(Priority::LOG_ERR.bits());
    assert_eq!(ret, false);

    let ret = correct.is_logmasked(Priority::LOG_CRIT.bits());
    assert_eq!(ret, true);
}

#[test]
fn test_set_priority2()
{
    use crate::LOG_MASK;
    
    let mut correct = 
        SyncSyslogInternal::new(Some("test1"), LogStat::LOG_PID, LogFacility::LOG_DAEMON, None);

    let ret = correct.set_logmask(!LOG_MASK!(Priority::LOG_ERR));

    assert_eq!(ret, 0xff);

    let ret = correct.is_logmasked(Priority::LOG_WARNING.bits());
    assert_eq!(ret, false);

    let ret = correct.is_logmasked(Priority::LOG_ERR.bits());
    assert_eq!(ret, true);

    let ret = correct.is_logmasked(Priority::LOG_CRIT.bits());
    assert_eq!(ret, false);
}
