/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */



use std::io::ErrorKind;
use std::net::SocketAddr;
use std::os::unix::net::UnixDatagram;
use std::mem::MaybeUninit;
use std::path::{PathBuf, Path};
use std::{
    net::Shutdown,
    os::unix::{
        prelude::AsRawFd, 
        io::RawFd
    }
};

use nix::libc;
use nix::errno::Errno;

use crate::{map_error, throw_error};

use crate::common::*;
use crate::error::{SyRes, SyslogError, SyslogErrCode};

/// A trait for implemetation of a common functionality of sockets
pub(crate) trait SyslogTap
{
    /// Returns a socket's FD
    fn get_raw_fd(&self) -> RawFd;

    /// Opens a connection to target.
    fn connectlog(&mut self) -> SyRes<()>;

    /// Writes provided data to socket
    fn send(&mut self, msg: &[u8]) -> std::io::Result<usize>;

    /// Disconnects the connection.
    fn disconnectlog(&mut self) -> std::io::Result<()>;

    /// Tells if connection was opened.
    fn is_connected(&self) -> bool;

    /// Tells if connected to priv syslog socket (unixdatagram only)
    fn is_priv(&self) -> bool;
}

#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub(crate) enum TapType
{
    /// Not initialized
    None,

    /// Unprivileged socket used
    UnPriv,

    /// Privileged socket used
    Priv,

    /// A compatibility socket used
    OldLog,

    /// Custom
    CustomUnkn,

    /// Network (remote) todo
    Net,
}

pub(crate) struct TapNet
{
    /// Nameserver address and port
    remote_addr: SocketAddr, 
    /// Local addriss to use to bind socket
    bind_addr: SocketAddr,
}

pub(crate) struct TapLocal
{
    /// Nameserver address and port
    sock_path: PathBuf,
}

impl From<&Path> for TapLocal
{
    fn from(p: &Path) -> Self 
    {
        return Self{ sock_path: p.to_path_buf() };
    }
}

pub(crate) struct Tap<T>
{
    /// Channel
    sock: Option<T>,

    /// Tap settings and state
    tap_type: TapType,

    /// A settings for local tap
    tap_local: Option<TapLocal>,

    /// A settings for remote tap
    tap_remote: Option<TapNet>,
}

impl Tap<UnixDatagram>
{
    pub(crate) 
    fn new(opt_path: Option<&Path>) -> Box<dyn SyslogTap + Send>
    {
        let ret = 
            Self
            {
                sock: None, 
                tap_type: TapType::None,
                tap_local: opt_path.and_then(|p| Some(TapLocal::from(p.as_ref()))),
                tap_remote: None
            };

        return Box::new(ret);
    }
}

impl SyslogTap for Tap<UnixDatagram>
{
    fn get_raw_fd(&self) -> RawFd 
    {
        return self.sock.as_ref().unwrap().as_raw_fd();
    }

    fn connectlog(&mut self) -> SyRes<()> 
    {
        let sock = 
            UnixDatagram::unbound().map_err(|e|
                map_error!("unbounded unix datagram initialization failure: {}", e)
            )?;

        let tap_type = 
            if self.tap_local.is_some() == true && 
                sock.connect(self.tap_local.as_ref().unwrap().sock_path.as_path()).is_ok() == true
            {
                TapType::CustomUnkn
            }
            else if let Ok(_) = sock.connect(PATH_LOG_PRIV)
            {
                TapType::Priv
            }
            else if let Ok(_) = sock.connect(PATH_LOG)
            {
                TapType::UnPriv
            }
            else if let Ok(_) = sock.connect(PATH_OLDLOG)
            {
                TapType::OldLog
            }
            else if let Ok(_) = sock.connect(PATH_OSX)
            {
                TapType::Priv
            }
            else
            {
                // failed to open socket
                throw_error!("failed to open connection to syslog server: {} ", 
                            Errno::last());
            };

        // setting sendbuf
        let mut len: MaybeUninit<libc::socklen_t> = std::mem::MaybeUninit::uninit();

        // set the sndbuf len
        let res = 
            unsafe
            {
                libc::getsockopt(
                    sock.as_raw_fd(), 
                    libc::SOL_SOCKET, 
                    libc::SO_SNDBUF, 
                    len.as_mut_ptr() as *mut libc::c_void, 
                    &mut { 
                        std::mem::size_of::<libc::socklen_t>() as libc::socklen_t
                    } 
                )
            };
 
        if res == 0
        {
            let mut len = unsafe { len.assume_init() };

            if len < MAXLINE
            {
                len = MAXLINE;

                unsafe {
                    libc::setsockopt(
                        sock.as_raw_fd(), 
                        libc::SOL_SOCKET, 
                        libc::SO_SNDBUF, 
                        &len as *const _ as *const libc::c_void, 
                        std::mem::size_of::<libc::socklen_t>() as libc::socklen_t
                    )
                };
            }
        }

        self.sock = Some(sock);
        self.tap_type = tap_type;

        return Ok(());
    }

    fn send(&mut self, msg: &[u8]) -> std::io::Result<usize> 
    {
        let sock = 
            self.sock
                .as_mut()
                .ok_or_else(||
                    std::io::Error::new(ErrorKind::NotConnected, "no connection")
                )?;

        return sock.send(msg);
    }

    fn disconnectlog(&mut self) -> std::io::Result<()>
    {
        match self.sock.take()
        {
            Some(s) => 
            {
                self.tap_type = TapType::None;

                s.shutdown(Shutdown::Both)
            },
            None =>
            {
                self.tap_type = TapType::None;

                Ok(())
            }
        }
    }

    fn is_connected(&self) -> bool
    {
        return self.sock.is_some();
    }

    fn is_priv(&self) -> bool 
    {
        return self.tap_type == TapType::Priv;
    }
}

