/*-
 * syslog-rs - a syslog client translated from libc to rust
 * Copyright (C) 2020  Aleksandr Morozov, RELKOM s.r.o
 * Copyright (C) 2021-2022  Aleksandr Morozov
 * 
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 *  file, You can obtain one at https://mozilla.org/MPL/2.0/.
 */


use std::ffi::CStr;

use nix::libc;

#[cfg(target_os = "linux")]
use std::path::Path;

#[cfg(any(
    target_os = "freebsd",
    target_os = "dragonfly",
    target_os = "openbsd",
    target_os = "netbsd",
    target_os = "macos"
))]
#[link(name = "c")]
extern "C" {
    fn getprogname() -> *const libc::c_char;
}

#[cfg(target_os = "linux")]
#[link(name = "c")]
extern "C" {
    pub static mut program_invocation_name : *mut libc::c_char ;
}

/// Reutns the current process name, if available
pub(crate) 
fn p_getprogname() -> Option<String>
{
    let pn =
        {
            #[cfg(any(
                target_os = "freebsd",
                target_os = "dragonfly",
                target_os = "openbsd",
                target_os = "netbsd",
                target_os = "macos"
            ))]
            unsafe { getprogname() }

            #[cfg(target_os = "linux")]
            unsafe{ program_invocation_name }
        };

    let temp = unsafe {CStr::from_ptr(pn)};

    match temp.to_str()
    {
        Ok(r) => 
        {
            #[cfg(target_os = "linux")]
            {
                let path = Path::new(r);
                match path.file_name()
                {
                    Some(r) => return Some(r.to_string_lossy().into()),
                    None => return None,
                }
            }

            #[cfg(any(
                target_os = "freebsd",
                target_os = "dragonfly",
                target_os = "openbsd",
                target_os = "netbsd",
                target_os = "macos"
            ))]
            return Some(r.to_string());
        },
        Err(_) => return None,
    }
   
}

/// Returns pid of current process. Not thread id!
pub 
fn get_pid() -> u32
{
    return std::process::id();
}

#[test]
fn test_get_procname()
{
    println!("Processname is: {:?}", p_getprogname());
}
