/*-
* syslog-rs - a syslog client translated from libc to rust
* Copyright (C) 2021  Aleksandr Morozov
* Copyright (C) 2021  RELKOM s.r.o
* 
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public
* License as published by the Free Software Foundation; either
* version 3 of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
* Lesser General Public License for more details.
* 
* You should have received a copy of the GNU Lesser General Public License
* along with this program; if not, write to the Free Software Foundation,
* Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

use std::fmt;

/// Error code
#[derive(Clone, Copy, Debug, PartialEq)]
pub enum SyslogErrCode
{
    /// A message which can be output to stderr
    InternalError,

    /// Only for syslog_sync_queue.rs and returned only when syslog thread is stopped
    /// and someone is trying to do some operation
    #[cfg(feature = "use_sync_queue")]
    SyslogThreadNotAvailable,

    /// Only for syslog_sync_queue.rs and returned only when syslog thread is failed
    /// to send back data
    #[cfg(feature = "use_sync_queue")]
    UnboundedChannelError,
}

/// A syslog crate error
pub struct SyslogError 
{
    errcode: SyslogErrCode,
    message: String,
}

impl SyslogError
{
    /// Creates new error
    pub(crate)
    fn new(errcode: SyslogErrCode, msg: String) -> Self
    {
        return SyslogError{errcode: errcode, message: msg};
    }

    /// Retuns a clone of [SyslogErrCode]
    pub 
    fn get_errcode(&self) -> SyslogErrCode
    {
        return self.errcode;
    }

    /// Moves out the error description
    pub 
    fn into_inner(self) -> String
    {
        return self.message;
    }
}

impl fmt::Display for SyslogError 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "{}", self.message)
    }
}
impl fmt::Debug for SyslogError 
{
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result 
    {
        write!(f, "{}", self.message)
    }
}

pub type SyRes<T> = Result<T, SyslogError>;

#[macro_export]
macro_rules! throw_error 
{
    ($($arg:tt)*) => (
        return std::result::Result::Err(SyslogError::new(SyslogErrCode::InternalError, format!($($arg)*)))
    )
}

#[macro_export]
macro_rules! map_error 
{
    ($($arg:tt)*) => (
        return SyslogError::new(SyslogErrCode::InternalError, format!($($arg)*))
    )
}

#[macro_export]
macro_rules! throw_error_code
{
    ($code:expr, $($arg:tt)*) => (
        return std::result::Result::Err(SyslogError::new($code, format!($($arg)*)))
    )
}

#[macro_export]
macro_rules! map_error_code
{
    ($code:expr, $($arg:tt)*) => (
        return SyslogError::new($code, format!($($arg)*))
    )
}
